package org.lixm.core.list;

import java.util.Vector;

import org.lixm.core.common.XMLType;
import org.lixm.core.model.AbstractModel;
import org.lixm.core.model.AttributeModel;

/**
 * <p>
 * 属性モデル一覧を表します。
 * </p>
 * 
 * @author tasogare
 * 
 */
public class AttributesList extends AbstractModel {

    private Vector attrs;

    /**
     * <p>
     * 既存の属性モデル配列から初期化します。
     * </p>
     * 
     * @param array
     *                任意の<code>AttributeModel[]</code>
     */
    public AttributesList(AttributeModel[] array) {
	super(XMLType.ATTRIBUTES_LIST);

	attrs = new Vector(array.length);
	for (int i = 0; i < array.length; i++) {
	    attrs.addElement(array[i]);
	}
    }

    /**
     * 属性モデル数を指定して初期化します。
     * 
     * @param length
     *                モデルの最大数
     */
    public AttributesList(int length) {
	super(XMLType.ATTRIBUTES_LIST);
	attrs = new Vector(length);
    }

    /**
     * {@inheritDoc}
     */
    public boolean equals(Object o) {

	if (!(o instanceof AttributesList))
	    return false;

	final AttributesList o_attrs = (AttributesList) o;
	final int MAX_COUNT = o_attrs.getLength();

	if (!(getLength() == MAX_COUNT))
	    return false;

	AttributeModel attr = null;
	for (int i = 0; i < MAX_COUNT; i++) {
	    attr = this.getIndex(i);
	    if (!o_attrs.getIndex(i).equals(attr)) {
		return false;
	    }
	}
	return true;
    }

    /**
     * リストの指定した位置のモデルを取得
     * 
     * @deprecated {@link #get(int)}を利用します。
     * @param index
     *                取得するモデルのリスト内での絶対位置
     * @return 指定した位置に存在するモデル
     * 
     */
    public AttributeModel getIndex(int index) {
	return ((AttributeModel) attrs.elementAt(index));
    }

    /**
     * リストの指定した位置のモデルを取得
     * 
     * @since 1.1.0
     * @param index
     *                取得するモデルのリスト内での絶対位置
     * @return 指定した位置に存在するモデル
     * 
     */
    public AttributeModel get(int index) {
	return ((AttributeModel) attrs.elementAt(index));
    }

    /**
     * リスト内の総モデル数を返します。
     * 
     * @deprecated {@link #size()}を利用します。
     * @return 実際に存在するモデル数より１大きい数値が返ります。
     */
    public int getLength() {
	return attrs.size();
    }

    /**
     * <p>
     * この属性リストにある全ての属性のローカル名を配列で取得します。
     * </p>
     * 
     * @return ローカル名文字列配列
     */
    public String[] getLocalNameList() {

	String[] list = new String[getLength()];
	for (int i = 0; i < getLength(); i++) {
	    list[i] = ((AttributeModel) attrs.elementAt(i)).toLocalName();
	}
	return list;
    }

    /**
     * <p>
     * このリストにある全ての属性のローカル名と属性値がペアになった 文字列表現を配列で取得します。
     * </p>
     * 
     * <p>
     * このメソッドで得られる文字列表現は実装依存です。<br>
     * その使用はデバッグやエンドユーザー向けメッセージのような用途にのみに留めてください。
     * </p>
     * 
     * @return ペア文字列の配列
     * @see org.lixm.core.model.AttributeModel#getLocalPair()
     */
    public String[] getLocalPairList() {
	String[] list = new String[getLength()];
	for (int i = 0; i < getLength(); i++) {
	    list[i] = ((AttributeModel) attrs.elementAt(i)).getLocalPair();
	}
	return list;
    }

    /**
     * <p>
     * ローカル名を指定して属性を取得します。
     * </p>
     * 
     * @param name
     *                属性のローカル名
     * @return 見つかった属性
     */
    public AttributeModel getName(String name) {

	AttributeModel hitAttr = null;
	for (int i = 0; i < getLength(); i++) {

	    AttributeModel model = ((AttributeModel) attrs.elementAt(i));
	    String attrName = model.getName();
	    if (attrName.equals(name)) {
		hitAttr = ((AttributeModel) attrs.elementAt(i));
	    }
	}
	return hitAttr;
    }

    /**
     * プレフィックスとローカル名を指定して属性を取得します。
     * @param prefix プレフィックス
     * @param name ローカル名
     * @return 見つかった属性
     */
    public AttributeModel getName(
	    String prefix, String name)
    {

	AttributeModel hitAttr = null;
	for (int i = 0; i < getLength( ); i++) {

	    AttributeModel model = ((AttributeModel)attrs.elementAt(i));
	    String attrPrefix = model.getPrefix();
	    String attrName = model.getName( );
	    if (attrName.equals(name) && attrPrefix.equals(prefix)) {
		hitAttr = ((AttributeModel)attrs.elementAt(i));
	    }
	}
	return hitAttr;
    }

    /**
     * <p>
     * この属性リストにある全ての属性の修飾名を配列で取得します。
     * </p>
     * 
     * @return 修飾名文字列配列
     */
    public String[] getQNameList() {

	String[] list = new String[getLength()];
	for (int i = 0; i < getLength(); i++) {
	    list[i] = ((AttributeModel) attrs.elementAt(i)).toQName();

	    if (list[i] == null) {
		list[i] = ((AttributeModel) attrs.elementAt(i)).toLocalName();
	    }
	}
	return list;
    }

    /**
     * <p>
     * このリストにある全ての属性のローカル名と属性値がペアになった 文字列表現を配列で取得します。
     * </p>
     * 
     * <p>
     * このメソッドで得られる文字列表現は実装依存です。<br>
     * その使用はデバッグやエンドユーザー向けメッセージのような用途にのみに留めてください。
     * </p>
     * 
     * @return ペア文字列の配列
     * @see org.lixm.core.model.AttributeModel#getLocalPair()
     */
    public String[] getQPairList() {
	String[] list = new String[getLength()];
	for (int i = 0; i < getLength(); i++) {
	    list[i] = ((AttributeModel) attrs.elementAt(i)).getQPair();
	}
	return list;
    }

    /**
     * <p>
     * このリストにある全ての属性の属性値を配列で取得します。
     * </p>
     * 
     * @return 属性値文字列配列
     */
    public String[] getValueList() {
	String[] list = new String[getLength()];
	for (int i = 0; i < getLength(); i++) {
	    list[i] = ((AttributeModel) attrs.elementAt(i)).getValue();
	}
	return list;
    }

    /**
     * <p>
     * 指定した属性のリスト内の位置を返します。
     * </p>
     * 
     * @param attr
     *                任意の{@link org.lixm.core.model.AttributeModel AttributeModel}
     * @return このリスト内で存在する位置。見つからなかった場合は<code>-1</code>
     */
    public int indexOf(AttributeModel attr) {
	return attrs.indexOf(attr);
    }

    /**
     * @deprecated {@link #set(AttributeModel, int)}を利用します。
     * @param attr
     * @param index
     * @throws ArrayIndexOutOfBoundsException
     */
    public void insertAttribute(AttributeModel attr, int index)
	    throws ArrayIndexOutOfBoundsException {
	attrs.insertElementAt(attr, index);
    }

    /**
     * <p>リストの指定した位置のモデルを設定</p>
     * @since 1.1.0
     * @param attr
     *                属性モデル
     * @param index
     *                インデックス
     * @throws ArrayIndexOutOfBoundsException
     */
    public void set(AttributeModel attr, int index)
	    throws ArrayIndexOutOfBoundsException {
	attrs.insertElementAt(attr, index);
    }

    /**
     * <p>
     * 指定した属性のリスト内の最後から数えた位置を返します。
     * </p>
     * 
     * @param attr
     *                任意の{@link org.lixm.core.model.AttributeModel AttributeModel}
     * @return このリスト内で存在する位置。見つからなかった場合は<code>-1</code>
     */
    public int lastIndexOf(AttributeModel attr) {
	return attrs.lastIndexOf(attr);
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
	String str = "";
	for (int i = 0; i < getLength(); i++) {
	    str += " " + ((AttributeModel) attrs.elementAt(i)).getLocalPair();
	}
	return str;
    }

    /**
     * <p>
     * 指定された属性モデルがあればそれを属性リストから削除します。
     * </p>
     * 
     * @throws ClassCastException
     *                 <code>model</code>が<code>AttributeModel</code>型ではない場合
     * @throws NullPointerException
     *                 モデルが<code>null</code>の場合
     */
    public boolean remove(Object model) throws ClassCastException,
	    NullPointerException {
	if (model == null) {
	    throw new NullPointerException();
	}

	return attrs.removeElement((AttributeModel) model);
    }

    /**
     * <p>指定した属性モデルを属性リストに追加します。</p>
     * 
     * @return モデルが追加された場合に<code>true</code>を返します。
     * @throws NullPointerException
     *                 <code>model</code>が<code>null</code>の場合
     */
    public boolean add(AttributeModel model) throws NullPointerException {
	if (model == null) {
	    throw new NullPointerException();
	}
	attrs.addElement(model);
	return true;
    }

    /**
     * リスト内の総モデル数を返します。
     * 
     * @since 1.1.0
     * @return 実際に存在するモデル数より１大きい数値が返ります。
     */
    public int size() {
	return attrs.size();
    }

    /**
     * リストにモデルが格納されているか？
     * 
     * @since 1.1.0
     * @return 空の場合は真、そうでない場合は偽を返します。
     */
    public boolean isEmpty() {
	return attrs.isEmpty();
    }

    /**
     * <p>この属性リストからすべての要素を削除します。
     * このメソッドが戻ると、属性リストは空になります。</p>
     * 
     * @since 1.1.0
     */
    public void clear() {
	attrs = new Vector();
    }

}
