package org.lixm.core.list;

import java.util.Vector;

import org.lixm.core.common.XMLType;
import org.lixm.core.model.AbstractModel;


/**
 * <h4>格納されるべきモデル</h4>
 *  
 *  <p>このリストには整形式または妥当なXMLのみが格納されます。<br>
 *  つまり実装ではモデライズの時点で整形式または妥当なXMLである事を確認し、
 *  生成されるリストではそのどちらかである事を保障しなければいけません。</p>
 *
 *
 *  <h4>参照</h4>
 *
 *  <p>LIXMではXMLにおける参照はモデライズの際に解決しておかなければいけません。<br>
 *  <strong>リストの中に実体参照が出てきてはいけない</strong>と言う事です。
 *  しかし、外部実体の扱いはこの仕様では未定義です。</p>
 *  
 *  <p>その理由は以下の通りです。</p>
 *
 *  <h5>本当に必要なものを直感的に扱える事</h5>
 *  
 *  <p>実体参照はXML内に直接含める事の出来ない特殊文字を現すための仕組みであり、
 *  XMLにアクセスし実体参照を取得するという事はつまりは、
 *  <em>その実体参照が現している文字そのものが必要である</em>という事です。</p>
 *
 *
 *  <p>このため、XMLを扱う過程で最終的にXMLの必要な情報にアクセスする段階では
 *  特殊文字とその他の文字を区別する必要はありません。</p>
 *
 *  <p>さらに、単なる文字一つを取得するために実体参照から文字への変換を取得するコードを
 *  逐一挿入していては煩雑で見通しの悪いソースコードが出来上がってしまうでしょう。</p>
 *
 *  <p>このため、LIXMでは実体参照は扱いません。</p>
 *  
 *  <p>このような弱点はDOMにもあります。
 *  DOMにはプラットフォーム中立性ために<em>ノード</em>と呼ばれる独自の概念が存在します。<br>
 *  このノードは<em>LIXMではモデル</em>に該当しますが、DOMはこの任意のノードの取得方法が
 *  とても直感的ではなく、かつ煩雑になりがちです。</p>
 * 
 *  <p>LIXMはDOMとは違ったアプローチによりXML内の情報へ
 *  直感的に低レベルで一般的なアクセスを可能とする事が目的です。</p>
 *  
 *  <h4>文書型やカタログはモデルとしてリストに含まない</h4>
 *  
 *  <p>LIXMでは文書型やカタログは特別にモデル化しないという事です。
 *  結果的に文書型モデルやカタログモデルはリストには存在しないという事です。</p>
 *  
 *  <p>これらのモデル化が必要なのはバリデータや高レベルなものではスキーマコンパイラ等になるでしょう。
 *  </p>
 * 
 * @author tasogare
 * @version 1.0M1
 */
public class XMLModelList {

    private Vector    vect;

    private XMLCursor cur;

    /**
     * 初期化します。
     */
    public XMLModelList() {
        super( );
        cur = new XMLCursor(this);
        vect = new Vector( );
    }

    /**
     * モデル数を指定して初期化します。
     * @param initialCapacity モデル数
     */
    public XMLModelList(
        int initialCapacity)
    {
        cur = new XMLCursor(this);
        vect = new Vector(initialCapacity);
    }

    /**
     * {@link org.lixm.core.model.AbstractModel AbstractModel}
     * 配列を利用して初期化します。
     * @param array モデルの配列
     */
    public XMLModelList(
        AbstractModel[] array)
    {
        cur = new XMLCursor(this);
        vect = new Vector(array.length);
        vect.copyInto(array);
    }

    /**
     * <p>リストの指定した位置のモデルを取得</p>
     * @param index 取得するモデルのリスト内での絶対位置
     * @return 指定した位置に存在するモデル
     */
    public AbstractModel get(
        int index)
    {
        return (AbstractModel) vect.elementAt(index);
    }

    /**
     * <p>ルート要素の開始タグを取得</p>
     * 
     * @return
     * <p>ルート要素のモデルを見つけた場合の
     * 戻り値は{@link org.lixm.core.model.StartTagModel StartTagModel}です。
     * それ以外は<code>null</code>です。<br>メソッドの戻り値の型は
     * {@link org.lixm.core.model.AbstractModel AbstractModel}
     * なので適切にキャストしてください。</p>
     */
    public AbstractModel getRootStartTag() {

        AbstractModel model = null;
        for (int idx = cur.getFirst( ); idx < size( ); idx++) {

            model = (AbstractModel) get(idx);
            if (model.matchTypes(XMLType.START_TAG)) {
                return model;
            }
        }
        return null;
    }

    /**
     * <p>ルート要素の終了タグを取得</p>
     * 
     * @return
     * <p>ルート要素のモデルを見つけた場合の
     * 戻り値は{@link org.lixm.core.model.StartTagModel EndTagModel}です。
     * それ以外は<code>null</code>です。<br>
     * メソッドの戻り値の型は{@link org.lixm.core.model.AbstractModel AbstractModel}
     * なので適切にキャストしてください。</p>
     */
    public AbstractModel getRootEndTag() {

        AbstractModel model = null;
        for (int idx = cur.getLast( ); idx > -1; idx--) {

            model = (AbstractModel) get(idx);
            if (model.matchTypes(XMLType.END_TAG)) {
                return model;
            }
        }
        return null;
    }

    /*
     */
    // function getCurrent();

    /**
     * <p>リストの指定位置にモデルを追加</p>
     * 
     * @param model 追加するモデル
     */
    public void add(
        AbstractModel model)
    {
        vect.addElement(model);
    }

    /*
     */
    // function remove(index);
    /*
     */
    // function clear();

    /**
     * <p>リストにモデルが格納されているか？</p>
     * 
     * @return 空の場合は真、そうでない場合は偽を返します。
     */
    public boolean isEmpty() {
        return vect.isEmpty( );
    }

    /**
     * <p>リスト内の総モデル数を返します。</p>
     * 
     * @return 実際に存在するモデル数より１大きい数値が返ります。
     */
    public int size() {
        return vect.size( );
    }

    /**
     * <p>このリストに関連付けられているカーソルを返します。</p>
     * 
     * @return 関連付けられているカーソル
     */
    public XMLCursor getCursor() {
        return cur;
    }
}
