package LISM::Handler::Rewrite;

use strict;
use base qw(LISM::Handler);
use Net::LDAP;
use Net::LDAP::Constant qw(:all);
use URI;
use Sys::Syslog;
use Sys::Syslog qw(:macros);
use Data::Dumper;

=head1 NAME

LISM::Handler::Rewrite - Handler to do script

=head1 DESCRIPTION

This class implements the L<LISM::Hanlder> interface to do script.

=head1 METHODS

=pod

=head2 pre_bind($binddnp)

Rewrite bind dn before bind operation is done.

=cut

sub pre_bind
{
    my $self = shift;
    my ($binddnp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my $substitution = $rule->{substitution};
            $substitution =~ s/%0/${$binddnp}/;

            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$binddnp});
            if (!$str) {
                $self->log(level => 'error', message => "bind rewrite \"${$binddnp}\" failed");
                return -1;
            }
            ${$binddnp} = $str;
        }
    }

    return 0;
}

=head2 pre_compare($dnp, $avaStrp)

Rewrite dn and attribute, value before compare operation is done.

=cut

sub pre_compare
{
    my $self = shift;
    my ($dnp, $avaStrp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my %rwcache;
            my $substitution = $rule->{substitution};
            $substitution =~ s/%0/${$dnp}\n${$avaStrp}/;

            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$dnp}, \%rwcache);
            if (!$str) {
                $self->log(level => 'error', message => "compare rewrite \"${$dnp}\" failed");
                return -1;
            }
            ${$dnp} = $str;

            $str = $self->_rewriteParse($rule->{match}, $substitution, ${$avaStrp}, \%rwcache);
            if (!$str) {
                $self->log(level => 'error', message => "compcare rewrite \"${$dnp}\" failed");
                return -1;
             }
             ${$avaStrp} = $str;
        }
    }

    return 0;
}

=head2 pre_search($basep, $filterStrp)

Rewrite base dn and filter before search operation is done.

=cut

sub pre_search
{
    my $self = shift;
    my ($basep, $filterStrp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my %rwcache;
            my $substitution = $rule->{substitution};

            $substitution =~ s/%0/${$basep}\n${$filterStrp}/;
            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$basep}, \%rwcache);
            if (!defined($str)) {
                $self->log(level => 'error', message => "search rewrite \"${$basep}\" failed");
                return -1;
            }
            ${$basep} = $str;

            $str = $self->_rewriteParse($rule->{match}, $substitution, ${$filterStrp}, \%rwcache);
            if (!defined($str)) {
                $self->log(level => 'error', message => "search rewrite \"${$basep}\" failed");
                return -1;
            }
            ${$filterStrp} = $str;
        }
    }

    return 0;
}

=head2 post_search($entriesp)

Rewrite base dn and filter after search operation is done.

=cut

sub post_search
{
    my $self = shift;
    my ($entriesp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'searchResult') {
            for (my $i = 0; $i < @{$entriesp}; $i++) {
                my %rwcache;
                my $rc = 0;
                my $substitution = $rule->{substitution};
                $substitution =~ s/%0/${$entriesp}[$i]/;

                my (@line) = split(/\n/, ${$entriesp}[$i]);
                my ($dn) = ($line[0] =~ /^dn: (.*)$/);
                for (my $j = 0; $j < @line; $j++) {
                    $line[$j] = $self->_rewriteParse($rule->{match}, $substitution, $line[$j], \%rwcache);
                    if (!defined($line[$j])) {
                        $self->log(level => 'error', message => "search result rewrite \"$dn\" failed");
                        $rc = -1;
                        last;
                    }
                }
                if (!$rc) {
                    ${$entriesp}[$i] = join("\n", @line)."\n";
                }
            }
        }
    }

    return 0;
}

=pod

=head2 pre_modify($dnp, $listp)

Rewrite dn and attributes, values before modify operation is done.

=cut

sub pre_modify
{
    my $self = shift;
    my ($dnp, $listp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my %rwcache;
            my $substitution = $rule->{substitution};
            my $modlist = "${$dnp}\n".join("\n", @{$listp});
            $substitution =~ s/%0/$modlist/;

            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$dnp}, \%rwcache);
            if (!$str) {
                $self->log(level => 'error', message => "modify rewrite \"${$dnp}\" failed");
                return -1;
            }
            ${$dnp} = $str;

            for (my $i = 0; $i < @{$listp};) {
                # skip action
                $i++;
                my $attr = ${$listp}[$i++];

                while ($i < @{$listp} && !(${$listp}[$i] =~ /ADD|DELETE|REPLACE/)) {
                    $str = $self->_rewriteParse($rule->{match}, $substitution, "$attr: ${$listp}[$i]", \%rwcache);
                    if (!$str) {
                        $self->log(level => 'error', message => "modify rewrite \"${$dnp}\" failed");
                        return -1;
                    }
                    ($attr, ${$listp}[$i]) = split(/: */, $str);
                    $i++;
                }
            }
        }
    }

    return 0;
}

=pod

=head2 pre_add($dnp, $entryStrp)

Rewrite entry before add operation is done.

=cut

sub pre_add
{
    my $self = shift;
    my ($dnp, $entryStrp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my %rwcache;
            my $substitution = $rule->{substitution};
            $substitution =~ s/%0/${$dnp}\n${$entryStrp}/;

            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$dnp}, \%rwcache);
            if (!$str) {
                $self->log(level => 'error', message => "add rewrite \"${$dnp}\" failed");
                return -1;
            }
            ${$dnp} = $str;

            my (@line) = split(/\n/, ${$entryStrp}[0]);

            for (my $i = 0; $i < @line; $i++) {
                $line[$i] = $self->_rewriteParse($rule->{match}, $substitution, $line[$i], \%rwcache);
                if (!$line[$i]) {
                    $self->log(level => 'error', message => "add rewrite \"${$dnp}\" failed");
                    return -1;
                }
            }
            ${$entryStrp}[0] = join("\n", @line)."\n";
         }
    }

    return 0;
}

=head2 pre_modrdn($dnp, $argsp)

Rewrite dn and new rdn before modrdn operation is done.

=cut

sub pre_modrdn
{
    my $self = shift;
    my ($dnp, $argsp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my %rwcache;
            my $substitution = $rule->{substitution};
            $substitution =~ s/%0/${$dnp}\n${$argsp}[0]/;

            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$dnp}, \%rwcache);
            if (!$str) {
                $self->log(level => 'error', message => "modrdn rewrite \"${$dnp}\" failed");
                return -1;
            }
            ${$dnp} = $str;

            $str = $self->_rewriteParse($rule->{match}, $substitution, ${$argsp}[0], \%rwcache);
            if (!$str) {
                $self->log(level => 'error', message => "modrdn rewrite \"${$dnp}\" failed");
                return -1;
            }
            ${$argsp}[0] = $str;
        }
    }

    return 0;
}

=pod

=head2 pre_delete($dnp)

Rewrite dn before delete operation is done.
    
=cut
    
sub pre_delete
{
    my $self = shift;
    my ($dnp) = @_;
    my $conf = $self->{_config};

    foreach my $rule (@{$conf->{rewrite}}) {
        if ($rule->{context} eq 'request') {
            my $substitution = $rule->{substitution};
            $substitution =~ s/%0/${$dnp}/;

            my $str = $self->_rewriteParse($rule->{match}, $substitution, ${$dnp});
            if (!$str) {
                $self->log(level => 'error', message => "delete rewrite \"${$dnp}\" failed");
                return -1;
            }
        }
    }

    return 0;
}


sub _checkConfig
{
    my $self = shift;
    my $conf = $self->{_config};
    my $rc = 0;

    if ($rc = $self->SUPER::_checkConfig()) {
        return $rc;
    }

    if (defined($conf->{libload})) {
        foreach my $lib (@{$conf->{libload}}) {
            eval "require \'$lib\'";
            if ($@) {
                $self->log(level => 'alert', message => "rewrite handler require $lib: $@");
                return 1;
            }
        }
    }

    if (defined($conf->{rewritemap})) {
        foreach my $map_name (keys %{$conf->{rewritemap}}) {
            if ($conf->{rewritemap}{$map_name}->{type} eq 'ldap') {
                if (!defined($self->{ldapmap})) {$self->{ldapmap} = {}};
                my $ldapmap = {};
                $ldapmap->{uri} = $conf->{rewritemap}{$map_name}->{attrs};
                my $uri = URI->new($ldapmap->{uri});
                $ldapmap->{base} = $uri->dn;
                ($ldapmap->{attr}) = $uri->attributes;
                my %extn = $uri->extensions;
                $ldapmap->{binddn} = $extn{binddn};
                $ldapmap->{bindpw} = $extn{bindpw};
                $ldapmap->{ldap} = undef;
                $self->{ldapmap}{$map_name} = $ldapmap;
            }
        }
    }

    return $rc;
}

sub _rewriteParse
{
    my $self = shift;
    my ($match, $substitution, $str, $rwcache) = @_;

    my @matches = ($str =~ /$match/gi);

    # replace variables
    for (my $i = 0; $i < @matches; $i++) {
        my $num = $i + 1;
        $substitution =~ s/%$num/$matches[$i]/g;
    }

    if ($str =~ /$match/i) {
        # do functions
        my @rwmaps = ($substitution =~ /%{([^}]*)}/g);
        foreach my $rwmap (@rwmaps) {
            my $value;
            my $key = lc($rwmap);
            if (defined(${$rwcache}{$key})) {
                $value = ${$rwcache}{$key};
            } else {
                my ($map_name, $map_args) = ($rwmap =~ /^([^(]*)\((.*)\)$/);
                $value = $self->_rewriteMap($map_name, $map_args);
                if (!defined($value)) {
                    return undef;
                }

                ${$rwcache}{$key} = $value;
            }

            if (!$value) {
                return $str;
            }

            $rwmap =~ s/([()])/\\$1/g;
            $substitution =~ s/%{$rwmap}/$value/;
        }

        $str =~ s/$match/$substitution/gi;
    }

    return $str;
}

sub _rewriteMap
{
    my $self = shift;
    my ($map_name, $map_args) = @_;
    my $conf = $self->{_config};
    my $value = '';

    if (defined($conf->{rewritemap}{$map_name})) {
        my $method = '_'.$conf->{rewritemap}{$map_name}->{type}.'Map';
        $value = $self->$method($map_name, $map_args);
    }

    return $value;
}

sub _ldapMap
{
    my $self = shift;
    my ($map_name, $map_args) = @_;
    my $ldapmap = $self->{ldapmap}{$map_name};
    my $msg;
    my $rc;
    my $value = '';

    if (!$ldapmap->{ldap}) {
        $ldapmap->{ldap} = Net::LDAP->new($ldapmap->{uri});
        $msg = $ldapmap->{ldap}->bind($ldapmap->{binddn}, password => $ldapmap->{bindpw});
        $rc = $msg->code;
        if ($rc) {
            $self->log(level => 'level', message => "rewriteMap $map_name bind by $ldapmap->{binddn} failed($rc)");
            return undef;
        }
    }

    $msg = $ldapmap->{ldap}->search(base => $ldapmap->{base}, filter => $map_args, attrs => [$ldapmap->{attr}]);

    $rc = $msg->code;
    if ($rc) {
        $self->log(level => 'level', message => "rewriteMap $map_name search by $map_args failed($rc)");
        if ($rc == LDAP_SERVER_DOWN) {
            $ldapmap->{ldap}->unbind();
            undef($ldapmap->{ldap});
        }

        return undef;
    }

    if ($msg->count) {
        my $entry = $msg->entry(0);
        $value = $entry->get_value($ldapmap->{attr});
        if (!defined($value)) {
            $value = '';
        }
    }

    return $value;    
}

sub _functionMap
{
    my $self = shift;
    my ($map_name, $map_args) = @_;
    my $value;

    eval "\$value = $map_name($map_args)";
    if ($@) {
        $self->log(level => 'error', message => "rewriteMap $map_name failed: $@");
        return undef;
    }

    return $value;
}

=head1 SEE ALSO

L<LISM>,
L<LISM::Handler>

=head1 AUTHOR

Kaoru Sekiguchi, <sekiguchi.kaoru@secioss.co.jp>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by Kaoru Sekiguchi

This library is free software; you can redistribute it and/or modify
it under the GNU LGPL.

=cut

1;
