﻿/*
 * FormSeriesImage.cs
 * Copyright (c) 2007-2009 kbinani
 *
 * This file is part of LipSync.
 *
 * LipSync is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * LipSync is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.IO;
using System.Drawing;
using System.Windows.Forms;
using System.Reflection;

using Boare.Lib.AppUtil;

namespace LipSync {

    public partial class FormSeriesImage : Form, IMultiLanguageControl {
        float m_start = 0f;
        float m_end = 0f;

        public FormSeriesImage() {
            InitializeComponent();
            ApplyLanguage();
            ApplyFont( AppManager.Config.Font.GetFont() );
            Type type = typeof( System.Drawing.Imaging.ImageFormat );
            PropertyInfo[] properties = type.GetProperties();
            int i = -1;
            foreach ( PropertyInfo pi in properties ) {
                if ( pi.PropertyType.Equals( typeof( System.Drawing.Imaging.ImageFormat ) ) ) {
                    System.Drawing.Imaging.ImageFormat ifm = (System.Drawing.Imaging.ImageFormat)pi.GetValue( null, null );
                    string ext = Misc.GetExtensionFromImageFormat( ifm );
                    if ( ext.Length > 0 ) {
                        i++;
                        comboFormat.Items.Add( pi.Name );
                        if ( i == 0 ) {
                            comboFormat.SelectedIndex = 0;
                        }
                        if ( ext == "bmp" ) {
                            comboFormat.SelectedIndex = i;
                        }
                    }
                }
            }
        }

        public void ApplyFont( Font font ) {
            this.Font = font;
            foreach ( Control c in this.Controls ) {
                Boare.Lib.AppUtil.Misc.ApplyFontRecurse( c, font );
            }
        }

        public string ParserString {
            get {
                if ( chkAutomaticallyAddExtension.Checked ) {
                    return txtParser.Text + "." + Misc.GetExtensionFromImageFormat( ImageFormat );
                } else {
                    return txtParser.Text;
                }
            }
        }

        public System.Drawing.Imaging.ImageFormat ImageFormat {
            get {
                if ( comboFormat.SelectedItem == null ){
                    return System.Drawing.Imaging.ImageFormat.Bmp;
                }
                string title = (string)comboFormat.SelectedItem;
#if DEBUG
                Console.WriteLine( "ImageFormat.get()" );
                Console.WriteLine( "    System.Drawing.Imaging.ImageFormat.Bmp.ToString()=" + System.Drawing.Imaging.ImageFormat.Bmp.ToString() );
#endif
                PropertyInfo[] properties = typeof( System.Drawing.Imaging.ImageFormat ).GetProperties();
                foreach ( PropertyInfo pi in properties ) {
                    if ( pi.PropertyType.Equals( typeof( System.Drawing.Imaging.ImageFormat ) ) ) {
                        if ( pi.Name == title ) {
                            return (System.Drawing.Imaging.ImageFormat)pi.GetValue( null, null );
                        }
                    }
                }
                return System.Drawing.Imaging.ImageFormat.Bmp;
            }
        }

        public string DirectoryName {
            get {
                return txtDirectory.Text;
            }
        }

        public void ApplyLanguage() {
            this.Text = _( "Series Image" );
            btnCancel.Text = _( "Cancel" );
            btnOK.Text = _( "Save" );
            lblFileName.Text = _( "Directory" );
            groupStartEnd.Text = _( "Specify output range" );
            chkStart.Text = _( "Start" );
            chkEnd.Text = _( "End" );
            groupFileName.Text = _( "File Name" );
            lblParser.Text = _( "Parser String" );
            chkAutomaticallyAddExtension.Text = _( "Automatically Add Extension" );
            lblFormat.Text = _( "Image Format" );
        }

        public static string _( string s ) {
            return Messaging.GetMessage( s );
        }

        public float Start {
            get {
                return m_start;
            }
        }

        public float End {
            get {
                return m_end;
            }
        }

        private void btnOK_Click( object sender, EventArgs e ) {
            try {
                m_start = float.Parse( txtStart.Text );
                m_end = float.Parse( txtEnd.Text );
                this.DialogResult = DialogResult.OK;
            } catch ( Exception ex ) {
                MessageBox.Show(
                    _( "Invalid value has been entered" ),
                    _( "Error" ),
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Exclamation
                );
                Common.LogPush( ex );
            }
        }

        private void btnFile_Click( object sender, EventArgs e ) {
            using ( FolderBrowserDialog fbd = new FolderBrowserDialog() ) {
                if ( fbd.ShowDialog() == DialogResult.OK ) {
                    txtDirectory.Text = fbd.SelectedPath;
                }
            }
        }

        private void chkStart_CheckedChanged( object sender, EventArgs e ) {
            txtStart.Enabled = chkStart.Checked;
            if ( txtStart.Enabled ) {
                txtStart.Focus();
            }
        }

        private void checkBox1_CheckedChanged( object sender, EventArgs e ) {
            txtEnd.Enabled = chkEnd.Checked;
            if ( txtEnd.Enabled ) {
                txtEnd.Focus();
            }
        }

        public bool StartSpecified {
            get {
                return chkStart.Checked;
            }
        }

        public bool EndSpecified {
            get {
                return chkEnd.Checked;
            }
        }

        private void txtParser_TextChanged( object sender, EventArgs e ) {
            UpdateSampleFileNames();
        }

        private void comboFormat_SelectedIndexChanged( object sender, EventArgs e ) {
            UpdateSampleFileNames();
        }

        private void UpdateSampleFileNames() {
            string ret = _( "Sample File Name" ) + Environment.NewLine;
            System.Drawing.Imaging.ImageFormat ifm = ImageFormat;
            string ext = Misc.GetExtensionFromImageFormat( ifm );
            string parser = ParserString;
            try {
                ret += string.Format( parser, 0 ) + Environment.NewLine;
                ret += string.Format( parser, 1 ) + Environment.NewLine;
                ret += " ... " + Environment.NewLine;
                ret += string.Format( parser, 1234 );
            } catch {
                ret += _( "Error: Invalid parser string." );
            }
            txtPreview.Text = ret;
        }

        private void chkAutomaticallyAddExtension_CheckedChanged( object sender, EventArgs e ) {
            UpdateSampleFileNames();
        }
    }

}
