﻿/*
 * FormCommandHistory.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of LipSync.
 *
 * LipSync is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * LipSync is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Drawing;
using System.Windows.Forms;

using Boare.Lib.AppUtil;

namespace LipSync {

    public partial class FormCommandHistory : Form, IMultiLanguageControl {
        private const int _LABEL_HEIGT = 15;
        private ToolStripMenuItem m_redo;
        private ToolStripMenuItem m_undo;
        private MenuStrip m_menu;

        public FormCommandHistory() {
            InitializeComponent();
            
            m_redo = new ToolStripMenuItem();
            m_undo = new ToolStripMenuItem();

            m_menu = new MenuStrip();
            m_menu.Visible = false;
            m_menu.Items.AddRange( new ToolStripItem[] { m_redo, m_undo } );

            m_redo.Visible = false;
            m_redo.ShortcutKeys = Keys.Control | Keys.Shift | Keys.Z;
            m_redo.Click += new EventHandler( m_redo_Click );

            m_undo.Visible = false;
            m_undo.ShortcutKeys = Keys.Control | Keys.Z;
            m_undo.Click += new EventHandler( m_undo_Click );

            this.Controls.Add( m_menu );

            ApplyLanguage();
            ApplyFont( AppManager.Config.Font.GetFont() );
            pictCommands.Height = _LABEL_HEIGT;
            SettingsEx.CommandExecuted += new CommandExecutedEventHandler( SettingsEx_CommandExecuted );
            pictCommands.MouseWheel += new MouseEventHandler( pictCommands_MouseWheel );
        }

        public void ApplyFont( Font font ) {
            this.Font = font;
            foreach ( Control c in this.Controls ) {
                Boare.Lib.AppUtil.Misc.ApplyFontRecurse( c, font );
            }
        }

        private void m_undo_Click( object sender, EventArgs e ) {
            if ( AppManager.IsUndoAvailable ) {
                AppManager.Undo();
            }
        }

        private void m_redo_Click( object sender, EventArgs e ) {
            if ( AppManager.IsRedoAvailable ) {
                AppManager.Redo();
            }
        }

        public void ApplyLanguage() {
            this.Text = _( "Command History" );
            this.Invalidate();
        }

        private static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        private void SettingsEx_CommandExecuted( TimeTableType command_target, CommandType command_type ) {
            pictCommands.Height = (AppManager.m_commands.Count + 2) * _LABEL_HEIGT;
            //lblLookMe.Location = new Point( 0, (AppManager.m_command_position + 2) * _LABEL_HEIGT );
            //panel.ScrollControlIntoView( lblLookMe );
            pictCommands.Invalidate();
        }

        private void pictCommands_Paint( object sender, PaintEventArgs e ) {
            e.Graphics.FillRectangle( Brushes.Pink, new Rectangle( 0, 0, pictCommands.Width, _LABEL_HEIGT ) );
            e.Graphics.DrawString( _( "Command History" ), SystemFonts.MenuFont, Brushes.Black, new PointF( 0, 0 ) );

            e.Graphics.FillRectangle( Brushes.White, new Rectangle( 0, _LABEL_HEIGT, pictCommands.Width, _LABEL_HEIGT ) );
            Font font = SystemFonts.MenuFont;
            if ( AppManager.m_command_position < 0 ) {
                font = new Font( font.FontFamily, font.Size, FontStyle.Bold );
            }
            e.Graphics.DrawString( "Root", font, Brushes.Black, new PointF( 0, _LABEL_HEIGT ) );

            for ( int i = 0; i < AppManager.m_commands.Count; i++ ) {
                Brush brs;
                if ( i % 2 == 0 ) {
                    brs = Brushes.LightGray;
                } else {
                    brs = Brushes.White;
                }
                font = SystemFonts.MenuFont;
                if ( i == AppManager.m_command_position ) {
                    font = new Font( font.FontFamily, font.Size, FontStyle.Bold );
                }
                e.Graphics.FillRectangle( brs, new Rectangle( 0, (i + 2) * _LABEL_HEIGT, pictCommands.Width, _LABEL_HEIGT ) );
                e.Graphics.DrawString( StringFromCommand( AppManager.m_commands[i] ),
                                       font,
                                       Brushes.Black,
                                       new PointF( 0, (i + 2) * _LABEL_HEIGT ) );
            }
        }

        private void pictCommands_MouseWheel( object sender, MouseEventArgs e ) {

        }

        private void pictCommands_MouseDown( object sender, MouseEventArgs e ) {
            pictCommands.Focus();
        }

        private string StringFromCommand( Command command ) {
            if ( command.target == TimeTableType.another ) {
                if ( command.type == CommandType.addEntry ) {
                    return string.Format( _( "Delete entry of 'Another Image' at Index {0}" ), command.track );
                } else if ( command.type == CommandType.deleteEntry ) {
                    return string.Format( _( "Add entry of 'Another Image' at Index {0}" ), command.track );
                } else if ( command.type == CommandType.editEntry ) {
                    return string.Format( _( "Edit entry of 'Another Image' at Track {0}, Index {1}" ), command.track, command.entry );
                }
            } else if ( command.target == TimeTableType.telop ) {
                if ( command.type == CommandType.addTelop ) {
                    return string.Format( _( "Delete telop '{0}'" ), command.telop.Text );
                } else if ( command.type == CommandType.editTelop ) {
                    return string.Format( _( "Edit telop '{0}' at Index {1}" ), command.telop.Text, command.entry );
                } else if ( command.type == CommandType.deleteTelop ) {
                    return string.Format( _( "Add telop '{0}'" ), command.telop.Text );
                }
            }
            return command.target + ", " + command.type;
        }

        private void FormCommandHistory_Load( object sender, EventArgs e ) {
            SettingsEx_CommandExecuted( TimeTableType.none, CommandType.nothing );
        }
    }

}
