﻿/*
 * EditEntry.cs
 * Copyright (c) 2007-2009 kbinani
 *
 * This file is part of LipSync.
 *
 * LipSync is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * LipSync is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Drawing;
using System.Windows.Forms;

using Boare.Lib.AppUtil;

namespace LipSync {

    public partial class EditEntry : Form, IMultiLanguageControl {
        private float m_start;
        private float m_end;
        private float m_min_start;
        private float m_max_end;
        /// <summary>
        /// On timeがテキストボックスによって編集されたかどうか
        /// </summary>
        private bool m_start_edited = false;
        /// <summary>
        /// Off timeがテキストボックスによって編集されたかどうか
        /// </summary>
        private bool m_end_edited = false;

        public EditEntry( float start, float end, float min_start, float max_end ) {
            InitializeComponent();
            ApplyLanguage();
            ApplyFont( AppManager.Config.Font.GetFont() );
            txtStart.Text = start.ToString();
            txtEnd.Text = end.ToString();
            txtMinStart.Text = min_start.ToString();
            txtMaxEnd.Text = max_end.ToString();
            m_min_start = min_start;
            m_max_end = max_end;
            m_start_edited = false;
            m_end_edited = false;
        }

        public void ApplyFont( Font font ) {
            this.Font = font;
            foreach ( Control c in this.Controls ) {
                Boare.Lib.AppUtil.Misc.ApplyFontRecurse( c, font );
            }
        }

        public void ApplyLanguage() {
            this.lblOnTime.Text = _( "ON time (sec)" );
            this.lblOffTime.Text = _( "OFF time (sec)" );
            this.btnCancel.Text = _( "Cancel" );
            this.btnOK.Text = _( "OK" );
            this.btnUseThisValue.Text = _( "Use this value" );
            this.groupBox1.Text = _( "Expandable range of this entry" );
            this.Text = _( "Numeric entry" );
        }

        public static string _( string s ) {
            return Messaging.GetMessage( s );
        }

        public float Start {
            get {
                return m_start;
            }
        }

        public float End {
            get {
                return m_end;
            }
        }

        private void btnOK_Click( object sender, EventArgs e ) {
            /*if ( !m_start_edited ) {
                m_start = m_start;
            }
            if ( !m_end_edited ) {
                m_end = m_end;
            }*/
            if ( m_start >= m_end || m_start < m_min_start || m_max_end < m_end ) {
                MessageBox.Show( _( "Invalid value has been entered" ), _( "Error" ), MessageBoxButtons.OK, MessageBoxIcon.Exclamation );
                this.DialogResult = DialogResult.Cancel;
            } else {
                this.DialogResult = (!m_end_edited && !m_start_edited) ? DialogResult.Cancel : DialogResult.OK;
            }
            this.Close();
        }

        private void btnUseThisValue_Click( object sender, EventArgs e ) {
            if ( m_start != m_min_start || m_end != m_max_end ) {
                txtStart.Text = m_min_start.ToString();
                txtEnd.Text = m_max_end.ToString();
                m_start = m_min_start;
                m_end = m_max_end;
                m_end_edited = true;
                m_start_edited = true;
            }
        }

        private void txtStart_TextChanged( object sender, EventArgs e ) {
            float old_begin = m_start;
            m_start_edited = true;
            try {
                m_start = float.Parse( txtStart.Text );
            } catch ( Exception ex ) {
                m_start = old_begin;
                txtStart.Text = m_start.ToString();
                txtStart.SelectAll();
                Common.LogPush( ex );
            }
        }

        private void txtEnd_TextChanged( object sender, EventArgs e ) {
            float old_end = m_end;
            m_end_edited = true;
            try {
                m_end = float.Parse( txtEnd.Text );
            } catch ( Exception ex ) {
                m_end = old_end;
                txtEnd.Text = m_end.ToString();
                txtEnd.SelectAll();
            }
        }

        private void txtStart_KeyPress( object sender, KeyPressEventArgs e ) {
            if ( (e.KeyChar < '0' || e.KeyChar > '9') && e.KeyChar != '\b' && e.KeyChar != '.' ) {
                e.Handled = true;
            }
        }

        private void txtEnd_KeyPress( object sender, KeyPressEventArgs e ) {
            if ( (e.KeyChar < '0' || e.KeyChar > '9') && e.KeyChar != '\b' && e.KeyChar != '.' ) {
                e.Handled = true;
            }
        }

        private void txtStart_PreviewKeyDown( object sender, PreviewKeyDownEventArgs e ) {
            if ( (e.Modifiers & Keys.Control) == Keys.Control ) {
                if ( (e.KeyCode & Keys.X) == Keys.X ) {
                    Clipboard.SetText( txtStart.Text );
                    txtStart.Text = "";
                } else if ( (e.KeyCode & Keys.C) == Keys.C ) {
                    Clipboard.SetText( txtStart.Text );
                } else if ( (e.KeyCode & Keys.V) == Keys.V ) {
                    if ( Clipboard.ContainsText() ) {
                        txtStart.Text = Clipboard.GetText();
                    }
                }
            }
        }

        private void txtEnd_PreviewKeyDown( object sender, PreviewKeyDownEventArgs e ) {
            if ( (e.Modifiers & Keys.Control) == Keys.Control ) {
                if ( (e.KeyCode & Keys.X) == Keys.X ) {
                    Clipboard.SetText( txtEnd.Text );
                    txtEnd.Text = "";
                } else if ( (e.KeyCode & Keys.C) == Keys.C ) {
                    Clipboard.SetText( txtEnd.Text );
                } else if ( (e.KeyCode & Keys.V) == Keys.V ) {
                    if ( Clipboard.ContainsText() ) {
                        txtEnd.Text = Clipboard.GetText();
                    }
                }
            }
        }
    }

}
