/*
 * AviOutput.cs
 * Copyright (c) 2007-2009 kbinani
 *
 * This file is part of LipSync.
 *
 * LipSync is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * LipSync is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.IO;
using System.Windows.Forms;
using System.Drawing;

using Boare.Lib.AppUtil;

namespace LipSync {

    public partial class AviOutput : Form, IMultiLanguageControl {
        private bool m_raw_mode = false;
        private float m_start = 0f;
        private float m_end = 0f;

        public AviOutput( bool raw_mode ) {
            InitializeComponent();
            ApplyLanguage();
            ApplyFont( AppManager.Config.Font.GetFont() );
            m_raw_mode = raw_mode;
            if ( groupAudio.Enabled ) {
                if ( File.Exists( AppManager.SaveData.m_audioFile ) ) {
                    this.chkMergeAudio.Enabled = true;
                } else {
                    this.chkMergeAudio.Enabled = false;
                }
            }
            if ( m_raw_mode ) {
                btnVideoCompression.Enabled = false;
                txtDescription.Enabled = false;
            } else {
                btnVideoCompression.Enabled = true;
                txtDescription.Enabled = true;
            }
            txtFile.Text = AppManager.Config.LastAviPath;
            JudgeWritable();
        }

        public void ApplyFont( Font font ) {
            this.Font = font;
            foreach ( Control c in this.Controls ) {
                Boare.Lib.AppUtil.Misc.ApplyFontRecurse( c, font );
            }
        }

        public void ApplyLanguage() {
            btnCancel.Text = _( "Cancel" );
            btnOK.Text = _( "Save" );
            lblFileName.Text = _( "File Name" );
            if ( AppManager.Config.PathFFmpeg != "" && File.Exists( AppManager.Config.PathFFmpeg ) ) {
                groupAudio.Text = _( "Audio" );
                groupAudio.Enabled = true;
            } else {
                groupAudio.Text = _( "Audio" ) + " (" + _( "Set the path of ffmpeg to enable this option" ) + ")";
                groupAudio.Enabled = false;
            }
            if ( AppManager.Config.PathMEncoder != "" && File.Exists( AppManager.Config.PathMEncoder ) ) {
                groupFlv.Text = _( "FLV and MP4" );
                groupFlv.Enabled = true;
            } else {
                groupFlv.Text = _( "FLV and MP4" ) + " (" + _( "Set the path of mencoder and ffmpeg to enable this option" ) + ")";
                groupFlv.Enabled = false;
            }
            chkFLV.Text = _( "Convert to FLV" );
            chkMP4.Text = _( "Convert to MP4" );
            chkMergeAudio.Text = _( "Merge WAVE to AVI" );
            chkDeleteIntermediate.Text = _( "Delete Intermediate File" );
            btnVideoCompression.Text = _( "Video Compression" );
            groupStartEnd.Text = _( "Specify Output Range" );
            chkStart.Text = _( "Start" );
            chkEnd.Text = _( "End" );
            checkContainsAlpha.Text = _( "Add Alpha" );
        }

        public static string _( string s ) {
            return Messaging.GetMessage( s );
        }

        public AviOutputArguments Arguments {
            get {
                AviOutputArguments aoa = new AviOutputArguments();
                aoa.AviFile = this.FileName;
                aoa.End = m_end;
                aoa.EndSpecified = chkEnd.Checked;
                aoa.FileNameParser = "";
                aoa.ImageFormat = null;
                aoa.IsDeleteIntermediateRequired = chkDeleteIntermediate.Checked;
                aoa.IsFlvEncodeRequired = chkFLV.Checked;
                aoa.IsMp4EncodeRequired = chkMP4.Checked;
                if ( aoa.IsMp4EncodeRequired && aoa.IsFlvEncodeRequired ) {
                    aoa.IsFlvEncodeRequired = false;
                }
                aoa.IsTransparent = checkContainsAlpha.Checked;
                aoa.IsWaveMergeRequired = chkMergeAudio.Checked;
                aoa.Start = m_start;
                aoa.StartSpecified = chkStart.Checked;
                aoa.UseVfwEncoder = radioVfw.Checked;
                return aoa;
            }
        }

        private void btnOK_Click( object sender, EventArgs e ) {
            //ディレクトリが存在するかどうかを確認
            string name = FileName;
            if ( !Directory.Exists( Path.GetDirectoryName( name ) ) ) {
                MessageBox.Show( string.Format( _( "Directory {0} does not exist." ), Path.GetDirectoryName( name ) ),
                                 _( "Error" ),
                                 MessageBoxButtons.OK,
                                 MessageBoxIcon.Exclamation );
                return;
            }

            //既にファイルが存在することを警告
            if ( File.Exists( name ) ) {
                if ( MessageBox.Show( string.Format( _( "{0} already exists.\nDo you want to replace it?" ), name ),
                                      "LipSync",
                                      MessageBoxButtons.YesNo,
                                      MessageBoxIcon.Exclamation ) == DialogResult.No ) {
                    return;
                }
            }
            try {
                m_start = float.Parse( txtStart.Text );
                m_end = float.Parse( txtEnd.Text );
                this.DialogResult = DialogResult.OK;
            } catch ( Exception ex ) {
                MessageBox.Show( _( "Invalid value has been entered" ),
                                 _( "Error" ),
                                 MessageBoxButtons.OK,
                                 MessageBoxIcon.Exclamation );
                Common.LogPush( ex );
            }
        }

        private string FileName {
            get {
                if ( Path.GetExtension( txtFile.Text ).ToLower() != ".avi" ) {
                    string name = txtFile.Text;
                    txtFile.Text = Path.Combine( Path.GetDirectoryName( name ), Path.GetFileNameWithoutExtension( name ) + ".avi" );
                }
                return txtFile.Text;
            }
        }

        private void btnFile_Click( object sender, EventArgs e ) {
            using ( SaveFileDialog dlg = new SaveFileDialog() ) {
                if ( AppManager.Config.LastAviPath != "" ) {
                    try {
                        dlg.InitialDirectory = Path.GetDirectoryName( AppManager.Config.LastAviPath );
                    } catch {
                    }
                }
                try {
                    dlg.Filter = _( "Avi file(*.avi)|*.avi" ) + "|" + _( "All Files(*.*)|*.*" );
                } catch {
                    dlg.Filter = "Avi file(*.avi)|*.avi|All Files(*.*)|*.*";
                }
                dlg.OverwritePrompt = false;
                if ( dlg.ShowDialog() == DialogResult.OK ) {
                    AppManager.Config.LastAviPath = dlg.FileName;
                    txtFile.Text = dlg.FileName;
                    JudgeWritable();
                }
            }
        }

        private void JudgeWritable() {
            if ( txtFile.Text != "" ) {
                btnOK.Enabled = true;
            } else {
                btnOK.Enabled = false;
            }
        }

        private void chkStart_CheckedChanged( object sender, EventArgs e ) {
            txtStart.Enabled = chkStart.Checked;
            if ( txtStart.Enabled ) {
                txtStart.Focus();
            }
        }

        private void checkBox1_CheckedChanged( object sender, EventArgs e ) {
            txtEnd.Enabled = chkEnd.Checked;
            if ( txtEnd.Enabled ) {
                txtEnd.Focus();
            }
        }

        private void chkFLV_CheckedChanged( object sender, EventArgs e ) {
            if ( chkFLV.Checked && chkMP4.Checked ) {
                chkMP4.Checked = false;
            }
            this.chkDeleteIntermediate.Enabled = chkFLV.Checked | chkMP4.Checked | chkMergeAudio.Checked;
        }

        private void chkMP4_CheckedChanged( object sender, EventArgs e ) {
            if ( chkMP4.Checked && chkFLV.Checked ) {
                chkFLV.Checked = false;
            }
            this.chkDeleteIntermediate.Enabled = chkFLV.Checked | chkMP4.Checked | chkMergeAudio.Checked;
        }

        private void chkMergeAudio_CheckedChanged( object sender, EventArgs e ) {
            this.chkDeleteIntermediate.Enabled = chkFLV.Checked | chkMP4.Checked | chkMergeAudio.Checked;
        }
    }

}
