﻿/*
 * SingerConfig.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.IO;
using System.Collections.Generic;

namespace Boare.Lib.Vsq {

    public class SingerConfig {
        private string m_id = "VOCALOID:VIRTUAL:VOICE";
        private string m_format = "2.0.0.0";
        private string m_voiceidstr = "";
        private string m_voicename = "Miku";
        private int m_breathiness = 0;
        private int m_brightness = 0;
        private int m_clearness = 0;
        private int m_opening = 0;
        private int m_gender_factor = 0;
        private int m_original = 0;

        public static void decode_vvd_bytes( ref byte[] dat ) {
            for ( int i = 0; i < dat.Length; i++ ) {
                byte M = (byte)(dat[i] >> 4);
                byte L = (byte)(dat[i] - (M << 4));
                byte newM = endecode_vvd_m( M );
                byte newL = endecode_vvd_l( L );
                dat[i] = (byte)((newM << 4) | newL);
            }
        }

        private static byte endecode_vvd_l( byte value ) {
            switch ( value ) {
                case 0x0:
                    return 0xa;
                case 0x1:
                    return 0xb;
                case 0x2:
                    return 0x8;
                case 0x3:
                    return 0x9;
                case 0x4:
                    return 0xe;
                case 0x5:
                    return 0xf;
                case 0x6:
                    return 0xc;
                case 0x7:
                    return 0xd;
                case 0x8:
                    return 0x2;
                case 0x9:
                    return 0x3;
                case 0xa:
                    return 0x0;
                case 0xb:
                    return 0x1;
                case 0xc:
                    return 0x6;
                case 0xd:
                    return 0x7;
                case 0xe:
                    return 0x4;
                case 0xf:
                    return 0x5;
            }
            return 0x0;
        }

        private static byte endecode_vvd_m( byte value ) {
            switch ( value ) {
                case 0x0:
                    return 0x1;
                case 0x1:
                    return 0x0;
                case 0x2:
                    return 0x3;
                case 0x3:
                    return 0x2;
                case 0x4:
                    return 0x5;
                case 0x5:
                    return 0x4;
                case 0x6:
                    return 0x7;
                case 0x7:
                    return 0x6;
                case 0x8:
                    return 0x9;
                case 0x9:
                    return 0x8;
                case 0xa:
                    return 0xb;
                case 0xb:
                    return 0xa;
                case 0xc:
                    return 0xd;
                case 0xd:
                    return 0xc;
                case 0xe:
                    return 0xf;
                case 0xf:
                    return 0xe;
            }
            return 0x0;
        }

        public SingerConfig( string file, int original ) {
            m_original = original;
            using ( FileStream fs = new FileStream( file, FileMode.Open, FileAccess.Read ) ) {
                int length = (int)fs.Length;
                byte[] dat = new byte[length];
                fs.Read( dat, 0, length );
                decode_vvd_bytes( ref dat );
                for ( int i = 0; i < dat.Length - 1; i++ ) {
                    if ( dat[i] == 0x17 && dat[i + 1] == 0x10 ) {
                        dat[i] = 0x0d;
                        dat[i + 1] = 0x0a;
                    }
                }
                string str = bocoree.cp932.convert( dat );
                string crlf = ((char)0x0d).ToString() + ((char)0x0a).ToString();
                string[] spl = str.Split( new string[] { crlf }, StringSplitOptions.RemoveEmptyEntries );

                foreach ( string s in spl ) {
                    int first = s.IndexOf( '"' );
                    int first_end = get_quated_string( s, first );
                    int second = s.IndexOf( '"', first_end + 1 );
                    int second_end = get_quated_string( s, second );
                    char[] chs = s.ToCharArray();
                    string id = new string( chs, first, first_end - first + 1 );
                    string value = new string( chs, second, second_end - second + 1 );
                    id = id.Substring( 1, id.Length - 2 );
                    value = value.Substring( 1, value.Length - 2 );
                    value = value.Replace( "\\\"", "\"" );
                    switch ( id ) {
                        case "ID":
                            m_id = value;
                            break;
                        case "FORMAT":
                            m_format = value;
                            break;
                        case "VOICEIDSTR":
                            m_voiceidstr = value;
                            break;
                        case "VOICENAME":
                            m_voicename = value;
                            break;
                        case "Breathiness":
                            try {
                                m_breathiness = int.Parse( value );
                            } catch {
                            }
                            break;
                        case "Brightness":
                            try {
                                m_brightness = int.Parse( value );
                            } catch {
                            }
                            break;
                        case "Clearness":
                            try {
                                m_clearness = int.Parse( value );
                            } catch {
                            }
                            break;
                        case "Opening":
                            try {
                                m_opening = int.Parse( value );
                            } catch {
                            }
                            break;
                        case "Gender:Factor":
                            try {
                                m_gender_factor = int.Parse( value );
                            } catch {
                            }
                            break;
                    }
                }
            }
        }

        /// <summary>
        /// 位置positionにある'"'から，次に現れる'"'の位置を調べる．エスケープされた\"はスキップされる．'"'が見つからなかった場合-1を返す
        /// </summary>
        /// <param name="s"></param>
        /// <param name="position"></param>
        /// <returns></returns>
        private static int get_quated_string( string s, int position ) {
            if ( position < 0 ) {
                return -1;
            }
            char[] chs = s.ToCharArray();
            if ( position >= chs.Length ) {
                return -1;
            }
            if ( chs[position] != '"' ) {
                return -1;
            }
            int end = -1;
            for ( int i = position + 1; i < chs.Length; i++ ) {
                if ( chs[i] == '"' && chs[i - 1] != '\\' ) {
                    end = i;
                    break;
                }
            }
            return end;
        }

        public new string[] ToString() {
            List<string> ret = new List<string>();
            ret.Add( "\"ID\":=:\"" + m_id + "\"" );
            ret.Add( "\"FORMAT\":=:\"" + m_format + "\"" );
            ret.Add( "\"VOICEIDSTR\":=:\"" + m_voiceidstr + "\"" );
            ret.Add( "\"VOICENAME\":=:\"" + m_voicename.Replace( "\"", "\\\"" ) + "\"" );
            ret.Add( "\"Breathiness\":=:\"" + m_breathiness + "\"" );
            ret.Add( "\"Brightness\":=:\"" + m_brightness + "\"" );
            ret.Add( "\"Clearness\":=:\"" + m_clearness + "\"" );
            ret.Add( "\"Opening\":=:\"" + m_opening + "\"" );
            ret.Add( "\"Gender:Factor\":=:\"" + m_gender_factor + "\"" );
            return ret.ToArray();
        }

        public int Original {
            get {
                return m_original;
            }
            set {
                m_original = value;
            }
        }

        public string ID {
            get {
                return m_id;
            }
            set {
                m_id = value;
            }
        }

        public string FORMAT {
            get {
                return m_format;
            }
            set {
                m_format = value;
            }
        }

        public string VOICEIDSTR {
            get {
                return m_voiceidstr;
            }
            set {
                m_voiceidstr = value;
            }
        }

        public string VOICENAME {
            get {
                return m_voicename;
            }
            set {
                m_voicename = value;
            }
        }

        public int Breathiness {
            get {
                return m_breathiness;
            }
            set {
                m_breathiness = value;
            }
        }

        public int Brightness {
            get {
                return m_brightness;
            }
            set {
                m_brightness = value;
            }
        }

        public int Clearness {
            get {
                return m_clearness;
            }
            set {
                m_clearness = value;
            }
        }

        public int Opening {
            get {
                return m_opening;
            }
            set {
                m_opening = value;
            }
        }

        public int GenderFactor {
            get {
                return m_gender_factor;
            }
            set {
                m_gender_factor = value;
            }
        }
    }

}
