/************************************************************
* Copyright (C) 2006-2007 Masahiko SAWAI All Rights Reserved. 
************************************************************/
#ifndef WIIREMOTE_H
#define WIIREMOTE_H

/** \mainpage
* This is a libwiiremote API document.
* <p>
* libwiiremote is a multi-platform C library for communicating
* with the Nintendo Wii Remote controler. 
* </p>
* @See wiiremote.h
**/

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h> /* size_t */

#define WRMT_MAX_DEVICES 32

#define WRMT_MASK_BUTTON_TWO    0x0001
#define WRMT_MASK_BUTTON_ONE    0x0002
#define WRMT_MASK_BUTTON_B      0x0004
#define WRMT_MASK_BUTTON_A      0x0008
#define WRMT_MASK_BUTTON_MINUS  0x0010
#define WRMT_MASK_BUTTON_HOME   0x0080
#define WRMT_MASK_BUTTON_LEFT   0x0100
#define WRMT_MASK_BUTTON_RIGHT  0x0200
#define WRMT_MASK_BUTTON_DOWN   0x0400
#define WRMT_MASK_BUTTON_UP     0x0800
#define WRMT_MASK_BUTTON_PLUS   0x1000

#define WRMT_MASK_LED_1   0x0001
#define WRMT_MASK_LED_2   0x0002
#define WRMT_MASK_LED_3   0x0004
#define WRMT_MASK_LED_4   0x0008

/**
* type definitions 
**/

/**
* Function Type ID
*
* @See WRMT_WiiRemote_IsEnabled(), WRMT_WiiRemote_SetEnabled()
**/
typedef enum 
{
	WRMT_FUNCTION_CONTINUOUS = 0,
	WRMT_FUNCTION_MOTION,
	WRMT_FUNCTION_IR,
	WRMT_FUNCTION_SPEAKER,
	WRMT_NUMBER_OF_FUNCTIONS,
} WRMT_FunctionType;

/**
* Data Type ID
*
* @See WRMT_WiiRemote_GetState(), WRMT_WiiRemote_SetState()
**/
typedef enum 
{
	/* read write data */
	WRMT_DATA_FORCE_FEEDBACK = 0,
	WRMT_DATA_LEDS,
	WRMT_DATA_SPEAKER_FORMAT,
	WRMT_DATA_SPEAKER_VOLUME,
	WRMT_DATA_SPEAKER_SAMPLE_RATE,
	/* read only data */
	WRMT_DATA_BUTTONS,
	WRMT_DATA_MOTION_X,
	WRMT_DATA_MOTION_Y,
	WRMT_DATA_MOTION_Z,
	WRMT_DATA_IR1_FOUND,
	WRMT_DATA_IR1_SIZE,
	WRMT_DATA_IR1_X,
	WRMT_DATA_IR1_Y,
	WRMT_DATA_IR2_FOUND,
	WRMT_DATA_IR2_SIZE,
	WRMT_DATA_IR2_X,
	WRMT_DATA_IR2_Y,
	WRMT_DATA_IR_FOUND,
	WRMT_DATA_IR_SIZE,
	WRMT_DATA_IR_X,
	WRMT_DATA_IR_Y,
	WRMT_DATA_BATTERY_LEVEL, /* 0 - 200(?) */
	WRMT_NUMBER_OF_DATA_TYPE,
} WRMT_DataType;

typedef enum 
{
	WRMT_IO_ERROR = -1,
	WRMT_IO_SUCCESS = 0,
	WRMT_IO_TIMEOUT = 1,
} WRMT_IOReturn;

struct wrmt_wiiremote;
typedef struct wrmt_wiiremote WRMT_WiiRemote;

/**
* function declarations
**/

/**
* Set Error.
**/
extern
void
WRMT_SetError(const char *message);

/**
* Get Error.
**/
extern
char *
WRMT_GetError();

/**
* Initialize this library system. 
* It is necessary to call this function before
* the call of any other functions. 
**/
extern
int
WRMT_Init();

/**
* Quit library system.
**/
extern
void
WRMT_Quit();

/**
* Sleep.
**/
extern
void
WRMT_Sleep(int ms);

/**
* Update all opened Wii Remote devices.
**/
extern
void
WRMT_Update();

/**
* Checks whether there is input data from the Wii Remote device.
* The state is updated if there is data. 
**/
extern
WRMT_IOReturn
WRMT_Poll(int *updated_device_index_pointer);

/**
* Get number of Wii Remote devices.
**/
extern
int
WRMT_GetNumWiiRemote();

/**
* Get Wii Remote device at specified index.
**/
extern
WRMT_WiiRemote *
WRMT_GetWiiRemoteAt(int device_index);

/**
* Open connection to Wii Remote
**/
extern
WRMT_IOReturn
WRMT_WiiRemote_Open(WRMT_WiiRemote *self);

/**
* Checks whether the connection to Wii Remote has opened. 
**/
extern
int
WRMT_WiiRemote_IsOpened(WRMT_WiiRemote *self);

/**
* Close connection to Wii Remote.
**/
extern
void
WRMT_WiiRemote_Close(WRMT_WiiRemote *self);

/**
* Checks whether the specified function is enabled. 
**/
extern
int 
WRMT_WiiRemote_IsEnabled(WRMT_WiiRemote *self, WRMT_FunctionType type);

/**
* Enable or disable the specified function. 
**/
extern
void
WRMT_WiiRemote_SetEnabled(WRMT_WiiRemote *self, WRMT_FunctionType type, int value);

/**
* Get device state from WiiRemote.
**/
extern
int 
WRMT_WiiRemote_GetState(WRMT_WiiRemote *self, WRMT_DataType type);

/**
* Sets the state of the Wii Remote device. 
**/
extern
void
WRMT_WiiRemote_SetState(WRMT_WiiRemote *self, WRMT_DataType type, int value);

/**
* Writes data to the specified memory area. 
**/
extern
int
WRMT_WiiRemote_WriteDataToMemory(WRMT_WiiRemote *self,
	int address, const unsigned char *data, size_t data_size);


#ifdef __cplusplus
}
#endif

#endif /* WIIREMOTE_H */

