/*
  Some global functions for libOftp.
  
  Copyright (c) 2009-2009 hirohito higashi. All rights reserved.
  This file is distributed under BSD license.
*/


/***** Feature test switches ************************************************/
/***** System headers *******************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>
#include <string.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <sys/select.h>


/***** Local headers ********************************************************/
#include "liboftp.h"
#include "sub.h"


/***** Constat values *******************************************************/
/***** Macros ***************************************************************/
/***** Typedefs *************************************************************/
/***** Function prototypes **************************************************/
/***** Local variables ******************************************************/
/***** Global variables *****************************************************/
/***** Signal catching functions ********************************************/
/***** Local functions ******************************************************/
/****************************************************************************/
/*! read a line
 *
 *@param	sd	socket descriptor
 *@param	buf	pointer of receive buffer
 *@param	bufsiz	buffer size
 *@param	flags	flags of recv()
 *@retval	int	number of bytes received, or -1 if an error.
 *@note
 *
 */
static int recv_line( int sd, char *buf, int bufsiz, int flags )
{
    int n;
    char c;
    char *p = buf;
    int total_len = 0;

    bufsiz--;

    while( 1 ) {
	n = recv( sd, &c, 1, flags );
	if( n < 0 ) {
	    if( errno == EINTR ) continue;
	    *p = 0;
	    DEBUGPRINT1( "recv_line: get minus. %s\n", strerror(errno) );
	    return n;
	}
	if( n == 0 ) {
	    DEBUGPRINT1( "recv_line: get zero. %s\n", strerror(errno) );
	    break;
	}
	
	*p++ = c;
	total_len++;

	if( c == '\n' ) break;
	if( bufsiz < total_len ) break;
    }
    *p = 0;

    return total_len;
}



/***** Global functions *****************************************************/

/****************************************************************************/
/*! send n bytes.
 *
 *@param	sd	socket descriptor
 *@param	buf	pointer of send buffer
 *@param	len	message length
 *@param	flags	flags of send()
 *@retval	int	number of bytes sent, or -1 if an error.
 *@note
 *
 */
int sendn( int sd, const char *buf, int len, int flags )
{
    int left = len;
    int n;

    while( left > 0 ) {
	n = send( sd, buf, left, flags );
	if( n < 0 ) {
	    if( errno == EINTR ) continue;
	    DEBUGPRINT1( "sendn: get minus. %s\n", strerror(errno) );
	    return n;
	}
	buf += n;
	left -= n;
    }

    return len;
}



/****************************************************************************/
/*! send ftp command
 *
 *@param	ftp	pointer of LIBOFTP.
 *@param	cmd	ftp command.
 *@retval	int	0 is no error, or -1 if an error.
 *@note
 *
 */
int ftp_send_command( LIBOFTP *ftp, const char *cmd )
{
    int ret;
    DEBUGPRINT1( "SENDC: %s", cmd );
    ret = sendn( ftp->socket, cmd, strlen( cmd ), 0 );
    if( ret < 0 ) {
	copy_strerror();
	return ret;
    }

    return 0;
}



/****************************************************************************/
/*! receive response
 *
 *@param	ftp	pointer of LIBOFTP.
 *@param	res_buf	response buffer ( null ok )
 *@param	bufsiz	response buffer size.
 *@retval	int	reply code, or error code (minus value)
 *@note
 * set reply code to ftp attribute reply_code;
 */
int ftp_receive_response( LIBOFTP *ftp, char *res_buf, int bufsiz )
{
    char str1[256];
    int n;

    ftp->reply_code = 0;
    if( res_buf ) {
	*res_buf = 0;
    }

    /*
     * receive 1st line.
     */
    if( (n = recv_line( ftp->socket, str1, sizeof(str1), 0 )) < 0 ) {
	if( errno == EAGAIN ) {
	    return LIBOFTP_ERROR_TIMEOUT;
	}
	copy_strerror();
	return LIBOFTP_ERROR_OS;
    }
    DEBUGPRINT1( "RESP1: %s\n", str1 );
    if( n < 3 ) {
	return LIBOFTP_ERROR_PROTOCOL;
    }
    ftp->reply_code = atoi( str1 );

    /*
     * if exist next line, receive all response.
     */
    if( str1[3] == '-' ) {
	while( 1 ) {
	    if( (n = recv_line( ftp->socket, str1, sizeof(str1), 0 )) < 0 ) {
		if( errno == EAGAIN ) {
		    return LIBOFTP_ERROR_TIMEOUT;
		}
		copy_strerror();
		return LIBOFTP_ERROR_OS;
	    }
	    DEBUGPRINT1( "RESP2: %s\n", str1 );
	    if( n < 3 ) {
		return LIBOFTP_ERROR_PROTOCOL;
	    }

	    if( atoi(str1) == ftp->reply_code && str1[3] == ' ' ) {
		break;
	    }
	}
    }

    /*
     * copy a response strings to user buffer, if need
     */
    if( res_buf ) {
	strncpy( res_buf, str1, bufsiz );
	res_buf[bufsiz-1] = 0;
    }

    return ftp->reply_code;
}



/****************************************************************************/
/*! 送受信準備　アクティブモード
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	cmd	FTPコマンド (ex: RETR, STOR, APPE)
 *@param	fname	サーバ上のファイル名
 *@retval	int	ソケット, マイナス値ならエラーコード
 *@note
 アクティブモードで送受信の際の、以下の３ステートをサポートする。
 例) 受信時
       (port open) --> PORT --> RETR
 */
int ftp_getready_active( LIBOFTP *ftp, const char *cmd, const char *fname )
{
    int sock_listen, sock_accept = -1;
    struct sockaddr_in saddr;
    int saddr_len;
    unsigned char *ip, *pt;
    char str1[256];
    int res;
    struct timeval timeout;
    fd_set rfds, wfds;
    int flag_status_ok = 0;

    /*
     * open data port.
     * (note)
     *  same command port ip and automatic random port.
     */
    sock_listen = socket( PF_INET, SOCK_STREAM, IPPROTO_TCP );
    if( sock_listen < 0 ) {
	DEBUGPRINT1( "getready_active: can't open socket. %s\n", strerror(errno) );
	goto ERROR_OS;
    }

    memset( &saddr, 0, sizeof( saddr ) );
    saddr_len = sizeof( saddr );
    if( getsockname( ftp->socket, (struct sockaddr *)&saddr, &saddr_len ) < 0 ) {
	DEBUGPRINT1( "getready_active: can't get control socket name. %s\n", strerror(errno) );
	goto ERROR_OS;
    }
    saddr.sin_port = htons( 0 );

    if( bind( sock_listen, (struct sockaddr *)&saddr, sizeof( saddr ) ) < 0 ) {
	DEBUGPRINT1( "getready_active: can't bind socket. %s\n", strerror(errno) );
	goto ERROR_OS;
    }

    if( listen( sock_listen, 1 ) < 0 ) {
	DEBUGPRINT1( "getready_active: can't listen socket. %s\n", strerror(errno) );
	goto ERROR_OS;
    }

    /*
     * make PORT command.
     */
    memset( &saddr, 0, sizeof( saddr ) );
    saddr_len = sizeof( saddr );
    if( getsockname( sock_listen, (struct sockaddr *)&saddr, &saddr_len ) < 0 ) {
	DEBUGPRINT1( "getready_active: can't get data socket name. %s\n", strerror(errno) );
	goto ERROR_OS;
    }
    ip = (unsigned char*)&saddr.sin_addr.s_addr;
    pt = (unsigned char*)&saddr.sin_port;
    snprintf( str1, sizeof(str1), "PORT %d,%d,%d,%d,%d,%d\r\n", ip[0], ip[1], ip[2], ip[3], pt[0], pt[1] );

    /*
     * send PORT command and get status.
     */
    if( ftp_send_command( ftp, str1 ) < 0 ) {
	DEBUGPRINT1( "getready_active: %s command sending error.\n", "PORT" );
	close( sock_listen );
	return LIBOFTP_ERROR_OS;
    }
    res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message) - 1 );
    if( res != 200 ) {					/* 200: Command okay. */
	DEBUGPRINT1( "getready_active: get PORT response code %d\n", res );
	close( sock_listen );
	return LIBOFTP_ERROR_PROTOCOL;
    }
    
    /*
     * send ftp command.
     */
    if( fname ) {
	snprintf( str1, sizeof(str1), "%s %s\r\n", cmd, fname );
    } else {
	snprintf( str1, sizeof(str1), "%s\r\n", cmd );
    }
    if( ftp_send_command( ftp, str1 ) < 0 ) {
	DEBUGPRINT1( "getready_active: command sending error. %s", str1 );
	close( sock_listen );
	return LIBOFTP_ERROR_OS;
    }

    /*
     * accept data connection with timeout, or get status by control connection.
     */
    timeout.tv_sec = ftp->timeout_sec;
    timeout.tv_usec = 0;
    do {
	int nfds = (sock_listen > ftp->socket)? sock_listen: ftp->socket;
	FD_ZERO( &rfds );
	FD_ZERO( &wfds );
	if( sock_listen >= 0) {
	    FD_SET( sock_listen, &rfds );
	    FD_SET( sock_listen, &wfds );
	}
	FD_SET( ftp->socket, &rfds );
	
	res = select( nfds+1, &rfds, &wfds, 0, &timeout );
	/* ckeck timeout */
	if( res == 0 ) {
	    DEBUGPRINT1( "getready_active: waiting connection timeout.%s\n", "" );
	    close( sock_listen );
	    strncpy( ftp->error_message, "connection timeout.", sizeof( ftp->error_message )-1 );
	    return LIBOFTP_ERROR_TIMEOUT;
	}

	/* check some OS errors. */
	if( res < 0 ) {
	    if( errno == EINTR ) continue;
	    DEBUGPRINT1( "getready_active: select error. %s\n", strerror(errno) );
	    goto ERROR_OS;
	}

	/* accept data connection. */
	if( FD_ISSET( sock_listen, &rfds ) || FD_ISSET( sock_listen, &wfds ) ) {
	    sock_accept = accept( sock_listen, (struct sockaddr *)&saddr, &saddr_len );
	    close( sock_listen );
	    sock_listen = -1;
	    if( sock_accept < 0 ) {
		DEBUGPRINT1( "getready_active: socket accept error. %s\n", strerror(errno) );
		goto ERROR_OS;
	    }
	    DEBUGPRINT1( "getready_active: socket accept. fd=%d\n", sock_accept );
	}

	/* get status. */
	if( FD_ISSET( ftp->socket, &rfds ) ) {
	    res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message) - 1 );
	    DEBUGPRINT1( "getready_active: get STOR/RETR response code %d\n", res );
	    if( res != 150 ) {				/* 150: File status okay; about to open data connection. */
		if( sock_listen >= 0 ) close( sock_listen );
		if( sock_accept >= 0 ) close( sock_accept );
		return LIBOFTP_ERROR_PROTOCOL;
	    }
	    flag_status_ok = 1;
	}
	
    } while( !( sock_accept >= 0 && flag_status_ok ) );

    return sock_accept;

    
ERROR_OS:
    copy_strerror();
    if( sock_listen >= 0 ) {
	close( sock_listen );
    }
    return LIBOFTP_ERROR_OS;
}



/****************************************************************************/
/*! 送受信準備　パッシブモード
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	cmd	FTPコマンド (ex: RETR, STOR, APPE)
 *@param	fname	サーバ上のファイル名
 *@retval	int	ソケット, マイナス値ならエラーコード
 *@note
 パッシブモードで送受信の際の、以下の３ステートをサポートする。
 例) 受信時
       PASV --> RETR --> (port open)
 */
int ftp_getready_pasv( LIBOFTP *ftp, const char *cmd, const char *fname )
{
    char str1[256];
    int res;
    char *p;
    int h1, h2, h3, h4, p1, p2;
    struct sockaddr_in saddr;
    int sock;
    struct timeval tval;

    /*
     * send "PASV" command and get status.
     */
    str1[255] = 0;
    snprintf( str1, sizeof(str1)-1, "PASV\r\n" );
    if( ftp_send_command( ftp, str1 ) < 0 ) {
	return LIBOFTP_ERROR_OS;
    }
    res = ftp_receive_response( ftp, str1, sizeof(str1) - 1 );
    if( res != 227 ) {					/* 227: Entering Passive Mode */
	DEBUGPRINT1( "getready_pasv: get ftp response code %d\n", res );
	strncpy( ftp->error_message, str1, sizeof( ftp->error_message ) - 1 );
	return LIBOFTP_ERROR_PROTOCOL;
    }

    /*
     * parse (h1,h2,h3,h4,p1,p2)
     */
    p = strchr( str1, '(' );
    if( p == NULL ) {
	DEBUGPRINT1( "getready_pasv: response parse error. %d", 1 );
	strncpy( ftp->error_message, "Illegal pasv response", sizeof(ftp->error_message) - 1 );
	return LIBOFTP_ERROR_PROTOCOL;
    }
    if( sscanf( p+1, "%d,%d,%d,%d,%d,%d", &h1, &h2, &h3, &h4, &p1, &p2 ) != 6 ) {
	DEBUGPRINT1( "getready_pasv: response parse error. %d", 2 );
	strncpy( ftp->error_message, "Illegal pasv response", sizeof(ftp->error_message) - 1 );
	return LIBOFTP_ERROR_PROTOCOL;
    }


    /*
     * send ftp command.
     */
    if( fname ) {
	snprintf( str1, sizeof(str1), "%s %s\r\n", cmd, fname );
    } else {
	snprintf( str1, sizeof(str1), "%s\r\n", cmd );
    }
    if( ftp_send_command( ftp, str1 ) < 0 ) {
	DEBUGPRINT1( "getready_pasv: command sending error. %s", str1 );
	return LIBOFTP_ERROR_OS;
    }

    /*
     * open data port.
     */
    sock = socket( PF_INET, SOCK_STREAM, IPPROTO_TCP );
    if( sock < 0 ) {
	DEBUGPRINT1( "getready_pasv: can't open socket. %s\n", strerror(errno) );
	copy_strerror();
	return LIBOFTP_ERROR_OS;
    }

    memset( &saddr, 0, sizeof( saddr ) );
    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = htonl( (((unsigned long)h1) << 24) | (h2 << 16) | (h3 << 8) | h4 );
    saddr.sin_port = htons( (p1 << 8) | p2 );

    if( connect( sock, (struct sockaddr *)&saddr, sizeof( saddr ) ) < 0 ) {
	DEBUGPRINT1( "getready_pasv: can't connect socket. %s\n", strerror(errno) );
	copy_strerror();
	close( sock );
	return LIBOFTP_ERROR_OS;
    }

    tval.tv_sec = ftp->timeout_sec;
    tval.tv_usec = 0;
    if( setsockopt( sock, SOL_SOCKET, SO_RCVTIMEO, &tval, sizeof(struct timeval) ) < 0 ) {
	DEBUGPRINT1( "setsockopt(SO_RCVTIMEO) failed. %s\n", strerror(errno) );
	copy_strerror();
	close( sock );
	return LIBOFTP_ERROR_OS;
    }
    
    /*
     * check response.
     */
    res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message) - 1 );
    if( res == 150 || res == 125 ) {
	return sock;
    } else {
	close( sock );
	return LIBOFTP_ERROR_PROTOCOL;
    }
}
