/* Copyright 1999-2003 Red Hat, Inc.
 *
 * This software may be freely redistributed under the terms of the GNU
 * public license.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/time.h>

#include "kudzu.h"
#include "psaux.h"

#define MOUSE_CMD_RESET                  0xFF 
#define MOUSE_CMD_RESEND                 0xFE
#define MOUSE_CMD_DEFAULTS               0xF6
#define MOUSE_CMD_DISABLE                0xF5
#define MOUSE_CMD_ENABLE                 0xF4
#define MOUSE_CMD_SET_SAMPLE_RATE        0xF3
#define MOUSE_CMD_GET_DEVICE_ID          0xF2
#define MOUSE_CMD_SET_REMOTE_MODE        0xF0
#define MOUSE_CMD_SET_WRAP_MODE          0xEE
#define MOUSE_CMD_RESET_WRAP_MODE        0xEC
#define MOUSE_CMD_READ_DATA              0xEB
#define MOUSE_CMD_SET_STREAM_MODE        0xEA
#define MOUSE_CMD_STATUS_REQUEST         0xE9
#define MOUSE_CMD_SET_RESOLUTION         0xE8
#define MOUSE_CMD_SET_SCALING_2          0xE7
#define MOUSE_CMD_SET_SCALING_1          0xE6

#define MOUSE_RESP_RESENT 0xFE
#define MOUSE_RESP_ERROR  0xFC
#define MOUSE_RESP_ACK    0xFA
#define MOUSE_RESP_TESTOK 0xAA

static void psauxFreeDevice(struct psauxDevice *dev)
{
    freeDevice((struct device *) dev);
}

static void psauxWriteDevice(FILE *file, struct psauxDevice *dev)
{
    writeDevice(file, (struct device *)dev);
}

static int psauxCompareDevice(struct psauxDevice *dev1, struct psauxDevice *dev2)
{
    return compareDevice( (struct device *)dev1, (struct device *)dev2);
}

struct psauxDevice *psauxNewDevice(struct psauxDevice *old)
{
    struct psauxDevice *ret;
    
    ret = malloc(sizeof(struct psauxDevice));
    memset(ret, '\0', sizeof(struct psauxDevice));
    ret = (struct psauxDevice *) newDevice((struct device *) old, (struct device *) ret);
    ret->bus = BUS_PSAUX;
    ret->newDevice = psauxNewDevice;
    ret->freeDevice = psauxFreeDevice;
    ret->writeDevice = psauxWriteDevice;
    ret->compareDevice = psauxCompareDevice;
    return ret;
}

static int mouse_read(int fd)
{
    struct timeval tv;
    unsigned char ch;
    fd_set fds;
    int ret;
    
    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    tv.tv_sec = 0;
    tv.tv_usec = 600000;
    ret = select(fd+1, &fds, NULL, NULL, &tv);
    if (-1 == ret) {
	DEBUG("select error: %s\n",strerror(errno));
	return -1;
    }

    if (0 == ret) {
	DEBUG("Timeout waiting for response\n");
	return -1;
    }
    
    ret = read(fd, &ch, 1);
    if (1 == ret) {
	DEBUG("mouse_read: %04x\n",ch);
	return ch;
    }

    DEBUG("error reading mouse data: %s\n",strerror(errno));
    return -1;
}

static int mouse_cmd(int fd, unsigned char cmd)
{
    int ret;
    
    DEBUG("mouse_cmd: %02x\n",cmd);
    ret = write(fd, &cmd, 1);
    if (ret != 1) {
	return -1;
    }

    ret = mouse_read(fd);
    if (ret == MOUSE_RESP_ACK)
	return 0;
    else
	return -1;
}

struct device *psauxProbe(enum deviceClass probeClass, int probeFlags,
			  struct device *devlist)
{
    int portfd, ret;
    struct psauxDevice *ps2dev;
    
    if (probeFlags & PROBE_SAFE) return devlist;

    if (probeClass & CLASS_MOUSE) {
	portfd=open("/dev/psaux", O_RDWR|O_NONBLOCK);
	if (portfd < 0) {
	    DEBUG("error opening mouse port.\n");
	    return devlist;
	}

	mouse_cmd(portfd, MOUSE_CMD_RESET);
	
	if ((ret = mouse_read(portfd)) != MOUSE_RESP_TESTOK)
	    DEBUG("Mouse self-test failed.\n");
	
	if ((ret = mouse_read(portfd)) != 0x00)
	    DEBUG("Mouse did not finish reset response.\n");

	mouse_cmd(portfd, MOUSE_CMD_ENABLE);
	    
	if (mouse_cmd(portfd, MOUSE_CMD_GET_DEVICE_ID)) {
	    DEBUG("mouse device id command failed: no mouse\n");
	    goto out;
	}

	ret = mouse_read(portfd);
	DEBUG("got mouse type %02x\n",ret);
	if (-1 == ret) {
	    DEBUG("Failed to read initial mouse type\n");
	    goto out;
	}

	/* attempt to enable IntelliMouse 3-button mode */
	DEBUG("Attempting to enable IntelliMouse 3-button mode\n");
	mouse_cmd(portfd, MOUSE_CMD_SET_SAMPLE_RATE);
	mouse_cmd(portfd, 0xc8);
	mouse_cmd(portfd, MOUSE_CMD_SET_SAMPLE_RATE);
	mouse_cmd(portfd, 0x64);
	mouse_cmd(portfd, MOUSE_CMD_SET_SAMPLE_RATE);
	mouse_cmd(portfd, 0x50);

	/* now issue get device id command */
	mouse_cmd(portfd, MOUSE_CMD_GET_DEVICE_ID);
	
	ret = mouse_read(portfd);
	DEBUG("Device ID after IntelliMouse initialization: %02x\n", ret);

	if (0x03 == ret) {
	    /* attempt to enable IntelliMouse 5-button mode */
	    DEBUG("Attempting to enable IntelliMouse 5-button mode\n");
	    mouse_cmd(portfd, MOUSE_CMD_SET_SAMPLE_RATE);
	    mouse_cmd(portfd, 0xc8);
	    mouse_cmd(portfd, MOUSE_CMD_SET_SAMPLE_RATE);
	    mouse_cmd(portfd, 0xc8);
	    mouse_cmd(portfd, MOUSE_CMD_SET_SAMPLE_RATE);
	    mouse_cmd(portfd, 0x50);

	    /* now issue get device id command */
	    mouse_cmd(portfd, MOUSE_CMD_GET_DEVICE_ID);
	    ret = mouse_read(portfd);
	    DEBUG("Device ID after IntelliMouse 5 button initialization: %02x\n", ret);
	}
	
	ps2dev=psauxNewDevice(NULL);
	ps2dev->device=strdup("psaux");
	ps2dev->type=CLASS_MOUSE;
	if (devlist)
	    ps2dev->next = devlist;
	devlist = (struct device *) ps2dev;
	switch (ret) {
	case 0x03:
	case 0x04:
	case 0x05:
	    ps2dev->driver=strdup("msintellips/2");
	    ps2dev->desc=strdup("Generic PS/2 Wheel Mouse");
	    break;
	case 0x02:
	    /* a ballpoint something or other */
	case 0x06:
	    /* A4 Tech 4D Mouse ? */
	case 0x08:
	    /* A4 Tech 4D+ Mouse ? */
	case 0x00:
	default:
	    DEBUG("mouse type: %x\n",ret);
	    ps2dev->driver=strdup("genericps/2");
	    ps2dev->desc=strdup("Generic Mouse (PS/2)");
	    break;
	}
out:    
	DEBUG("resetting mouse\n");
	mouse_cmd(portfd, MOUSE_CMD_RESET);

	if ((ret = mouse_read(portfd)) != MOUSE_RESP_TESTOK)
	    DEBUG("Mouse self-test failed.\n");

	if ((ret = mouse_read(portfd)) != 0x00)
	    DEBUG("Mouse did not finish reset response.\n");
	
	if (mouse_cmd(portfd, MOUSE_CMD_ENABLE)) {
	    DEBUG("mouse enable failed: no mouse?\n");
	}

	if (mouse_cmd(portfd, MOUSE_CMD_GET_DEVICE_ID)) {
	     DEBUG("mouse type command failed: no mouse\n");
	}
	    
	if (0x00 != mouse_read(portfd)) {
	     DEBUG("initial mouse type check strange: no mouse\n");
	}
	    
	close(portfd);
    }
    return devlist;
}
