/****************************************************************************/
/* The developnent of this program is partly supported by IPA.              */
/* (Infomation-Technology Promotion Agency, Japan).                         */
/****************************************************************************/

/****************************************************************************/
/*  cloopreadahead.c                                                        */
/*  Copyright : Copyright (C) 2006 ALPHA SYSTEMS INC.                       */
/*  Authors   : TAN Hideyuki (tanh@alpha.co.jp)                             */
/*              ALPHA SYSTEMS INC. knoppix team (knoppix@alpha.co.jp)       */
/*                                                                          */
/*  This is free software; you can redistribute it and/or modify            */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 2 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This software is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this software; if not, write to the Free Software            */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,  */
/*  USA.                                                                    */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <ctype.h>
#include <sched.h>
#include <dirent.h>

#include "compressed_loop.h"
#include "cloop_optimizer.h"

#define CLOOP_DAEMONIZE   "background"
#define CLOOP_BURST_START "burst_start"
#define CLOOP_BURST_END   "burst_end"

int is_running(const char *cmdname) {
  struct dirent *entry;
  int i, fd;
  char path[256], buf[1024];
  int cmdlen = strlen(cmdname);

  DIR *dir = opendir("/proc");
  if (! dir) return 0;

  while((entry = readdir(dir)) != NULL) {
    int len = strlen(entry->d_name);
    for (i = 0; i < len; i++) if (! isdigit(entry->d_name[i])) break;
    if (i != len) continue;
    sprintf(path, "/proc/%s/cmdline", entry->d_name);
    fd = open(path, O_RDONLY);
    if (fd < 0) continue;
    if (read(fd, &buf, 1024) < 0 || buf[0] == '\0') {
      close(fd);
      continue;
    }
    //    printf("%s: %s\n", entry->d_name, buf);
    if (strncmp(cmdname, buf, cmdlen) == 0) return 1;
    close(fd);
  }
  closedir(dir);

  return 0;
}

unsigned int wait_modprobe(void) {
  unsigned int count = 0;

  while(is_running("modprobe")) {
    count++;
    usleep(500000);
    //    usleep(125000);
  }
  return count;
}

int cloop_read(int clofd, loff_t startpos, size_t read_len) {

  int ret = 0;
  unsigned int wait = 0;

  wait = wait_modprobe();
  if (wait != 0)
    printf("wait = %u\n", wait);

  void *dummy = malloc(read_len);
  if (! dummy) {
    fprintf(stderr, "dummy malloc error. %u bytes.\n", read_len);
    exit(1);
  }

  if (lseek(clofd, startpos, SEEK_SET) < 0) {
    fprintf(stderr, "seek error.\n");
    exit(1);
  }
  if (read(clofd, dummy, read_len) < 0) {
    fprintf(stderr, "read error.\n");
    exit(1);
  }

  free(dummy);

  return ret;
}

int main(int argc, char **argv) {

  int clofd;
  struct cloop_head header;
  unsigned int total_offsets;

  struct block_info *blk_info_tbl;
  int i;
  FILE *listfp;
  char tmpbuf[512];
  int len;
  int blk_num = 0, preblk_num = 0, startblk_num = 0;
  int read_count = 0;
  unsigned int blk_len = 0, read_len = 0, read_sum = 0, max_read_len = 0;
  int skip = 0;
  int burst = 0;

  if (argc != 3) {
    fprintf(stderr, "%s : Usage <optimized cloop file> <block list file>\n", argv[0]);
    exit(1);
  }

  clofd = open(argv[1],O_RDONLY);

  if (clofd < 0) {
    fprintf(stderr, "cloop file open error.\n");
    exit(1);
  }

  if (read(clofd, &header, sizeof(struct cloop_head)) != sizeof(struct cloop_head)) {
    fprintf(stderr, "header read error.\n");
    exit(1);
  }

  if (strcmp(header.preamble, CLOOP_OPT_PREAMBLE) != 0) {
    fprintf(stderr, "%s is not optimized cloop image.\n", argv[1]);
    exit(1);
  }
  
  fprintf(stderr, "blocksize: %u, blocknumber: %u\n",
	 ntohl(header.block_size), ntohl(header.num_blocks));

  total_offsets = ntohl(header.num_blocks) + 1;

  blk_info_tbl = malloc(sizeof(struct block_info) * total_offsets);
  if (! blk_info_tbl) {
    fprintf(stderr, "block_info table malloc error.\n");
    exit(1);
  }
  if (read(clofd, blk_info_tbl, (sizeof(struct block_info) * total_offsets))
	  != (sizeof(struct block_info) * total_offsets)) {
    fprintf(stderr, "block_info table read error.\n");
    exit(1);
  }

  listfp = fopen(argv[2], "r");
  if (! listfp) {
    fprintf(stderr, "block list file open error.\n");
    exit(1);
  }

  // buffering 
  while(fgets(tmpbuf, sizeof(tmpbuf), listfp) != 0) {
    i = strlen(tmpbuf);
    i++;
  }
  rewind(listfp);

  preblk_num = -1;
  while(fgets(tmpbuf, sizeof(tmpbuf), listfp) != 0) {

    len = strlen(tmpbuf) -1;
    if (len <= 0) continue;
    tmpbuf[len] = '\0';

    if (strcmp(tmpbuf, CLOOP_DAEMONIZE) == 0) {
      printf("daemonize.\n");
      daemon(1, 0);
      nice(10);
      continue;
    }

    if (strcmp(tmpbuf, CLOOP_BURST_START) == 0) {

      burst = 1;
      continue;

    } else if (strcmp(tmpbuf, CLOOP_BURST_END) == 0) {

      printf("BURSTREAD start_pos=%Lu, length=%u\n",
	     be64_to_cpu(blk_info_tbl[startblk_num].offset), read_len);

      cloop_read(clofd, be64_to_cpu(blk_info_tbl[startblk_num].offset), read_len);
      read_count++;
      max_read_len = (max_read_len > read_len ? max_read_len : read_len);

      read_sum += read_len;
      skip = 0;
      startblk_num = -1;
      read_len = 0;

      sched_yield();

      continue;

    } else {

      for(i = 0; i < len; i++)
	if (! isdigit(tmpbuf[i])) break;
      if (i != len) continue;

    }

    // tmpbuf == block number

    blk_num = atoi(tmpbuf);
    if (blk_num > ntohl(header.num_blocks)) continue;

    blk_len = ntohl(blk_info_tbl[blk_num].size);

    if (blk_num == preblk_num +1 || burst) {

      printf("%05u: size=%u, pos=%Lu : %s\n",
	     blk_num, blk_len, be64_to_cpu(blk_info_tbl[blk_num].offset),
	     (burst ? "BURST SKIP" : "SKIP"));
      read_len += blk_len;
      skip = 1;

      if (startblk_num == -1) startblk_num = blk_num;

    } else {

      printf("%05u: size=%u, pos=%Lu : ",
	     blk_num, blk_len, be64_to_cpu(blk_info_tbl[blk_num].offset));
      printf("READ start_pos=%Lu, length=%u\n",
	     be64_to_cpu(blk_info_tbl[startblk_num].offset), read_len);

      cloop_read(clofd, be64_to_cpu(blk_info_tbl[startblk_num].offset), read_len);
      read_count++;
      max_read_len = (max_read_len > read_len ? max_read_len : read_len);

      read_sum += read_len;

      startblk_num = blk_num;
      read_len = blk_len;
      skip = 0;

      sched_yield();
    }

    preblk_num = blk_num;
  }

  if (skip) { // for remaining blocks.

    printf("%05u: size=%u, pos=%Lu : ",
	   blk_num, blk_len, be64_to_cpu(blk_info_tbl[blk_num].offset));
    printf("READ start_pos=%Lu, length=%u\n",
	   be64_to_cpu(blk_info_tbl[startblk_num].offset), read_len);
    
    cloop_read(clofd, be64_to_cpu(blk_info_tbl[startblk_num].offset), read_len);

  }

  fprintf(stderr, "read_sum : %u Bytes (%uKB)\n", read_sum, read_sum/1024);
  fprintf(stderr, "max_read_length : %u Bytes (%uKB)\n", max_read_len, max_read_len/1024);
  fprintf(stderr, "read count : %d times\n", read_count);

  free(blk_info_tbl);

  fclose(listfp);
  close(clofd);

  exit(0);
}
