﻿using System;
using System.Collections.Generic;
using System.Data.SQLite;
using com.andoutomo.kybernetes.data.connection;
using com.andoutomo.kybernetes.data.DAO;
using System.Linq;


namespace com.andoutomo.kybernetes.data.accessor
{
    public class TaskDataAccessor
    {
        private static TaskDataAccessor accessor;
        private TaskDataAccessor()
        {
        }
        /// <summary>
        /// アクセッサを取得します。
        /// </summary>
        public static TaskDataAccessor getObject
        {
            get
            {
                if (accessor == null)
                {
                    accessor = new TaskDataAccessor();
                }
                return accessor;
            }
        }
        /// <summary>
        /// 今日のタスクリストを取得します。「今日」はシステムテーブルから取得します。
        /// </summary>
        /// <returns></returns>
        public List<TaskData> getToDayTaskList()
        {
            return getTaskList(SqlDictionary.getTodayList,new List<object>());
        }

        /// <summary>
        /// 日付ごとのタスクリストを取得します。
        /// </summary>
        /// <param name="date"></param>
        /// <returns></returns>
        public List<TaskData> getDailyTaskList(DateComponent date)
        {
            List<object> arg = new List<object>();
            arg.Add(date.getDateString("yyyy-MM-dd"));

            return getTaskList(SqlDictionary.getDailyList, arg);
        }
        /// <summary>
        /// 現在実施中のタスクリストを抽出します。日付はシステムテーブルから、実施中の判断は
        /// StartしていてEndしていないもの、とします。
        /// </summary>
        /// <returns></returns>
        public List<TaskData> getPwdTaskList()
        {
            return getTaskList(SqlDictionary.getPwdList, new List<object>());
        }


        /// <summary>
        /// IDを元に単体タスクを抽出します。
        /// </summary>
        /// <param name="sortId"></param>
        /// <returns></returns>
        public TaskData getTask(int sortId)
        {
            List<object> args = new List<object>();
            args.Add(sortId);
            TaskData data = null;
            TaskDBConnection.Connection.selectData(SqlDictionary.getOneTask, args,
                (SQLiteDataReader reader) =>
                {
                    if (reader.HasRows)
                    {
                        data = new TaskData();
                        reader.Read();
                        data.TaskID = Convert.ToInt32(reader["TASKID"].ToString());
                        data.SortID = Convert.ToInt32(reader["SORTID"].ToString());
                        data.DoDate = new DateComponent(reader["DODATE"].ToString(), '/');
                        data.TimeArea = reader["TIMEAREA"].ToString();
                        data.CategID = Convert.ToInt32(reader["CATEGID"].ToString());
                        data.Category = reader["CATEGORY"].ToString();
                        data.Contents = reader["CONTENTS"].ToString();
                        data.ForecastMin = Convert.ToInt32(reader["FORCASTMIN"].ToString());
                        data.StartTime = new TimeComponent(reader["STARTTIME"].ToString());
                        data.EndTime = new TimeComponent(reader["ENDTIME"].ToString());
                        data.IsRepeat = Convert.ToInt32(reader["REPEATFLG"].ToString());
                        data.Project = reader["PROJECT"].ToString();
                        data.Tags = reader["TAGS"].ToString();
                    }
                });
            return data;
        }

        /// <summary>
        /// 内部メソッド。タスクリストを取得します。
        /// </summary>
        /// <param name="sql"></param>
        /// <param name="args"></param>
        /// <returns></returns>
        private List<TaskData> getTaskList(string sql,List<object> args)
        {
            List<TaskData> resultList = new List<TaskData>();
            TaskDBConnection.Connection.selectData(sql, args,
            (SQLiteDataReader reader) =>
            {
                while (reader.Read())
                {
                    TaskData data = new TaskData();
                    data.TaskID = Convert.ToInt32(reader["TASKID"].ToString());
                    data.SortID = Convert.ToInt32(reader["SORTID"].ToString());
                    data.DoDate = new DateComponent(reader["DODATE"].ToString(),'/');
                    data.TimeArea = reader["TIMEAREA"].ToString();
                    data.Category = reader["CATEGORY"].ToString();
                    data.Contents = reader["CONTENTS"].ToString();
                    data.ForecastMin = Convert.ToInt32(reader["FORCASTMIN"].ToString());
                    data.StartTime = new TimeComponent(reader["STARTTIME"].ToString());
                    data.EndTime = new TimeComponent(reader["ENDTIME"].ToString());
                    data.IsRepeat = Convert.ToInt32(reader["REPEATFLG"].ToString());
                    data.HasComment = Convert.ToInt32(reader["HASCOMMENT"].ToString());
                    data.Project = reader["PROJECT"].ToString();
                    data.Tags = reader["TAGS"].ToString();
                    data.Comment = reader["COMMENT"].ToString();
                    if (data.Category != string.Empty)
                    {
                        //data.BgColor = Convert.ToInt32(reader["BGCOLOR"].ToString());
                        data.FrColor = Convert.ToInt32(reader["FRCOLOR"].ToString());
                    }
                    resultList.Add(data);
                }
            });
            return resultList;
        }
        /// <summary>
        /// タスクの最大採番を行います。<br/>
        /// その時点での最大の採番は、(タスクIDの最大値+1)×10とします。
        /// (1の位に1～9が入っているとしても、次のタスクに採番されるのは10多いから重ならない)
        /// </summary>
        /// <returns></returns>
        public int getMaxTaskID()
        {
            int maxID =0;
            TaskDBConnection.Connection.selectData(SqlDictionary.getMaxIDofTask, 
                (SQLiteDataReader reader) =>
                {

                    if (reader.HasRows)
                    {
                        reader.Read();
                        string nextID = reader["NEXTID"].ToString();
                        if (string.IsNullOrEmpty(nextID))
                        {
                            maxID = 1;
                        }
                        else
                        {
                            maxID = Convert.ToInt32(nextID);
                        }
                    }
                });
            return maxID;
        }
        /// <summary>
        /// ソート番号の最大を算出します。
        /// このメソッドは、renum実行時に過去分のタスクはマイナスになっていることを前提とします。
        /// </summary>
        /// <returns></returns>
        public int getMaxSortID()
        {
            int maxSortID = 0;
            TaskDBConnection.Connection.selectData(SqlDictionary.getMaxSortID, 
                (SQLiteDataReader reader) => 
                {
                    if (reader.HasRows)
                    {
                        reader.Read();
                        string nextSortID = reader["NEXTSORTID"].ToString();
                        if (string.IsNullOrEmpty(nextSortID))
                        {
                            maxSortID = 10;
                        }
                        else
                        {
                            maxSortID = Convert.ToInt32(nextSortID);
                        }
                    }
 
                });
            return maxSortID;

        }

        /// <summary>
        /// 次タスクのスタートを決めるため、当日の最大End時間を求めます。
        /// 当日にEnd時間がなければNullを返却します。
        /// </summary>
        /// <param name="date"></param>
        /// <returns></returns>
        public TimeComponent getNextStartTime(DateComponent date)
        {
            if (date == null)
            {
                return null;
            }
            List<object> arg = new List<object>();
            arg.Add(date.getDateString("yyyy-MM-dd"));
            TimeComponent endDate = null;

            TaskDBConnection.Connection.selectData(SqlDictionary.getNextStartTime, arg, 
                (SQLiteDataReader reader) => {
                    if (reader.HasRows)
                    {
                        reader.Read();
                        endDate = new TimeComponent(reader["MAXENDTIME"].ToString());
                    }
                });
            return endDate;
        }

        /// <summary>
        /// タスクを追加します。
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public bool insertTask(TaskData data)
        {
            List<object> parameters = new List<object>();

            parameters.Add(data.TaskID);
            parameters.Add(data.SortID);
            parameters.Add(data.DoDate.getDateString("yyyy-MM-dd"));
            parameters.Add(data.TimeArea.ToUpper());
            parameters.Add(data.CategID);
            parameters.Add(data.Contents);
            parameters.Add(data.ForecastMin);

            return TaskDBConnection.Connection.updateData(SqlDictionary.insertTask, parameters);

        }
        /// <summary>
        /// タスクを更新します。
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public bool updateTask(TaskData data)
        {

            List<object> parameters = new List<object>();
            parameters.Add(data.DoDate.getDateString("yyyy-MM-dd"));
            parameters.Add(data.TimeArea.ToUpper());
            parameters.Add(data.CategID);
            parameters.Add(data.Contents);
            parameters.Add(data.ForecastMin);
            parameters.Add(data.SortID);

            return TaskDBConnection.Connection.updateData(SqlDictionary.updateTask, parameters);
        }

        /// <summary>
        /// タスクを開始します。
        /// </summary>
        /// <param name="id">ソートID</param>
        /// <param name="startTime"></param>
        /// <returns></returns>
        public bool startTask(int id ,TimeComponent startTime)
        {
            List<object> parameters = new List<object>();
            parameters.Add(startTime.ToString());
            parameters.Add(id);

            return TaskDBConnection.Connection.updateData(SqlDictionary.startTask, parameters);
        }

        /// <summary>
        /// タスクを終了します。
        /// </summary>
        /// <param name="id">ソートID</param>
        /// <param name="endTime"></param>
        /// <returns></returns>
        public bool endTask(int id, TimeComponent endTime)
        {
            List<object> parameters = new List<object>();
            parameters.Add(endTime.ToString());
            parameters.Add(id);

            return TaskDBConnection.Connection.updateData(SqlDictionary.endTask, parameters);
        }

        /// <summary>
        /// タスクがリピート対象であるかどうかを照合します。
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public bool isRepeatTask(int id)
        {
            bool retval=false;
            List<object> parameters = new List<object>();
            parameters.Add(id);
            TaskDBConnection.Connection.selectData(SqlDictionary.isRepeatTask, parameters,
                (SQLiteDataReader reader) =>
                {
                    if (reader.HasRows)
                    {
                        reader.Read();
                        int dbVal = Convert.ToInt32(reader["REPEATFLG"].ToString());
                        if (dbVal == 1)
                        {
                            retval = true;

                        }
                        else
                        {
                            retval = false;
                        }
                    }
                });
            return retval;
        }
        /// <summary>
        /// 繰り返しの設定を行います。
        /// </summary>
        /// <param name="id"></param>
        /// <param name="isRepeat"></param>
        /// <returns></returns>
        public bool setRepeatTask(int id, bool isRepeat)
        {
            List<object> parameters = new List<object>();

            int settingData = (isRepeat ? 1 : 0);

            parameters.Add(settingData);
            parameters.Add(id);

            return TaskDBConnection.Connection.updateData(SqlDictionary.setRepeatTask, parameters);

        }


        /// <summary>
        /// タスクを削除します。
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public bool deleteTask(int from,int to)
        {
            List<object> parameters = new List<object>();
            parameters.Add(from);
            parameters.Add(to);

            return TaskDBConnection.Connection.updateData(SqlDictionary.deleteTasks, parameters);
        }

        /// <summary>
        /// 削除する前に何件あるか確認します。
        /// </summary>
        /// <param name="from"></param>
        /// <param name="to"></param>
        /// <returns></returns>
        public int countDeleteTargetTask(int from, int to)
        {
            List<object> parameters = new List<object>();
            parameters.Add(from);
            parameters.Add(to);

            int retVal = 0;

            TaskDBConnection.Connection.selectData(SqlDictionary.countBeforeDelete, parameters,
                (SQLiteDataReader reader) =>
                {
                    if (reader.HasRows)
                    {
                        reader.Read();
                        retVal = Convert.ToInt32(reader["COUNT"].ToString());
                    }
                });
            return retVal;
        }

        /// <summary>
        /// 全タスクを表示します。
        /// </summary>
        /// <returns></returns>
        public List<TaskData> getAllTaskList()
        {
            return getTaskList(SqlDictionary.getAllList, new List<object>());
        }

        /// <summary>
        /// 終了していないタスクをすべて表示します。
        /// </summary>
        /// <returns></returns>
        public List<TaskData> getRemainTaskList()
        {
            return getTaskList(SqlDictionary.getRemainList, new List<object>());
        }

        /// <summary>
        /// 今後のタスクをすべて表示します。
        /// </summary>
        /// <returns></returns>
        public List<TaskData> getGoingTaskList()
        {
            return getTaskList(SqlDictionary.getGoingList, new List<object>());
        }

        /// <summary>
        /// タスクの検索を行います。
        /// </summary>
        /// <param name="searchWord"></param>
        /// <returns></returns>
        public List<TaskData> findTask(string searchWord)
        {
            string wkSearchWord = "%" + searchWord + "%";

            List<object> args = new List<object> { wkSearchWord ,wkSearchWord,wkSearchWord,wkSearchWord};
            return getTaskList(SqlDictionary.searchTasks, args);
        }


        /// <summary>
        /// ソート番号の再設定を行います。
        /// </summary>
        /// <param name="data"></param>
        /// <returns></returns>
        public bool renumAllTasks(List<TaskData> data)
        {
            List<MultiSQLRunnningBean> sqlList = new List<MultiSQLRunnningBean>();

            for (int i = 0; i < data.Count; i++)
            {
                int newNumber = (i + 1) * 10;
                string sql = SqlDictionary.changeTaskNo;
                List<object> paramList = new List<object>();
                paramList.Add(newNumber);
                paramList.Add(data[i].TaskID);

                //連続実行するSQLを設定
                MultiSQLRunnningBean bean = new MultiSQLRunnningBean(sql, paramList);
                sqlList.Add(bean);

            }
            //最後に、過去分タスクをマイナスに変換する(これをやらないと次に振り出されるIDがどんどん増えていく)
            MultiSQLRunnningBean lastBean = new MultiSQLRunnningBean(SqlDictionary.suppressCompleteTask,new List<object>());
            sqlList.Add(lastBean);
            return TaskDBConnection.Connection.updateData(sqlList);
        }

        /// <summary>
        /// タスクを延期します。日をまたぎます。
        /// </summary>
        /// <param name="id"></param>
        /// <param name="nextDay"></param>
        /// <returns></returns>
        public bool delayTask(int id, DateComponent nextDay)
        {
            List<object> paramList = new List<object>();
            paramList.Add(nextDay.getDateString("yyyy-MM-dd"));
            paramList.Add(id);

            return TaskDBConnection.Connection.updateData(SqlDictionary.delayTask, paramList);
        }

        /// <summary>
        /// ソート番号を変更します。
        /// </summary>
        /// <param name="targetTaskID"></param>
        /// <param name="after"></param>
        /// <returns></returns>
        public bool changeNumber(int targetTaskID, int after)
        {
            List<object> paramList = new List<object>();
            paramList.Add(after);
            paramList.Add(targetTaskID);

            return TaskDBConnection.Connection.updateData(SqlDictionary.changeTaskNo, paramList);
        }
        /// <summary>
        /// タイムエリアを変更します。
        /// </summary>
        /// <param name="targetSortID"></param>
        /// <param name="nextArea"></param>
        /// <returns></returns>
        public bool changeArea(int targetSortID, string nextArea)
        {
            List<object> paramList = new List<object>();
            paramList.Add(nextArea);
            paramList.Add(targetSortID);

            return TaskDBConnection.Connection.updateData(SqlDictionary.changeAreaOfTask, paramList);
        }
        /// <summary>
        /// 時間をクリアし、タスクを初期状態に戻します。
        /// </summary>
        /// <param name="targetSortID"></param>
        /// <returns></returns>
        public bool clearStartEnd(int targetSortID)
        {
            List<object> paramList = new List<object>();
            paramList.Add(targetSortID);

            return TaskDBConnection.Connection.updateData(SqlDictionary.clearTime, paramList);
        }

        /// <summary>
        /// タスクIDからソート番号を逆引きします。主にコメント用です。
        /// </summary>
        /// <param name="taskID"></param>
        /// <returns></returns>
        public int getSortIDFromTaskID(int taskID)
        {
            List<object> paramList = new List<object>(){taskID};
            int retVal=0;
            TaskDBConnection.Connection.selectData(SqlDictionary.getSortIDFromTaskID, paramList,
                (SQLiteDataReader reader) =>
                {
                    if (reader.HasRows)
                    {
                        reader.Read();
                        retVal = Convert.ToInt32(reader["SORTID"].ToString());
                    }
                });
            return retVal;
        }
        /// <summary>
        /// 対象タスクのプロジェクトを更新します。
        /// </summary>
        /// <param name="targetSortID"></param>
        /// <param name="projectName"></param>
        /// <returns></returns>
        public bool updateProjectofTask(int targetSortID, string projectName)
        {
            List<object> paramList = new List<object>() { projectName, targetSortID };
            return TaskDBConnection.Connection.updateData(SqlDictionary.updateProject, paramList);
        }
        /// <summary>
        /// 現在登録されているプロジェクトを重複なしで取得します。
        /// </summary>
        /// <returns></returns>
        public List<SimpleDataBean> getProjectList()
        {
            List<SimpleDataBean> retList = new List<SimpleDataBean>();
            TaskDBConnection.Connection.selectData(SqlDictionary.findProject, 
                (SQLiteDataReader reader) =>
            {
                while (reader.Read())
                {
                    string prjName = reader["PROJECT"].ToString();
                    if (!string.IsNullOrEmpty(prjName))
                    {
                        SimpleDataBean bean = new SimpleDataBean(prjName, prjName);
                        retList.Add(bean);
                    }
                }
            });
            return retList;
        }
        /// <summary>
        /// タスクにタグを付与します。
        /// </summary>
        /// <param name="targetSortID"></param>
        /// <param name="tag"></param>
        /// <returns></returns>
        public bool updateTagofTask(int targetSortID, string tag)
        {
            string tagLikeStr = '%' + tag + '%';
            List<object> paramList = new List<object>() { tag, tagLikeStr, tag, targetSortID };
            return TaskDBConnection.Connection.updateData(SqlDictionary.addTag, paramList);
        }
        /// <summary>
        /// タグ一覧を取得します。取得時点でC#側でカンマ区切りを配慮して分解します。
        /// </summary>
        /// <returns></returns>
        public List<SimpleDataBean> getTagList()
        {
            List<string> wkList = new List<string>();
            TaskDBConnection.Connection.selectData(SqlDictionary.findTags,
                (SQLiteDataReader reader) =>
                {
                    while (reader.Read())
                    {
                        string tagName = reader["TAGS"].ToString();
                        foreach (string eachTag in tagName.Split(','))
                        {
                            wkList.Add(eachTag);
                        }
                    }
                });

            //重複を排除
            string[] wkArray = wkList.Distinct().ToArray();

            List<SimpleDataBean> retList = new List<SimpleDataBean>();
            foreach (string tag in wkArray)
            {
                retList.Add(new SimpleDataBean(tag, tag));
            }
            return retList;
            
        }

        /// <summary>
        /// タグを全クリアします。
        /// </summary>
        /// <param name="targetSortID"></param>
        /// <returns></returns>
        public bool clearTagOfTask(int targetSortID)
        {
            List<object> paramList = new List<object>() { targetSortID };
            return TaskDBConnection.Connection.updateData(SqlDictionary.clearTags, paramList);
        }
    }
}
