/*
 mbuffer -- buffer module
 Copyright (c) 2005,2007 Hitachi,Ltd.,
 Created by Satoru Moriya <satoru.moriya.br@hitachi.com>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <linux/module.h>
#include <linux/vmalloc.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>
#include <asm/fcntl.h>

#include "mbuffer_arch.h"
#include "mbuffer.h"
#include "mcontrol.h"

#ifdef _IA64_KSTRAX_
static atomic64_t kstrax_serial = ATOMIC64_INIT(0);
#else  /* _IA64_KSTRAX_ */
static atomic_t kstrax_serial = ATOMIC_INIT(0);
#endif /* _IA64_KSTRAX_ */

//static kstrax_serial_t kstrax_serial;

#define KSTRAX_WAKEUP    1
#define KSTRAX_SLEEP     0

/*
 *  global variable
 */
trace_stat_t kstrax_status;

/* smp code */
static DEFINE_PER_CPU(buf_head_t, kernel_buffer_head) = { NULL };

/* tasklet function */
static void kstrax_wake_up_tasklet(unsigned long addr)
{
	wake_up_interruptible((wait_queue_head_t *)addr);
}

/*--------------------------------------------------------------------------
 * module interface
 */
int init_mbuffer(int entry)
{
	__u32 cpu;
	
	if (entry <= 0) {
		printk("invalid parameter(kbuffer_entry)\n");
		return -EINVAL;
	}

        /* initialize buffer header and tasklet for each cpu */
        for_each_online_cpu(cpu) {
		buf_head_t *head;

                head = &per_cpu(kernel_buffer_head, cpu);
     	
		head->buffer=(sys_call_t*)vmalloc(sizeof(sys_call_t) * entry);
		if (head->buffer == NULL) {
			return -ENOMEM;
		}
		head->size = entry;
		head->w_index = 0;
		head->r_index = 0;
		head->limit_index = -1;
		head->wakeup_index = -1;
		head->wakeup_flag = KSTRAX_WAKEUP;
		init_waitqueue_head(&head->wq_head);
		tasklet_init(&head->kst_tasklet, kstrax_wake_up_tasklet,
			     (unsigned long)&head->wq_head);
	}	
	return 0;
}

/* clear buffer header */
void clear_mbuffer(void)
{
	__u32 cpu;
	buf_head_t *head;
	
	for_each_online_cpu(cpu) {
		head = &per_cpu(kernel_buffer_head, cpu);
		if (head->buffer != NULL)
			vfree(head->buffer);
	}
}

int kstrax_init_status(int nr_entry)
{
	int i;

	kstrax_status.nr_cpu = num_online_cpus();
	kstrax_status.nr_syscalls = NR_syscalls;
	kstrax_status.flag_spec_syscall = 0;
	for (i = 0; i < NR_TRACE_PROCESS_MAX; i++)
		kstrax_status.trace_pid[i] = INIT_PID;
	for (i = 0; i < NR_syscalls; i++)
		kstrax_status.trace_syscall[i] = 0;

	__kstrax_init_ipc_socketcall(&kstrax_status);

	kstrax_status.nr_buf_entry = nr_entry;

	return 0;
}

void pre_sys_call_record(const ctr_arg_t args)
{
	int i;
	buf_head_t *head;
	__u32 cpu;  /* unsigned int */
	struct list_head *w_list;
	sys_call_t *buf;

	/* check tracing pid or not */
	if (kstrax_status.trace_pid[0] != INIT_PID) {
		for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
			if (current->pid == kstrax_status.trace_pid[i]) {
				break;
			} else if (kstrax_status.trace_pid[i] == INIT_PID) {
				return;
			}
		}
	}

	if (kstrax_status.flag_spec_syscall != 0) {
		if(__check_tracing_syscall(&args, &kstrax_status) == NOT_TRACE)
			return;
	}

	/* buffer header of current cpu */
	cpu = get_cpu();
	head = &per_cpu(kernel_buffer_head, cpu);
	
	buf = &head->buffer[head->w_index];
	w_list = &head->wq_head.task_list;

	/* update index */	
	head->w_index = (head->w_index + 1) % head->size;
	if (head->w_index == head->r_index) {
		head->r_index = (head->r_index + 1) % head->size;
	}
	
	__record_sys_call_information_pre(&args, buf, &kstrax_serial, cpu);

	if (head->wakeup_flag == KSTRAX_SLEEP && 
	    (head->wakeup_index == head->w_index ||
	     kstrax_force_flag == 1)) {
		head->wakeup_flag = KSTRAX_WAKEUP;

		if (!list_empty(w_list)) {
			tasklet_schedule(&head->kst_tasklet);
		}
	}
	put_cpu();
}

void post_sys_call_record(const ctr_arg_t args)
{
	int i;
	buf_head_t *head;
	__u32 cpu;  /* unsigned int */
	struct list_head *w_list;
	sys_call_t *buf;

	/* check tracing pid or not */
	if (kstrax_status.trace_pid[0] != INIT_PID) {
		for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
			if (kstrax_status.trace_pid[i] == current->pid) {
				break;
			} else if (kstrax_status.trace_pid[i] == INIT_PID) {
				return ;
			}
		}
	}

	if (kstrax_status.flag_spec_syscall != 0) {
		if(__check_tracing_syscall(&args, &kstrax_status) == NOT_TRACE)
			return;
	}

	/* buffer header of current cpu */
	cpu = get_cpu();
	head = &per_cpu(kernel_buffer_head, cpu);  
	
	buf = &head->buffer[head->w_index];
	w_list = &head->wq_head.task_list;

	__kstrax_get_filename(&args, head, &buf);

	/* update index */
	head->w_index = (head->w_index + 1) % head->size;
	if (head->w_index == head->r_index) {
		head->r_index = (head->r_index + 1) % head->size;
	}

	__record_sys_call_information_post(&args, buf, &kstrax_serial, cpu);

	if (head->wakeup_flag == KSTRAX_SLEEP &&
	    (head->wakeup_index == head->w_index ||
	     kstrax_force_flag == 1)) {
		head->wakeup_flag = KSTRAX_WAKEUP;
		
		if (!list_empty(w_list)) { 
			tasklet_schedule(&head->kst_tasklet);
		}
	}
	put_cpu();
}

/*--------------------------------------------------------------------------------
 *  called by ioctl function
 */
static int unregister_trace_pid(pid_t, int);
static void kstrax_trace_all(void);
static void kstrax_trace_file(void);
static void kstrax_trace_process(void);
static void kstrax_trace_signal(void);
static void kstrax_trace_demo(void);

int kstrax_copy_status(trace_stat_t *status)
{
	int retval = 0;

	if (copy_to_user((trace_stat_t *)status, &kstrax_status, 
			 sizeof(trace_stat_t))) {
	      	printk(KERN_ERR "error(ioctl) : cannot copy date\n");
		retval = -EFAULT;
	}
	return retval;
}

/* reset read index */
int kstrax_reset_index(void)
{
	__u32 cpu;
	
	for_each_online_cpu(cpu) {
		buf_head_t *head;
		head = &per_cpu(kernel_buffer_head, cpu);
		head->r_index = head->w_index;
	}
	return 0;
}

/* user_buf is not pointer to int but sys_call_t */
int kstrax_read_basetime(int *buf)
{
	int retval = 0;
	sys_call_t tm_info;
	__set_basetime(&tm_info);
	if (copy_to_user((sys_call_t *)buf, &tm_info, sizeof(sys_call_t))) {
		printk(KERN_ERR "error(ioctl) : cannot copy basetime");
		retval = -EFAULT;
	}
	return retval;
}

int kstrax_pid_spec(pid_t pid)
{
	int i;
	int retval = 0;

	if (pid < -1) {
		retval = -EINVAL;
	} else if (pid == -1) {
		/* delete all pid from trace table --> from now trace all pid */
		for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
			kstrax_status.trace_pid[i] = INIT_PID;
		}
	} else {
		/* 0 <= pid */
		for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
			if (kstrax_status.trace_pid[i] == pid) {
				/* delete pid from trace table */
				retval = unregister_trace_pid(pid, i);
				break;
			} else if (kstrax_status.trace_pid[i] == INIT_PID) {
				/* add pid to trace table */
				kstrax_status.trace_pid[i] = pid;
				break;
			}
		}
		if (i == NR_TRACE_PROCESS_MAX) {
			printk("error:cannot register pid"
			       "(too much pid registered)\n");
			retval = -EINVAL;
		}
	}
	return retval;
}

static int unregister_trace_pid(pid_t pid, int index)
{
	int j;
	
	for (j = index + 1; j < NR_TRACE_PROCESS_MAX; j++) {
		if (kstrax_status.trace_pid[j] == INIT_PID)
			break;
	}
	
	if (j == index + 1) {
		kstrax_status.trace_pid[index] = INIT_PID;
	} else {
		kstrax_status.trace_pid[index] = kstrax_status.trace_pid[j-1];
		kstrax_status.trace_pid[j-1] = INIT_PID;
	}
	
	return 0;
}

int kstrax_syscall_spec(int syscall_num)
{
	int retval = 0;

	if (syscall_num < 0) { /* specified by syscall type */
		switch (syscall_num) {
		case TRACE_ALL:
			kstrax_trace_all();
			break;
		case TRACE_FILE:
			kstrax_trace_file();
			break;
		case TRACE_NETWORK:
			__kstrax_trace_socketcall(TRACE_NETWORK, &kstrax_status);
			break;
		case TRACE_IPC:
			__kstrax_trace_ipc(TRACE_IPC, &kstrax_status);
			break;
		case TRACE_PROCESS:
			kstrax_trace_process();
			break;
		case TRACE_SIGNAL:
			kstrax_trace_signal();
			break;
		case TRACE_DEMO:
			kstrax_trace_demo();
			break;
		default:
			retval = __kstrax_syscall_spec_default(syscall_num, 
							       &kstrax_status);			
		}
	} else { /* syscall_num >= 0 */
		__kstrax_syscall_spec_by_syscall_num(syscall_num, &kstrax_status);
	}
	return retval;
}

static void kstrax_trace_all(void)
{
	int i;

	kstrax_status.flag_spec_syscall = 0;
	for (i = 0; i < NR_syscalls; i++)
		kstrax_status.trace_syscall[i] = 0;
	__kstrax_init_ipc_socketcall(&kstrax_status);
}

static void kstrax_trace_file(void)
{
	int i;
	if (kstrax_status.flag_spec_syscall == 1 && SPECIFIED_FILE == 1) {
		/* stop tracing FILE system calls */
		for (i = 0; i < NR_syscalls; i++) {
			IS_FILE_SYSCALL(i) 
				kstrax_status.trace_syscall[i] = 0;
		}
		test_flag_spec_syscall();	
	} else {
		/* start tracing FILE system calls */
		for (i = 0; i < NR_syscalls; i++) {
			IS_FILE_SYSCALL(i)
				kstrax_status.trace_syscall[i] = 1;
		}
		kstrax_status.flag_spec_syscall = 1;
	}
}

static void kstrax_trace_process(void)
{
	int i;
	if (kstrax_status.flag_spec_syscall == 1 &&
	    SPECIFIED_PROCESS == 1) {
		/* stop tracing PROCESS management system calls */
		for (i = 0; i < NR_syscalls; i++) {
			IS_PROCESS_SYSCALL(i) 
				kstrax_status.trace_syscall[i] = 0;
		}
		test_flag_spec_syscall();	
	} else {
		/* start tracing PROCESS management system calls */
		for (i = 0; i < NR_syscalls; i++) {
			IS_PROCESS_SYSCALL(i)
				kstrax_status.trace_syscall[i] = 1;
		}
		kstrax_status.flag_spec_syscall = 1;
	}
}

static void kstrax_trace_signal(void)
{
	int i;
	if (kstrax_status.flag_spec_syscall == 1 &&
	    SPECIFIED_SIGNAL == 1) {
		/* stop tracing SIGNAL system calls */
		for (i = 0; i < NR_syscalls; i++) {
			IS_SIGNAL_SYSCALL(i) 
				kstrax_status.trace_syscall[i] = 0;
		}
		test_flag_spec_syscall();	
	} else {
		/* start tracing SIGNAL system calls */
		for (i = 0; i < NR_syscalls; i++) {
			IS_SIGNAL_SYSCALL(i)
				kstrax_status.trace_syscall[i] = 1;
		}
		kstrax_status.flag_spec_syscall = 1;
	}
}

static void kstrax_trace_demo(void)
{
	int i;
	if (kstrax_status.flag_spec_syscall == 1 &&
	    SPECIFIED_DEMO == 1) {
		/* stop tracing system calls for demo */
		for (i = 0; i < NR_syscalls; i++) {
			IS_DEMO_SYSCALL(i)
				kstrax_status.trace_syscall[i] = 0;
		}
		test_flag_spec_syscall();
	} else {
		/* start tracing system calls for demo */
		for (i = 0; i < NR_syscalls; i++) {
			IS_DEMO_SYSCALL(i)
				kstrax_status.trace_syscall[i] = 1;
		}
		kstrax_status.flag_spec_syscall = 1;
	}
}

static int kstrax_trace_each_syscall(int syscall_num)
{	
	int retval = 0;

	if (syscall_num > NR_syscalls) {
		printk("error(syscall_spec):bad argument\n");
		retval = -EINVAL;
	} else {
		if (kstrax_status.trace_syscall[syscall_num] == 0) {
			kstrax_status.trace_syscall[syscall_num] = 1;
		} else {
			kstrax_status.trace_syscall[syscall_num] = 0;
		}
		test_flag_spec_syscall();
	}
	return retval;
}

static void test_flag_spec_syscall(void)
{
	int i, nr_spec_syscalls = 0;
	
	for (i = 0; i < NR_syscalls; i++)
		nr_spec_syscalls += kstrax_status.trace_syscall[i];
	if (nr_spec_syscalls == 0)
		kstrax_status.flag_spec_syscall = 0;
	else
		kstrax_status.flag_spec_syscall = 1;
}

int kstrax_set_current_index(void)
{
	__u32 cpu;
	buf_head_t *head;

	for_each_online_cpu(cpu) {
		head = &per_cpu(kernel_buffer_head, cpu);
		head->limit_index = head->w_index;
	}
	return 0;
}

/*----------------------------------------------------------------------------------
 *  read
 */
ssize_t kstrax_read(struct file *filp, char __user *buf, 
		     size_t count, loff_t *f_pos)
{
	int request_nr_entry, copied_nr_entry;
	int start = 0, end = 0;
	int copy_ret;
	__u32 cpu;  /* unsigned int */
	buf_head_t *head;
	
	if (filp ==  NULL || buf == NULL || f_pos == NULL) 
		return -EINVAL;

	if (current->tgid != kstrax_owner)
		return -EPERM;

	/* buffer header of current cpu */
	cpu = smp_processor_id();
	head = &per_cpu(kernel_buffer_head, cpu);

	/* translate byte into index */
	request_nr_entry = count / sizeof(sys_call_t);

 retry:
	start = head->r_index;
	end = head->w_index;
	
	if (start > end)
		end = head->size;

	copied_nr_entry = end - start;
	if (request_nr_entry < copied_nr_entry) {
		end = (start + request_nr_entry) % head->size;
		copied_nr_entry = end - start;
	}

	if (start < head->limit_index && head->limit_index < end) {
		end = head->limit_index;
		copied_nr_entry = end - start;
	}

	/* copy */
	copy_ret = copy_to_user(buf, (head->buffer + start), 
				sizeof(sys_call_t) * copied_nr_entry);
	if (copy_ret) {
		printk("Error at copy_to_user, cpu:%d\n", smp_processor_id());
		goto out;
	}

	request_nr_entry -= copied_nr_entry;
	head->r_index = end % head->size;
	buf += copied_nr_entry * sizeof(sys_call_t);
	*f_pos += copied_nr_entry * sizeof(sys_call_t);

	/* if kstrax command is executed with -R option --> limit_index != -1
	 * This option read kernel buffer only one round.
	 */
	if (head->limit_index != -1 && head->limit_index == end)
		goto out;

	/* if need one more copy */
	if (0 < request_nr_entry) {
		int r_to_w, r_to_up;
		
		/* calculate distance from r_index to wake up */
		r_to_up = head->size / 2;
		if (r_to_up > request_nr_entry)
			r_to_up = request_nr_entry;
		
		if (head->w_index >= head->r_index) {
			r_to_w = head->w_index - head->r_index;
		} else {
			r_to_w = head->size - head->r_index;
			r_to_w += head->w_index;
		}

		/* sleep */
		if (r_to_w < r_to_up) {
			/* termination */
			if (filp->f_flags & O_NONBLOCK) {
				if (head->r_index == head->w_index) 
					goto out;
				goto retry;
			}

			head->wakeup_index = (head->r_index + r_to_up) % head->size;
			head->wakeup_flag = KSTRAX_SLEEP;

			wait_event_interruptible(head->wq_head, 
						 head->wakeup_flag == KSTRAX_WAKEUP); 
		}
		goto retry;
	}
	
 out:
	return (count - ((ssize_t)request_nr_entry * sizeof(sys_call_t)));
}
