/*
 kstrax_buffer
 Copyright (c) 2005,2006 Hitachi,Ltd.,
 Created by Satoru Moriya <s-moriya@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <pthread.h>
#include <limits.h>
#include <sys/time.h>

#include "kstrax.h"
#include "kstrax_buf.h"
#include "syscall_name.h"
#include "kstrax_ioc.h"
#include "kstrax_syscall_list.h"

#define READ_FILE   1
#define WRITE_FILE  2
#define KS_BASETIME -1
#define KS_FILE_END -2
#define KS_FINISHED -3
#define BIN_HEAD_ID 893
#define ERR_LIMIT   530

/* max value of long long int */
#define LLONG_MAX         ((unsigned long long)(1LL << 63) - 1)

#define kstrax_for_each_online_cpu(cpu, max_nr_cpu) \
for (cpu = 0; cpu < max_nr_cpu; cpu++)

typedef struct kstrax_thread_param {
	pthread_t th;
	int cpu_id;
	int input_fd;
	int output_fd;
	int nr_entry;
	int limit;
} tparam_t;

typedef struct sys_call_info{
	pid_t pid;
	short sys_call_number;
	unsigned long time;
	unsigned long utime;
	union {
		long arg_1;
		long return_value;
	};
	long arg_2;
	long arg_3;
	long arg_4;
	long arg_5;
	long arg_6;
	unsigned long serial; /* unsigned long */
	int cpu;
} buffer_t;

typedef struct kstrax_basetime {
	long long g_t_d;         /* time of gettimeofday */
	long long t_s_c;         /* time of tsc */
	long khz;                /* cpu freq (khz) */
} basetime_t;

typedef struct kstrax_per_cpu {
	buffer_t buf;
	basetime_t basetime;
	int fd;
} per_cpu_t;

typedef struct kstrax_list_entry {
	struct kstrax_list_entry *next;
	buffer_t buf;
} kst_list_entry_t;

typedef struct kstrax_list_head {
	kst_list_entry_t *next;
} kst_list_head_t;

typedef struct kstrax_binary_header {
	int id;
	char version[10];
	buffer_t buf_basetime;
} bin_header_t;

typedef struct kstrax_statistics {
	unsigned long count;
	unsigned long nr_error;
	unsigned long nr_no_entry;
	unsigned long nr_no_return;
	long long max_time;
	long long min_time;
	long long total_time;
} statistics_t;

typedef struct kstrax_print_data {
	buffer_t *buf;
	kst_list_entry_t **hash_table;
	kst_list_entry_t *free_list;
	statistics_t *stats;
	FILE *w_fp;
} print_data_t;

/*
 *  global variable
 */
static int read_file_handle;
static tparam_t *kstrax_tparam;

/*---------------------------------------------------------------------------------
 *  signal handler 
 */
static void kstrax_sig_handler(int signum)
{
	int cpu;
	int flags;
	struct kstrax_status sparam;

	if (ioctl(read_file_handle, KSTRAX_IOC_SET_FORCE_WAKEUP)) {
		perror("kstrax_sig_handler(ioctl:force wakeup)");
		exit(1);
	}
	flags = fcntl(read_file_handle, F_GETFL);
	fcntl(read_file_handle, F_SETFL, flags|O_NONBLOCK);
	
	if (ioctl(read_file_handle, KSTRAX_IOC_STATUS, &sparam) < 0) {
		perror("kstrax_sig_handler(ioctl:get status)");
		exit(1);
	}
	kstrax_for_each_online_cpu(cpu, sparam.nr_cpu) {
		pthread_kill(kstrax_tparam[cpu].th, SIGUSR1);
	}
	
	fprintf(stderr, "terminating...\n");
	return;
}

static void kstrax_dummy_handler(int signum)
{
	return;
}

/*-------------------------------------------------------------------------------
 *  file operation
 */
/* open */
static int kstrax_file_open(const char *filename, int flag)
{
	int fd;

	if (flag == READ_FILE) {
		fd = open(filename, O_RDONLY);
	} else if (flag == WRITE_FILE) {
		fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, S_IRWXU);
	} else {
		fprintf(stderr, "Bad Argument : kstrax_file_open\n");
		exit(1);
	}
	
	if (fd < 0) {
		perror("kstrax_file_open(open)");
		exit(1);
	}
	return fd; /* file handler */
}

/*----------------------------------------------------------------------------
 *  read system call information
 *  "-b" option
 */
static void kstrax_read_per_cpu(tparam_t *);
static void output_to_bin_file(int, int, int, int);
static void kstrax_format_bin_file(int, int);

void kstrax_read(int nr_entry, int limit_time, 
		  const char *input_f, const char *output_f)
{
	int cpu, pid;
	struct kstrax_status sparam;

	/* register signal action */
	signal(SIGINT,  &kstrax_sig_handler);
	signal(SIGALRM, &kstrax_sig_handler);
	signal(SIGUSR1, &kstrax_dummy_handler);

	read_file_handle = kstrax_file_open(input_f, READ_FILE); 

	/* set owner process */
	pid = getpid();
	if (ioctl(read_file_handle, KSTRAX_IOC_SET_OWNER, &pid) < 0) {
		perror("kstrax_read(ioctl:set owner)");
		exit(1);
	}
	
	/* reset read index */
	if (kstrax_kflag != 1) {
		if (ioctl(read_file_handle, KSTRAX_IOC_INIT_INDEX) < 0) {
			perror("kstrax_read(ioctl:init index)");
			exit(1);
		}
	}
	
 	/* read system status */
	if (ioctl(read_file_handle, KSTRAX_IOC_STATUS, &sparam) < 0) {
		perror("kstrax_read(ioctl:read status)");
		exit(1);
	}
	
	kstrax_tparam = (tparam_t *)malloc(sizeof(tparam_t) * sparam.nr_cpu);
	if (kstrax_tparam == NULL) {
		perror("kstrax_read(malloc)");
		exit(1);
	}

	/* fork per processor */
	/* create threads */
	kstrax_for_each_online_cpu(cpu, sparam.nr_cpu) {
		char s[1024];
		sprintf(s, "%s_%02d", output_f, cpu);
		kstrax_tparam[cpu].cpu_id    = cpu;
		kstrax_tparam[cpu].input_fd  = read_file_handle;
		kstrax_tparam[cpu].output_fd = kstrax_file_open(s, WRITE_FILE);
		kstrax_tparam[cpu].nr_entry  = nr_entry; 
		kstrax_tparam[cpu].limit     = limit_time;

		pthread_create(&kstrax_tparam[cpu].th, NULL, 
			       (void *)kstrax_read_per_cpu, 
			       (void *)&kstrax_tparam[cpu]);
	}

	pause();
	
	/* wait threads */
	kstrax_for_each_online_cpu(cpu, sparam.nr_cpu) {
		pthread_join(kstrax_tparam[cpu].th, NULL);
		close(kstrax_tparam[cpu].output_fd);
	}		

	/* clear owner process */
	if (ioctl(read_file_handle, KSTRAX_IOC_CLR_OWNER) < 0) {
		perror("kstrax_read(ioctl:clear owner)");
		exit(1);
	}

	close(read_file_handle);
}

/*
 *  this function is executed by each thread
 */
static void kstrax_read_per_cpu(tparam_t *tparam)
{
	sigset_t mask_set;
	if (ioctl(tparam->input_fd, KSTRAX_IOC_SET_CPU, &tparam->cpu_id)) {
		perror("kstrax_read_per_cpu(ioctl:set cpu affinity)");
		exit(1);
	}

	/* block signals */
	if (sigfillset(&mask_set) < 0) {
		perror("kstrax_read_per_cpu(sigfillset)");
		exit(1);
	}
	if (sigdelset(&mask_set, SIGUSR1) < 0) {
		perror("kstrax_read_per_cpu(sigdelset)");
		exit(1);
	}

	if (pthread_sigmask(SIG_SETMASK, &mask_set, NULL) != 0) {
		perror("kstrax_read_per_cpu(pthread_sigmask)");
		exit(1);
	}

	output_to_bin_file(tparam->nr_entry, tparam->limit, 
			   tparam->input_fd, tparam->output_fd);
}

static void output_to_bin_file(int entry, int limit,
			       int read_handle, int write_handle)
{
	ssize_t request_size;
	buffer_t *buf;
		
	/* format write file */
	kstrax_format_bin_file(read_handle, write_handle);
	
	buf = (buffer_t *)malloc(sizeof(buffer_t) * entry);
	if (buf == NULL) {
		perror("output_to_bin_file(malloc)");
		exit(1);
	}

	alarm(limit);

	/* read and then write */
	request_size = entry * sizeof(buffer_t);
	while (1) {
		int read_size, write_size;
		if ((read_size = read(read_handle, buf, request_size)) < 0) {
			perror("output_to_bin_file(read)");
			exit(1);
		}
		if ((write_size = write(write_handle, buf, read_size)) < 0) {
			perror("output_to_bin_file(write)");
			exit(1);
		}
		
		if (read_size != write_size) {
			fprintf(stderr, "Error write failed\n");
			exit(1);
		}

		if (read_size < request_size) break;
	}
	
	/* free user space buffer */
	free(buf);
}

static void kstrax_format_bin_file(int read_handle, int write_handle)
{
	bin_header_t bin_head;
	
	/* init binary file header */
	bin_head.id = BIN_HEAD_ID;
	strncpy(bin_head.version, KSTRAX_VERSION, 10);
	if (ioctl(read_handle, KSTRAX_IOC_READ_BASETIME, 
		  &bin_head.buf_basetime) < 0) {
		perror("kstrax_format_bin_file(ioctl:read basetime)");
		exit(1);
	}

	if (write(write_handle, &bin_head, sizeof(bin_header_t)) 
	    != sizeof(bin_header_t)) {
		perror("kstrax_format_bin_file(write)");
		exit(1);
	}
}

/*--------------------------------------------------------------------------------
 *  print system call information
 *  "-c", "-r", "-t" option
 */
static int kstrax_count_nr_file(const char *);
static void check_binary_header(per_cpu_t *);
static void kstrax_marge_print(per_cpu_t *, int, FILE *);
static basetime_t get_basetime(const buffer_t *);

void kstrax_print(const char *input_f, const char *output_f)
{
	int i;
	int nr_file;
	per_cpu_t *kst_data;
	FILE *w_fp;
	char filename[1024];

	nr_file = kstrax_count_nr_file(input_f);

	kst_data = (per_cpu_t *)malloc(sizeof(per_cpu_t) * nr_file);
	if (kst_data == NULL) {
		perror("kstrax_print(malloc)");
		exit(1);
	}
	for (i = 0; i < nr_file; i++) {
		sprintf(filename, "%s_%02d", input_f, i);
		kst_data[i].fd = kstrax_file_open(filename, READ_FILE);
		
		check_binary_header(&kst_data[i]);
	}

	if (strcmp(output_f, "\0") == 0) {
		w_fp = stdout;
	} else {
		w_fp = fopen(output_f, "w");
		if (w_fp == NULL) {
			perror("kstrax_print(fopen)");
			exit(1);
		}
	}

	kstrax_marge_print(kst_data, nr_file, w_fp);

	if (w_fp != stdout)
		fclose(w_fp);
	for (i = 0; i < nr_file; i++) {
		close(kst_data[i].fd);
	}
	free(kst_data);
}

static int kstrax_count_nr_file(const char *input_f)
{
	char filename[1024];
	int nr_file = 0;
		
	while (1) {
		int success = 0;
		sprintf(filename, "%s_%02d", input_f, nr_file);
		success = open(filename, O_RDONLY);
		if (success < 0) {
			break;
		}
		close(success);
		nr_file++;
	}
	if (nr_file == 0) {
		fprintf(stderr, "file not found\n");
		exit(1);
	}
	return nr_file;
}

static void check_binary_header(per_cpu_t *kst_data)
{
	bin_header_t bin_head;

	/* check binary file header */
	if (read(kst_data->fd, &bin_head, sizeof(bin_header_t)) 
	    != sizeof(bin_header_t)) {
		perror("check_binary_header(read)");
		exit(1);
	}
	if (bin_head.id != BIN_HEAD_ID) {
		fprintf(stderr, "error:invalid header\n");
		exit(1);
	}

	if (strcmp(bin_head.version, KSTRAX_VERSION)) {
		fprintf(stderr, "Warning:Another version binary file\n");
	}
	kst_data->basetime = get_basetime(&bin_head.buf_basetime);
}

static basetime_t get_basetime(const buffer_t *buf)
{
	unsigned long *high, *low;
	basetime_t retval;

	if (buf->pid != KS_BASETIME) {
		fprintf(stderr, "error:invalid header\n");
		exit(1);
	}
	
	retval.g_t_d = (long long)buf->time * 1000000 + (long long)buf->utime; /* mu sec */
	high = (unsigned long *)&buf->arg_1;
	low  = (unsigned long *)&buf->arg_2;
	retval.t_s_c = ((long long)(*high) << 32) | (long long)(*low);
	retval.khz = buf->arg_3;
	
	return retval;
}

/*
 *  print main routine
 */
static void kstrax_read_entry(per_cpu_t *);
static void tsc_to_gtd(per_cpu_t *);
static long long kstrax_long_to_longlong(long, long);
static int kstrax_earliest_entry(const per_cpu_t *, int);
static unsigned long count_lost_syscall(unsigned long, unsigned long);

static void kstrax_pre_print_normal(print_data_t *);
static void kstrax_pre_print_raw(print_data_t *);
static void kstrax_pre_print_stats(print_data_t *);
static void kstrax_main_print_normal(print_data_t *);
static void kstrax_main_print_raw(print_data_t *);
static void kstrax_main_print_stats(print_data_t *);
static void kstrax_post_print_normal(print_data_t *);
static void kstrax_post_print_raw(print_data_t *);
static void kstrax_post_print_stats(print_data_t *);
static void kstrax_lost_syscall_normal(print_data_t *, unsigned long);
static void kstrax_lost_syscall_raw(print_data_t *, unsigned long);
static void kstrax_lost_syscall_stats(print_data_t *, unsigned long);

static void kstrax_marge_print(per_cpu_t *kst_data, int nr_file, FILE *w_fp)
{
	int i;
	int min = 0;
	int option = 0;
	int nr_round = 0;
	unsigned long first_serial, last_serial = 0;
	long long total_syscall = 0,  total_nr_lost = 0;
	print_data_t p_data;
	void (*pre_print[])(print_data_t *) = {kstrax_pre_print_normal, 
					       kstrax_pre_print_raw, 
					       kstrax_pre_print_stats};
	void (*main_print[])(print_data_t *) = {kstrax_main_print_normal,
						kstrax_main_print_raw,
						kstrax_main_print_stats};
	void (*post_print[])(print_data_t *) = {kstrax_post_print_normal,
						kstrax_post_print_raw,
						kstrax_post_print_stats};
	void (*lost_syscall[])(print_data_t *, unsigned long) = {
		kstrax_lost_syscall_normal,
		kstrax_lost_syscall_raw,
		kstrax_lost_syscall_stats};

	if (kstrax_rflag == 1) {
		option = 1;
	} else if (kstrax_cflag == 1) {
		option = 2;
	}

	p_data.w_fp = w_fp;
	(*pre_print[option])(&p_data);
	for (i = 0; i < nr_file; i++) {
		kstrax_read_entry(&kst_data[i]);
		tsc_to_gtd(&kst_data[i]);
	}

	/* record first serial */
	min = kstrax_earliest_entry(kst_data, nr_file);
	first_serial = kst_data[min].buf.serial;
	last_serial = first_serial - 1;

	/* main routine of print data */
	while (1) {
		unsigned long nr_lost = 0;

		min = kstrax_earliest_entry(kst_data, nr_file);
		if (min == KS_FINISHED)
			break;

		if ((nr_lost = 
		     count_lost_syscall(last_serial, kst_data[min].buf.serial)) != 0) {
			total_nr_lost += (long long)nr_lost;
			(*lost_syscall[option])(&p_data, nr_lost);
		}
		
		last_serial = kst_data[min].buf.serial;
		p_data.buf = &kst_data[min].buf;
		(*main_print[option])(&p_data);
		
		kstrax_read_entry(&kst_data[min]);
		tsc_to_gtd(&kst_data[min]);

		if (last_serial == ULONG_MAX)
			nr_round++;
	}

	if (nr_round) {
		total_syscall = (long long)((nr_round - 1) * ULONG_MAX + 
					    (ULONG_MAX - first_serial) + last_serial);
	} else {
		total_syscall = (long long)(last_serial - first_serial);
	}
	
	(*post_print[option])(&p_data);
	fprintf(p_data.w_fp, 
		"\n\ntotal syscalls      :: %lld\n"
		"total lost syscalls :: %lld\n",
		total_syscall / 2, total_nr_lost / 2);
}

static void kstrax_read_entry(per_cpu_t *kst_data)
{
	int retval;

	retval = read(kst_data->fd, &kst_data->buf, sizeof(buffer_t));
	if (retval < 0) {
		perror("kstrax_read_entry(read)");
		exit(1);
	} else if (retval == 0) {
		kst_data->buf.pid = KS_FILE_END;
	} 
}

static void tsc_to_gtd(per_cpu_t *kst_data)
{
	long long tmp_tsc, tmp_gtd;

	tmp_tsc = kstrax_long_to_longlong(kst_data->buf.time, kst_data->buf.utime);
	tmp_tsc -= kst_data->basetime.t_s_c;	
	tmp_tsc = (long long)(tmp_tsc * 10000 / kst_data->basetime.khz + 5) / 10;

	tmp_gtd = kst_data->basetime.g_t_d;
	tmp_gtd += tmp_tsc;

	/* gtd (gettimeofday)*/
	kst_data->buf.time  = (long)(tmp_gtd / (long long)1000000);
	kst_data->buf.utime = (long)(tmp_gtd % (long long)1000000);
}

static long long kstrax_long_to_longlong(long high, long low)
{
	long long retval;
	unsigned long *tmp_h, *tmp_l;

	tmp_h = (unsigned long *)&high;
	tmp_l = (unsigned long *)&low;

	retval = ((long long)(*tmp_h) << 32) | (long long)(*tmp_l);
	return retval;
}

static int kstrax_earliest_entry(const per_cpu_t *kst_data, int nr_file)
{
	int i;
	int min = KS_FINISHED;

	for (i = 0; i < nr_file; i++) {
		if (kst_data[i].buf.pid != KS_FILE_END) {
			min = i;
			break;
		}
	}		
	/* no data for print */
	if (min == KS_FINISHED) {
		goto out;
	}
	
	for (i = min+1; i < nr_file; i++) {
		if (kst_data[i].buf.pid == KS_FILE_END) continue;

		if (kst_data[min].buf.serial > kst_data[i].buf.serial)
			min = i;
#if 0
		if (kst_data[min].buf.time > kst_data[i].buf.time) {
			min = i;
		} else if (kst_data[min].buf.time == kst_data[i].buf.time) {
			if (kst_data[min].buf.utime > kst_data[i].buf.utime) {
				min = i;
			} else if (kst_data[min].buf.utime == kst_data[i].buf.utime) {
				if (kst_data[min].buf.serial > kst_data[i].buf.serial)
					min = i;
			}
		}
#endif 
	}
 out:
	return min;
}

static unsigned long int count_lost_syscall(unsigned long last, unsigned long now)
{
	unsigned long int nr_lost = 0;

	if (now < last) {
		nr_lost = (ULONG_MAX - last + 1) + now;
	} else {
		nr_lost = now - last - 1;
	}
	return nr_lost;
}

/*---------------------------------------------------------------------------------
 *  pre processing for each option
 */
static kst_list_entry_t **kstrax_alloc_hash_table(int);
static statistics_t *kstrax_alloc_statistics_array(int);

/* "-t" */
static void kstrax_pre_print_normal(print_data_t *p_data)
{
	p_data->free_list = NULL;
	p_data->hash_table = kstrax_alloc_hash_table(NR_syscalls);
	p_data->stats = NULL;
	
	fprintf(p_data->w_fp, "----------------------------------------------------------------------------------\n");
	fprintf(p_data->w_fp, " pid     start            end             system_call \"name,argument,return_value\"\n");
	fprintf(p_data->w_fp, "----------------------------------------------------------------------------------\n");
}

/* "-r" */
static void kstrax_pre_print_raw(print_data_t *p_data)
{
	p_data->free_list = NULL;
	p_data->hash_table = NULL;
	p_data->stats = NULL;

	fprintf(p_data->w_fp, "----------------------------------------------------------------------------------------------\n");
	fprintf(p_data->w_fp, "        serial  cpu   pid      time          system_call \"number,name,argument/return_value\"\n");
	fprintf(p_data->w_fp, "----------------------------------------------------------------------------------------------\n");	
}
/* "-c" */
static void kstrax_pre_print_stats(print_data_t *p_data)
{
	p_data->free_list = NULL;
	p_data->hash_table = kstrax_alloc_hash_table(NR_syscalls);
	p_data->stats = kstrax_alloc_statistics_array(NR_syscalls);

	fprintf(p_data->w_fp, "-------------------------------------------------------------------------------------------------------------------------------\n");
	fprintf(p_data->w_fp, " ID  NAME                        count        average            max            min          total   error  no entry  no return\n");
	fprintf(p_data->w_fp, "-------------------------------------------------------------------------------------------------------------------------------\n");
}

static kst_list_entry_t **kstrax_alloc_hash_table(int nr_entry)
{
	int i;
	kst_list_entry_t **hash_table;

	hash_table = (kst_list_entry_t **)malloc(
		sizeof(kst_list_entry_t *) * nr_entry);
	if (hash_table == NULL) {
		perror("kstrax_alloc_hash_table(malloc)");
		exit(1);
	}
	for (i = 0; i < nr_entry; i++) {
		hash_table[i] = NULL;
	}
	return hash_table;
}

static statistics_t *kstrax_alloc_statistics_array(int nr_entry)
{
	int i;
	statistics_t *stats;

	stats = (statistics_t *)malloc(sizeof(statistics_t) * nr_entry);
	if (stats == NULL) {
		perror("kstrax_alloc_statistics_array(malloc)");
		exit(1);
	}
	for (i = 0; i < nr_entry; i++) {
		stats[i].count    = 0;
		stats[i].nr_error = 0;
		stats[i].nr_no_entry  = 0;
		stats[i].nr_no_return = 0;
		stats[i].max_time     = 0;
		stats[i].min_time     =  LLONG_MAX;
		stats[i].total_time   = 0;
	}
	return stats;
}

/*--------------------------------------------------------------------------------
 *  main processing for each option
 */
static void print_call_entry(const buffer_t *, FILE *);
static void print_return_entry(const buffer_t *, FILE *);
static void print_pair_entry(const buffer_t *, const buffer_t *, FILE *);
static void print_arg(const buffer_t *, FILE *);
static kst_list_entry_t *kstrax_copy_entry(const buffer_t *, kst_list_entry_t **);
static void add_pair_entry(const buffer_t *, const buffer_t *, statistics_t *);
static void add_no_pair_entry(const buffer_t *, statistics_t *);
static long long kstrax_calc_processing_time(const buffer_t *, const buffer_t *);
static void kstrax_free_list(kst_list_entry_t **);
static void kstrax_list_add(kst_list_entry_t *, kst_list_entry_t **);
static kst_list_entry_t *kstrax_list_del(kst_list_entry_t **);

/* "-t" */
static void kstrax_main_print_normal(print_data_t *p_data)
{
	int hash;
	buffer_t *buf;
	kst_list_entry_t *copy_entry;
	kst_list_entry_t **free_list, **hash_table;
	
	buf = p_data->buf;
	free_list = &p_data->free_list;
	hash_table = p_data->hash_table;

	if (buf->sys_call_number > 0) { /* call entry */
		if (buf->sys_call_number == EXIT ||
		    buf->sys_call_number == SIGRETURN ||
		    buf->sys_call_number == RT_SIGRETURN ||
		    buf->sys_call_number == EXIT_GROUP) {
			/* these have no return entry */
			print_call_entry(buf, p_data->w_fp);
			return;
		}	
		copy_entry = kstrax_copy_entry(buf, free_list);
		hash = copy_entry->buf.sys_call_number % NR_syscalls;
		kstrax_list_add(copy_entry, &hash_table[hash]);
	} else { /* return entry */
		kst_list_entry_t **now;

		hash = (-buf->sys_call_number) % NR_syscalls;
		now = &hash_table[hash];
		
		while (*now) {
			if ((*now)->buf.pid == buf->pid) {
				kst_list_entry_t *del;
				print_pair_entry(&(*now)->buf, buf, p_data->w_fp);
				del = kstrax_list_del(now);
				kstrax_list_add(del, free_list);
				return;
			} else {
				now = &(*now)->next;
			}
		}
		if (*now == NULL) {
			if (buf->sys_call_number == 0) {
				copy_entry = kstrax_copy_entry(buf, 
								free_list);
				kstrax_list_add(copy_entry, &hash_table[hash]);
				return;
			}
			/* there is not call entry but only return entry */
			print_return_entry(buf, p_data->w_fp);
		}
	}
}

/* "-r" */
static void kstrax_main_print_raw(print_data_t *p_data)
{
	struct tm exe_time;
	buffer_t *buffer;
	
	buffer = p_data->buf;
	
	if(NR_syscalls < abs(buffer->sys_call_number)) {
		fprintf(p_data->w_fp, "Unknown syscall number : %d\n", 
			buffer->sys_call_number);
		return;
	}
#if 0	
	/* if syscall is ipc or socketcall call special function */
	if (abs(buffer->sys_call_number) == IPC) {
		print_ipc_raw(p_data); /* not implemented */
		return ;
	} else if (abs(buffer->sys_call_number) == SOCKETCALL) {
		print_socketcall_raw(p_data); /* not implemented */
		return ;
	}
#endif

	exe_time = *localtime((const time_t *)&buffer->time);	
	if (buffer->sys_call_number < 0) {
		/* return entry */
		fprintf(p_data->w_fp, "RAW   %8ld %3d  %5hd  "
			"%02d:%02d:%02d.%06ld  %4hd  <%s(",
		       buffer->serial,
		       buffer->cpu,
		       buffer->pid,
		       exe_time.tm_hour,
		       exe_time.tm_min,
		       exe_time.tm_sec,
		       buffer->utime,
		       -buffer->sys_call_number,
		       syscall_name[-buffer->sys_call_number].name);
		print_arg(buffer, p_data->w_fp);
		if (-ERR_LIMIT <= buffer->return_value && buffer->return_value < 0) {
			fprintf(p_data->w_fp, ") = -1 (errno %d)>\n", 
				abs(buffer->return_value));
		} else {
			fprintf(p_data->w_fp, ") = 0x%x>\n", 
			       (unsigned int)buffer->return_value);
		}
	} else {
		fprintf(p_data->w_fp, "RAW   %8ld %3d  %5hd  "
		       "%02d:%02d:%02d.%06ld  %4hd  %s(",
		       buffer->serial,
		       buffer->cpu,
		       buffer->pid,
		       exe_time.tm_hour,
		       exe_time.tm_min,
		       exe_time.tm_sec,
		       buffer->utime,
		       buffer->sys_call_number,
		       syscall_name[buffer->sys_call_number].name);
		print_arg(buffer, p_data->w_fp);
		fprintf(p_data->w_fp, ") = ??\n");
	}
}

static void kstrax_main_print_stats(print_data_t *p_data)
{
	int hash;
	buffer_t *buffer;
	kst_list_entry_t *copy_entry;
	kst_list_entry_t **free_list, **hash_table;
	statistics_t *stats;

	buffer = p_data->buf;
	free_list = &p_data->free_list;
	hash_table = p_data->hash_table;
	stats = p_data->stats;

	if (buffer->sys_call_number > 0) { /* call entry */
		if (buffer->sys_call_number == EXIT ||
		    buffer->sys_call_number == SIGRETURN ||
		    buffer->sys_call_number == RT_SIGRETURN ||
		    buffer->sys_call_number == EXIT_GROUP) {
			/* these have no return entry */
			add_no_pair_entry(buffer, stats);
			return;
		}	
		copy_entry = kstrax_copy_entry(buffer, free_list);
		hash = copy_entry->buf.sys_call_number % NR_syscalls;
		kstrax_list_add(copy_entry, &hash_table[hash]);
	} else { /* return entry */
		kst_list_entry_t **now;

		hash = (-buffer->sys_call_number) % NR_syscalls;
		now = &hash_table[hash];
		
		while (*now) {
			if ((*now)->buf.pid == buffer->pid) {
				kst_list_entry_t *del;
				add_pair_entry(&(*now)->buf, buffer, stats);
				del = kstrax_list_del(now);
				kstrax_list_add(del, free_list);
				return;
			} else {
				now = &(*now)->next;
			}
		}
		if (*now == NULL) {
			if (buffer->sys_call_number == 0) {
				copy_entry = kstrax_copy_entry(buffer, free_list);
				kstrax_list_add(copy_entry, &hash_table[hash]);
				return;
			}
			/* there is not call entry but only return entry */
			add_no_pair_entry(buffer, stats);
		}
	}
}

static void print_call_entry(const buffer_t *buf, FILE *w_fp)
{
	struct tm call_time;
	
	if (NR_syscalls < abs(buf->sys_call_number)) {
		fprintf(w_fp, "Unknown syscall number : %d\n", 
			buf->sys_call_number);
		return;
	}

	call_time = *localtime((const time_t *)&buf->time);

	fprintf(w_fp, "%5hd  %02d:%02d:%02d.%06ld  --:--:--.------   %s(",
	       buf->pid,
	       call_time.tm_hour,
	       call_time.tm_min,
	       call_time.tm_sec,
	       buf->utime,
	       syscall_name[buf->sys_call_number].name);
	print_arg(buf, w_fp);
	fprintf(w_fp, ") = ??\n");
}

static void print_return_entry(const buffer_t *buf, FILE *w_fp)
{
	struct tm ret_time;

	if (NR_syscalls < abs(buf->sys_call_number)) {
		fprintf(w_fp, "Unknown syscall number : %d\n", -buf->sys_call_number);
		return;
	}

	ret_time = *localtime((const time_t *)&buf->time); 

	fprintf(w_fp, "%5hd  --:--:--.------  %02d:%02d:%02d.%06ld   %s(------",
	       buf->pid,
	       ret_time.tm_hour,
	       ret_time.tm_min,
	       ret_time.tm_sec,
	       buf->utime,
	       syscall_name[-buf->sys_call_number].name);
	if (-ERR_LIMIT <= buf->return_value && buf->return_value < 0) {
		fprintf(w_fp, ") = -1 (%s)\n", strerror(abs(buf->return_value)));
	} else {
		fprintf(w_fp, ") = 0x%x\n", (unsigned int)buf->return_value);
	}
}

static void print_pair_entry(const buffer_t *call, const buffer_t *ret, FILE *w_fp)
{
	struct tm call_time, ret_time;

	if (NR_syscalls < abs(call->sys_call_number)) {
		fprintf(w_fp, "Unknown syscall number : %d\n", call->sys_call_number);
		return;
	}

	call_time = *localtime((const time_t *)&call->time);
	ret_time = *localtime((const time_t *)&ret->time);

	fprintf(w_fp, "%5hd  %02d:%02d:%02d.%06ld  %02d:%02d:%02d.%06ld   %s(",
	       call->pid,
	       call_time.tm_hour,
	       call_time.tm_min,
	       call_time.tm_sec,
	       call->utime,
	       ret_time.tm_hour,
	       ret_time.tm_min,
	       ret_time.tm_sec,
	       ret->utime,
	       syscall_name[call->sys_call_number].name);
	print_arg(call, w_fp);
	if (-ERR_LIMIT <= ret->return_value && ret->return_value < 0) {
		fprintf(w_fp, ") = -1 (%s)\n", strerror(abs(ret->return_value)));
	} else {
		fprintf(w_fp, ") = 0x%x\n", (unsigned int)ret->return_value);
	}
}

static void print_arg(const buffer_t *buf, FILE *w_fp)
{
	int i;
	long *ptr;
	int num = buf->sys_call_number;
	
	if (num < 0) {
		fprintf(w_fp, " --- ");
	} else {
		for(i = 0; i < syscall_name[num].args_number; i++) {
			if (i == 0) 
				ptr = (long *)&buf->arg_1;
			else
				fprintf(w_fp, ", ");
			
			fprintf(w_fp, "0x%lx", *ptr);
			ptr++;
		}
	}
}

static kst_list_entry_t *kstrax_copy_entry(const buffer_t *buf, 
					    kst_list_entry_t **free_list)
{
	kst_list_entry_t *ret;
	
	/* create cache */
	if (*free_list == NULL) {
		int i;
		for (i = 0; i < 50; i++) { /* create 50 entry for one time */
			kst_list_entry_t *tmp;
			tmp = (kst_list_entry_t *)malloc(sizeof(kst_list_entry_t));
			if (tmp == NULL) {
				perror("kstrax_copy_entry(malloc)");
				exit(1);
			}
			if (i == 0) {
				tmp->next = NULL;
			} else {
				tmp->next = *free_list;
			}
			*free_list = tmp;
		}
	}

	ret = *free_list;
	*free_list = ret->next;
	ret->buf = *buf;
	ret->next = NULL;
	return ret;
}

static void add_pair_entry(const buffer_t *call, const buffer_t *ret, 
			   statistics_t *stats)
{
	int number = call->sys_call_number;
	long long proc_time;

	if (NR_syscalls < abs(number)) {
		fprintf(stderr, "Unknown syscall number : %d\n", number);
		return;
	}

	proc_time = kstrax_calc_processing_time(call, ret);

	stats[number].count++;
	stats[number].total_time += proc_time;
	if (stats[number].max_time < proc_time) 
		stats[number].max_time = proc_time;
	if (stats[number].min_time > proc_time)
		stats[number].min_time = proc_time;
	if (-ERR_LIMIT <= ret->return_value && ret->return_value < 0)
		stats[number].nr_error++;	
}

static void add_no_pair_entry(const buffer_t *buf, statistics_t *stats)
{
	int number = buf->sys_call_number;
	
	if (NR_syscalls < abs(number)) {
		fprintf(stderr, "Unknown syscall number : %d\n", abs(number));
		return;
	}

	if (number > 0)
		stats[number].nr_no_return++;
	else
		stats[-number].nr_no_entry++;

	stats[abs(number)].count++;
}

static long long kstrax_calc_processing_time(const buffer_t *call, 
					      const buffer_t *ret)
{
	long long call_time, ret_time;

	call_time = (long long)call->time * 1000000 + (long long)call->utime;
	ret_time  = (long long)ret->time * 1000000 + (long long)ret->utime;

	return (ret_time - call_time);
}


/*---------------------------------------------------------------------------------
 *  post processing for each option
 */

static void kstrax_flush_hash_table(print_data_t *);
static void kstrax_stats_flush_hash_table(print_data_t *);
static void kstrax_stats_print(print_data_t *);

/* "-t" */
static void kstrax_post_print_normal(print_data_t *p_data)
{
	kstrax_flush_hash_table(p_data);
	kstrax_free_list(&p_data->free_list);
	free(p_data->hash_table);
}

/* "-r" */
static void kstrax_post_print_raw(print_data_t *p_data)
{
	/* dummy function */
}

/* "-c" */
static void kstrax_post_print_stats(print_data_t *p_data)
{
	kstrax_stats_flush_hash_table(p_data);
	kstrax_stats_print(p_data);

	free(p_data->stats);
	kstrax_free_list(&p_data->free_list);
	free(p_data->hash_table);
}

static void kstrax_flush_hash_table(print_data_t *p_data)
{
	int i;

	while (1) {
		int count = 0;
		kst_list_entry_t **tmp, **min = NULL;

		for (i = 0; i < NR_syscalls; i++) {
			tmp = &p_data->hash_table[i];
			if (*tmp == NULL) {
				count++;
				continue;
			}

			while ((*tmp)->next != NULL) {
				tmp = &(*tmp)->next;
			}

			if (min == NULL || 
			    (*min)->buf.serial > (*tmp)->buf.serial) {
				min = tmp;
			} 
		}
		
		/* hash_table is empty */
		if (count == NR_syscalls) break;

		print_call_entry(&(*min)->buf, p_data->w_fp);
		kstrax_list_add(*min, &p_data->free_list);
		*min = NULL;
	}
}

static void kstrax_stats_flush_hash_table(print_data_t *p_data)
{
	int i;

	while (1) {
		int count = 0;
		kst_list_entry_t **tmp, **min = NULL;

		for (i = 0; i < NR_syscalls; i++) {
			tmp = &p_data->hash_table[i];
			
			if (*tmp == NULL) {
				count++;
				continue;
			}

			while ((*tmp)->next != NULL) {
				tmp = &(*tmp)->next;
			}

			if (min == NULL || 
			    (*min)->buf.serial > (*tmp)->buf.serial) {
				min = tmp;
			} 
		}
		
		/* hash_table is empty */
		if (count == NR_syscalls) break;

		add_no_pair_entry(&(*min)->buf, p_data->stats);
		kstrax_list_add(*min, &p_data->free_list);
		*min = NULL;
	}
}

static void kstrax_stats_print(print_data_t *p_data)
{
	int i;
	statistics_t *stats = p_data->stats;
	unsigned long success;
	long long average;
	
	for (i = 0; i < NR_syscalls; i++) {
		if (stats[i].count == 0)
			continue;
		
		fprintf(p_data->w_fp, "%3d  %-24s  %7lu  ", 
			i, syscall_name[i].name, 
			stats[i].count);  
		if (stats[i].count  == 
		    stats[i].nr_no_entry + stats[i].nr_no_return) {
			int j;
			for (j = 0; j < 4; j++) 
				fprintf(p_data->w_fp, "------.------  ");
		} else {
			success = stats[i].count - 
				stats[i].nr_no_entry -
				stats[i].nr_no_return;
			average = stats[i].total_time / success;
			fprintf(p_data->w_fp, "%6lld.%06u  ",
				average / 1000000,
				(unsigned int)average % 1000000);
			fprintf(p_data->w_fp, "%6lld.%06u  ",
				stats[i].max_time / 1000000,
				(unsigned int)stats[i].max_time % 1000000);
			fprintf(p_data->w_fp, "%6lld.%06u  ",
				stats[i].min_time / 1000000,
				(unsigned int)stats[i].min_time % 1000000);
			fprintf(p_data->w_fp, "%6lld.%06u  ", 
				stats[i].total_time / 1000000,
				(unsigned int)stats[i].total_time % 1000000);
		}
		
		fprintf(p_data->w_fp, "%6lu  %6lu  %6lu\n",
			stats[i].nr_error,
			stats[i].nr_no_entry,
			stats[i].nr_no_return);
	}
}

static void kstrax_free_list(kst_list_entry_t **list)
{
	while (*list != NULL) {
		kst_list_entry_t *del;
		del = kstrax_list_del(list);
		free(del);
	}
}

static void kstrax_list_add(kst_list_entry_t *new, kst_list_entry_t **head)
{
	new->next = *head;
	*head = new;
}

static kst_list_entry_t *kstrax_list_del(kst_list_entry_t **now)
{
	kst_list_entry_t *ret;
	ret = *now;
	*now = (*now)->next;
	return ret;
}


/*----------------------------------------------------------------------------------
 *  over write processing
 */
static void kstrax_lost_syscall_normal(print_data_t *p_data, unsigned long nr_lost)
{
	kstrax_flush_hash_table(p_data);
	fprintf(p_data->w_fp, 
		"\n------- over write %lu syscalls ------\n\n", (nr_lost+1) / 2);
}

static void kstrax_lost_syscall_raw(print_data_t *p_data, unsigned long nr_lost)
{
	fprintf(p_data->w_fp, 
		"\n------- over write %lu entries ------\n\n", nr_lost);
}

static void kstrax_lost_syscall_stats(print_data_t *p_data, unsigned long nr_lost)
{
	kstrax_stats_flush_hash_table(p_data);
}








