/*
Copyright (C) 2010-2011 Taisuke Kobayashi
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define GSC_VERSION 1.00

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "glut.h"

#define GSC_MAX 512
#define GSC_TEX_MAX 10

#define gsc_vcp(r, a) {int i;for(i=0;i<3;i++){r[i]=a[i];}} // vector copy
#define gsc_vpl(r, a, b) {int i;for(i=0;i<3;i++){r[i]=a[i]+b[i];}} // vector plus
#define gsc_vplo(r, a) {int i;for(i=0;i<3;i++){r[i]+=a[i];}} // vector plus overwrite
#define gsc_vmi(r, a, b) {int i;for(i=0;i<3;i++){r[i]=a[i]-b[i];}} // vector minus
#define gsc_vmio(r, a) {int i;for(i=0;i<3;i++){r[i]-=a[i];}} // vector minus overwrite
#define gsc_vmu(r, a, b) {int i;for(i=0;i<3;i++){r[i]=a[i]*b;}} // vector multi
#define gsc_vmuo(r, a) {int i;for(i=0;i<3;i++){r[i]*=a;}} // vector multi overwrite
#define gsc_vdi(r, a, b) {int i;if(b!=0){for(i=0;i<3;i++){r[i]=a[i]/b;}}} // vector divide
#define gsc_vdio(r, a) {int i;if(a!=0){for(i=0;i<3;i++){r[i]/=a;}}} // vector divide overwrite
#define gsc_vln(a) sqrt(a[0]*a[0]+a[1]*a[1]+a[2]*a[2]) // vector length
#define gsc_vnr(r, a) {double ln=gsc_vln(a);gsc_vdi(r,a,ln);} // vector normalize
#define gsc_vnro(r)  {double ln=gsc_vln(r);gsc_vdio(r,ln);} // vector normalize overwrite
#define gsc_vdo(a,b) (a[0]*b[0]+a[1]*b[1]+a[2]*b[2]) // vector dot
#define gsc_vcr(r,a,b) {r[0]=a[1]*b[2]-a[2]*b[1]; \
                    r[1]=a[2]*b[0]-a[0]*b[2]; \
                    r[2]=a[0]*b[1]-a[1]*b[0];} // vector cross

#define gsc_error(format, ...) \
        gsc_print("gsc_error %s %d\n  ", __FILE__, __LINE__, format, __VA_ARGS__);
#define gsc_debug(format, ...) \
        gsc_print("gsc_debug %s %d\n  ", __FILE__, __LINE__, format,  __VA_ARGS__);

void gsc_draw(void);
void gsc_init_lighting(void);

static struct {
  int width, height;
  struct {
    double eye[3];
    double center[3];
    double up[3];
  } look;
  struct {
    double fovy, aspect, near, far;
  } pers;
  struct {
    double eye_center_length;
  } bound;
  enum {
    gsc_im_3d_normal, gsc_im_2d_pixel, gsc_im_2d_pixel_no_animation,
    gsc_im_2d_pixel_no_clear
  } init_mode;
  enum {
    gsc_bm_none, gsc_bm_left, gsc_bm_middle, gsc_bm_right
  } button_mode;
  struct {
    int horizontal_counter, vertical_counter;
  } for_learning;
  int modifier; // GLUT_ACTIVE_CTRL GLUT_ACTIVE_ALT GLUT_ACTIVE_SHIFT
  int button[3][2][2]; // [left middle right][x y][down motion]
  struct {
    struct {
      double w, h, tw, th;
      unsigned char* img;
    } c[GSC_TEX_MAX];
    GLuint name[GSC_TEX_MAX];
    int cnt;
  } tex;
} gsc_gv = {
  -1, -1, // width height
  { -1, -1, -1, -1, -1, -1, -1, -1, -1 }, // look gsc_set_default_look
  { 50, -1, 1, 100*1000 }, // pers
  { 10 }, // bound
  gsc_im_3d_normal, // init_mode
  gsc_bm_none, // button_mode
  { 0, 0 }, // for_learning
};

void gsc_set_default_look()
{
  static double d[2][9] = { 0, 0, 300, 0, 0, 0, 0, 1, 0,
                            0, 0, -300, 0, 0, 0, 0, -1, 0 };
  double *p;

  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
    p = gsc_gv.look.eye; p[0] = d[0][0]; p[1] = d[0][1]; p[2] = d[0][2]; 
    p = gsc_gv.look.center; p[0] = d[0][3]; p[1] = d[0][4]; p[2] = d[0][5]; 
    p = gsc_gv.look.up; p[0] = d[0][6]; p[1] = d[0][7]; p[2] = d[0][8]; 
    break;
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
  case gsc_im_2d_pixel_no_clear:
    p = gsc_gv.look.eye; p[0] = d[1][0]; p[1] = d[1][1]; p[2] = d[1][2]; 
    p = gsc_gv.look.center; p[0] = d[1][3]; p[1] = d[1][4]; p[2] = d[1][5]; 
    p = gsc_gv.look.up; p[0] = d[1][6]; p[1] = d[1][7]; p[2] = d[1][8]; 
    break;
  }
}

void print_help()
{
  void gsc_print(char* s, char* f, int l, char* format, ...);
  static char s[] = "\n" \
    "hbO: _S_𒆐SƂ鋅ʏړ\n" \
    "Control+hbO: _璆S_̕ւ̑Oړ Y+ O Y- \n" \
    "Alt+hbO: upxNgEɕύX X+ E X- \n" \
    "Shift+hbO: _ƒS_𕽍sړ\n" \
    "EhbO: _ƒS_̊Ԃ̋ύX Y+ ߂Ȃ Y- Ȃ\n" \
    "HOME L[: _AS_AupxNgN̒lɕύX\n" \
    "EJ[\L[: gsc_get_horizongtal_counter()̒lύX l 0  -1 E +1\n" \
    "㉺J[\L[: gsc_get_vertical_counter()̒lύX l 0  +1  -1\n" \
    "h,H L[: wvR\[ɕ\\n";

  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
    gsc_debug(s);
    break;
  case gsc_im_2d_pixel_no_clear:
    break;
  }
}
void gsc_print(char* s, char* f, int l, char* format, ...)
{
  va_list arg;

  va_start(arg, format);
  printf(s, f, l);
  vprintf(format, arg);
  printf("\n");
  va_end(arg);
}

void gsc_print_info()
{
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
  case gsc_im_2d_pixel:
    gsc_debug("look: eye: %.1lf , %.1lf , %.1lf"
	   " center: %.1lf , %.1lf , %.1lf"
	   " up: %.3lf , %.3lf , %.3lf",
	   gsc_gv.look.eye[0], gsc_gv.look.eye[1], gsc_gv.look.eye[2], 
	   gsc_gv.look.center[0], gsc_gv.look.center[1], gsc_gv.look.center[2], 
	   gsc_gv.look.up[0], gsc_gv.look.up[1], gsc_gv.look.up[2]);
    break;
  case gsc_im_2d_pixel_no_animation:
  case gsc_im_2d_pixel_no_clear:
    break;
  }
}

void gsc_idle(void)
{
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_clear:
    glutPostRedisplay();
    break;
  case gsc_im_2d_pixel_no_animation:
    break;
  }
}

void gsc_special(int key, int x, int y)
{
  void gsc_change_projection(void);

  switch(key){
  case GLUT_KEY_HOME:
    gsc_set_default_look();
    gsc_print_info();
    gsc_change_projection();
    break;
  case GLUT_KEY_LEFT:
    gsc_gv.for_learning.horizontal_counter--;
    break;
  case GLUT_KEY_RIGHT:
    gsc_gv.for_learning.horizontal_counter++;
    break;
  case GLUT_KEY_UP:
    gsc_gv.for_learning.vertical_counter++;
    break;
  case GLUT_KEY_DOWN:
    gsc_gv.for_learning.vertical_counter--;
    break;
  }
  glutPostRedisplay();
}

void gsc_keyboard(unsigned char key, int x, int y)
{
  switch(key){
  case 'h':
  case 'H':
    print_help();
    break;
  }
}

void gsc_change_projection(void)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gsc_gv.pers.aspect = (double)gsc_gv.width / gsc_gv.height;
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
    gluPerspective(gsc_gv.pers.fovy, gsc_gv.pers.aspect, gsc_gv.pers.near, gsc_gv.pers.far);
    break;
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
  case gsc_im_2d_pixel_no_clear:
    glOrtho(0, gsc_gv.width, -gsc_gv.height, 0, -1, 1000);
    break;
  }
  gluLookAt(gsc_gv.look.eye[0], gsc_gv.look.eye[1], gsc_gv.look.eye[2], 
	    gsc_gv.look.center[0], gsc_gv.look.center[1], gsc_gv.look.center[2],
	    gsc_gv.look.up[0], gsc_gv.look.up[1], gsc_gv.look.up[2]);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  gsc_init_lighting();
  glPopMatrix();
}

void gsc_change_eye_center_length(int y, int sy)
{
  int dy = y - sy;
  double min = gsc_gv.bound.eye_center_length, len;
  double a[3], b[3];

  gsc_vmi(a, gsc_gv.look.center, gsc_gv.look.eye);  
  len = gsc_vln(a);
  len += (double)dy;
  gsc_vnro(a);
  gsc_vmu(b, a, dy);
  if(min < len)
    gsc_vplo(gsc_gv.look.center, b);  
}

void gsc_move_around(int x, int y, int sx, int sy)
{
  int dx = x - sx,  dy = sy - y;
  double a[3], b[3], c[3];

  gsc_vmi(a, gsc_gv.look.center, gsc_gv.look.eye);  
  gsc_vcp(b, gsc_gv.look.up);
  gsc_vcr(c, a, b);
  gsc_vnro(b);
  gsc_vnro(c);
  gsc_vmuo(b, dy);
  gsc_vmuo(c, dx);
  gsc_vplo(c, b);
  gsc_vplo(gsc_gv.look.eye, c);
  gsc_vmi(a, gsc_gv.look.center, gsc_gv.look.eye);  
  gsc_vcr(b, a, gsc_gv.look.up);
  gsc_vcr(gsc_gv.look.up, b, a);
  gsc_vnro(gsc_gv.look.up);
}

void gsc_parallel_shift(int x, int y, int sx, int sy)
{
  int dx = x - sx,  dy = sy - y;
  double a[3], b[3], c[3];

  gsc_vmi(a, gsc_gv.look.center, gsc_gv.look.eye);  
  gsc_vcp(b, gsc_gv.look.up);
  gsc_vcr(c, a, b);
  gsc_vnro(b);
  gsc_vnro(c);
  gsc_vmuo(b, dy);
  gsc_vmuo(c, dx);
  gsc_vplo(b, c);
  gsc_vplo(gsc_gv.look.center, b);
  gsc_vplo(gsc_gv.look.eye, b);
}

void gsc_rotate_up_vector(int x, int sx)
{
  double dx = x - sx;
  double a[3], b[3];

  dx *= 0.01;
  gsc_vmi(a, gsc_gv.look.center, gsc_gv.look.eye);  
  gsc_vcr(b, a, gsc_gv.look.up);
  gsc_vnro(b);
  gsc_vmuo(b, dx);
  gsc_vplo(gsc_gv.look.up, b);  
  gsc_vnro(gsc_gv.look.up);
}

void gsc_back_and_forth(int y, int sy)
{
  int dy = sy - y;
  double a[3], b[3];

  gsc_vmi(a, gsc_gv.look.center, gsc_gv.look.eye);  
  gsc_vnro(a);
  gsc_vmu(b, a, dy);
  gsc_vplo(gsc_gv.look.eye, b);  
  gsc_vplo(gsc_gv.look.center, b);  
}

void gsc_motion(int x, int y)
{
  switch(gsc_gv.button_mode){
  case gsc_bm_left:
    if(gsc_gv.modifier & GLUT_ACTIVE_CTRL){
      gsc_back_and_forth(y, gsc_gv.button[0][1][1]);
    }else if(gsc_gv.modifier & GLUT_ACTIVE_ALT){
      gsc_rotate_up_vector(x, gsc_gv.button[0][0][1]);
    }else if(gsc_gv.modifier & GLUT_ACTIVE_SHIFT){
      gsc_parallel_shift(x, y, gsc_gv.button[0][0][1], gsc_gv.button[0][1][1]);
    }else{
      gsc_move_around(x, y, gsc_gv.button[0][0][1], gsc_gv.button[0][1][1]);
    }
    gsc_gv.button[0][0][1] = x;
    gsc_gv.button[0][1][1] = y;
    break;
  case gsc_bm_middle:
    gsc_gv.button[1][0][1] = x;
    gsc_gv.button[1][1][1] = y;
    break;
  case gsc_bm_right:
    gsc_change_eye_center_length(y, gsc_gv.button[2][1][1]);
    gsc_gv.button[2][0][1] = x;
    gsc_gv.button[2][1][1] = y;
    break;
  }
  if(gsc_gv.button_mode > gsc_bm_none){
    gsc_print_info();
    gsc_change_projection();
    glutPostRedisplay();
  }
}

void gsc_mouse(int button, int state, int x, int y)
{
  gsc_gv.modifier = glutGetModifiers();
  switch(state){
  case GLUT_DOWN:
    switch (button) {
    case GLUT_LEFT_BUTTON:
      gsc_gv.button[0][0][0] = gsc_gv.button[0][0][1] = x;
      gsc_gv.button[0][1][0] = gsc_gv.button[0][1][1] = y;
      gsc_gv.button_mode = gsc_bm_left;
      break;
    case GLUT_MIDDLE_BUTTON:
      gsc_gv.button[1][0][0] = gsc_gv.button[1][0][1] = x;
      gsc_gv.button[1][1][0] = gsc_gv.button[1][1][1] = y;
      gsc_gv.button_mode = gsc_bm_middle;
      break;
    case GLUT_RIGHT_BUTTON:
      gsc_gv.button[2][0][0] = gsc_gv.button[2][0][1] = x;
      gsc_gv.button[2][1][0] = gsc_gv.button[2][1][1] = y;
      gsc_gv.button_mode = gsc_bm_right;
      break;
    }
    break;
  case GLUT_UP:
    gsc_gv.button_mode = gsc_bm_none;
    glutPostRedisplay();
    break;
  }
}

void gsc_resize(int w, int h)
{
  glViewport(0, 0, w, h);
  gsc_gv.width = w;
  gsc_gv.height = h;
  gsc_change_projection();
}

void gsc_draw_2d_bm(int mode)
{
  int bm = gsc_gv.button_mode - gsc_bm_left,
    ox = gsc_gv.width /2, oy = gsc_gv.height /2, dx, dy;

  if(bm < 0 || bm > 2){
    gsc_error("bm < 0 || bm > 2 gsc_draw_2d_bm");
    return;
  }
  dx = gsc_gv.button[bm][0][1] - gsc_gv.button[bm][0][0];
  dy = gsc_gv.button[bm][1][0] - gsc_gv.button[bm][1][1];
  glLineWidth(1);
  switch(gsc_gv.button_mode){
  case gsc_bm_left:
    glColor3d(1, 0, 0);
    break;
  case gsc_bm_middle:
    glColor3d(0, 1, 0);
    break;
  case gsc_bm_right:
    glColor3d(0, 0, 1);
    break;
  }
  glBegin(GL_LINES);
  glVertex2i(ox, oy);
  switch(mode){
  case 0:
    glVertex2i(ox + dx, oy + dy);
    break;
  case 1:
    glVertex2i(ox + dx, oy);
    break;
  case 2:
    glVertex2i(ox, oy + dy);
    break;
  }
  glEnd();
}

void gsc_draw_2d(void)
{
  if(gsc_gv.button_mode > gsc_bm_none){
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_NORMALIZE);
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    glOrtho(0, gsc_gv.width, 0, gsc_gv.height, -1, 1);
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    switch(gsc_gv.button_mode){
    case gsc_bm_left:
      if(gsc_gv.modifier & GLUT_ACTIVE_CTRL){
        gsc_draw_2d_bm(2);
      }else if(gsc_gv.modifier & GLUT_ACTIVE_ALT){
        gsc_draw_2d_bm(1);
      }else{
        gsc_draw_2d_bm(0);
      }
      break;
    case gsc_bm_middle:
      gsc_draw_2d_bm(0);
      break;
    case gsc_bm_right:
      gsc_draw_2d_bm(2);
      break;
    }
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
    glEnable(GL_NORMALIZE);
    glEnable(GL_TEXTURE_2D);
    switch(gsc_gv.init_mode){
    case gsc_im_3d_normal:
      glEnable(GL_DEPTH_TEST);
      break;
    case gsc_im_2d_pixel:
    case gsc_im_2d_pixel_no_animation:
    case gsc_im_2d_pixel_no_clear:
      glDisable(GL_DEPTH_TEST);
      break;
    }
  }
}

void gsc_display(void)
{
  static int first = 1;
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    break;
  case gsc_im_2d_pixel_no_clear:
    if(first){
      first = 0;
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    }
    break;
  }
  gsc_draw();
  gsc_draw_2d();
  glFlush();
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
    glutSwapBuffers();
    break;
  case gsc_im_2d_pixel_no_clear:
    glFlush();
    break;
  }
}

int gsc_load_ppm_texture(int i, char* file_name)
{
  FILE* fp = fopen(file_name, "rb");
  char b[GSC_MAX];
  int first = 1, w = 0, h, t1 = 0, t2, size, y;

  if(fp == 0){
    gsc_error("fopen(file_name, \"rb\") %s", file_name);
    return -1;
  }
  while((int)fgets(b, GSC_MAX, fp) != EOF){
    if(first){
      first = 0;
      if(strcmp(b, "P6\n") != 0){
        gsc_error("strcmp(b, \"P6\\n\") != 0");
        fclose(fp);
        return -2;
      }else{
        continue;
      }
    }
    if(b[0] == '#')
      continue;
    if(w == 0){
      if(sscanf(b, "%d %d\n", &w, &h) != 2){
        gsc_error("sscanf(b, \"%d %d\\n\", &w, &h) != 2");
        fclose(fp);
        return -3;
      }else{
        continue;
      }
    }
    if(sscanf(b, "%d\n", &t1) != 1){
      gsc_error("sscanf(b, \"%d\\n\", &t1) != 1");
      fclose(fp);
      return -4;
    }else{
      if(t1 != 255){
        printf("t1 != 255 error\n");
        fclose(fp);
        return -5;
      }
    }
    break;
  }
  gsc_gv.tex.c[i].w = (double)w;
  gsc_gv.tex.c[i].h = (double)h;
  t1 = w;
  t2 = 1;
  while(t2 < t1)
    t2 *= 2;
  gsc_gv.tex.c[i].tw = (double)t2;
  t1 = h;
  t2 = 1;
  while(t2 < t1)
    t2 *= 2;
  gsc_gv.tex.c[i].th = (double)t2;
#if 0
  gsc_debug("%d,%.1lf,%.1lf,%.1lf,%.1lf",
            i, gsc_gv.tex.c[i].w, gsc_gv.tex.c[i].h, gsc_gv.tex.c[i].tw, gsc_gv.tex.c[i].th);
#endif
  size = (int)(gsc_gv.tex.c[i].tw * gsc_gv.tex.c[i].th *3);
  gsc_gv.tex.c[i].img = (unsigned char*)malloc(size);
  if(gsc_gv.tex.c[i].img == 0){
    gsc_error("gsc_gv.tex.c[i].img == 0");
    fclose(fp);
    return -6;
  }
  for(y = 0; y < (int)gsc_gv.tex.c[i].h; y++){
    if(fread(gsc_gv.tex.c[i].img + (int)(gsc_gv.tex.c[i].tw *3 * y),
	     (int)(gsc_gv.tex.c[i].w *3), 1, fp) != 1){
      gsc_error("fread(gsc_gv.tex.c[i].img + gsc_gv.tex.c[i].tw *3 * y, " \
             "gsc_gv.tex.c[i].w *3, 1, fp) != 1");
      fclose(fp);
      return -7;
    }
  }
  fclose(fp);
  return 0;
}

int gsc_gen_texture(int cnt)
{
  if(cnt > GSC_TEX_MAX){
    gsc_error("cnt > GSC_TEX_MAX");
    return -1;
  }
  gsc_gv.tex.cnt = cnt;
  glGenTextures(gsc_gv.tex.cnt, gsc_gv.tex.name);
  return 0;
}

int gsc_init_texture(int tex_count, char** tex_file_name)
{
  int i;
  if(gsc_gen_texture(tex_count) < 0){
    gsc_error("gsc_gen_texture");
    return -1;
  }
  for(i = 0; i < tex_count; i++){
    void gsc_bind_texture(int i);

    if(gsc_load_ppm_texture(i, tex_file_name[i]) < 0){
      gsc_error("gsc_load_ppm_texture");
      return -1;
    }
    gsc_bind_texture(i);
    glTexImage2D(GL_TEXTURE_2D, 0, 3, gsc_gv.tex.c[i].tw, gsc_gv.tex.c[i].th, 0,
                 GL_RGB, GL_UNSIGNED_BYTE, gsc_gv.tex.c[i].img);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST); 
  }
  return 0;
}

int gsc_init(int tex_count, char** tex_file_name)
{
  gsc_set_default_look();
  print_help();
  if(gsc_init_texture(tex_count, tex_file_name) < 0){
    gsc_error("gsc_init_texture");
    return -1;
  }
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    glEnable(GL_DEPTH_TEST);
    break;
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
  case gsc_im_2d_pixel_no_clear:
    glDisable(GL_CULL_FACE);
    glDisable(GL_DEPTH_TEST);
    break;
  }
  glDisable(GL_BLEND);
  glClearColor(0, 0, 0, 1);
  glEnable(GL_TEXTURE_2D);
  glEnable(GL_NORMALIZE);
  return 0;
}

// ===

void gsc_init_and_loop(int* argc, char *argv[],
                       int init_mode, int tex_count, char** tex_file_name)
{
  gsc_gv.init_mode = init_mode;
  glutInitWindowPosition(0, 0);
  glutInitWindowSize(500, 500);
  glutInit(argc, argv);
  switch(gsc_gv.init_mode){
  case gsc_im_3d_normal:
  case gsc_im_2d_pixel:
  case gsc_im_2d_pixel_no_animation:
    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);
    break;
  case gsc_im_2d_pixel_no_clear:
    glutInitDisplayMode(GLUT_RGBA | GLUT_DEPTH);
    break;
  }
  glutCreateWindow(argv[0]);
  if(gsc_init(tex_count, tex_file_name) < 0){
    gsc_error("gsc_init");
    return;
  }
  glutDisplayFunc(gsc_display);
  glutReshapeFunc(gsc_resize);
  glutMouseFunc(gsc_mouse);
  glutMotionFunc(gsc_motion);
  glutKeyboardFunc(gsc_keyboard);
  glutSpecialFunc(gsc_special);
  glutIdleFunc(gsc_idle);
  glutMainLoop();
}

int gsc_tex_width(int i)
{
  return gsc_gv.tex.c[i].w;
}

int gsc_tex_height(int i)
{
  return gsc_gv.tex.c[i].h;
}

void gsc_bind_texture(int i)
{
  glBindTexture(GL_TEXTURE_2D, gsc_gv.tex.name[i]);
}

void gsc_tex_coord(int i, double x, double y)
{
  glTexCoord2d(x / gsc_gv.tex.c[i].tw, y / gsc_gv.tex.c[i].th);
}

int gsc_get_horizontal_counter(void)
{
  return gsc_gv.for_learning.horizontal_counter;
}

int gsc_get_vertical_counter(void)
{
  return gsc_gv.for_learning.vertical_counter;
}
