/* **************************************************************************
 * Copyright (C) 2008 BJoRFUAN. All Right Reserved
 * **************************************************************************
 * This module, contains source code, binary and documentation, is in the
 * BSD License, and comes with NO WARRANTY.
 *
 *                                                 torao <torao@bjorfuan.com>
 *                                                       http://www.moyo.biz/
 * $Id: Xml.java,v 1.2 2009/04/11 04:22:31 torao Exp $
*/
package org.koiroha.xml;

import java.nio.charset.*;
import java.util.regex.*;

import org.xml.sax.ext.LexicalHandler;

// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Xml: XML 用ユーティリティクラス
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * XML 用のユーティリティクラスです。
 * <p>
 * @version $Revision: 1.2 $ $Date: 2009/04/11 04:22:31 $
 * @author torao
 * @since 2009/04/06 Java SE 6
 */
public final class Xml {

	// ======================================================================
	// ログ出力先
	// ======================================================================
	/**
	 * このクラスのログ出力先です。
	 * <p>
	 */
	private static final java.util.logging.Logger logger = java.util.logging.Logger.getLogger(Xml.class.getName());

	// ======================================================================
	// 標準接頭辞
	// ======================================================================
	/**
	 * 標準機能の接頭辞です。
	 * <p>
	 */
	private static final String FEATURE_PREFIX = "http://xml.org/sax/features/";

	// ======================================================================
	// 名前空間認識
	// ======================================================================
	/**
	 * 解析時に名前空間を認識するかどうかの機能フラグ名です。
	 * 定数値 {@value} を示します。
	 * <p>
	 */
	public static final String FEATURE_NAMESPACES = FEATURE_PREFIX + "namespaces";

	// ======================================================================
	// DTD 検証認識
	// ======================================================================
	/**
	 * 解析時に DTD 検証を行うかどうかの機能フラグ名です。
	 * 定数値 {@value} を示します。
	 * <p>
	 */
	public static final String FEATURE_VALIDATION = FEATURE_PREFIX + "validation";

	// ======================================================================
	// 標準接頭辞
	// ======================================================================
	/**
	 * 標準機能の接頭辞です。
	 * <p>
	 */
	private static final String PROPERTY_PREFIX = "http://xml.org/sax/properties/";

	// ======================================================================
	// ドキュメント XML バージョン
	// ======================================================================
	/**
	 * ドキュメント XML バージョンを参照するための SAX パーサのプロパティ名です。
	 * 定数値 {@value} を示します。
	 * <p>
	 */
	public static final String PROPERTY_DOCUMENT_XML_VERSION = PROPERTY_PREFIX + "document-xml-version";

	// ======================================================================
	// 構文ハンドラプロパティ名
	// ======================================================================
	/**
	 * 解析時に{@link LexicalHandler 構文ハンドラ}を指定する場合の SAX パーサのプロパティ
	 * 名です。定数値 {@value} を示します。
	 * <p>
	 */
	public static final String PROPERTY_LEXICAL_HANDLER = PROPERTY_PREFIX + "lexical-handler";

	// ======================================================================
	// 文字セット解析パターン
	// ======================================================================
	/**
	 * Content-Type から文字セット部分を取得するためのパターンです。
	 * <p>
	 */
	private static final Pattern CHARSET_ATTRIBUTE = Pattern.compile(";\\s*charset\\s*=\\s*[\"\']?([^\"\'\\s;]*)[\"\']?", Pattern.CASE_INSENSITIVE);

	// ======================================================================
	// 実体参照パターン
	// ======================================================================
	/**
	 * 実体参照のパターンです。
	 * <p>
	 */
	private static final Pattern ENTITY_REFERENCE = Pattern.compile("&([^;&]*);");

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * コンストラクタはクラス内に隠蔽されています。
	 * <p>
	 */
	private Xml() {
		return;
	}

	// ======================================================================
	// 空白文字の判定
	// ======================================================================
	/**
	 * 指定された文字が XML での空白を表すかどうかを判定します。空白文字 ({@code ' '})、
	 * 水平タブ ({@code '\t'})、復帰 ({@code '\r'})、改行 ({@code '\n'}) を空白文字
	 * とします。
	 * <p>
	 * @param ch 判定する文字
	 * @return 空白文字の場合 true
	 */
	public static boolean isWhitespace(int ch) {
		return (ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n');
	}

	// ======================================================================
	// 文字のエスケープ
	// ======================================================================
	/**
	 * 指定された文字列を XML のテキストとしてそのまま使用できるようにエスケープします。
	 * <p>
	 * @param text エスケープする文字列
	 * @return エスケープした文字列
	 */
	public static String escape(String text){
		StringBuilder buffer = new StringBuilder(text.length());
		for(int i=0; i<text.length(); i++){
			char ch = text.charAt(i);
			switch(ch){
			case '<':	buffer.append("&lt;");		break;
			case '>':	buffer.append("&gt;");		break;
			case '&':	buffer.append("&amp;");		break;
			case '\"':	buffer.append("&quot;");	break;
			case '\'':	buffer.append("&apos;");	break;
			case '\n':	buffer.append('\n');		break;
			case '\t':	buffer.append('\t');		break;
			case '\r':
				buffer.append('\n');
				if(i+1<text.length() && text.charAt(i+1) == '\n'){
					i ++;
				}
				break;
			default:
				if(Character.isDefined(ch) && ! Character.isISOControl(ch)){
					buffer.append(ch);
				} else {
					buffer.append("&#" + (int)ch + ";");
				}
				break;
			}
		}
		return buffer.toString();
	}

	// ======================================================================
	// 文字のエスケープ解除
	// ======================================================================
	/**
	 * 指定された文字列を XML エンコードを解除します。
	 * <p>
	 * @param text エスケープを解除する文字列
	 * @return エスケープを解除した文字列
	 */
	public static String unescape(String text){
		StringBuilder buffer = new StringBuilder(text.length());
		Matcher matcher = ENTITY_REFERENCE.matcher(text);
		int begin = 0;
		while(matcher.find()){

			// 実体参照より前の文字列を連結
			int end = matcher.start();
			if(begin != end){
				buffer.append(text, begin, end);
			}
			begin = matcher.end();

			// 実体参照を解析
			String name = matcher.group(1);
			if(name.equals("amp")){
				buffer.append('&');
			} else if(name.equals("lt")){
				buffer.append('<');
			} else if(name.equals("gt")){
				buffer.append('>');
			} else if(name.equals("quot")){
				buffer.append('\"');
			} else if(name.equals("apos")){
				buffer.append('\'');
			} else if(name.length() > 1 && name.charAt(0) == '#'){
				boolean hex = (name.length() > 2 && Character.toLowerCase(name.charAt(1)) == 'x');
				int ch = 0;
				try{
					if(hex){
						ch = Integer.parseInt(name.substring(2), 16);
					} else {
						ch = Integer.parseInt(name.substring(1), 10);
					}
					if(ch < Character.MIN_VALUE || ch > Character.MAX_VALUE){
						throw new NumberFormatException(String.valueOf(ch));
					}
					buffer.append((char)ch);
				} catch(NumberFormatException ex){
					logger.fine("invalid number entity reference: &" + name + ";");
					buffer.append('&').append(name).append(';');
				}
			} else {
				logger.fine("unrecognized entity character reference: &" + name + ";");
				buffer.append('&').append(name).append(';');
			}
		}

		// 残った文字列を連結
		if(begin != text.length()){
			buffer.append(text, begin, text.length());
		}
		return buffer.toString();
	}

	// ======================================================================
	// 文字セットの参照
	// ======================================================================
	/**
	 * <code>"text/html; charset=UTF-8"</code> のような Content-Type 値から
	 * 文字セットを参照します。Content-Type に charset 属性が付けられていない場合や、値を
	 * 文字セットとして認識できない場合は null を返します。
	 * <p>
	 * @param contentType Content-Type
	 * @return charset 属性に対する文字セット
	 */
	public static Charset getCharset(String contentType){
		logger.finest("getCharset(" + contentType + ")");

		// 正規表現を使用して文字セットを取得
		Matcher matcher = CHARSET_ATTRIBUTE.matcher(contentType);
		if(! matcher.find()){
			logger.finest("no charset specified: " + contentType);
			return null;
		}
		String charset = matcher.group(1);

		// 指定された文字セットがサポートされているかを確認
		boolean supported = false;
		try{
			supported = Charset.isSupported(charset);
		} catch(IllegalCharsetNameException ex){/* */}
		if(! supported){
			logger.finest("unsupported charset specified: " + charset + ": " + contentType);
			return null;
		}

		logger.finest("charset retrieved: " + charset + ": " + contentType);
		return Charset.forName(charset);
	}

}
