/* **************************************************************************
 * Copyright (C) 2009 koiroha.org All Right Reserved
 * **************************************************************************
 * This module, contains source code, binary and documentation, is in the
 * BSD License, and comes with NO WARRANTY.
 *
 *                                        takami torao <torao@mars.dti.ne.jp>
 *                                                     http://www.koiroha.org
 * $Id:$
*/
package org.koiroha.xml;

import java.util.*;
import java.util.regex.*;

// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Html: HTML ユーティリティクラス
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * HTML を扱うためのユーティリティクラスです。
 * <p>
 * @version $Revision:$ $Date:$
 * @author torao
 * @since 2010/01/19 Java SE 6
 */
public final class Html {

	// ======================================================================
	// エンティティ → 文字マップ
	// ======================================================================
	/**
	 * エンティティを文字に変換するためのマップです。
	 * <p>
	 */
	private static final Map<String,Character> ENTITY2CHAR = new HashMap<String, Character>();

	// ======================================================================
	// 文字 → エンティティマップ
	// ======================================================================
	/**
	 * 文字をエンティティに変換するためのマップです。
	 * <p>
	 */
	private static final Map<Character,String> CHAR2ENTITY = new HashMap<Character, String>();

	// ======================================================================
	// スタティックイニシャライザ
	// ======================================================================
	/**
	 * リソースの読み込みを行います。
	 * <p>
	 */
	static{

		// エンティティ定義の読み込み
		ResourceBundle res = ResourceBundle.getBundle("org.koiroha.xml.htmlentity");
		Enumeration<String> en = res.getKeys();
		while(en.hasMoreElements()){
			String key = en.nextElement();
			String value = res.getString(key);
			String entity = "&" + key + ";";
			Character ch = Character.valueOf((char)Integer.parseInt(value));
			ENTITY2CHAR.put(entity, ch);
			CHAR2ENTITY.put(ch, entity);
		}
	}

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * コンストラクタはクラス内に隠蔽されています。
	 * <p>
	 */
	private Html() {
		return;
	}

	// ======================================================================
	// 文字の参照
	// ======================================================================
	/**
	 * 指定された HTML 実体参照に対応する文字を返します。指定された実体参照が
	 * 定義されていない場合は負の値を返します。
	 * <p>
	 * @param entity 実体参照の文字列 ("&amp;nbsp;" など)
	 * @return 実体参照に対応する文字
	 */
	public static int getCharacter(String entity){
		Character ch = ENTITY2CHAR.get(entity);
		if(ch == null){
			return -1;
		}
		return ch;
	}

	// ======================================================================
	// 実体参照の参照
	// ======================================================================
	/**
	 * 指定された文字に対する実体参照または数値参照を参照します。
	 * <p>
	 * @param ch 実体参照を参照する文字
	 * @return 実体参照
	 */
	public static String getEntityReference(char ch){
		String entity = CHAR2ENTITY.get(ch);
		if(entity != null){
			return entity;
		}
		return "&#x" + Integer.toHexString(ch).toUpperCase() + ";";
	}

	// ======================================================================
	// HTML エスケープの解除
	// ======================================================================
	/**
	 * 指定された文字列内の HTML エスケープ (実際参照) を解除します。
	 * <p>
	 * @param text 実体参照を戻す文字列
	 * @return 戻した文字列
	 */
	public static String unescape(String text){
		StringBuilder buffer = new StringBuilder();
		Pattern pattern = Pattern.compile("&[^;&]+;");
		Matcher matcher = pattern.matcher(text);
		int end = 0;
		while(matcher.find()){
			int begin = matcher.start();

			// 一致箇所前方の文字列を連結
			buffer.append(text.substring(end, begin));

			// 実体参照を取得
			end = matcher.end();
			String entity = text.substring(begin, end);
			int ch = getCharacter(entity);
			if(ch < 0){
				buffer.append(entity);
			} else {
				buffer.append((char)ch);
			}
		}

		// 残った文字列を連結
		buffer.append(text.substring(end));

		return buffer.toString();
	}

}
