/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "thread.h"

#include <config.h>

#include <stdlib.h>
#include <unistd.h>

#include <kurl.h>
#include <kdebug.h>
#include <kprotocolmanager.h>
#include <kglobal.h>
#include <kstandarddirs.h>

#include <kio/slaveconfig.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>
#include <kio/netaccess.h>

#include <qwidget.h>
#include <qapplication.h>
#include <qregexp.h>

#include "board.h"
#include "kita_misc.h"
#include "qcp932codec.h"
#include "threadinfo.h"

using namespace Kita;

QDict<Thread>* Thread::m_threadDict = 0;

Thread::Thread()
{}

Thread::Thread( const QString& datURL, const QString& threadName )
        : m_datURL( datURL ), m_threadName( threadName )
{}

Thread::~Thread()
{}

const QString& Thread::datURL() const
{
    return m_datURL;
}

const QString Thread::url() const
{
    return datToThread( m_datURL );
}

const QString Thread::datID() const
{
    return KURL( m_datURL ).filename().section( ".", 0, 0 );
}

int Thread::resNum() const
{
    return KitaThreadInfo::resNum( m_datURL );
}

const QString& Thread::name() const
{
    if ( ! Kita::Thread::getByURL( m_datURL ) ->m_threadName.isEmpty() ) {
        return Kita::Thread::getByURL( m_datURL ) ->m_threadName;
    }
    return m_threadName;
}

void Thread::setResNum( int resNum )
{
    KitaThreadInfo::setResNum( m_datURL, resNum );
}

const QString& Thread::boardName() const
{
    Board * board = Board::getByURL( boardURL() );
    return board->name();
}

const QString Thread::boardURL() const
{
    return datToBoard( m_datURL );
}

const QString Thread::boardID() const
{
    return KURL( datToBoard( m_datURL ) ).fileName();
}

const QString Thread::toXmlFragment() const
{
    QString ret;

    ret += "<thread xmlns=\"http://kita.sourceforge.jp/ns/thread\">\n";
    ret += QString( "<daturl>%1</daturl>\n" ).arg( m_datURL );
    ret += QString( "<name>%1</name>\n" ).arg( m_threadName );
    ret += QString( "<resnum>%1</resnum>\n" ).arg( resNum() );

    // FIXME: Board饹饳ԡ
    ret += "<board xmlns=\"http://kita.sourceforge.jp/ns/board\">\n";
    ret += QString( "<url>%1</url>\n" ).arg( boardURL() );
    ret += QString( "<name>%1</name>\n" ).arg( boardName() );
    ret += "</board>\n";

    ret += "</thread>\n";

    return ret;
}

Thread* Thread::fromXml( const QString& xml )
{
    Thread * thread;

    ThreadXmlParser parser;
    QXmlSimpleReader reader;
    QXmlInputSource source;
    source.setData( xml );
    reader.setContentHandler( &parser );
    reader.parse( &source );

    if ( parser.isValid() ) {
        thread = parser.getThread();
    } else {
        kdError() << "inValid" << endl;
    }

    return thread;
}

Thread* Thread::getByURL( const QString& datURL )
{
    if ( m_threadDict == 0 ) {
        m_threadDict = new QDict<Thread>();
    }

    if ( m_threadDict->find( datURL ) ) {
        return m_threadDict->find( datURL );
    }

    Thread* newThread = new Thread( datURL );
    m_threadDict->insert( datURL, newThread );

    return newThread;
}

void Thread::setName( const QString& datURL, const QString& threadName )
{
    if ( datURL.isNull() ) {
        return ;
    }

    if ( m_threadDict == 0 ) {
        m_threadDict = new QDict<Thread>();
    }

    if ( m_threadDict->find( datURL ) ) {
        Thread * thread = getByURL( datURL );
        thread->m_threadName = threadName;
    } else {
        Thread* newThread = new Thread( datURL, threadName );
        m_threadDict->insert( datURL, newThread );
    }

    return ;
}

NullThread::NullThread()
{}

NullThread::~NullThread()
{}

ThreadXmlParser::ThreadXmlParser()
        : m_boardParser( 0 ), m_inBoard( false ), m_isValid( false ), m_inThread( false )
{};

bool ThreadXmlParser::startElement( const QString& namespaceURI, const QString& localName, const QString& qName, const QXmlAttributes& atts )
{
    if ( m_inBoard ) {
        m_boardParser->startElement( namespaceURI, localName, qName, atts );
    } else if ( m_inThread ) {
        if ( localName == "daturl" ) {} else if ( localName == "name" ) {} else if ( localName == "resnum" ) {} else if ( localName == "board" ) {
            m_inBoard = true;
            m_boardParser = new BoardXmlParser();
            m_boardParser->startElement( namespaceURI, localName, qName, atts );
        } else {
            // error
            return false;
        }
    } else {
        if ( localName == "thread" ) {
            m_inThread = true;
        } else {
            // error
            return false;
        }
    }

    return true;
}

// TODO: <resnum>FavoritesǤϻȤʤȤˤʤäΤǺɬפꡣ
bool ThreadXmlParser::endElement( const QString& namespaceURI, const QString& localName, const QString& qName )
{
    if ( localName == "board" ) {
        m_inBoard = false;
        m_boardParser->endElement( namespaceURI, localName, qName );

        delete m_boardParser;
        m_boardParser = 0;

        // create board;
    } else if ( m_inBoard ) {
        m_boardParser->endElement( namespaceURI, localName, qName );
    } else if ( localName == "thread" ) {
        m_inThread = false;
        Kita::Thread::setName( m_datUrlStr, m_threadName );
        m_thread = Kita::Thread::getByURL( m_datUrlStr );
        KitaThreadInfo::setResNum( m_datUrlStr, m_resNum );
        m_isValid = true;
        // create thread;
    } else if ( localName == "daturl" ) {
        m_datUrlStr = m_characters;
    } else if ( localName == "name" ) {
        m_threadName = m_characters;
    } else if ( localName == "resnum" ) {
        m_resNum = m_characters.toInt();
    } else {
        // error
        return false;
    }
    return true;
}

bool ThreadXmlParser::characters( const QString& ch )
{
    if ( m_inBoard ) {
        m_boardParser->characters( ch );
    } else {
        m_characters = ch;
    }
    return true;
}

