/***************************************************************************
 *   Copyright (C) 2003 by Hideki Ikemoto                                  *
 *   ikemo@wakaba.jp                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qfile.h>
#include <qtextstream.h>
#include <qtextcodec.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include "qcp932codec.h"
#include <qsjiscodec.h>
#include <qlabel.h>

#include <time.h>

#include <ktextbrowser.h>
#include <kurl.h>
#include <kio/netaccess.h>
#include <kio/slaveconfig.h>
#include <khtml_part.h>
#include <khtmlview.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>
#include <kmessagebox.h>
#include <krun.h>

#include "kitathreadview.h"
#include "kitawritedialog.h"

static QString linkURL(QString str)
{
  QString ret;
  // see RFC 1738.
  QRegExp url_rx("(h?ttp://([-.0-9a-zA-Z]+(/[;:@&=$-_.+!*'(),%~/?#0-9a-zA-Z]*)?))");
  
  int i;
  while ((i = str.find(url_rx)) != -1) {
    if (i == 0) {
      // starts with URL.
      ret += "<a href='http://";
      ret += url_rx.cap(2);
      ret += "'>";
      ret += url_rx.cap(1);
      ret += "</a>";
      str.remove(0, url_rx.cap(1).length());
    } else {
      ret += str.left(i);
      str.remove(0, i);
    }
  }
  ret += str;
  return ret;
}

KitaThreadView::KitaThreadView(QWidget *parent, const char *name)
    : KitaThreadViewBase(parent, name)
{
  threadPart = new KHTMLPart(threadFrame);
  QHBoxLayout* aLayout = new QHBoxLayout(threadFrame);
  threadPart->setStandardFont("Gothic [Mona]");
  threadPart->setZoomFactor(120); // 12px;
  aLayout->addWidget(threadPart->view());

  connect(writeButton, SIGNAL(clicked()), SLOT(slotWriteButtonClicked()));
  connect(threadPart, SIGNAL(onURL(const QString&)), SLOT(slotOnURL(const QString&)));
  connect(threadPart->browserExtension(),
          SIGNAL(openURLRequest(const KURL&, const KParts::URLArgs&)),
          SLOT(slotOpenURLRequest(const KURL&)));
}

KitaThreadView::~KitaThreadView() {}

void KitaThreadView::slotShowThread(const KURL& _datURL, const KURL& _boardURL, const QString& boardName)
{
  KIO::SlaveConfig::self()->setConfigData("http", _datURL.host() ,
                                          "UserAgent", "Monazilla/1.00 (test)");
  m_datURL = _datURL;     // ex. http://pc.2ch.net/linux/dat/XXXXXXXXXX.dat
  m_boardURL = _boardURL; // ex. http://pc.2ch.net/linux/
  m_boardID = m_boardURL.fileName();  // ex. linux
  m_datName = QStringList::split(".", m_datURL.fileName())[0];  // ex. XXXXXXXXXX
  // ex. http://pc.2ch.net/test/read.cgi/linux/XXXXXXXXXX
  m_threadURL = KURL(m_boardURL, QString("../test/read.cgi/") + QString(m_boardID) + QString("/") + QString(m_datName));
  m_boardName = boardName; // ex. Linux
  
  KIO::TransferJob* job = KIO::get(m_datURL, true, true);

  connect(job, SIGNAL(data(KIO::Job*, const QByteArray&)),
          SLOT(slotReceiveThreadData(KIO::Job*, const QByteArray&)));
  connect(job, SIGNAL(result(KIO::Job*)), SLOT(slotThreadResult(KIO::Job*)));

  // use 'HTTP-Headers' metadata.
  job->addMetaData("PropagateHttpHeader", "true");
}

void KitaThreadView::slotReceiveThreadData(KIO::Job* job, const QByteArray& data)
{
  QCString cstr(data.data(), data.size()+1);
  m_threadData.append(cstr);
}

void KitaThreadView::slotThreadResult(KIO::Job* job)
{
  if(job->error()) {
    job->showErrorDialog();
    return;
  }

  QCp932Codec codec;
  QTextStream stream(m_threadData, IO_ReadOnly);
  stream.setCodec(&codec);
  QString line;
  QString text;
  int num = 1;

  // parse HTTP headers
  QStringList headerList = QStringList::split("\n", job->queryMetaData("HTTP-Headers"));
  QRegExp regexp("Date: (...), (..) (...) (....) (..:..:..) .*");
  QString dateStr = headerList.grep(regexp)[0];
  if(regexp.search(dateStr) == -1) {
    // invalid date format
    qDebug("slotThreadResult() invalid Date:");
    return;
  }

  // I hate this format ;p
  QString usLocalDateStr = regexp.cap(1) + " " + regexp.cap(3) + " " +
                           regexp.cap(2) + " " + regexp.cap(5) + " " + regexp.cap(4);

  // 1970/01/01 00:00:00 GMT
  QDateTime zeroTime(QDate(1970, 1, 1), QTime(0, 0));

  serverTime = zeroTime.secsTo(QDateTime::fromString(usLocalDateStr));

  while(!stream.atEnd()) {
    line = stream.readLine();
    QStringList list = QStringList::split("<>", line, true);

    if(list.size() != 5) {
      // probably eof
      break;
    }
    QString name = list[0];
    QString address = list[1];
    QString dateId = list[2];
    QString body = list[3];
    QString subject = list[4];

    if(num == 1) {
      subjectLabel->setText(QString("[") + m_boardName + QString("] ") + subject);
    }
    text += "<dl><dt>";
    text += QString::number(num);
    text += " ";
    if(address != "") {
      text += "<a href='mailto:" + address + "' title='" + address + "'>" + name + "</a>";
    } else {
      text += name;
    }
    text += " ";
    text += dateId;
    text += "</dt><dd>";
    text += linkURL(body);
    text += "<br><br></dd></dl>";

    num++;
  }
  threadPart->begin(KURL(m_threadURL));
  threadPart->write(text);
  threadPart->end();

  m_threadData.truncate(0);
}

void KitaThreadView::slotWriteButtonClicked()
{
  KitaWriteDialog dialog;

  QSjisCodec cp932Codec;
  int cp932MIB = cp932Codec.mibEnum();
  KURL bbscgiURL = KURL(m_boardURL, "../test/bbs.cgi");

  if(dialog.exec() != QDialog::Accepted) {
    return;
  }

  QString postStr;
  QTextStream stream(&postStr, IO_WriteOnly);
  QString name = KURL::encode_string(dialog.name(), cp932MIB);
  QString mail = KURL::encode_string(dialog.mail(), cp932MIB);
  QString body = KURL::encode_string(dialog.body(), cp932MIB);

  stream << "submit=%91%53%90%d3%94%43%82%f0%95%89%82%a4%82%b1%82%c6"; // zennsekininnwooukoto
  stream << "%82%f0%8f%b3%91%f8%82%b5%82%c4%8f%91%82%ab%8d%9e%82%de&"; // wosyoudakusitekakikomu
  stream << "subject=&";
  stream << "FROM=" << name << "&";
  stream << "mail=" << mail << "&";
  stream << "MESSAGE=" << body << "&";
  stream << "bbs=" << m_boardID << "&";
  stream << "key=" << m_datName << "&";
  stream << "time=" << serverTime;

  KIO::TransferJob* job = KIO::http_post(bbscgiURL, postStr.utf8(), true);

  // see kdelibs/kioslave/http/http.cc:2216
  job->addMetaData("content-type", "Content-type: application/x-www-form-urlencoded");

  job->addMetaData("referrer", KURL(m_boardURL,"./index2.html").prettyURL());

  connect(job, SIGNAL(data(KIO::Job*, const QByteArray&)),
          SLOT(slotReceiveWriteResult(KIO::Job*, const QByteArray&)));
  connect(job, SIGNAL(result(KIO::Job*)), SLOT(slotWriteResult(KIO::Job*)));
}

void KitaThreadView::slotReceiveWriteResult(KIO :: Job * job,  const QByteArray & data)
{
  QCString cstr(data.data(), data.size()+1);
  m_writeResult += cstr;
}

void KitaThreadView::slotWriteResult(KIO::Job* job)
{
  if(job->error()) {
    job->showErrorDialog();
  } else {
    QSjisCodec codec;
    QTextStream stream(m_writeResult, IO_ReadOnly);
    stream.setCodec(&codec);
    QString resultHtml = stream.read();

    // parse result

    // parse 2ch_X
    QString _2ch_X;
    QRegExp regexp_2ch_X("<!-- 2ch_X:(.*) -->");
    int pos = regexp_2ch_X.search(resultHtml);
    if(pos != -1) {
      _2ch_X = regexp_2ch_X.cap(1);
    } else {
      // not match
      _2ch_X = QString::null;
    }

    QString title;
    QRegExp regexp_title("<title>(.*)</title>");
    pos = regexp_title.search(resultHtml);
    if(pos != -1) {
      title = regexp_title.cap(1);
    } else {
      // not match
      title = QString::null;
    }

    QString errMsg;
    QRegExp regexp_message("<b>([^<]*)</b>"); // Probably, the first bold-decorated text is error message.
    pos = regexp_message.search(resultHtml);
    if(pos != -1) {
      errMsg = regexp_message.cap(1);
    } else {
      // not match
      errMsg = QString::null;
    }

    if(_2ch_X == "true") {} else if(_2ch_X == "false") {} else if(_2ch_X == "error") {
      KMessageBox::error(0, errMsg, title);
    } else if(_2ch_X == "check") {} else if(_2ch_X == "cookie") {} else {
      // probably ok.
    }

    qDebug("#%s#", (const char *)resultHtml.local8Bit());
    m_writeResult.truncate(0);
  }
}

void KitaThreadView::slotCanceled(KIO::Job* job)
{}

void KitaThreadView::slotOnURL(const QString& url)
{
  emit signalChangeStatusbar(url);
}

void KitaThreadView::slotOpenURLRequest(const KURL& url)
{
  new KRun(url);
}
