/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*   linux 󥫥פ餰ο                                        *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitathreadview.h"

#include <kpopupmenu.h>
#include <krun.h>
#include <kurl.h>
#include <khtmlview.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kaction.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kapplication.h>
#include <ktextbrowser.h>
#include <kcombobox.h>

#include <dom/html_inline.h>
#include <dom/html_base.h>
#include <dom/html_list.h>

#include <qclipboard.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <qprogressdialog.h>
#include <qcursor.h>
#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qdatetime.h>
#include <qeucjpcodec.h>
#include <qsimplerichtext.h>
#include <qpopupmenu.h>

#include "kitahtmlpart.h"

#include "libkita/kitaconfig.h"
#include "libkita/kita-utf8.h"
#include "libkita/qcp932codec.h"
#include "libkita/favoritethreads.h"
#include "libkita/favoriteboards.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"
#include "libkita/signalcollection.h"
#include "libkita/kita_misc.h"

#define MAX_LABEL_LENGTH 60


KitaThreadView::KitaThreadView( QWidget* parent, const char* name )
        : KitaThreadViewBase( parent, name )
{
    m_threadPart = new KitaHTMLPart( threadFrame );
    QHBoxLayout* aLayout = new QHBoxLayout( threadFrame );
    aLayout->addWidget( m_threadPart->view() );

    {
        subjectLabel = new Kita::KitaSubjectLabel( Subjectframe );
        QHBoxLayout* LBox = new QHBoxLayout( Subjectframe );
        LBox->addWidget( subjectLabel );
        connect( subjectLabel, SIGNAL( urlClick( const QString& ) ),
                 SLOT( subjectlinkClicked( const QString& ) ) );
        connect( subjectLabel, SIGNAL( urlMClick( const QString& ) ),
                 SLOT( subjectlinkMClicked( const QString& ) ) );
    }

    {
        SearchButton->setPixmap( SmallIcon( "find" ) );
        HighLightButton->setPixmap( SmallIcon( "idea" ) );
        ReloadButton->setPixmap( SmallIcon( "reload" ) );
        GobackAnchorButton->setPixmap( SmallIcon( "2leftarrow" ) );
        BookmarkButton->setPixmap( SmallIcon( "bookmark_add" ) );
        writeButton->setPixmap( SmallIcon( "pencil" ) );
        deleteButton->setPixmap( SmallIcon( "edittrash" ) );
        closeButton->setPixmap( SmallIcon( "fileclose" ) );
    }

    setAcceptDrops( true ); // DND Drop eneble: 2nd stage. - enable on "KitaThreadView" widget and disable on the others(child widgets of "KitaThreadView").
    threadFrame->setAcceptDrops( false ); // don't treat Drop event on child.
    m_threadPart->view() ->setAcceptDrops( false ); // don't treat Drop event on child.

    m_threadPart->enableMetaRefresh( false ); //disable <meta refresh="...">

    connect( deleteButton, SIGNAL( clicked() ),
             SLOT( slotDeleteButtonClicked() ) );
    connect( writeButton, SIGNAL( clicked() ),
             SLOT( slotWriteButtonClicked() ) );
    connect( m_threadPart, SIGNAL( openWriteDialog( QString ) ),
             SLOT( slotWriteButtonClicked ( QString ) ) );
    connect( m_threadPart, SIGNAL( finishReload() ),
             SLOT( slotUpdateInfo() ) );
    connect( m_threadPart, SIGNAL( nodeActivated( const DOM::Node& ) ),
             SLOT( slotDOMNodeActivated( const DOM::Node& ) ) );
    connect( m_threadPart, SIGNAL( mousePressed() ),
             SLOT( setFocus() ) );

    connect( BookmarkButton, SIGNAL( toggled( bool ) ),
             SLOT( slotBookmarkButtonClicked( bool ) ) );
    connect( SearchButton, SIGNAL( clicked() ),
             SLOT( slotSearchButton() ) );
    connect( SearchCombo, SIGNAL( activated( int ) ),
             SLOT( slotSearchButton() ) );
    connect( GobackAnchorButton, SIGNAL( clicked() ),
             m_threadPart, SLOT( slotGobackAnchor() ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( slotReloadButton() ) );
    connect( gotoCombo, SIGNAL( activated( int ) ),
             SLOT( slotComboActivated( int ) ) );
    connect( closeButton, SIGNAL( clicked() ),
             SLOT( slotCloseButton() ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( this, SIGNAL( setMainStatusbar( const QString& ) ),
             signalCollection, SIGNAL( setMainStatusbar ( const QString& ) ) );
    connect( this, SIGNAL( bookmarked( const QString&, bool ) ),
             signalCollection, SIGNAL( bookmarked( const QString&, bool ) ) );
    connect( this, SIGNAL( setMainURLLine( const KURL& ) ),
             signalCollection, SIGNAL( setMainURLLine( const KURL& ) ) );
    connect( this, SIGNAL( updateSubjectTab( const KURL& ) ),
             signalCollection, SIGNAL( updateSubjectTab( const KURL& ) ) );
    connect( this, SIGNAL( updateThreadTab( const KURL& ) ),
             signalCollection, SIGNAL( updateThreadTab( const KURL& ) ) );
    connect( this, SIGNAL( openURLRequestExt(
                               const KURL&, const KParts::URLArgs&, QString, int, int,
                               const KURL&, const KURL&, const QString&, const QString& ) ),
             signalCollection, SIGNAL( openURLRequestExt(
                                           const KURL& , const KParts::URLArgs&, QString, int, int,
                                           const KURL&, const KURL&, const QString&, const QString& ) ) );
    connect( this, SIGNAL( setMainCaption( const QString& ) ),
             signalCollection, SIGNAL( setMainCaption( const QString& ) ) );
    connect( this, SIGNAL( closeThreadTab( const KURL& ) ),
             signalCollection, SIGNAL( closeThreadTab( const KURL& ) ) );

    connect( this, SIGNAL( activateThreadView( const KURL& ) ),
             signalCollection, SIGNAL( activateThreadView( const KURL& ) ) );

    /* Kita Navi */
    connect( this, SIGNAL( showKitaNaviByWord( const KURL& , QString ) ),
             signalCollection, SIGNAL( showKitaNaviByWord( const KURL& , QString ) ) );

    /* write dock */
    connect( this, SIGNAL( showWriteView( const KURL& , const QString& ) ),
             signalCollection, SIGNAL( showWriteView( const KURL& , const QString& ) ) );
    connect( this, SIGNAL( createNewThread( const KURL& , const QString&, const QString& ) ),
             signalCollection, SIGNAL( createNewThread( const KURL& , const QString&, const QString& ) ) );
    

    m_viewmode = VIEWMODE_MAINVIEW;
    m_rescode = 200;
    m_serverTime = 0;
    m_datURL = QString::null;

}

KitaThreadView::~KitaThreadView()
{

    if ( m_threadPart ) {
        delete m_threadPart;
        m_threadPart = NULL;
    }
}

const QString KitaThreadView::threadName() const
{
    return Kita::DatManager::threadName( m_datURL );
}

const KURL KitaThreadView::threadURL() const
{
    return Kita::getThreadURL( m_datURL );
}

const KURL KitaThreadView::datURL() const
{
    return m_datURL;
}

void KitaThreadView::slotDOMNodeActivated( const DOM::Node& node )
{
    { //process Anchor tags. Anchor tags not proccessed here cause 'emit KParts::BrowserExtention::openURLRequest()'
        DOM::HTMLAnchorElement anchor = node;

        if ( ! anchor.href().isEmpty() ) {
            kdDebug() << "AnchorNodeActivated::" << endl;
        } // end: anchor.href().isEmpty()
    } // end of Anchor tags.
}

void KitaThreadView::setSubjectLabel( const QString& boardName, const QString& threadName, const QString boardURL )
{
    QString disp;
    if ( boardName.isEmpty() ) {
        disp = threadName;
    } else {
        disp = QString( "<a href=\"%1\">[%2]</a> %3" ).arg( boardURL ).arg( boardName ).arg( threadName );
    }

    //disp.truncate( MAX_LABEL_LENGTH );
    subjectLabel->setText( disp );
}

void KitaThreadView::subjectlinkClicked( const QString& URL )
{
    kdDebug() << QString( "Jump board: \"%1\"" ).arg( URL ) << endl;
    emit openURLRequestExt( URL, KParts::URLArgs(), "kita_open_2chboard", KitaConfig::alwaysUseTab() );
}

void KitaThreadView::subjectlinkMClicked( const QString &URL )
{
    kdDebug() << QString( "Jump board as new tab: \"%1\"" ).arg( URL ) << endl;
    emit openURLRequestExt( URL, KParts::URLArgs(), "kita_open_2chboard", 1 );
}

void KitaThreadView::updateButton()
{
    writeButton->setEnabled( true );
    BookmarkButton->setEnabled( true );
    ReloadButton->setEnabled( true );
    deleteButton->setEnabled( true );
    closeButton->setEnabled( true );

    if ( HighLightButton->isOn() ) {
        HighLightButton->toggle();
    }

    // don't emit SIGNAL( toggled() )
    disconnect( BookmarkButton, SIGNAL( toggled( bool ) ), this, SLOT( slotBookmarkButtonClicked( bool ) ) );
    if ( FavoriteThreads::getInstance() ->contains( m_datURL.prettyURL() ) ) {
        BookmarkButton->setOn( true );
    } else {
        BookmarkButton->setOn( false );
    }
    connect( BookmarkButton, SIGNAL( toggled( bool ) ), SLOT( slotBookmarkButtonClicked( bool ) ) );
}


/*--------------------*/
/* write response     */
/*--------------------*/  /* private slots */
void KitaThreadView::slotWriteButtonClicked( QString resstr )
{
    emit showWriteView( m_datURL, resstr );
}


void KitaThreadView::insertSearchCombo()
{
    for ( int count = 0; count < SearchCombo->count(); ++count ) {
        if ( SearchCombo->text( count ) == SearchCombo->currentText() ) {
            // found
            return ;
        }
    }
    SearchCombo->insertItem( SearchCombo->currentText() );
}

QStringList KitaThreadView::parseSearchQuery( const QString& input ) const
{
    QStringList tmp = QStringList::split( ' ', input );
    QStringList ret_list;
    QRegExp truncSpace( "\\s*$" );
    QStringList::iterator it = tmp.begin();
    for ( ; it != tmp.end(); ++it )
        ret_list += ( *it ).replace( truncSpace, "" );
    return ret_list;
}


void KitaThreadView::slotPopupMenu( KXMLGUIClient* client, const QPoint& global, const KURL& url, const QString& mimeType, mode_t mode )
{
    KActionCollection * collection = client->actionCollection();
    KAction* action;
    action = new KAction( i18n( "goback anchor" ), SmallIcon( "idea" ), KShortcut(), m_threadPart, SLOT( gobackAnchor() ), collection, "goback_anchor" );
    emit popupMenu( client, global, url, mimeType, mode );
}

void KitaThreadView::setFont( const QFont& font )
{
    //  m_threadPart->setStandardFont(font.family());
    subjectLabel->setFont( font );

    DOM::CSSStyleDeclaration style = m_threadPart->htmlDocument().body().style();
    style.setProperty( "font-family", font.family(), "" );
    style.setProperty( "font-size", QString( "%1pt" ).arg( font.pointSize() ), "" );
}


void KitaThreadView::slotBookmarkButtonClicked( bool on )
{
    emit bookmarked( m_datURL.prettyURL(), on );
}


void KitaThreadView::focusSearchCombo()
{
    if ( ! SearchCombo->hasFocus() ) {
        SearchCombo->setFocus();
    } else {
        setFocus();
    }
}


/* public slot */ /* virtual */
void KitaThreadView::setFocus()
{
    emit activateThreadView( m_datURL );
    showStatusBar( QString::null );
    m_threadPart->view() ->setFocus();
}

namespace Kita
{

    KitaSubjectLabel::KitaSubjectLabel( QWidget* parent, const char * name )
            : KTextBrowser( parent, name )
    {
        setTextFormat( Qt::RichText );
        setVScrollBarMode( QScrollView::AlwaysOff );
        setHScrollBarMode( QScrollView::AlwaysOff );
        setSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum );
        setNotifyClick( TRUE );
        setFrameStyle( QFrame::NoFrame );
        paletteChanged();
        if ( kapp ) {
            connect( kapp, SIGNAL( kdisplayPaletteChanged() ),
                     this, SLOT( paletteChanged() ) );
        }
    }

    void KitaSubjectLabel::paletteChanged()
    {
        QPalette p = kapp ? kapp->palette() : palette();
        p.setBrush( QColorGroup::Base, p.brush( QPalette::Normal, QColorGroup::Background ) );
        p.setColor( QColorGroup::Text, p.color( QPalette::Normal, QColorGroup::Foreground ) );
        setPalette( p );
    }

    QSize KitaSubjectLabel::minimumSizeHint() const
    {
        QSize ms = minimumSize();
        if ( ( ms.width() > 0 ) && ( ms.height() > 0 ) )
            return ms;

        int w = 400;
        if ( ms.width() > 0 )
            w = ms.width();

        QString txt = text();
        QSimpleRichText rt( txt, font() );
        rt.setWidth( w - 2 * frameWidth() - 10 );
        w = 10 + rt.widthUsed() + 2 * frameWidth();
        if ( w < ms.width() )
            w = ms.width();
        int h = rt.height() + 2 * frameWidth();
        if ( h < ms.height() )
            h = ms.height();

        return QSize( w, h );
    }

    QSize KitaSubjectLabel::sizeHint() const
    {
        return minimumSizeHint();
    }
    void KitaSubjectLabel::contentsMousePressEvent ( QMouseEvent * e )
    {
        KTextBrowser::contentsMousePressEvent( e );
        kdDebug() << "Press Button:" << QString::number( e->button() ) << endl;
        if ( e->button() == Qt::MidButton ) {
            QString t = text();
            QSimpleRichText rt( t, font() );
            m_popupURL = rt.anchorAt( e->pos() );
        }
    }

    void KitaSubjectLabel::contentsMouseReleaseEvent ( QMouseEvent * e )
    {
        KTextBrowser::contentsMouseReleaseEvent( e );
        if ( !m_popupURL.isEmpty() && e->button() == Qt::MidButton )
            emit urlMClick( m_popupURL );
    }

    QPopupMenu * KitaSubjectLabel::createPopupMenu ( const QPoint & pos )
    {
        QString t = text();
        QSimpleRichText rt( t, font() );
        m_popupURL = rt.anchorAt( pos );
        kdDebug() << QString( "popupMenu for \"%1\"" ).arg( m_popupURL ) << endl
        ;
        if ( !m_popupURL.isEmpty() ) {
            QPopupMenu * popup = new QPopupMenu;
            popup->insertItem( i18n( "Open with new tab" ), this, SLOT( popupOpenAsNewTab() ) );
            popup->insertItem( i18n( "Copy title and URL" ), this, SLOT( popupCopyToClipboard() ) );
            QString ft = Kita::FavoriteBoards::boards().contains( KURL( m_popupURL ) ) ?
                         i18n( "Remove from Favorites" ) : i18n( "Add to Favorites" ) ;
            popup->insertItem( ft, this, SLOT( popupFavorites() ) );
            return popup;
        } else return NULL;
    }

    void KitaSubjectLabel::popupOpenAsNewTab()
    {
        if ( !m_popupURL.isEmpty() )
            emit urlMClick( m_popupURL );
    }

    void KitaSubjectLabel::popupCopyToClipboard()
    {
        QClipboard * clipboard = QApplication::clipboard();
        QString text = QString( "%1\n%2" ).arg( Kita::BoardManager::boardName( m_popupURL ) ).arg( m_popupURL );
        kdDebug() << "popupCopytoClipboard :" << endl << text << endl;
        clipboard->setText( text , QClipboard::Clipboard );
        clipboard->setText( text , QClipboard::Selection );
    }

    void KitaSubjectLabel::popupFavorites()
    {
        if ( !m_popupURL.isEmpty() ) {
            KURL u = KURL( m_popupURL );
            if ( Kita::FavoriteBoards::boards().contains( u ) )
                Kita::FavoriteBoards::remove( u );
            else
                Kita::FavoriteBoards::append( u );
        }
    }

}


/*-------------------------------------------------*/
/* shobon extension                                */
/*-------------------------------------------------*/


/*-------*/
/* setup */
/*-------*/
void KitaThreadView::setup( const KURL& datURL, int mode )
{
    /* config. */

    /*---------------------------------------*/
    /* setup                                 */

    m_datURL = Kita::getDatURL( datURL );
   
    /* setup HTMLPart */
    int partMode = HTMLPART_MODE_MAINPART;
    if ( mode == VIEWMODE_KITANAVI ) partMode = HTMLPART_MODE_NAVI;
    m_threadPart->setup( partMode, m_datURL );

    /* mode. Mode is defined in kitathreadview.h */
    m_viewmode = mode;

    /* reset search direction */
    m_revsearch = FALSE;

    /* load SETTING.TXT */
    Kita::BoardManager::loadBBSSetting( m_datURL, FALSE );
}


/*--------------------------------------------------------*/
/* Show thread                                            */
/* This function is called from KitaThreadTabWidget class */
/*--------------------------------------------------------*/
void KitaThreadView::showThread( const KURL& datURL, int num )
{
    /* If this widget is not parent, then do nothing. */
    if ( m_viewmode != VIEWMODE_MAINVIEW ) return ;

    if ( num == 0 ) num = Kita::DatManager::getViewPos( datURL );

    if ( topLevelWidget() ->isMinimized() ) topLevelWidget() ->showNormal();
    topLevelWidget() ->raise();
    setActiveWindow();

    /* setup */
    setup( datURL, VIEWMODE_MAINVIEW );

    /* get log from cahce */
    m_rescode = 200;
    if ( !m_threadPart->load( num ) ) showStatusBar( "" );

    /* update data */
    slotUpdateInfo();
    slotReloadButton();
}



/*---------*/
/* reload  */
/*---------*/ /* public slot */
void KitaThreadView::slotReloadButton( int jumpNum )
{
    topLevelWidget() ->raise();
    setActiveWindow();

    if ( m_threadPart->reload( jumpNum ) ) {
        showStatusBar( Kita::utf8ToUnicode( KITAUTF8_NOWRENEW ) );
    }
}


/*-----------------------------------*/
/* stop loading the thread           */ /* public slot */
void KitaThreadView::slotStopLoading()
{
    /* hide popup */
    if ( m_threadPart->isPopupVisible() ) {
        m_threadPart->slotDeletePopup();
        return ;
    }

    /* unforcus search combo */
    if ( SearchCombo->hasFocus() ) {
        setFocus();
        return ;
    }

    Kita::DatManager::stopLoading( m_datURL );
}


/*-----------------------------------------*/
/* show the information at the statusbar.  */
/*-----------------------------------------*/
void KitaThreadView::showStatusBar( QString info )
{
    if ( m_datURL.isEmpty() ) return ;
    QString captionStr = QString::null;
    QString infostr = QString::null;
    QString errstr = QString::null;
    int viewPos = Kita::DatManager::getViewPos( m_datURL );
    int resNum = Kita::DatManager::getResNum( m_datURL );
    bool broken = Kita::DatManager::isBroken( m_datURL );
    int datSize = Kita::DatManager::getDatSize( m_datURL );

    switch ( m_viewmode ) {

    case VIEWMODE_MAINVIEW:

        errstr = QString::null;
        if ( m_rescode != 200 && m_rescode != 206 && m_rescode != 0 )
            errstr = QString( "Error %1" ).arg( m_rescode );
        if ( broken ) info += " This thread is broken.";

        /* show status bar,caption, url  */
        infostr = Kita::DatManager::threadName( m_datURL ) +
                  QString( " [Total: %1 New: %2] %3 k" ).arg( resNum ).arg( resNum - viewPos ).arg( datSize / 1024 )
                  + info + " " + errstr;

        captionStr = Kita::DatManager::threadName( m_datURL ) + QString( " (%1)" ).arg( viewPos );

        emit setMainCaption( captionStr );
        emit setMainStatusbar( infostr );
        emit setMainURLLine( Kita::getThreadURL( m_datURL ) );

        return ;
        break;

    default:

        break;
    }
}




/*--------------------*/
/* update information */ /* public */
void KitaThreadView::slotUpdateInfo()
{
    m_rescode = Kita::DatManager::getResponseCode( m_datURL );
    m_serverTime = Kita::DatManager::getServerTime( m_datURL );

    /* uptate informations */
    setSubjectLabel( Kita::BoardManager::boardName( m_datURL ),
                     Kita::DatManager::threadName( m_datURL )
                     + QString( " (%1)" )
                     .arg( Kita::DatManager::getReadNum( m_datURL ) ),
                     Kita::BoardManager::boardURL( m_datURL ) );
    updateButton();

    gotoCombo->clear();
    gotoCombo->insertItem( Kita::utf8ToUnicode( KITAUTF8_GOTO ) );
    gotoCombo->insertItem( Kita::utf8ToUnicode( KITAUTF8_KOKOYON ) );
    for ( int i = 1; i < Kita::DatManager::getReadNum( m_datURL ); i += 100 ) {
        gotoCombo->insertItem( QString().setNum( i ) + "-" );
    }
    gotoCombo->insertItem( Kita::utf8ToUnicode( KITAUTF8_SAIGO ) );
    gotoCombo->adjustSize();

    emit updateSubjectTab( m_datURL );
    emit updateThreadTab( m_datURL );

    showStatusBar( "" );

    emit showThreadCompleted(); /* to KitaThreadPart */
}




/*------------------------*/
/* search function        */
/*------------------------*/  /* private slots */
void KitaThreadView::slotSearchButton()
{
    if ( m_datURL.isEmpty() ) return ; /* Nothing is shown on the screen.*/

    QString str = SearchCombo->currentText();
    if ( str.at( 0 ) == ':' ) {

        /* show res popup */
        if ( str.at( 1 ) == 'p' ) {
            int refNum = str.mid( 2 ).toInt();
            QPoint pos = mapToGlobal( SearchCombo->pos() );
            pos.setY( pos.y() + SearchCombo->height() / 2 );
            m_threadPart->slotShowResPopup( pos , refNum, refNum );
            return ;
        }
        /* extract responses by keywords */
        else if ( str.at( 1 ) == '?' ) {
            QString query = str.mid( 2 );
            emit showKitaNaviByWord( m_datURL, query );
            return ;
        }
        /* find by find dialog */
        else if ( str.at( 1 ) == 'f' ) {
            KAction * act = static_cast< KAction* >( m_threadPart->action( "find" ) );
            if ( act ) act->activate();
            return ;
        }
        /* create new thread */
        else if ( str.mid( 1, 3 ) == "new" ) {
	    emit createNewThread( m_datURL, QString::null, QString::null );
            return ;
        }	

        /* jump */
        QString anc = str.mid( 1 );
        m_threadPart->gotoAnchor( anc, FALSE );
        SearchCombo->setFocus();
        return ;
    }

    /* if this is parent, then show all responses, and search */
    if ( m_viewmode == VIEWMODE_MAINVIEW ) m_threadPart->showAll();

    slotSearchPrivate( FALSE );
}

/* public slot */
void KitaThreadView::slotSearchNext() { slotSearchPrivate( FALSE ); }
void KitaThreadView::slotSearchPrev() { slotSearchPrivate( TRUE ); }

/* private */
void KitaThreadView::slotSearchPrivate( bool rev )
{
    if ( m_datURL.isEmpty() ) return ; /* Nothing is shown on the screen.*/

    /* jump */
    QString str = SearchCombo->currentText();
    if ( str == QString::null ) return ;
    if ( str == "" ) return ;
    if ( str.at( 0 ) == ':' ) return ;
    if ( str.at( 0 ) == '?' ) return ;

    /* if this is parent, then show all responses, and search */
    if ( m_viewmode == VIEWMODE_MAINVIEW ) m_threadPart->showAll();

    insertSearchCombo();
    QStringList list = parseSearchQuery( SearchCombo->currentText() );
    m_threadPart->findText( SearchCombo->currentText(), rev );
    SearchCombo->setFocus();
}

/* public slot */
void KitaThreadView::slotGobackAnchor() { m_threadPart->slotGobackAnchor(); }
void KitaThreadView::slotGotoHeader() { m_threadPart->gotoAnchor( "header", FALSE );}
void KitaThreadView::slotGotoFooter() { m_threadPart->slotClickGotoFooter(); }

// vim:sw=2:

void KitaThreadView::slotComboActivated( int index )
{
    if ( index == gotoCombo->count() - 1 ) {
        // last
        m_threadPart->gotoAnchor( "footer", FALSE );
    } else if ( index == 1 ) {
        // kokomade yonda
        m_threadPart->gotoAnchor( "kokomade_yonda", FALSE );
    } else if ( index != 0 ) {
        QString numText = gotoCombo->text( index );
        numText.truncate( numText.length() - 1 );
        m_threadPart->gotoAnchor( numText, FALSE );
    }
}

void KitaThreadView::slotDeleteButtonClicked()
{
    if ( m_datURL.isEmpty() ) return ;

    int rescode = Kita::DatManager::getResponseCode( m_datURL );
    if ( ( rescode != 200 && rescode != 206 )
            || FavoriteThreads::getInstance() ->contains( m_datURL.prettyURL() ) ) {
        if ( QMessageBox::warning( this,
                                   "Kita",
                                   i18n( "Do you want to delete Log ?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                != QMessageBox::Ok ) return ;
    }

    if( Kita::DatManager::deleteCache( m_datURL ) ){
	emit closeThreadTab( m_datURL );
	emit updateSubjectTab( m_datURL );
    }
}

void KitaThreadView::slotCloseButton()
{
    emit closeCurrentTab();
}

const QString KitaThreadView::selectedText() const
{
    return m_threadPart->selectedText();
}

/*--------------------------------------------------------*/
/* These public functions are used for KitaNavi & preview */
/*--------------------------------------------------------*/

/*---------------------*/
/* set data & parse it */
/*---------------------*/ /* public */
void KitaThreadView::domAppendData( QString& , int bufnum , int )
{
    m_threadPart->showResponses( bufnum, bufnum );
}


/* public */
void KitaThreadView::domDeleteRes( int )
{}


/* public */
void KitaThreadView::domAppendBelt( QString idstr )
{
    m_threadPart->insertBeltNode( idstr );
}

/* public */
void KitaThreadView::domSetInnerHTML( const QString& html )
{
    m_threadPart->setInnerHTML( html );
    showStatusBar( "" );
}

/* public */
void KitaThreadView::domApplyChange( QString labelStr,
                                     QString headerStr,
                                     QString footerStr,
                                     QString anchorToJump,
                                     QString comboStr
                                   )
{

    /* label, header & footer */
    if ( headerStr != NULL ) m_threadPart->insertHeaderNode( headerStr );
    if ( footerStr != NULL ) m_threadPart->insertFooterNode( footerStr );
    if ( comboStr != NULL ) SearchCombo->insertItem( comboStr );
    if ( labelStr != QString::null ) {
        QString str = QString( "<a href=\"%1\">[%2]</a> " )
                      .arg( Kita::BoardManager::boardURL( m_datURL ) )
                      .arg( Kita::BoardManager::boardName( m_datURL ) ) + labelStr;
        subjectLabel->setText( str );
    }

    /* update display */
    m_threadPart->updateScreen( FALSE, FALSE );
    if ( anchorToJump != NULL ) m_threadPart->gotoAnchor( anchorToJump, FALSE );

    updateButton();
    showStatusBar( "" );
}
