/***************************************************************************
*   Copyright (C) 2004 by Hideki Ikemoto , (c) 2004 by 421                *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* Image Viewer Widget */

#include "kitaimgview.h"
#include "libkita/imgmanager.h"

#include <kpopupmenu.h>
#include <kaction.h>
#include <khtml_part.h>
#include <khtmlview.h>

#include <qpainter.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qlayout.h>

/*---------------------------------------------------------------*/


KitaImgView::KitaImgView( const KURL& url, const KURL& datURL, KActionCollection* action,
                          QWidget* parent, const char* name, WFlags fl )
        : QScrollView( parent, name, fl )
{
    m_url = url;
    m_datURL = datURL;
    m_htmlPart = NULL;
    m_layout = NULL;
    m_action = action;
    m_popup = NULL;
    m_message = QString::null;
    m_pixmap = NULL;

    m_size = -100;
    m_imgwd = m_imght = 0;
    m_custom_wd = m_custom_ht = 0;

    viewport() ->setEraseColor( Qt::black );
    viewport() ->setPaletteBackgroundColor( Qt::black );
}


KitaImgView::~KitaImgView()
{
    if ( m_pixmap ) delete m_pixmap;
    if ( m_popup ) delete m_popup;
    if ( m_layout ) delete m_layout;
    if ( m_htmlPart ) delete m_htmlPart;
}


/* public */
const KURL KitaImgView::url() const
{
    return m_url;
}


/* public */
const KURL KitaImgView::datURL() const
{
    return m_datURL;
}


/* public */
void KitaImgView::showMessage( QString message )
{
    m_message = message;
    repaintContents();
}


/* show image */ /* public slot */
void KitaImgView::slotShowImage()
{
    const int defaultsize = 0;

    if ( m_size < -1 ) m_size = defaultsize; /* size is not initialized. */

    bool usekhtmlpart = Kita::ImgManager::isAnimationGIF( m_url );
    if ( m_htmlPart ) usekhtmlpart = TRUE;
    if ( Kita::ImgManager::mosaic( m_url ) ) usekhtmlpart = FALSE;

    if ( usekhtmlpart ) showImageWithKHTML();

    if ( resizeImage() ) {
        myResizeContents();
        repaintContents();
    }
}


/* public slot */
void KitaImgView::slotFitToWin()
{
    if ( m_size == 0 ) return ;

    m_size = 0;
    slotShowImage();
}


/* public slot */
void KitaImgView::slotOriginalSize()
{
    if ( m_size == 100 ) return ;

    m_size = 100;
    slotShowImage();
}


/* public slot */
void KitaImgView::slotSetSize( int size )
{
    if ( m_size == size ) return ;

    m_size = size;
    slotShowImage();
}


/* public slot */
void KitaImgView::slotCustomSize( int wd, int ht )
{
    const int minsize = 8;

    if ( wd == m_custom_wd && ht == m_custom_ht ) return ;

    m_custom_wd = QMAX( minsize, wd );
    m_custom_ht = QMAX( minsize, ht );
    m_size = -1;
    slotShowImage();
}


/* public slot */
void KitaImgView::slotZoomIn()
{
    const int maxsize = 400;

    if ( m_size >= maxsize ) return ;

    /* current mode is "fit to win" or "custom size" */
    if ( m_size <= 0 && m_imgwd ) m_size = QMIN( 100, m_imgwd * 100 / Kita::ImgManager::width( m_url ) );

    m_size += 10;
    slotShowImage();
}


/* public slot */
void KitaImgView::slotZoomOut()
{
    const int minsize = 10;

    if ( m_size > 0 && m_size <= minsize ) return ;

    /* current mode is "fit to win" or "custom size" */
    if ( m_size <= 0 && m_imgwd ) m_size = QMIN( 100, m_imgwd * 100 / Kita::ImgManager::width( m_url ) );

    m_size -= 10;
    slotShowImage();
}



/* virtual protected */
void KitaImgView::resizeEvent( QResizeEvent * e )
{
    QScrollView::resizeEvent( e );

    if ( m_htmlPart ) showImageWithKHTML();

    else if ( m_pixmap ) {
        if ( m_size == 0 ) resizeImage();  /* fit to window */
        myResizeContents();

        /* drawContents is called later. */
    }
}


/* virtual protected */
void KitaImgView::drawContents( QPainter * p, int clipx, int clipy, int clipw, int cliph )
{
    if ( m_htmlPart ) return ; /* use KHTMLPart */

    if ( m_pixmap == NULL ) {
        p->setPen( Qt::white );
        if ( m_message != QString::null ) p->drawText( 16, 16, m_message );
    } else {

        p->fillRect( clipx, clipy, clipw, cliph, QColor( "black" ) );

        int x = QMAX( 0, ( QMAX( width(), m_imgwd ) - m_imgwd ) / 2 );
        int y = QMAX( 0, ( QMAX( height(), m_imght ) - m_imght ) / 2 );
        int sx = QMAX( 0, clipx - x );
        int sy = QMAX( 0, clipy - y );
        int sw = QMIN( clipw, x + m_imgwd - clipx );
        int sh = QMIN( cliph, y + m_imght - clipy );

        if ( x + m_imgwd <= clipx || x >= clipx + clipw
                || y + m_imght <= clipy || y >= clipy + cliph ) return ;

        x = QMAX( x, clipx );
        y = QMAX( y, clipy );
        p->drawPixmap( x, y, *m_pixmap, sx, sy, sw, sh );
    }
}


/* virtual protected */
void KitaImgView::contentsMousePressEvent( QMouseEvent * e )
{
    if ( e->button() & Qt::RightButton ) {
        QString dummy;
        slotPopupMenu( dummy, e->globalPos() );
        return ;
    }

    QScrollView::contentsMousePressEvent( e );
}


/* private */
bool KitaImgView::resizeImage()
{
    const int mosaic_wd = 48;
    const int mosaic_ht = 48;

    /*-------------------------------------------*/

    if ( Kita::ImgManager::code( m_url ) != 200 ) return FALSE;

    QString path = Kita::ImgManager::getPath( m_url );
    QImage orgimg = QImage( path );
    if ( orgimg.isNull() ) return FALSE;

    int imgwd = Kita::ImgManager::width( m_url );
    int imght = Kita::ImgManager::height( m_url );

    /* mosaic image */
    if ( Kita::ImgManager::mosaic( m_url ) )
        orgimg = orgimg.scale( mosaic_wd, mosaic_ht, QImage::ScaleMin ).scale( imgwd, imght, QImage::ScaleMin );

    /* resize */
    int wd;
    int ht;
    getImgSize( wd, ht );

    if ( wd != imgwd || ht != imght ) orgimg = orgimg.scale( wd, ht, QImage::ScaleMin );

    /* convert to pixmap */
    if ( m_pixmap == NULL ) m_pixmap = new QPixmap();
    m_pixmap->convertFromImage( orgimg );
    m_imgwd = m_pixmap->width();
    m_imght = m_pixmap->height();

    return TRUE;
}




/* private */
void KitaImgView::getImgSize( int& wd, int& ht )
{
    int imgwd = Kita::ImgManager::width( m_url );
    int imght = Kita::ImgManager::height( m_url );

    wd = imgwd;
    ht = imght;

    if ( m_size == -1 ) {    /* custom size */
        wd = m_custom_wd;
        ht = m_custom_ht;
    } else if ( m_size == 0 ) { /* fit to window */

        if ( width() < imgwd || height() < imght ) {

            const int mrg = 16;

            wd = width() - mrg;
            ht = height() - mrg;
        }
    } else if ( m_size != 100 ) { /* % */
        wd = imgwd * m_size / 100;
        ht = imght * m_size / 100;
    }
}



/* private */
void KitaImgView::myResizeContents()
{
    int wd = QMAX( width(), m_imgwd );
    int ht = QMAX( height(), m_imght );

    resizeContents( wd, ht );

    if ( wd == width() ) setHScrollBarMode( AlwaysOff );
    else setHScrollBarMode( AlwaysOn );

    if ( ht == height() ) setVScrollBarMode( AlwaysOff );
    else setVScrollBarMode( AlwaysOn );

    if ( m_htmlPart ) {

        if ( wd == width() ) m_htmlPart->view() ->setHScrollBarMode( AlwaysOff );
        else m_htmlPart->view() ->setHScrollBarMode( AlwaysOn );

        if ( ht == height() ) m_htmlPart->view() ->setVScrollBarMode( AlwaysOff );
        else m_htmlPart->view() ->setVScrollBarMode( AlwaysOn );
    }
}


/* private slot */
void KitaImgView::slotPopupMenu( const QString& , const QPoint& global )
{
    if ( !m_action ) return ;

    if ( m_popup ) delete m_popup;
    m_popup = new KPopupMenu( this );

    if ( Kita::ImgManager::mosaic( m_url ) ) {
        if ( m_action->action( "imgviewer_cancelmosaic" ) )
            m_action->action( "imgviewer_cancelmosaic" ) ->plug( m_popup );
        m_popup->insertSeparator();
    }

    if ( m_action->action( "imgviewer_fittowin" ) )
        m_action->action( "imgviewer_fittowin" ) ->plug( m_popup );

    if ( m_action->action( "imgviewer_original" ) )
        m_action->action( "imgviewer_original" ) ->plug( m_popup );

    if ( m_action->action( "imgviewer_selectsize" ) )
        m_action->action( "imgviewer_selectsize" ) ->plug( m_popup );

    if ( m_action->action( "imgviewer_zoomin" ) )
        m_action->action( "imgviewer_zoomin" ) ->plug( m_popup );

    if ( m_action->action( "imgviewer_zoomout" ) )
        m_action->action( "imgviewer_zoomout" ) ->plug( m_popup );

    m_popup->insertSeparator();
    if ( m_action->action( "imgviewer_openbrowser" ) )
        m_action->action( "imgviewer_openbrowser" ) ->plug( m_popup );
    if ( m_action->action( "imgviewer_openthread" ) )
        m_action->action( "imgviewer_openthread" ) ->plug( m_popup );
    if ( m_action->action( "imgviewer_save" ) )
        m_action->action( "imgviewer_save" ) ->plug( m_popup );

    m_popup->exec( global );
}


/*----------------------------------------------*/

/* use KHTMLPart ( to show an Animation GIF ) */


bool KitaImgView::showImageWithKHTML()
{
    if ( Kita::ImgManager::code( m_url ) != 200 ) return FALSE;
    QString path = Kita::ImgManager::getPath( m_url );

    /* create KHTMLPart */
    if ( !m_htmlPart ) {
        m_htmlPart = new KHTMLPart( this );
        m_layout = new QVBoxLayout( this, 0, 0, "BoxLayout" );
        m_layout->addWidget( m_htmlPart->view() );

        connect( m_htmlPart, SIGNAL( popupMenu( const QString&, const QPoint& ) ),
                 SLOT( slotPopupMenu( const QString&, const QPoint& ) ) );
    }

    /* show image */
    if ( m_htmlPart ) {

        QString sizetxt = QString::null;
        int wd, ht;
        int imgwd = Kita::ImgManager::width( m_url );
        int imght = Kita::ImgManager::height( m_url );
        m_imgwd = imgwd;
        m_imght = imght;

        getImgSize( wd, ht );

        if ( m_size <= 0 ) { /* "fit to win" or "custom size" */

            if ( ( double ) imght * wd / imgwd >= ht ) {
                sizetxt = "height=" + QString().setNum( ht );
                m_imgwd = ( int ) ( ( double ) imgwd * ht / imght );
                m_imght = ht;
            } else {
                sizetxt = "width=" + QString().setNum( wd );
                m_imgwd = wd;
                m_imght = ( int ) ( ( double ) imght * wd / imgwd );
            }
        } else if ( m_size != 100 ) { /* % */
            sizetxt = "width=" + QString().setNum( wd );
            m_imgwd = wd;
            m_imght = ( int ) ( ( double ) imght * wd / imgwd );
        }

        int x = QMAX( 0, ( QMAX( width(), m_imgwd ) - m_imgwd ) / 2 );
        int y = QMAX( 0, ( QMAX( height(), m_imght ) - m_imght ) / 2 );

        QString style = QString( "body { margin:%1px %2px; background-color:black; }" )
                        .arg( y ).arg( x );

        QString text = "<html><head><style>" + style + "</style></head><body>";
        text += "<img src=\"file:" + path + "\" " + sizetxt + " >";
        text += "</body></html>";

        m_htmlPart->setJScriptEnabled( false );
        m_htmlPart->setJavaEnabled( false );
        m_htmlPart->begin( "file:/dummy.htm" );
        m_htmlPart->write( text );
        m_htmlPart->end();
        m_htmlPart->view() ->show();
    }

    myResizeContents();

    return TRUE;
}

