/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/
#include "threadindex.h"

#include <kconfig.h>

#include "cache.h"
#include "thread.h"
#include "threadinfo.h"
#include "kita_misc.h"

using namespace Kita;

QString ThreadIndex::getSubject( const KURL& url )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    return getSubjectPrivate( config );
}

void ThreadIndex::setSubject( const KURL& url, const QString& str )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    setSubjectPrivate( str, config );
}

int ThreadIndex::getResNum( const KURL& url )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    return getResNumPrivate( url, config );
}

void ThreadIndex::setResNum( const KURL& url, int resNum )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    setResNumPrivate( resNum, config );
}

int ThreadIndex::getReadNum( const KURL& url )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    return getReadNumPrivate( url, config );
}

void ThreadIndex::setReadNum( const KURL& url, int readNum )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    setReadNumPrivate( readNum, config );
}

int ThreadIndex::getViewPos( const KURL& url )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    return getViewPosPrivate( config );
}

void ThreadIndex::setViewPos( const KURL& url, int viewPos )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
    setViewPosPrivate( viewPos, config );
}


/*------------------------------------------------------------*/


/* load thread information */  /* public */ /* static */
void ThreadIndex::loadIndex( Kita::Thread* thread, const KURL& url )
{
    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );

    /* load read number */
    int readNum = getReadNumPrivate( url, config );
    if( readNum == 0 ) return;  /* cache does not exist. */
    thread->setReadNum( readNum );
   
    /* load thread name */
    QString subject = getSubjectPrivate( config );
    if( subject == QString::null && thread->threadName() != QString::null ){
       subject = thread->threadName();
       config.writeEntry( "Subject", subject );
    }
    if( subject == QString::null ) thread->setThreadName( "?" );
    else thread->setThreadName( subject );

    /* load res number */
    thread->setResNum( getResNumPrivate( url, config ) );

    /* load view pos */
    thread->setViewPos( getViewPosPrivate( config ) );
    if( thread->viewPos() > thread->readNum() ) thread->setReadNum( thread->viewPos() );
}



/* save thread information */  /* public */ /* static */
void ThreadIndex::saveIndex( const Kita::Thread* thread, const KURL& url )
{
    if( thread->readNum() == 0 ) return;

    QString indexPath = Kita::Cache::getIndexPath( url );
    KConfig config( indexPath );
	
    /* save thread name */
    setSubjectPrivate( thread->threadName(), config );
	
    /* save res number */
    setResNumPrivate( thread->resNum(), config );

    /* save read number */
    setReadNumPrivate( thread->readNum(), config );

    /* save view pos */
    setViewPosPrivate( thread->viewPos(), config );

    /* save "cache" */
    KURL datURL = Kita::getDatURL( url );
    int num = ( thread->viewPos() ? thread->viewPos() : thread->readNum() );
    KitaThreadInfo::setResNum( datURL.prettyURL(), thread->resNum() );
    KitaThreadInfo::setReadNum( datURL.prettyURL(), num );
}


/*------------------------------------------------------------------*/

/* private */ /* static */
QString ThreadIndex::getSubjectPrivate( KConfig& config )
{
    return config.readEntry( "Subject" );
}

/* private */ /* static */
void ThreadIndex::setSubjectPrivate( const QString& str, KConfig& config )
{
    config.writeEntry( "Subject", str );
}

/*-------*/

/* private */ /* static */
int ThreadIndex::getResNumPrivate( const KURL& url, KConfig& config )
{
    int resNum = config.readNumEntry( "ResNum" );
    
    /* use obsoleted "cache" file */
    if( !resNum ){
	KURL datURL = Kita::getDatURL( url );
	resNum = KitaThreadInfo::readNum( datURL.prettyURL() );
	if( resNum ) config.writeEntry( "ResNum", resNum );
    }

    return resNum;
}

/* private */ /* static */
void ThreadIndex::setResNumPrivate( int resNum, KConfig& config )
{
    config.writeEntry( "ResNum", resNum );
}

/*-------*/

/* private */ /* static */
int ThreadIndex::getReadNumPrivate( const KURL& url, KConfig& config )
{
    int readNum = config.readNumEntry( "ReadNum" );

    if( !readNum ){

	/* use ViewPos instead of ReadNum. */
	readNum = config.readNumEntry( "ViewPos" );
			
	/* use obsoleted "cache" file */
	if( !readNum ){
	    KURL datURL = Kita::getDatURL( url );
	    readNum = KitaThreadInfo::readNum( datURL.prettyURL() );
	}

	if( readNum ) config.writeEntry( "ReadNum", readNum );
    }

    return readNum;
}

/* private */ /* static */
void ThreadIndex::setReadNumPrivate( int readNum, KConfig& config )
{
    config.writeEntry( "ReadNum", readNum );
}

/*-------*/

/* private */ /* static */
int ThreadIndex::getViewPosPrivate( KConfig& config )
{
    return config.readNumEntry( "ViewPos" );
}

/* private */ /* static */
void ThreadIndex::setViewPosPrivate( int viewPos, KConfig& config )
{
    config.writeEntry( "ViewPos", viewPos );
}
