/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@wakaba.jp                                                       *
*                                                                         *
*   Permission is hereby granted, free of charge, to any person obtaining *
*   a copy of this software and associated documentation files (the       *
*   "Software"), to deal in the Software without restriction, including   *
*   without limitation the rights to use, copy, modify, merge, publish,   *
*   distribute, sublicense, and/or sell copies of the Software, and to    *
*   permit persons to whom the Software is furnished to do so, subject to *
*   the following conditions:                                             *
*                                                                         *
*   The above copyright notice and this permission notice shall be        *
*   included in all copies or substantial portions of the Software.       *
*                                                                         *
*   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *
*   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF    *
*   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.*
*   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR     *
*   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, *
*   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR *
*   OTHER DEALINGS IN THE SOFTWARE.                                       *
***************************************************************************/

#include "qcp932codec.h"
#include <qjpunicode.h>

#include <kdebug.h>

#define QValidChar(u)	((u) ? QChar((ushort)(u)) : QChar::replacement)

static unsigned short const jisx0208_ibm_extension_to_unicode[] =
    {
        /* 115 ku */
        0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176,
        0x2177, 0x2178, 0x2179, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164,
        0x2165, 0x2166, 0x2167, 0x2168, 0x2169, 0xffe2, 0xffe4, 0xff07,
        0xff02, 0x3231, 0x2116, 0x2121, 0x2235, 0x7e8a, 0x891c, 0x9348,
        0x9288, 0x84dc, 0x4fc9, 0x70bb, 0x6631, 0x68c8, 0x92f9, 0x66fb,
        0x5f45, 0x4e28, 0x4ee1, 0x4efc, 0x4f00, 0x4f03, 0x4f39, 0x4f56,
        0x4f92, 0x4f8a, 0x4f9a, 0x4f94, 0x4fcd, 0x5040, 0x5022, 0x4fff,
        0x501e, 0x5046, 0x5070, 0x5042, 0x5094, 0x50f4, 0x50d8, 0x514a,
        0x5164, 0x519d, 0x51be, 0x51ec, 0x5215, 0x529c, 0x52a6, 0x52c0,
        0x52db, 0x5300, 0x5307, 0x5324, 0x5372, 0x5393, 0x53b2, 0x53dd,
        0xfa0e, 0x549c, 0x548a, 0x54a9, 0x54ff, 0x5586, 0x5759, 0x5765,
        0x57ac, 0x57c8, 0x57c7, 0xfa0f, 0xfa10, 0x589e, 0x58b2,
        /* 116 ku */
        0x590b, 0x5953, 0x595b, 0x595d, 0x5963, 0x59a4, 0x59ba,
        0x5b56, 0x5bc0, 0x752f, 0x5bd8, 0x5bec, 0x5c1e, 0x5ca6, 0x5cba,
        0x5cf5, 0x5d27, 0x5d53, 0xfa11, 0x5d42, 0x5d6d, 0x5db8, 0x5db9,
        0x5dd0, 0x5f21, 0x5f34, 0x5f67, 0x5fb7, 0x5fde, 0x605d, 0x6085,
        0x608a, 0x60de, 0x60d5, 0x6120, 0x60f2, 0x6111, 0x6137, 0x6130,
        0x6198, 0x6213, 0x62a6, 0x63f5, 0x6460, 0x649d, 0x64ce, 0x654e,
        0x6600, 0x6615, 0x663b, 0x6609, 0x662e, 0x661e, 0x6624, 0x6665,
        0x6657, 0x6659, 0xfa12, 0x6673, 0x6699, 0x66a0, 0x66b2, 0x66bf,
        0x66fa, 0x670e, 0xf929, 0x6766, 0x67bb, 0x6852, 0x67c0, 0x6801,
        0x6844, 0x68cf, 0xfa13, 0x6968, 0xfa14, 0x6998, 0x69e2, 0x6a30,
        0x6a6b, 0x6a46, 0x6a73, 0x6a7e, 0x6ae2, 0x6ae4, 0x6bd6, 0x6c3f,
        0x6c5c, 0x6c86, 0x6c6f, 0x6cda, 0x6d04, 0x6d87, 0x6d6f,
        /* 117 ku */
        0x6d96, 0x6dac, 0x6dcf, 0x6df8, 0x6df2, 0x6dfc, 0x6e39,
        0x6e5c, 0x6e27, 0x6e3c, 0x6ebf, 0x6f88, 0x6fb5, 0x6ff5, 0x7005,
        0x7007, 0x7028, 0x7085, 0x70ab, 0x710f, 0x7104, 0x715c, 0x7146,
        0x7147, 0xfa15, 0x71c1, 0x71fe, 0x72b1, 0x72be, 0x7324, 0xfa16,
        0x7377, 0x73bd, 0x73c9, 0x73d6, 0x73e3, 0x73d2, 0x7407, 0x73f5,
        0x7426, 0x742a, 0x7429, 0x742e, 0x7462, 0x7489, 0x749f, 0x7501,
        0x756f, 0x7682, 0x769c, 0x769e, 0x769b, 0x76a6, 0xfa17, 0x7746,
        0x52af, 0x7821, 0x784e, 0x7864, 0x787a, 0x7930, 0xfa18, 0xfa19,
        0xfa1a, 0x7994, 0xfa1b, 0x799b, 0x7ad1, 0x7ae7, 0xfa1c, 0x7aeb,
        0x7b9e, 0xfa1d, 0x7d48, 0x7d5c, 0x7db7, 0x7da0, 0x7dd6, 0x7e52,
        0x7f47, 0x7fa1, 0xfa1e, 0x8301, 0x8362, 0x837f, 0x83c7, 0x83f6,
        0x8448, 0x84b4, 0x8553, 0x8559, 0x856b, 0xfa1f, 0x85b0,
        /* 118 ku */
        0xfa20, 0xfa21, 0x8807, 0x88f5, 0x8a12, 0x8a37, 0x8a79,
        0x8aa7, 0x8abe, 0x8adf, 0xfa22, 0x8af6, 0x8b53, 0x8b7f, 0x8cf0,
        0x8cf4, 0x8d12, 0x8d76, 0xfa23, 0x8ecf, 0xfa24, 0xfa25, 0x9067,
        0x90de, 0xfa26, 0x9115, 0x9127, 0x91da, 0x91d7, 0x91de, 0x91ed,
        0x91ee, 0x91e4, 0x91e5, 0x9206, 0x9210, 0x920a, 0x923a, 0x9240,
        0x923c, 0x924e, 0x9259, 0x9251, 0x9239, 0x9267, 0x92a7, 0x9277,
        0x9278, 0x92e7, 0x92d7, 0x92d9, 0x92d0, 0xfa27, 0x92d5, 0x92e0,
        0x92d3, 0x9325, 0x9321, 0x92fb, 0xfa28, 0x931e, 0x92ff, 0x931d,
        0x9302, 0x9370, 0x9357, 0x93a4, 0x93c6, 0x93de, 0x93f8, 0x9431,
        0x9445, 0x9448, 0x9592, 0xf9dc, 0xfa29, 0x969d, 0x96af, 0x9733,
        0x973b, 0x9743, 0x974d, 0x974f, 0x9751, 0x9755, 0x9857, 0x9865,
        0xfa2a, 0xfa2b, 0x9927, 0xfa2c, 0x999e, 0x9a4e, 0x9ad9,
        /* 119 ku */
        0x9adc, 0x9b75, 0x9b72, 0x9b8f, 0x9bb1, 0x9bbb, 0x9c00,
        0x9d70, 0x9d6b, 0xfa2d, 0x9e19, 0x9ed1, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
        0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    };

static uint cp932ToUnicode( QJpUnicodeConv* conv, uint h, uint l )
{
    uint ku;
    uint ten;

    if ( h >= 0x81 && h <= 0x9f ) {
        ku = h * 2 - 257;
    } else if ( h >= 0xe0 && h <= 0xfc ) {
        ku = h * 2 - 385;
    } else {
        return 0x0000;
    }

    if ( l >= 0x40 && l <= 0x7e ) {
        ten = l - 63;
    } else if ( l >= 0x80 && l <= 0x9e ) {
        ten = l - 64;
    } else if ( l >= 0x9f && l <= 0xfc ) {
        ten = l - 158;
        ku++;
    } else {
        return 0x0000;
    }

    if ( ku >= 115 && ku <= 119 && ten <= 94 ) {
        return jisx0208_ibm_extension_to_unicode[ ( ku - 115 ) * 94 + ( ten - 1 ) ];
    }
    return conv->jisx0208ToUnicode( ku + 0x20, ten + 0x20 );
}

QCp932Codec::QCp932Codec() : QTextCodec(),
        conv( QJpUnicodeConv::newConverter( QJpUnicodeConv::Microsoft_CP932 |
                                            QJpUnicodeConv::NEC_VDC |
                                            QJpUnicodeConv::IBM_VDC ) )
{}


QCp932Codec::~QCp932Codec()
{
    if ( conv ) {
        delete conv;
    }
}

const char * QCp932Codec::name() const
{
    return "CP932";
}

// see http://www.iana.org/assignments/character-sets
const char * QCp932Codec::mimeName() const
{
    return "Windows-31J";
}

int QCp932Codec::mibEnum() const
{
    return 2024;
}

QTextDecoder* QCp932Codec::makeDecoder() const
{
    return new QCp932Decoder( conv );
}

/*
QTextEncoder* QCp932Codec::makeEncoder() const
{
}*/

/*QString QCp932Codec::toUnicode(const char* chars, int len) const
{
}*/

static uint jisx0212ToSjis( uint h, uint l )
{
    if ( ( 0x0021 <= h ) && ( h <= 0x007e ) && ( 0x0021 <= l ) && ( l <= 0x007e ) ) {
        return ( ( ( ( h - 1 ) >> 1 ) + ( ( h <= 0x5e ) ? 0x71 : 0xb1 ) ) << 8 ) |
               ( l + ( ( h & 1 ) ? ( ( l < 0x60 ) ? 0x1f : 0x20 ) : 0x7e ) );
    }
    return 0x0000;
}

QCString QCp932Codec::fromUnicode( const QString& uc, int& lenInOut ) const
{
    int ulen = QMAX( ( int ) uc.length(), lenInOut );
    int clen = ulen * 2 + 1;
    QCString ret( clen );
    char *pos = ( char * ) ret.data();

    for ( int i = 0; i < ulen; i++ ) {
        uint c = uc[ i ].unicode();
        uint j;

        /* convert WAVE DASH to FULLWIDTH TILDE */
        if ( c == 0x301C ) c = 0xFF5E;

        if ( ( j = conv->unicodeToJisx0201( c ) ) != 0 ) {
            // ascii(jis x 0201) or hankaku-kana
            *pos++ = j;
        } else if ( ( j = conv->unicodeToSjis( c ) ) != 0 ) {
            *pos++ = ( j >> 8 );
            *pos++ = ( j & 0xff );
        } else if ( ( j = conv->unicodeToJisx0212( c ) ) != 0 ) {
            // support NEC and IBM extension...
            j = jisx0212ToSjis( ( j & 0xff00 ) >> 8, j & 0x00ff );
            *pos++ = ( j >> 8 );
            *pos++ = ( j & 0xff );
        } else {
            // invalid
            *pos++ = '?';
        }
    }
    lenInOut = pos - ( char * ) ret.data();
    ret.truncate( lenInOut );
    return ret;
}

/*bool QCp932Codec::canEncode(QChar ch) const
{
}
 
bool QCp932Codec::canEncode(const QString& s) const
{
}*/

// FIXME: implement
int QCp932Codec::heuristicContentMatch( const char*, int ) const
{
//    kdWarning( 7743 ) << "XXX heuristicContentMatch" << endl;
    return 0;
}

// FIXME: implement
int QCp932Codec::heuristicNameMatch( const char * hint ) const
{
//    kdWarning( 7743 ) << "XXX heuristicNameMatch hint = " << hint << endl;
    return 0;
}

QCp932Decoder::QCp932Decoder( QJpUnicodeConv* _conv ) : QTextDecoder(), conv( _conv )
{}

QCp932Decoder::~QCp932Decoder()
{}

QString QCp932Decoder::toUnicode( const char* chars, int len )
{
    QString ret;
    static int buf = -1;

    for ( int i = 0; i < len; i++ ) {
        unsigned char c = chars[ i ];
        if ( buf >= 0 ) { // 2-byte
            if ( ( c >= 0x40 && c <= 0x7e ) || ( c >= 0x80 && c <= 0xfc ) ) {
                // valid kanji
                uint u = cp932ToUnicode( conv, buf, c );
                ret += QValidChar( u );
            } else {
                // invalid
                ret += QChar::replacement;
            }
            buf = -1;
        } else {
            if ( ( c >= 0x81 && c <= 0x9f ) || ( c >= 0xe0 && c <= 0xfc ) ) {
                // 1st-byte of 2-byte character.
                buf = ( int ) c;
            } else if ( c >= 0xa1 && c <= 0xdf ) {
                // hankaku-kana
                uint u = conv->jisx0201ToUnicode( c );
                ret += QValidChar( u );
            } else {
                // 1-byte character.
                uint u = conv->asciiToUnicode( c );
                ret += QValidChar( u );
            }
        }
    }
    return ret;
}
