/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitaconfig.h"

#include <kconfig.h>

KitaConfig* KitaConfig::instance = 0;

KitaConfig* KitaConfig::getInstance()
{
    if ( !instance ) {
        instance = new KitaConfig();
    }
    return instance;
}

KitaConfig::KitaConfig()
        : m_threadColor( KitaConfig::defaultThreadColor() ),
        m_threadBackgroundColor( KitaConfig::defaultThreadBackgroundColor() ),
        m_popupColor( KitaConfig::defaultPopupColor() ),
        m_popupBackgroundColor( KitaConfig::defaultPopupBackgroundColor() ),
        m_listFont( KitaConfig::defaultListFont() ),
        m_threadFont( KitaConfig::defaultThreadFont() ),
        m_popupFont( KitaConfig::defaultPopupFont() ),
        // m_boardListURL
        // m_nameCompletion
        m_useStyleSheet( KitaConfig::defaultUseStyleSheet() ),
        m_styleSheetText( KitaConfig::defaultStyleSheetText() ),
        m_asciiArtList( KitaConfig::defaultAsciiArtList() ),
        m_showMailAddress( KitaConfig::defaultShowMailAddress() ),
        m_aboneNameList( KitaConfig::defaultAboneNameList() ),
        m_aboneIDList( KitaConfig::defaultAboneIDList() ),
        m_aboneWordList( KitaConfig::defaultAboneWordList() ),
        m_MarkTime( KitaConfig::defaultMarkTime() ),
        m_alwaysUseTab( KitaConfig::defaultAlwaysUseTab() ),
        m_sortOrder( static_cast<SortOrder>( KitaConfig::defaultSortOrder() ) ),
        m_showAA( KitaConfig::defaultShowAA() ),
        m_showNum( KitaConfig::defaultShowNum() ),
        m_usePart( KitaConfig::defaultUsePart() ),
        m_useImageViewer( KitaConfig::defaultUseImageViewer() ),
        m_useImagePopup( KitaConfig::defaultUseImagePopup() ),
        m_useMosaic( KitaConfig::defaultUseMosaic() ),
        m_useKitaNavi( KitaConfig::defaultUseKitaNavi() ),
        m_partMimeList( KitaConfig::defaultPartMimeList() ),
        m_userID( KitaConfig::defaultUserID() ),
        m_password( KitaConfig::defaultPassword() ),
        m_autoLogin( KitaConfig::defaultAutoLogin() )
{}

KitaConfig::~KitaConfig()
{}

void KitaConfig::writeConfig( KConfig* config )
{
    config->setGroup( "Global" );
    config->writeEntry( "Font", KitaConfig::listFont() );
    config->writeEntry( "ThreadFont", KitaConfig::threadFont() );
    config->writeEntry( "PopupFont", KitaConfig::popupFont() );
    config->writeEntry( "ShowMailAddress", KitaConfig::showMailAddress() );
    config->writeEntry( "MarkTime", KitaConfig::MarkTime() );
    config->writeEntry( "SortOrder", KitaConfig::sortOrder() );
    config->writeEntry( "AlwaysUseTab", KitaConfig::alwaysUseTab() );
    config->writeEntry( "ShowAA", KitaConfig::showAA() );
    config->writeEntry( "ShowNum", KitaConfig::showNum() );
    config->writeEntry( "UsePart", KitaConfig::usePart() );
    config->writeEntry( "UseImageViewer", KitaConfig::useImageViewer() );
    config->writeEntry( "UseImagePopup", KitaConfig::useImagePopup() );
    config->writeEntry( "UseMosaic", KitaConfig::useMosaic() );
    config->writeEntry( "UseKitaNavi", KitaConfig::useKitaNavi() );
    config->writeEntry( "UseStyleSheet", KitaConfig::useStyleSheet() );
    config->writeEntry( "PartMimeList", KitaConfig::partMimeList() );

    config->setGroup( "Color" );
    config->writeEntry( "Thread", KitaConfig::threadColor() );
    config->writeEntry( "ThreadBackground", KitaConfig::threadBackgroundColor() );
    config->writeEntry( "Popup", KitaConfig::popupColor() );
    config->writeEntry( "PopupBackground", KitaConfig::popupBackgroundColor() );

    config->setGroup( "Account" );
    config->writeEntry( "UserID", KitaConfig::userID() );
    config->writeEntry( "Password", encryptStr( KitaConfig::password() ) );
    config->writeEntry( "AutoLogin", KitaConfig::autoLogin() );
}

void KitaConfig::readConfig( KConfig* config )
{
    config->setGroup( "Global" );

    if ( config->hasKey( "Font" ) ) {
        KitaConfig::setListFont( config->readFontEntry( "Font" ) );
    } else {
        KitaConfig::setListFont( KitaConfig::defaultListFont() );
    }

    if ( config->hasKey( "ThreadFont" ) ) {
        KitaConfig::setThreadFont( config->readFontEntry( "ThreadFont" ) );
    } else {
        KitaConfig::setThreadFont( KitaConfig::defaultThreadFont() );
    }

    if ( config->hasKey( "PopupFont" ) ) {
        KitaConfig::setPopupFont( config->readFontEntry( "PopupFont" ) );
    } else {
        KitaConfig::setPopupFont( KitaConfig::defaultPopupFont() );
    }

    KitaConfig::setShowMailAddress( config->readBoolEntry( "ShowMailAddress", KitaConfig::defaultShowMailAddress() ) );
    KitaConfig::setMarkTime( config->readUnsignedNumEntry( "MarkTime", KitaConfig::defaultMarkTime() ) );
    KitaConfig::setSortOrder( config->readNumEntry( "SortOrder", KitaConfig::defaultSortOrder() ) );
    KitaConfig::setAlwaysUseTab( config->readBoolEntry( "AlwaysUseTab", KitaConfig::defaultAlwaysUseTab() ) );
    KitaConfig::setShowAA( config->readBoolEntry( "ShowAA", KitaConfig::defaultShowAA() ) );
    KitaConfig::setShowNum( config->readNumEntry( "ShowNum", KitaConfig::defaultShowNum() ) );
    KitaConfig::setUsePart( config->readBoolEntry( "UsePart", KitaConfig::defaultUsePart() ) );
    KitaConfig::setUseImageViewer( config->readBoolEntry( "UseImageViewer", KitaConfig::defaultUseImageViewer() ) );
    KitaConfig::setUseImagePopup( config->readBoolEntry( "UseImagePopup", KitaConfig::defaultUseImagePopup() ) );
    KitaConfig::setUseMosaic( config->readBoolEntry( "UseMosaic", KitaConfig::defaultUseMosaic() ) );

    KitaConfig::setUseKitaNavi( config->readBoolEntry( "UseKitaNavi", KitaConfig::defaultUseKitaNavi() ) );
    KitaConfig::setUseStyleSheet( config->readBoolEntry( "UseStyleSheet", KitaConfig::defaultUseStyleSheet() ) );

    {
        // MIME list
        QStringList mimeList = config->readListEntry( "PartMimeList" );
        if ( ! mimeList.isEmpty() ) {
            KitaConfig::setPartMimeList( mimeList );
        } else {
            KitaConfig::setPartMimeList( KitaConfig::defaultPartMimeList() );
        }
    }

    config->setGroup( "Color" );

    {
        // Thread text color
        QColor threadColor = config->readColorEntry( "Thread" );
        if ( threadColor.isValid() ) {
            KitaConfig::setThreadColor( threadColor );
        } else {
            KitaConfig::setThreadColor( KitaConfig::defaultThreadColor() );
        }
    }

    {
        // Thread background color
        QColor threadBackgroundColor = config->readColorEntry( "ThreadBackground" );
        if ( threadBackgroundColor.isValid() ) {
            KitaConfig::setThreadBackgroundColor( threadBackgroundColor );
        } else {
            KitaConfig::setThreadBackgroundColor( KitaConfig::defaultThreadBackgroundColor() );
        }
    }

    {
        // Popup text color
        QColor popupColor = config->readColorEntry( "Popup" );
        if ( popupColor.isValid() ) {
            KitaConfig::setPopupColor( popupColor );
        } else {
            KitaConfig::setPopupColor( KitaConfig::defaultPopupColor() );
        }
    }

    {
        // Popup background color
        QColor popupBackgroundColor = config->readColorEntry( "PopupBackground" );
        if ( popupBackgroundColor.isValid() ) {
            KitaConfig::setPopupBackgroundColor( popupBackgroundColor );
        } else {
            KitaConfig::setPopupBackgroundColor( KitaConfig::defaultPopupBackgroundColor() );
        }
    }

    config->setGroup( "Account" );
    KitaConfig::setUserID( config->readEntry( "UserID" , KitaConfig::defaultUserID() ) );
    KitaConfig::setPassword( decryptStr( config->readEntry( "Password" , KitaConfig::defaultPassword() ) ) );
    KitaConfig::setAutoLogin( config->readBoolEntry( "AutoLogin", KitaConfig::defaultAutoLogin() ) );
}

/*
 * encryptStr and decryptStr are copied from kdepim/kmail/kmaccount.{h,cpp}
 */
QString KitaConfig::encryptStr( const QString& aStr )
{
    QString result;
    for ( uint i = 0; i < aStr.length(); i++ )
        result += ( aStr[ i ].unicode() < 0x20 ) ? aStr[ i ] :
                  QChar( 0x1001F - aStr[ i ].unicode() );
    return result;
}

QString KitaConfig::decryptStr( const QString& aStr )
{
    return encryptStr( aStr );
}

QString KitaConfig::defaultStyleSheetText()
{
    return QString(
               "body,\n"
               "body.pop {\n"
               "}\n"
               "table.res_title,\n"
               "table.pop_res_title {\n"
               "  white-space: nowrap;\n"
               "}\n"
               "td.res_title_number,\n"
               "td.pop_res_title_number {\n"
               "}\n"
               "td.res_title_name,\n"
               "td.pop_res_title_name {\n"
               "}\n"
               "td.res_title_date,\n"
               "td.pop_res_title_date {\n"
               "}\n"
               "td.res_title_id,\n"
               "td.pop_res_title_id {\n"
               "}\n"
               "div.res_body,\n"
               "div.pop_res_body {\n"
               "  margin-left: 3.5em;\n"
               "}\n"
               "div.res_block,\n"
               "div.pop_res_block {\n"
               "  padding-bottom: 1.8em;\n"
               "}\n"
               "div.kokoyon {\n"
               "  background-color: #CCCCCC;\n"
               "  text-align: center;\n"
               "}\n"
               "\n"
           );
}

QStringList KitaConfig::defaultPartMimeList()
{
    QStringList list;
    list << "image/gif" << "image/jpeg" << "image/png" << "image/x-bmp";
    return list;
}
