/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kita_misc.h"
#include "qcp932codec.h"
#include "datinfo.h"  /* struct RESDAT is defined. */
#include "kita-utf8.h"
#include "kita-utf16.h"

#include <kurl.h>
#include <klocale.h>
#include <kdebug.h>

#include <qregexp.h>
#include <qeucjpcodec.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qmutex.h>
#include <qdatetime.h>
#include <dom/html_document.h>
#include <dom/html_element.h>
#include <dom/dom_text.h>

#define KITA_RESDIGIT 4


namespace Kita
{
    static QMutex codecMutex;
    static QCp932Codec* qcpCodec = NULL;
    static QTextCodec*  utf8Codec = NULL;
    static QTextCodec*  eucCodec = NULL; 

    static QString m_weekstr[ 7 ];
    static QString m_colonstr;
    static QString m_colonnamestr;

    /* fro convertURL */
    static int m_prevConvMode;
    static QString m_prevConvURL;
    static QString m_prevConvNewURL;
    static QString m_prevConvRefstr;    

    /* for ParseMachiBBSOneLine */
    static QString m_machiSubject;
    static QString m_machiLine;
}


/*---------------------------------------------------*/

/* Text codecs */


QString Kita::qcpToUnicode( const QString& str )
{

    QMutexLocker locker( & Kita::codecMutex ); /* QTextCodec is not reentrant. */

    if ( !Kita::qcpCodec ) Kita::qcpCodec = new QCp932Codec();

    return Kita::qcpCodec->toUnicode( str );
}


QString Kita::utf8ToUnicode( const QString& str )
{

    QMutexLocker locker( & Kita::codecMutex ); /* QTextCodec is not reentrant. */

    if ( !Kita::utf8Codec ) Kita::utf8Codec = QTextCodec::codecForName( "utf8" );

    return Kita::utf8Codec->toUnicode( str );
}


QString Kita::ecuToUnicode( const QString& str )
{
    QMutexLocker locker( & Kita::codecMutex ); /* QTextCodec is not reentrant. */

    if ( !Kita::eucCodec ) Kita::eucCodec = QTextCodec::codecForName( "eucJP" );

    return Kita::eucCodec->toUnicode( str );
}



/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* conversion of DAT */


/* get HTML from raw data   */
QString Kita::DatToHtml( const QString& rawData, int num, bool showAddr )
{
    QString tmpstr;
    RESDAT resdat;

    resdat.linestr = rawData;
    resdat.set = TRUE;
    resdat.parsed = FALSE;
    parseResDat( resdat, tmpstr );

    return ResDatToHtml( resdat, num, showAddr );
}


/* get HTML from RESDAT                       */
/* resdat should be parsed by parseResDat before
   calling this function.
   For example, see Kita::DatToHtml().
  
   struct RESDAT is defined in datinfo.h .     */
QString Kita::ResDatToHtml( const RESDAT& resdat, int num, bool showAddr )
{
    QString result, titletext, bodytext;

    parseTITLEdatText( PARSEMODE_HTML, num, showAddr, resdat, titletext );
    parseBODYdatText( PARSEMODE_HTML, resdat.body, bodytext );

    result = QString( "<div class=\"pop_res_block\"><table class=\"pop_res_title\"><tr>" );
    result += titletext;
    result += "</tr></table><div class=\"pop_res_body\">";
    result += bodytext;
    result += "</div></div>";

    return result;
}



/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* conversion of URL */

KURL Kita::getDatURL( const KURL& url , QString& refstr )
{
    return convertURL( URLMODE_DAT, url, refstr );
}

KURL Kita::getDatURL( const KURL& url )
{
    QString refstr;
    return convertURL( URLMODE_DAT, url, refstr );
}

QString Kita::getThreadURL( const KURL& url, QString& refstr )
{
    return convertURL( URLMODE_THREAD, url, refstr );
}

QString Kita::getThreadURL( const KURL& url )
{
    QString refstr;
    return convertURL( URLMODE_THREAD, url, refstr );
}


/* convert thread URL, and get reference. 
   If mode = URLMODE_DAT, output is URL of dat file.
   If mode = URLMODE_THREAD, output is URL of read.cgi .

   If url is NOT enrolled, return QString::null. 
   
(ex.1)

mode = 0
url  = http://pc5.2ch.net/linux/dat/1069738960.dat#20-30
->
return : http://pc5.2ch.net/linux/dat/1069738960.dat
retstr : 20-30

(ex.2)

mode = 1
url  = http://pc5.2ch.net/linux/dat/1069738960.dat#20-30
->
return : http://pc5.2ch.net/test/read.cgi/linux/1069738960
retstr : 20-30

(ex.3)

mode = 0
url  = http://pc5.2ch.net/test/read.cgi/linux/1069738960/-100
->
return : http://pc5.2ch.net/linux/dat/1069738960.dat
refstr : 1-100
                                                            */

QString Kita::convertURL(

    /* input */
    int mode,  /* if 0, output is dat URL. If 1, output is thread URL */
    const KURL& url ,

    /* output */
    QString& refstr )
{
    refstr = QString::null;
    
    if( url.isEmpty() ) return QString::null;
	
    /* cache */
    if( m_prevConvMode == mode &&  m_prevConvURL == url.prettyURL() ) {

	refstr = m_prevConvRefstr;
	return m_prevConvNewURL;
    }
    
    /* Is board enrolled ? */
    BoardData* bdata = Kita::BoardManager::getBoardData( url );
    if( bdata == NULL ) return QString::null;

    QString urlstr = url.prettyURL();

    /* get thread & reference */
    QString thread = QString::null;
    QString refBase = QString::null;;

    if( urlstr.contains( "/dat/" ) ){ 

	/* url = (hostname)/(rootPath)/(bbsPath)/dat/(thread_ID).(ext)#(refBase)  */
	thread = url.filename().remove( bdata->ext() );
	refBase = url.ref();
    }
    else if( urlstr.contains( bdata->delimiter() ) ) {

	QString tmpstr;
	switch( bdata->type() ){

	/* machi BBS */
	/* ex.) If url = http://kanto.machi.to/bbs/read.pl?BBS=kana&KEY=1096716679 ,
	   then, thread = 1096716679                                */
	case Board_MachiBBS:
	    thread = url.queryItem( "KEY" );
	    refBase = QString::null;
	    break;

        /* url = (hostname)/(rootPath)/(delimiter)/(bbsPath)/(thread_ID)/(refBase) */
	default:
	    tmpstr = urlstr.section( bdata->delimiter() + bdata->bbsPath(), 1, 1 );
            thread = tmpstr.section( '/', 1, 1 );
	    refBase = tmpstr.section( '/', 2, 2 );
	    break;
	}
    }

    if( thread == QString::null ) return QString::null;

    if ( refBase != QString::null ){
	
	if ( refBase.at( 0 ) == '-' ) refstr = "1" + refBase;
	else refstr = refBase;
    }

    /* create new URL */
    QString newURL;
    if( mode == URLMODE_DAT ) newURL = bdata->basePath() + "dat/" + thread + bdata->ext();
    else {
	newURL = bdata->cgiBasePath();
	    
	switch( bdata->type() ){

	case Board_MachiBBS:
	    newURL += "&KEY=" + thread;
	    break;
	    
	default:
	    newURL += thread;
	    break;
	}
    }

    /* cache */
    m_prevConvMode = mode;
    m_prevConvURL = url.prettyURL();
    m_prevConvNewURL = newURL;
    m_prevConvRefstr = refstr;
    
    return newURL;
}



/**
 * http://pc5.2ch.net/linux/dat/1089905503.dat
 * -> http://pc5.2ch.net/test/offlaw.cgi?bbs=linux&key=1089905503
 */
QString Kita::datToOfflaw( const KURL& datURL )
{
    /* TODO: not tested. */
    KURL url( datURL );
    QString root = url.host();

    QStringList list = QStringList::split( ".", url.fileName() );
    if ( list.size() != 2 ) {
        return QString::null;
    }
    QString datName = list[ 0 ];

    url.cd( ".." );
    if ( url.fileName() != "dat" ) {
        return QString::null;
    }

    url.cd( ".." );
    QString board = url.fileName();

    return QString( "http://%1/test/offlaw.cgi?raw=0.0&bbs=%2&key=%3" ).arg( root ).arg( board ).arg( datName );
}


/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* utilities */

/* create directory recursively */
bool Kita::mkdir( const QString& targetPath )
{
    QDir qdir( targetPath );
    if( !qdir.exists() ){

	QStringList pathList = QStringList::split( "/", targetPath );
	QString path = QString::null;

	for( unsigned int i = 0; i < pathList.count(); ++i ){
	    
	    path += "/" + pathList[ i ];

	    qdir =  path;
	    if( !qdir.exists() ){
		if( !qdir.mkdir( path ) ) return FALSE;
	    }
	}
    }

    return TRUE;
}


QString Kita::unescape( const QString& str )
{
    QString ret = str;
    return ret.replace( "&lt;", "<" ).replace( "&gt;", ">" ).replace( "&amp;", "&" );
}


uint Kita::datToSince( const KURL& datURL )
{
    return KURL( datURL ).fileName().section( '.', 0, 0 ).toInt();
}


/* if cdat == str, return str.length() */ 
int Kita::isEqual( const QChar *cdat, const QString& str )
{
    int i = 0;
    while ( str.at( i ) != '\0' ) {
        if ( *cdat != str.at( i ) ) return 0;
        cdat++;i++;
    }
    return i;
}


/* convert strings to positive number.  */
/* if cdat is not number, return -1.    */

/*  For example, if cdat = "1234", then
    ret = 1234. If cdat = "abcd", then
    ret = -1.                           */
int Kita::stringToPositiveNum( const QChar *cdat, const unsigned int length )
{
    int ret = 0;

    for ( unsigned int i = 0 ; i < length ; i++ ) {

        unsigned short c = cdat[ i ].unicode();

        if ( ( c < UTF16_0 || c > UTF16_9 ) && ( c < '0' || c > '9' ) ) return -1;

        ret *= 10;
        if ( c >= UTF16_0 ) ret += c - UTF16_0;
        else ret += c - '0';
    }

    return ret;
}



/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* internal parsing functions  */


/* for Machi BBS */


/* init parser. Don't forget to call this before parsing. */
void Kita::InitParseMachiBBS(){
    m_machiSubject = QString::null;
    m_machiLine = QString::null;
}

QString Kita::ParseMachiBBSOneLine( const QString& inputLine, int& nextNum )
{
    QString ret = QString::null;
    m_machiLine += inputLine;

    int num = 0;
    QString name = QString::null;
    QString mail = QString::null;
    QString date = QString::null;
    QString time = QString::null;
    QString id = QString::null;
    QString host = QString::null;
    QString message = QString::null;

    // Subject
    QRegExp title_regexp( "<title>(.*)</title>" );
    
    // pattern 1 (tokyo,kanagawa,...)
    QRegExp regexp ( "<dt>(\\d*) .*<font color=\"#......\"><b> (.*) </b></font> .* (..../../..).* (..:..:..) ID:([^<]*)<br><dd>(.*)" );
    QRegExp regexp2( "<dt>(\\d*) .*<a href=\"mailto:(.*)\"><b> (.*) </B></a> .* (..../../..).* (..:..:..) ID:([^<]*)<br><dd>(.*)" );

    // pattern 2 (hokkaido,...)
    QRegExp regexp3( "<dt>(\\d*) .*<font color=\"#......\"><b> (.*) </b></font> .* (..../../..).* (..:..:..) ID:([^<]*) <font size=.>\\[ ([^ ]*) \\]</font><br><dd>(.*)" );
    QRegExp regexp4( "<dt>(\\d*) .*<a href=\"mailto:(.*)\"><b> (.*) </B></a> .* (..../../..).* (..:..:..) ID:([^<]*) <font size=.>\\[ ([^ ]*) \\]</font><br><dd>(.*)" );

    /* abone */
    QRegExp regexp5( "<dt>(\\d*) .*<br><dd>.*" );

    
    if ( regexp.search( m_machiLine ) != -1 ) {
	    
	num = regexp.cap( 1 ).toInt();
	name = regexp.cap( 2 );
	date = regexp.cap( 3 );
	time = regexp.cap( 4 );
	id = regexp.cap( 5 );
	message = regexp.cap( 6 );
	    
    } else if ( regexp2.search( m_machiLine ) != -1 ) {
	    
	num = regexp2.cap( 1 ).toInt();
	mail = regexp2.cap( 2 );
	name = regexp2.cap( 3 );
	date = regexp2.cap( 4 );
	time = regexp2.cap( 5 );
	id = regexp2.cap( 6 );
	message = regexp2.cap( 7 );
	    
    } else if ( regexp3.search( m_machiLine ) != -1 ) {
	
	num = regexp3.cap( 1 ).toInt();
	name = regexp3.cap( 2 );
	date = regexp3.cap( 3 );
	time = regexp3.cap( 4 );
	id = regexp3.cap( 5 );
	host = regexp3.cap( 6 );
	message = regexp3.cap( 7 );
	
    } else if ( regexp4.search( m_machiLine ) != -1 ) {
	
	num = regexp4.cap( 1 ).toInt();
	mail = regexp4.cap( 2 );
	name = regexp4.cap( 3 );
	date = regexp4.cap( 4 );
	time = regexp4.cap( 5 );
	id = regexp4.cap( 6 );
	host = regexp4.cap( 7 );	
	message = regexp4.cap( 8 );

    } else if ( regexp5.search( m_machiLine ) != -1 ) { /* abone */
	
	num = regexp5.cap( 1 ).toInt();
	m_machiLine = QString::null;
	if( num == nextNum ) return "abone<><><>abone<>";
	else return QString::null;
	
    } else if ( title_regexp.search( m_machiLine ) != -1 ) { /* get title */
	
	m_machiSubject = title_regexp.cap( 1 );
	m_machiLine = QString::null;
	return QString::null;
    }
    
    else return QString::null;

    if( num >= nextNum ){
	
	if( num != 1 ) m_machiSubject = QString::null;
	ret += name + "<><>" + date + " " + time + " ID:" + id;
	if( host != QString::null ) ret += " HOST:" + host;
	ret += "<>" + message + "<>" + m_machiSubject;
	nextNum = num;
    }
    
    m_machiLine = QString::null;
    return ret;
}


/*---------------------------------*/

/* for JBBS */

QString Kita::ParseJBBSOneLine( const QString& line, int& nextNum )
{
  QString ret = QString::null;
  QStringList list = QStringList::split( "<>", line, true );
  if( list.size() != 7 ) return QString::null;

  int num = list[ 0 ].toInt();
  QString name = list[ 1 ];
  QString mail = list[ 2 ];
  QString date = list[ 3 ];
  QString body = list[ 4 ];
  QString subject = list[ 5 ];
  QString id = list[ 6 ];

  if( num < nextNum ) return QString::null;
  
  /* remove tag */
  QRegExp rex( "<[^<]*>" );
  name.remove( rex );

  /* remove week */
  rex = QRegExp( "\\(.*\\)" );
  date.remove( rex );
  
  ret += name + "<>" + mail + "<>" + date + " ID:" + id + "<>" + body + "<>" + subject;
  nextNum = num;
  
  return ret;
}


/*---------------------------------*/

/* for Flash CGI/Mini Thread  */

QString Kita::ParseFlashCGIOneLine( const QString& line )
{
  QString ret = QString::null;
  QStringList list = QStringList::split( "<>", line, true );
  if( list.size() != 13 ) return QString::null;

  QString name = list[ 0 ];
  QString mail = list[ 1 ];
  QString date = list[ 2 ];
  QString body = list[ 3 ];
  QString subject = list[ 4 ];
  QString id = list[ 6 ];
  QString host = list[ 7 ];

  /* remove tag */
  QRegExp rex( "<[^<]*>" );
  name.remove( rex );

  ret += name + "<>" + mail + "<>" + date + " ID:" + id;
  if( host != QString::null ) ret += " HOST:" + host;
  ret += "<>" + body + "<>" + subject;
  
  return ret;
}


/*-------------------------------------------------*/

/* parsing functions for 2ch */


/*------------------------------------------*/
/*                                          */
/*      Parsing Engine for Title            */
/*                                          */
/*------------------------------------------*/

/*
  struct RESDAT resdat should be parsed by parseResDat before
  calling this function. struct RESDAT is defined in datinfo.h

  If mode = PARSEMODE_DOM, titlenode is DOM tree of title node.
  titletext is ignored.
 
  If mode = PARSEMODE_HTML, titletext is HTML text of title.
  DOM tree is not created. Both hdoc and titlenode are ignored.
  
  If mode = PARSEMODE_TEXT, titletext is plain text of title.
  DOM tree is not created. Both hdoc and titlenode are ignored.
*/
void Kita::parseTITLEdat(

    /* input */
    int mode,                   /* mode */
    DOM::HTMLDocument& hdoc,    /* root node of DOM document*/
    int num,                    /* number of res */
    bool showMailAddress,
    const RESDAT& resdat,       /* RESDAT is defined in datinfo.h */

    /* output */
    DOM::Element &titlenode,    /* DOM tree of title */
    QString& titletext       /* HTML or plain text of title */

)
{

    /*-----------------------------*/
    /* init                        */
    /*-----------------------------*/

    if ( !resdat.parsed ) {
        titletext = QString::null;
        return ;
    }

    const QString& str_name = resdat.name;
    const QString& str_address = resdat.address;
    const QString& str_id = resdat.id;

    unsigned int i;
    DOM::Element tmpelm;
    QString linkstr, linkstr2, linkurl, addstr, tmpstr;

    if ( m_colonstr == QString::null ) {
        m_colonstr = utf8ToUnicode( KITAUTF8_COLON );
        m_colonnamestr = utf8ToUnicode( KITAUTF8_NAME );
    }

    titletext = QString::null;


    /*-----------------------------*/
    /* number                      */
    /*-----------------------------*/

    linkstr = QString( "%1" ).arg( num );
    linkurl = "#write" + linkstr;

    switch ( mode ) {

    case PARSEMODE_DOM:

        tmpelm = titlenode.appendChild( hdoc.createElement( "TD" ) );
        tmpelm.setAttribute( "class", "res_title_number" );
        tmpelm = tmpelm.appendChild( hdoc.createElement( "A" ) );
        {
            /* set anchor id = number */
            tmpelm.setAttribute( "href", linkurl );
            tmpelm.appendChild( hdoc.createTextNode( linkstr ) );
        }
        break;

    case PARSEMODE_HTML:
        titletext += "<td class=\"pop_res_title_number\"><a href=\"" + linkurl + "\">";
        titletext += linkstr;
        titletext += "</a>";
        break;

    case PARSEMODE_TEXT:
        titletext += linkstr;
        break;

    }


    /*-----------------------------*/
    /* name & mail address         */
    /*-----------------------------*/

    DOM::Element namenode;
    linkurl = QString::null;

    /* parse name & address strings */
    parseBODYdatText( PARSEMODE_TEXT, str_name, linkstr );
    if ( str_address != QString::null ) parseBODYdatText( PARSEMODE_TEXT, str_address, addstr );
    else addstr = QString::null;

    tmpstr = " " + m_colonnamestr;
    switch ( mode ) {

    case PARSEMODE_DOM:
        tmpelm = titlenode.appendChild( hdoc.createElement( "TD" ) );
        tmpelm.setAttribute( "class", "res_title_name" );
        tmpelm.appendChild( hdoc.createTextNode( tmpstr ) );
        namenode = tmpelm.appendChild( hdoc.createElement( "B" ) );
        break;

    case PARSEMODE_HTML:
        titletext += "<td class=\"pop_res_title_name\">" + tmpstr;
        break;
    case PARSEMODE_TEXT:
        titletext += tmpstr;
        break;
    }

    /* show name with mail address, or show name only */
    if ( showMailAddress || addstr == QString::null ) {

        const QChar * chpt = linkstr.unicode();
        unsigned int length = linkstr.length();

        unsigned int pos;
        int refNum[ 2 ];
        i = 0;

        bool ancChain = TRUE;
        /* ancChain is chain for anchor. For examle, if anchor "2"
           appeared, ancChain is set to TRUE. Moreover, if next strings
           are "=5", anchor for 5 is also set. Thus, we can obtain anchors
           for strings "2=5" as follows:

           <a href="#2">2</a><a href="#5">=5</a>
        */

        /* show name */
        for ( ;; ) {

            linkurl = QString::null;
            linkstr2 = QString::null;

            /* get strings & anchor for digits */
            if ( showMailAddress && ancChain ) {

                if ( ( ancChain = parseResAnchor( chpt + i, length - i, linkstr2, refNum, pos ) ) ) {

                    linkurl = QString( "#%1" ).arg( refNum[ 0 ] );
                    if ( refNum[ 1 ] ) linkurl += QString( "-%1" ).arg( refNum[ 1 ] );
                }

                i += pos;
            } else { /* get strings for non-digits */

                while ( i < length ) linkstr2 += chpt[ i++ ];
            }

            if ( linkstr2 != QString::null ) {

                switch ( mode ) {

                case PARSEMODE_DOM:

                    if ( linkurl != QString::null ) { /* create anchor */

                        tmpelm = namenode.appendChild( hdoc.createElement( "A" ) );
                        {
                            tmpelm.setAttribute( "href", linkurl );
                            tmpelm.appendChild( hdoc.createTextNode( linkstr2 ) );
                        }


                    } else {

                        tmpelm = namenode.appendChild( hdoc.createElement( "SPAN" ) );
                        {
                            tmpelm.setAttribute( "style", "color: green" );
                            tmpelm.appendChild( hdoc.createTextNode( linkstr2 ) );
                        }
                    }

                    break;

                case PARSEMODE_HTML:
                case PARSEMODE_TEXT:

                    if ( mode == PARSEMODE_HTML
                            && linkurl != QString::null ) { /* create anchor */

                        titletext += "<a href=\"" + linkurl + "\">";
                        titletext += linkstr2;
                        titletext += "</a>";

                    } else titletext += linkstr2;

                    break;
                }
            }

            if ( i >= linkstr.length() ) break;

        } /* for(;;) */


        /* show mail address */
        switch ( mode ) {

        case PARSEMODE_DOM:

            if ( showMailAddress && addstr != QString::null ) {
                tmpstr = QString( " [" ) + addstr + "]";
                namenode.appendChild( hdoc.createTextNode( tmpstr ) );
            }

            break;

        case PARSEMODE_HTML:
        case PARSEMODE_TEXT:

            if ( showMailAddress && addstr != QString::null ) {
                tmpstr = QString( " [" ) + addstr + "]";
                titletext += tmpstr;
            }

            break;
        }

    } /* if ( showMailAddress || addstr == QString::null ) */

    /* don't show mail address (i.e. showMailAddress == FALSE) */
    else {

        linkurl = QString( "mailto:" ) + addstr;

        switch ( mode ) {

        case PARSEMODE_DOM:
            tmpelm = namenode.appendChild( hdoc.createElement( "A" ) );
            {
                tmpelm.setAttribute( "href", linkurl );
                tmpelm.setAttribute( "title", addstr );
                tmpelm.appendChild( hdoc.createTextNode( linkstr ) );
            }
            break;

        case PARSEMODE_HTML:

            titletext += "<a href=\"" + linkurl.replace( ">", "&lt;" );
            titletext += " title=\"" + addstr.replace( ">", "&lt;" ) + "\">";
            titletext += linkstr;
            titletext += "</a>";

            break;

        case PARSEMODE_TEXT:
            titletext += linkstr;
            break;
        }
    }


    /*-----------------------------*/
    /* date & ID & HOST            */
    /*-----------------------------*/

    /* show date */
    tmpstr = " " + m_colonstr + resdat.date + " ";

    switch ( mode ) {

    case PARSEMODE_DOM:
        tmpelm = titlenode.appendChild( hdoc.createElement( "TD" ) );
        tmpelm.setAttribute( "class", "res_title_date" );
        tmpelm.appendChild( hdoc.createTextNode( tmpstr ) );
        break;

    case PARSEMODE_HTML:
        titletext += "<td class=\"pop_res_title_date\">" + tmpstr;
        break;

    case PARSEMODE_TEXT:
        titletext += tmpstr;
        break;

    }

    /* show ID */
    if ( str_id != QString::null ) {

        if ( str_id == "???" ) tmpstr = "ID:" + str_id;
        else tmpstr = ":" + str_id;

        switch ( mode ) {

        case PARSEMODE_DOM:

            tmpelm = titlenode.appendChild( hdoc.createElement( "TD" ) );
            tmpelm.setAttribute( "class", "res_title_id" );
            if ( tmpstr.left( 3 ) != "ID:" ) {
                DOM::Element aelm = tmpelm.appendChild( hdoc.createElement( "A" ) );
                {
                    aelm.setAttribute( "href", "#idpop" + str_id );
                    aelm.appendChild( hdoc.createTextNode( "ID" ) );
                }
            }

            tmpelm.appendChild( hdoc.createTextNode( tmpstr ) );
            break;

        case PARSEMODE_HTML:
            titletext += "<td class=\"pop_res_title_id\">";
            if ( tmpstr.left( 3 ) != "ID:" )
                titletext += "<a href=\"#idpop" + str_id + "\">ID</a>";
            titletext += tmpstr;
            break;

        case PARSEMODE_TEXT:
            if ( tmpstr.left( 3 ) != "ID:" ) titletext += "ID";
            titletext += tmpstr;
            break;
        }
    }

    /* show host */
    if( resdat.host != QString::null ){

	tmpstr = " HOST:" + resdat.host;

	switch ( mode ) {

	case PARSEMODE_DOM:
	    tmpelm = titlenode.appendChild( hdoc.createElement( "TD" ) );
	    tmpelm.setAttribute( "class", "res_title_host" );
	    tmpelm.appendChild( hdoc.createTextNode( tmpstr ) );
	    break;

	case PARSEMODE_HTML:
	    titletext += "<td class=\"pop_res_title_host\">" + tmpstr;
	    break;

	case PARSEMODE_TEXT:
	    titletext += tmpstr;
	    break;
	}
    }
}



/* For convenience.
   mode can be set to PARSEMODE_HTML or PARSEMODE_TEXT */
void Kita::parseTITLEdatText(

    /* input */
    int mode,                   /* mode = PARSEMODE_HTML or PARSEMODE_TEXT */
    int num,
    bool showMailAddress,
    const RESDAT& resdat,             /* RESDAT is defined in datinfo.h */

    /* output */
    QString& titletext       /* HTML or plain text of title */

)
{

    /* dummy */
    DOM::HTMLDocument hdoc;
    DOM::Element titlenode;

    parseTITLEdat( mode, hdoc, num, showMailAddress, resdat, titlenode, titletext );
}





/*------------------------------------------*/
/*                                          */
/*      Parsing Engine for Body             */
/*                                          */
/*------------------------------------------*/

/*
  If mode = PARSEMODE_DOM, bodynode is DOM tree of body.
  bodytext is ignored.
 
  If mode = PARSEMODE_HTML, bodytext is HTML text of body.
  DOM tree is not created. Both hdoc and bodynode are ignored.
  
  If mode = PARSEMODE_TEXT, bodytext is plain text of body.
  DOM tree is not created. Both hdoc and bodynode are ignored.
*/
void Kita::parseBODYdat(

    /* input */
    int mode,                   /* mode */
    const QString &rawStr,      /* raw strings of body text */
    DOM::HTMLDocument& hdoc,    /* root node of DOM document*/
    bool showAA,                /* show AA (for KDE3.1x) */

    /* output */
    DOM::Element &bodynode,    /* DOM tree of body */
    QString& bodytext       /* HTML or plain text of body */

)
{

    /*-----------------------------------------*/
    /* init                                    */

    unsigned int i, i2, index, pos, length = rawStr.length();
    DOM::Element tmpelm;
    QString linkstr, linkurl;

    const QChar *chpt = rawStr.unicode();
    QString lineStr = QString::null;
    bodytext = QString::null;

    bool ancChain = FALSE;
    /* ancChain is chain for anchor. For examle, if anchor "&gt;2"
       appeared, ancChain is set to TRUE. Moreover, if next strings
       are "=5", anchor for 5 is also set. Thus, we can obtain anchors
       for strings "&gt;2=5" as follows:

       <a href="#2">&gt;2</a><a href="#5">=5</a>
    */


    /*-----------------------------------------*/

    for ( i = index = 0 ; i < length ; i++ ) {

        switch ( chpt[ i ].unicode() ) {

        case '<':

            /* " <br> " */
            if ( chpt[ i + 1 ] == 'b' && chpt[ i + 2 ] == 'r' && chpt[ i + 3 ] == '>' ) {

                i2 = i - index;
                if ( i > 0 && chpt[ i - 1 ] == ' ' ) i2--; /* remove space before <br> */
                lineStr += rawStr.mid( index, i2 );

                switch ( mode ) {

                case PARSEMODE_DOM:

                    /* add BR node */
                    bodynode.appendChild( hdoc.createTextNode( lineStr ) );
                    bodynode.appendChild( hdoc.createElement( "BR" ) );

                    /* show Ascii Art (for KDE3.1*) */
                    if ( showAA ) {

                        /* put the span node after BR node */
                        tmpelm = bodynode.appendChild( hdoc.createElement( "SPAN" ) );
                        {
                            tmpelm.setAttribute( "style", "color: white" );
                            tmpelm.appendChild( hdoc.createTextNode( "" ) );
                        }
                    }

                    break;

                case PARSEMODE_HTML:

                    bodytext += lineStr;
                    bodytext += "<br>";
                    /* show Ascii Art (for KDE3.1*) */
                    if ( showAA ) {
                        bodytext += "<span style=\"color: white\"></span>";
                    }
                    break;

                case PARSEMODE_TEXT:

                    bodytext += lineStr;
                    bodytext += '\n';

                    break;
                }

                index = i + 4;
                if ( chpt[ index ] == ' ' ) index++; /* remove space after <br> */
                i = index - 1;
                lineStr = QString::null;
                ancChain = FALSE;
            }

            /*----------------------------------------*/

            /* remove HTML tags <[^>]*>  */
            else {
                lineStr += rawStr.mid( index, i - index );
                while ( chpt[ i ] != '>' && i < length ) i++;
                index = i + 1;
            }

            break;

            /*----------------------------------------*/

        case 'h':    /* "http://" or "ttp://" or "tp:" */
        case 't':

            if ( mode != PARSEMODE_TEXT
                    && parseLink( chpt + i, length - i, linkstr, linkurl, pos ) ) {

                lineStr += rawStr.mid( index, i - index );

                switch ( mode ) {

                case PARSEMODE_DOM:

                    /* create A node */
                    bodynode.appendChild( hdoc.createTextNode( lineStr ) );

                    tmpelm = bodynode.appendChild( hdoc.createElement( "A" ) );
                    {
                        tmpelm.setAttribute( "href", linkurl );
                        tmpelm.appendChild( hdoc.createTextNode( linkstr ) );
                    }

                    break;

                case PARSEMODE_HTML:
                    bodytext += lineStr;
                    bodytext += "<a href=\"" + linkurl + "\">";
                    bodytext += linkstr;
                    bodytext += "</a>";
                    break;
                }

                index = i + pos;
                i = index - 1;
                lineStr = QString::null;
            }

            break;

            /*----------------------------------*/

        case '&':

            /* &gt; */
            if ( ( mode == PARSEMODE_DOM || mode == PARSEMODE_HTML )
                    && chpt[ i + 1 ] == 'g' && chpt[ i + 2 ] == 't' && chpt[ i + 3 ] == ';' )
                ancChain = createResAnchor( mode, rawStr, hdoc, bodynode, bodytext, chpt, i, index, lineStr );
            /* special char */
            else if ( mode == PARSEMODE_DOM || mode == PARSEMODE_TEXT ) {

                QString tmpstr;
                tmpstr = parseSpecialChar( chpt + i, pos );

                if ( tmpstr != QString::null ) {
                    lineStr += rawStr.mid( index, i - index ) + tmpstr;
                    index = i + pos;
                    i = index - 1;
                }
            }

            break;

            /*----------------------------------------*/

            /* unicode '>'  */
        case UTF16_BRACKET:
            if ( mode != PARSEMODE_TEXT )
                ancChain = createResAnchor( mode, rawStr, hdoc, bodynode, bodytext, chpt, i, index, lineStr );
            break;

            /*----------------------------------*/

        default:
            if ( mode != PARSEMODE_TEXT && ancChain )
                ancChain = createResAnchor( mode, rawStr, hdoc, bodynode, bodytext, chpt, i, index, lineStr );
        }
    }


    /*---------------------------*/

    lineStr += rawStr.mid( index );

    switch ( mode ) {

    case PARSEMODE_DOM:
        bodynode.appendChild( hdoc.createTextNode( lineStr ) );
        break;

    case PARSEMODE_HTML:
    case PARSEMODE_TEXT:
        bodytext += lineStr;
        break;
    }

}



/* For convenience.
   mode can be set to PARSEMODE_HTML or PARSEMODE_TEXT */
void Kita::parseBODYdatText(

    /* input */
    int mode,                   /* mode = PARSEMODE_HTML or PARSEMODE_TEXT */
    const QString &rawStr,      /* raw strings of body text */

    /* output */
    QString& bodytext       /* HTML or plain text of body */

)
{
    /* dummy */
    DOM::HTMLDocument hdoc;
    DOM::Element bodynode;

    parseBODYdat( mode, rawStr, hdoc, FALSE, bodynode, bodytext );
}



/*-----------------------------------------------------*/
/* parsing function for special char (such as &hearts; */
/*-----------------------------------------------------*/

/* For example, if cdat = "&amp;", then
 
   pos (= length of cdat) = 5,
   retstr = "&".
*/

QString Kita::parseSpecialChar(

    /* input */
    const QChar *cdat,

    /* output */
    unsigned int& pos )
{
    QString retstr = QString::null;

    if ( ( pos = isEqual( cdat , "&gt;" ) ) ) retstr = ">";
    else if ( ( pos = isEqual( cdat , "&lt;" ) ) ) retstr = "<";
    else if ( ( pos = isEqual( cdat , "&nbsp;" ) ) ) retstr = " ";
    else if ( ( pos = isEqual( cdat , "&amp;" ) ) ) retstr = "&";
    else if ( ( pos = isEqual( cdat , "&quot;" ) ) ) retstr = "\"";

    else if ( ( pos = isEqual( cdat , "&hearts;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_HEART );

    else if ( ( pos = isEqual( cdat , "&diams;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_DIA );

    else if ( ( pos = isEqual( cdat , "&clubs;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_CLUB );

    else if ( ( pos = isEqual( cdat , "&spades;" ) ) )
        retstr = utf8ToUnicode( KITAUTF8_SPADE );

    return retstr;
}



/* parse date & ID */
void Kita::parseDateId(

    /* input */
    const QString& str,

    /* output */
    RESDAT& resdat )
{
    QRegExp regexp( "(\\d\\d(\\d\\d)?)/(\\d\\d)/(\\d\\d) (\\d\\d):(\\d\\d)(:\\d\\d)?( ID:([^ ]*))?( HOST:([^ ]*))?" );

    if ( regexp.search( str ) == -1 ) {
        resdat.date = str;
        resdat.id = QString::null;
        return ;
    }

    int year = regexp.cap( 1 ).toInt();
    if ( year >= 70 && year < 100 ) {
        year += 1900;
    } else if ( year < 70 ) {
        year += 2000;
    }

    if ( m_weekstr[ 0 ] == QString::null ) {
        m_weekstr[ 0 ] = utf8ToUnicode( KITAUTF8_MONDAY );
        m_weekstr[ 1 ] = utf8ToUnicode( KITAUTF8_TUESDAY );
        m_weekstr[ 2 ] = utf8ToUnicode( KITAUTF8_WEDNESDAY );
        m_weekstr[ 3 ] = utf8ToUnicode( KITAUTF8_THURSDAY );
        m_weekstr[ 4 ] = utf8ToUnicode( KITAUTF8_FRIDAY );
        m_weekstr[ 5 ] = utf8ToUnicode( KITAUTF8_SATURDAY );
        m_weekstr[ 6 ] = utf8ToUnicode( KITAUTF8_SUNDAY );
    }

    resdat.dateTime = QDateTime( QDate( year, regexp.cap( 3 ).toInt(), regexp.cap( 4 ).toInt() ),
                                 QTime( regexp.cap( 5 ).toInt(), regexp.cap( 6 ).toInt(), regexp.cap( 7 ).mid( 1 ).toInt() ) );

    resdat.date = regexp.cap( 1 ) + "/" + regexp.cap( 3 ) + "/" + regexp.cap( 4 )
                  + m_weekstr[ resdat.dateTime.date().dayOfWeek() - 1 ]
                  + regexp.cap( 5 ) + ":" + regexp.cap( 6 ) + regexp.cap( 7 );

    resdat.id = regexp.cap( 9 );
    if( resdat.id.length() == 0 ) resdat.id = QString::null;
    resdat.host = regexp.cap( 11 );
    if( resdat.host.length() == 0 ) resdat.host = QString::null;
}



/* parse struct ResDat.
   This function splits raw data into ID, name,
   date, body text, subject, etc.

   "resdat.linestr" and "resdat.set" should be set
   before calling this.

   struct RESDAT is defined in datinfo.h.
   See also Kita::DatToHtml() and DatInfo::parseDat().
   
   input:
   resdat.linestr : raw line data
   resdat.set     : = TRUE

   output:
   resdata.*
   subject
*/ 
bool Kita::parseResDat( RESDAT& resdat, QString& subject )
{
    if ( resdat.parsed ) return TRUE;
    if ( !resdat.set ) return FALSE;

    resdat.parsed = TRUE;

    /* split dat */
    QString idstr = "none";
    QStringList list = QStringList::split( "<>", resdat.linestr, true );

    if ( list.size() == 5 ) {

        resdat.broken = FALSE;

        resdat.name = list[ 0 ];
        parseBODYdatText( PARSEMODE_TEXT, resdat.name, resdat.parsedName );
        resdat.address = list[ 1 ];
        parseDateId( list[ 2 ], resdat );
        if ( list[ 3 ].at( 0 ) == ' ' ) resdat.body = list[ 3 ].mid( 1 ); /* remove space after <> */
        else resdat.body = list[ 3 ];

        /* get subject */
        if ( list[ 4 ] != QString::null ) {
            subject = list[ 4 ];
        }

    } else resdat.broken = TRUE;

    return TRUE;
}




/* parsing function for link   */
/* For example,
 
   cdat = "ttp://foo.com",
 
   then
 
   linkstr = "ttp://foo.com",
   linkurl = "http://foo.com",
   pos (= length of cdat) = 13,
 
   and return TRUE.
                                */
bool Kita::parseLink(

    /* input */
    const QChar *cdat, const unsigned int length,

    /* output */
    QString& linkstr, QString& linkurl, unsigned int& pos
)
{

    /*-----------------------------*/

    linkstr = QString::null;
    linkurl = QString::null;

    QString retlinkstr = QString::null;
    QString prefix = QString::null;

    if ( isEqual( cdat , "http://" ) ) prefix = "http://";
    else if ( isEqual( cdat , "ttp://" ) ) prefix = "ttp://";
    else if ( isEqual( cdat , "tp://" ) ) prefix = "tp://";
    if ( prefix == QString::null ) return FALSE;

    pos = prefix.length();
    while ( cdat[ pos ] >= '!' && cdat[ pos ] <= '~' &&
            cdat[ pos ] != ' ' && cdat[ pos ] != '<' && cdat[ pos ] != '>'
            && pos < length ) {
        retlinkstr += cdat[ pos++ ];
    }
    if ( pos > length ) return FALSE;

    if ( retlinkstr != QString::null ) parseBODYdatText( PARSEMODE_TEXT, retlinkstr, linkstr );
    linkurl = "http://" + linkstr;
    linkstr = prefix + linkstr;

    return TRUE;
}

/*------------------------------------------*/
/* parsing function for anchor (>>digits)   */
/*------------------------------------------*/

/* This fuction parses res anchor.
 
   For example, if cdat = "&gt;12-20", then
 
   linkstr = ">12-20",
   refNum[0] = 12,
   refNum[1] = 20,
   pos (= length of cdat ) = 9,
   ret = TRUE;
 
*/
bool Kita::parseResAnchor(

    /* input */
    const QChar *cdat, const unsigned int length,

    /* output */
    QString& linkstr, int* refNum, unsigned int& pos )
{

    struct LocalFunc {
        static bool isHYPHEN( unsigned short c )
        {

            /* UTF-16 */
            if ( c == '-'
                    || ( c >= 0x2010 && c <= 0x2015 )
                    || ( c == 0x2212 )
                    || ( c == 0xFF0D )         /* UTF8: 0xEFBC8D */
               ) {
                return TRUE;
            }

            return FALSE;
        }
    };

    bool ret = FALSE;
    int i;

    if ( length == 0 ) return FALSE;

    linkstr = QString::null;
    refNum[ 0 ] = 0;
    refNum[ 1 ] = 0;
    pos = 0;

    /* check '>' twice */
    for ( i = 0;i < 2;i++ ) {

        if ( cdat[ pos ].unicode() == UTF16_BRACKET ) {
            linkstr += cdat[ pos ];
            pos++;
        } else if ( cdat[ pos ] == '&' && cdat[ pos + 1 ] == 'g'  /* &gt; */
                    && cdat[ pos + 2 ] == 't' && cdat[ pos + 3 ] == ';' ) {
            linkstr += ">";
            pos += 4;
        }

    }

    /* check ',' */
    if ( !pos ) {
        if ( cdat[ pos ] == ',' || cdat[ pos ].unicode() == UTF16_COMMA ) {
            linkstr += ",";
            pos ++;
        }
    }

    /* check '=' */
    if ( !pos ) {
        if ( cdat[ pos ] == '=' || cdat[ pos ].unicode() == UTF16_EQ ) {
            linkstr += "=";
            pos ++;
        }
    }

    /* check digits */
    int hyphen = 0;

    for ( i = 0 ; i < KITA_RESDIGIT + 1 && pos < length ; i++, pos++ ) {

        unsigned short c = cdat[ pos ].unicode();

        if ( ( c < UTF16_0 || c > UTF16_9 )
                && ( c < '0' || c > '9' )
                && ( !LocalFunc::isHYPHEN( c )
                     || ( i == 0 && LocalFunc::isHYPHEN( c ) )
                     || ( hyphen && LocalFunc::isHYPHEN( c ) ) )
           ) break;

        linkstr += cdat[ pos ];

        if ( LocalFunc::isHYPHEN( c ) ) {
            hyphen = 1;
            i = -1;
        } else {
            if ( c >= UTF16_0 ) c = '0' + cdat[ pos ].unicode() - UTF16_0;
            refNum[ hyphen ] *= 10;
            refNum[ hyphen ] += c - '0';
        }

        ret = TRUE;
    }

    return ret;
}


/* create node of anchor  */
/* This function is called from parseBODYdat internally. So, see also parseBODYdat */   
bool Kita::createResAnchor(

    /* input */
    int mode,
    const QString &rawStr,
    DOM::HTMLDocument& hdoc ,

    /* output */
    DOM::Element &bodynode,
    QString& bodytext,

    /* internal variables */
    /* They are the same variables that ara used in parseBODYdat. */
    const QChar *chpt, unsigned int &i, unsigned int &index, QString& lineStr
)
{
    /*-----------------------*/

    QString linkstr, linkurl;
    DOM::Element tmpelm;
    int refNum[ 2 ];
    unsigned int pos;
    unsigned int length = rawStr.length();

    /* parse anchor */
    if ( !parseResAnchor( chpt + i, length - i, linkstr, refNum, pos ) ) {
        lineStr += rawStr.mid( index, i - index ) + linkstr;
        index = i + pos;
        i = index - 1;
        return FALSE;
    }

    /* create anchor */
    lineStr += rawStr.mid( index, i - index );
    linkurl = QString( "#%1" ).arg( refNum[ 0 ] );
    if ( refNum[ 1 ] ) linkurl += QString( "-%1" ).arg( refNum[ 1 ] );

    switch ( mode ) {

    case PARSEMODE_DOM:

        /* create 'A' element */
        bodynode.appendChild( hdoc.createTextNode( lineStr ) );

        tmpelm = bodynode.appendChild( hdoc.createElement( "A" ) );
        {
            tmpelm.setAttribute( "href", linkurl );
            tmpelm.appendChild( hdoc.createTextNode( linkstr ) );
        }

        break;

    case PARSEMODE_HTML:
        bodytext += lineStr;
        bodytext += "<a href=\"" + linkurl + "\">";
        bodytext += linkstr;
        bodytext += "</a>";
        break;
    }

    index = i + pos;
    i = index - 1;
    lineStr = QString::null;

    return TRUE;
}




/*--------------------------------*/
/*--------------------------------*/
/* obsolete */

QString Kita::datToBoard( const KURL& datURL ){ return BoardManager::boardURL( datURL ); } 
int Kita::boardType( const KURL& url ) { return BoardManager::type( url ); }
