/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* This is the class to download files. */

#include "downloadmanager.h"
#include "kita_misc.h"

#include <qregexp.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qapplication.h>

using namespace Kita;


DownloadManager* DownloadManager::instance = NULL;
QMutex DownloadManager::m_mutex;


/* Download file. Return value is pointer of FileLoader.
   If path = QString::null, DownloadManager does not open local file.

   ex.1 )

   If you just want to download "http://www.hoge.com/foo.jpg", then:

   Kita::DownloadManager::download( "http://www.hoge.com/foo.jpg", "/tmp/foo.jpg" );

   
   ex.2 )

   If you want to do someting when received the data and when done,
   then connect signals like this:

   FileLoader* loader = Kita::DownloadManager::download( "http://www.hoge.com/foo.jpg", "/tmp/foo.jpg" );
   connect( loader, SIGNAL( data( const Kita::LoaderData&, const QByteArray& ) ),
	     SLOT( slotData( const Kita::LoaderData&, const QByteArray& ) ) );
   connect( loader, SIGNAL( result( const Kita::LoaderData& ) ),
	     SLOT( slotResult( const Kita::LoaderData& ) ) );

   [ Notice ] DownloadManager deletes FileLoader when done.  Never delete it in slotResult().
              See also ImgManager::loadPrivate(), and ImgManager::slotResult().

*/ /* public */ /* static */
FileLoader* DownloadManager::download( const KURL& url, const QString& path,
				     const LoaderOption& option /* option user data */
    )
{
    QMutexLocker locker( &m_mutex );

    /* to use SIGNAL & SLOT, create instance of DownloadManager here. */
    /* ( You CANNOT use SIGNAL & SLOT without instance. )             */
    if( instance == NULL ) instance = new DownloadManager();
    
    return instance->downloadPrivate( url, path, option );
}


/* public */ /* static */
void DownloadManager::stopLoading( const KURL& url )
{
    QMutexLocker locker( &m_mutex );

    if( instance ) instance->stopLoadingPrivate( url );
}


/* public */ /* static */
bool DownloadManager::isLoadingNow( const KURL& url )
{
    if( instance == NULL ) return FALSE;

    return instance->isLoadingNowPrivate( url );
}




/*---------------------------------*/
/* internal functions              */

DownloadManager::DownloadManager()
{
    m_loaderList.clear();
}

DownloadManager::~DownloadManager()
{
    /* delete all loaders */
    FileLoader * loader;
    QPtrListIterator< FileLoader > it( m_loaderList );
    while ( ( loader = it.current() ) != NULL ) {
        delete loader;
        ++it;
    }
}

/* private */
FileLoader* DownloadManager::downloadPrivate( const KURL& url, const QString& path,
					      const LoaderOption& option )
{
    if ( isLoadingNowPrivate( url ) ) return NULL;

    FileLoader* loader;
    loader = new FileLoader( url, path, option );
    m_loaderList.append( loader );

    if( loader->get() ){
	connect( loader, SIGNAL( result( const Kita::LoaderData& ) ),
                 SLOT( slotResult( const Kita::LoaderData& ) ) );
    }
    else {
        deleteLoader( loader );
        return NULL;
    }
    
    return loader;
}


/* private */
void DownloadManager::stopLoadingPrivate( const KURL& url )
{
    FileLoader * loader = getLoader( url );
    if ( !loader ) return ;

    loader->stopJob(); /* slotResult() will be called later. */
}


/* private */
bool DownloadManager::isLoadingNowPrivate( const KURL& url )
{
    if ( getLoader( url ) ) return TRUE;

    return FALSE;
}


/* private */    
FileLoader* DownloadManager::getLoader( const KURL& url )
{
    if ( m_loaderList.count() == 0 ) return NULL;

    FileLoader* loader;
    QPtrListIterator< FileLoader > it( m_loaderList );
    while ( ( loader = it.current() ) != NULL ) {
        if ( loader->getData().url == url ) {
            return loader;
        }
        ++it;
    }

    return NULL;
}


/* private */
void DownloadManager::deleteLoader( FileLoader* loader )
{
    m_loaderList.remove( loader );
    delete loader;
}


/* This slot is called when loading is done. */ /* private slot */
void DownloadManager::slotResult( const LoaderData& data )
{
    FileLoader* loader = getLoader( data.url );

   /* Deleting the FileLoader here may crush kita,
      because loader may be also used in the other classes.
      So, delete it later.  See also customEvent().    */  
    DeleteLoaderEvent * e = new DeleteLoaderEvent( loader );
    QApplication::postEvent( this, e );  // Qt will delete it when done
}


/* protected */ /* virtual */
void DownloadManager::customEvent( QCustomEvent * e )
{
    /* delete loader */
    if ( e->type() == EVENT_DeleteLoader ) {
        FileLoader * loader = static_cast< DeleteLoaderEvent* >( e ) ->getLoader();
        if ( loader ) deleteLoader( loader );
    }
}



/*------------------------------------------------------------*/
/*------------------------------------------------------------*/

/* class FileLoader */


/*

url:     url of target
path:    path of save file. It can be QString::null. Then data is not saved in HDD. 
option:  user defined option. See also downloadmanager.h 

Call get() to start. Call stopJob() to stop.
SIGNAL data() is emitted when data is received.
SIGNAL result() is emitted when done.

The pointer of struct Kita::LoaderData is sent by SIGNAL data() and result().
For example,  to get result code:

void TmpClass::slotResult( const Kita::LoaderData& data )
{
   int result_code = data.code;
}

See also ImgManager::slotResult() and ImgManager::slotData().

*/
FileLoader::FileLoader( const KURL& url, const QString& path, const LoaderOption& option )
{
    m_data.url = url;
    m_data.path = path;
    m_data.tmppath = path;
    if( path != QString::null ) m_data.tmppath += ".tmp";
    m_data.option = option;

    m_ds = NULL;
    m_file = NULL;
    m_data.totalsize = 0;
    m_data.size = 0;

    m_currentJob = NULL;
    m_data.code = 0;
    m_data.header = QString::null;
}

FileLoader::~FileLoader()
{
    killJob();
}


/* terminate job.  slotResult is NOT called. */ /* public */
void FileLoader::killJob()
{
    /* now loading */
    if ( m_currentJob ){

	m_currentJob->kill(); /* without emitting result SIGNAL */
	m_currentJob = NULL;
	m_data.code = 0;
	closeFile();
    }
}


/* stop job. slotResult is called later. */ /* public */
void FileLoader::stopJob()
{
    if ( m_currentJob ) m_currentJob->kill( FALSE ); /* with emitting result SIGNL */
}


/* start downloading                              */
/* When done, slotResult() will be called.        */ /* public */
bool FileLoader::get()
{
    /* now loading */
    if ( m_currentJob ) return FALSE;

    /* create directory */
    if( m_data.tmppath != QString::null ){
	QFileInfo qi( m_data.tmppath );
	if( !Kita::mkdir( qi.dirPath( TRUE ) ) ) return FALSE;
    }

    m_data.totalsize = 0;
    m_data.size = 0;
    m_data.code = 0;
    m_data.header = QString::null;
    m_data.redirection = FALSE;
    m_file = NULL;
    m_ds = NULL;
    
    /* If path of file is not QString::null, open file here */
    /* Note that data is saved as "m_data.tmppath".
       This file will be renamed in FileLoader::slotResult(). */
    if( m_data.tmppath != QString::null ){
	m_file = new QFile( m_data.tmppath );
	if ( m_file == NULL || ! m_file->open( IO_WriteOnly ) ) {
	    closeFile();
	    return FALSE;
	}
	m_ds = new QDataStream( m_file );
    }

    /* create job */
    m_currentJob = KIO::get( m_data.url, TRUE, FALSE );
    if ( m_currentJob == NULL ) {
        closeFile();
        return FALSE;
    }

    connect( m_currentJob, SIGNAL( data( KIO::Job*, const QByteArray& ) ),
             SLOT( slotData( KIO::Job*, const QByteArray& ) ) );

    connect( m_currentJob, SIGNAL( result( KIO::Job* ) ),
             SLOT( slotResult( KIO::Job* ) ) );

    connect( m_currentJob, SIGNAL( redirection( KIO::Job *, const KURL& ) ),
             SLOT( slotRedirection( KIO::Job *, const KURL& ) ) );

    connect( m_currentJob, SIGNAL( totalSize( KIO::Job*, KIO::filesize_t ) ),
             SLOT( slotTotalSize( KIO::Job*, KIO::filesize_t ) ) );

    /* use 'HTTP-Headers' metadata.*/
    m_currentJob->addMetaData( "PropagateHttpHeader", "true" );

    return TRUE;
}

/* public */
const bool FileLoader::isRunning() const
{
    if ( m_currentJob ) return TRUE;
    return FALSE;
}

/* public */
const LoaderData& FileLoader::getData() const
{
    return m_data;
} 

/* close file                        */
/* if code != 200, then remove file. */ /* private */
void FileLoader::closeFile()
{
    if ( m_file ) m_file->close();
    if ( m_ds ) delete m_ds;
    if ( m_file ) delete m_file;

    m_ds = NULL;
    m_file = NULL;

    /* If loading failed, remove file */
    if ( m_data.code != 200 && m_data.tmppath != QString::null && QFile::exists( m_data.tmppath ) ){

	QFile::remove( m_data.tmppath );
    }
}

/* private */ /* copied from Access::responseCode() */
int FileLoader::responseCode()
{
    // parse HTTP headers
    QStringList headerList = QStringList::split( "\n", m_data.header );
    QRegExp regexp( "HTTP/1\\.[01] ([0-9]+) .*" );
    QString dateStr = headerList.grep( regexp ) [ 0 ];
    if ( regexp.search( dateStr ) == -1 ) {
        // invalid response
        return 0;
    } else {
        return regexp.cap( 1 ).toInt();
    }
}


/* slots called from job */ /* private slots */

void FileLoader::slotData( KIO::Job* job, const QByteArray& barray )
{
    if ( job->error() == 0 ) {

        m_data.size += barray.size();
        if( m_ds ) m_ds->writeRawBytes( barray.data(), barray.size() );

        emit data( m_data, barray );
    }
}


void FileLoader::slotTotalSize( KIO::Job * job, KIO::filesize_t size )
{
    if ( job->error() == 0 ) m_data.totalsize = size;
}


void FileLoader::slotRedirection( KIO::Job*, const KURL& url )
{
    stopJob();
    m_data.redirection = TRUE;
    m_data.redirectURL = url;
}


void FileLoader::slotResult( KIO::Job* job )
{
    m_currentJob = NULL;
    if ( job->error() == 0 ) m_data.header = job->queryMetaData( "HTTP-Headers" );
    m_data.code = responseCode();
    closeFile();

    /* rename */
    if( m_data.code == 200 && m_data.tmppath != QString::null ){
	if ( QFile::exists( m_data.path ) ) QFile::remove( m_data.path );
	QDir qdir;
	qdir.rename( m_data.tmppath, m_data.path );
    }
    
    emit result( m_data );
}
