/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#ifndef KITADATINFO_H
#define KITADATINFO_H

#include <qobject.h>
#include <kurl.h>
#include <qvaluevector.h>
#include <qdatetime.h>
#include <qmutex.h>

class QStringList;

namespace DOM
{
    class HTMLDocument;
    class Element;
}


enum{
    ABONECHK_BY_ID,
    ABONECHK_BY_NAME,
    ABONECHK_BY_WORD
};

/* ResDatVec is the Database for responses. */
/* For example, No.5 res has name = "foo", id "id1234", and anchors >>2-3 and >>4,
   then,
 
   m_resDatVec[5].name = "foo",
   m_resDatVec[5].id = "id1234",
   m_resDatVec[5].anclist[0].from = 2,
   m_resDatVec[5].anclist[0].to = 3,
   m_resDatVec[5].anclist[1].from = 4,
   m_resDatVec[5].anclist[1].to = 4.   
 
*/
struct ANCNUM
{
    int from;
    int to;
};
typedef QValueList<ANCNUM> AncList;

/* Data of one response */
struct RESDAT
{

    bool set; /* Is linestr set ? */

    QString linestr; /* raw line strings */

    /*-------*/

    bool parsed; /* Is dat parsed ? */

    bool broken; /* This dat is broken */

    bool marked;

    QString name;
    QString parsedName;
    QString address;
    QString body;
    QString id;
    QString host;
    QString date;
    QDateTime dateTime;

    /*-------*/

    bool setAnclist; /* Is anclist set ? */

    AncList anclist; /* anchor list. See also setAncList()  */

    /*-------*/

    bool checkAbone; /* Is abone checked ? */

    bool abone;
};

typedef QValueVector<RESDAT> ResDatVec;


namespace Kita
{

    class Access;
    class OfflawAccess;
    class Thread;

    /*-----------------------*/

    class DatInfo : public QObject
    {
        Q_OBJECT

        QMutex m_mutex;

	/* pointer of Thread class */
	Thread* m_thread; 

        /* basic information */
        QString m_rawData;
        KURL m_datURL;
        bool m_broken;
        ResDatVec m_resDatVec;
        int m_lock;

        /* for caching */
        Access* m_access;
        OfflawAccess* m_access2;
        QString m_lastLine;
        bool m_nowLoading;

        /* abone */
        bool m_aboneByID;
        bool m_aboneByName;
        bool m_aboneByBody;
        bool m_aboneByRes;


        /* japanese strings */
        QString m_spacestr;
        QString m_framestr1; /* '|' */
        QString m_framestr2; /* '|-' */
        QString m_framestr3; /* 'L' */

        /*-------------------------------*/

    public:

        DatInfo();
        DatInfo( const KURL& url );
        ~DatInfo();
        void init();
        void wait();	
        const KURL& url();

        /* for caching */
        bool updateCache( const QObject* parent );
        int getResponseCode();
        int getServerTime();
        bool deleteCache();
        bool isLoadingNow();
        void stopLoading();

        /* lock, unlock */
        void lock ();
        void unlock();
        int isLocked();

        /* string data */
        const QString& getRawDat();
        const QString& getDat( int num );
        const QString& getId( int num );
        const QString& getName( int num );
        const QString& getBody( int num );
        QString getPlainName( int num );
        QString getPlainBody( int num );
        QString getPlainTitle( int num );

        /* HTML data */
        QString getHtml( int startnum, int endnum );
        QString getHtmlByID( const QString& strid, int &count );
        QString getTreeByRes( const int rootnum, int& count );
        QString getTreeByResReverse( const int rootnum, int& count );

        /* DOM element */
        bool getDomElement( int num, DOM::HTMLDocument& hdoc, DOM::Element& retelm );

        /* numerical data */
        int getResNum();
        int getReadNum();
	int getViewPos();
        int getNumByID( const QString& strid );
        int getDatSize();

        /* several informations */
        bool isResValid( int num );
        bool isBroken();
        bool isResBroken( int num );
        bool checkID( const QString& strid, int num );
        bool checkWord( QStringList& stlist, int num, bool checkOR );
        bool isMarked( int num );
        void setMark( int num, bool mark );

        /* abone check */
        bool checkAbone( int num );
        void resetAbone();
	
        /*-------------------------*/

    private:

        void initPrivate( bool loadCache = TRUE );
        void resetResDat( RESDAT& resdat );
        void increaseResDatVec( int delta );
        void deleteAccessJob();

        /* copy data */
        bool copyOneLineToResDat( const QString& line, int num );
	
        /* HTML data */
        QString aboneHTML( int num );
        QString brokenHTML( int num );
        QString getTreeByResPrivate( const int rootnum, bool reverse, int& count );
        QString getTreeByResCore( const int rootnum, bool reverse, int& count, QString prestr );
        bool checkRes( const int num, const int target );
        void setAncList( int num );

        /* for abone */
        void resetAbonePrivate();
        bool checkAbonePrivate( int num );
        bool checkAboneCore( const QString& str, QStringList& strlist );

        /* parsing funtions */
        bool parseDat( int num );

        /*----------------------------*/

    private slots:

        void slotReceiveData( const QStringList& lineList );
        void slotFinishLoad();

    signals:
        void receiveData();
        void finishLoad();
    };

}

#endif
