/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <qregexp.h>
#include <qstringlist.h>
#include <klocale.h>
#include <dom/html_document.h>
#include <dom/html_element.h>
#include <dom/html_table.h>
#include <dom/dom_text.h>

#include "datinfo.h"
#include "datmanager.h"
#include "access.h"
#include "thread.h"
#include "kitaconfig.h"
#include "kita-utf8.h"
#include "kita-utf16.h"
#include "kita_misc.h"
#include "account.h"
#include "threadindex.h"
#include "cache.h"


using namespace Kita;

#define RESDAT_DEFAULTSIZE 10
#define RESDAT_DELTA       1000


/*------------------------------------------------------*/
/* DatInfo stores & handles all information about *.dat */

DatInfo::DatInfo( const KURL& url ) : m_access ( 0 ), m_access2( 0 )
{
    QString refstr;
    m_datURL = Kita::getDatURL( url, refstr );

    /* get the pointer of Thread class */
    m_thread = Kita::Thread::getByURLNew( m_datURL );
    if( m_thread == NULL ){

	/* create Thread */
	m_thread = Kita::Thread::getByURL( m_datURL );
	if( m_thread == NULL ) return;
	
	/* read idx file */
	ThreadIndex::loadIndex( m_thread, m_datURL );
    }

    m_thread = Thread::getByURL( m_datURL );
    
    /* japanese strings */
    m_spacestr = ". ";
    m_framestr1 = Kita::utf8ToUnicode( KITAUTF8_FRAME1 ); /* |  */
    m_framestr2 = Kita::utf8ToUnicode( KITAUTF8_FRAME2 ); /* |- */
    m_framestr3 = Kita::utf8ToUnicode( KITAUTF8_FRAME3 ); /* L  */

    /* make directory */
    QString cacheDir = Cache::baseDir() + Cache::serverDir( m_datURL ) + Cache::boardDir( m_datURL );
    if( !Kita::mkdir( cacheDir ) ) return;
    
    initPrivate();
}

DatInfo::~DatInfo()
{
    deleteAccessJob();
}


/* Init  */
/* Usually, don't call this. */ /* public */
void DatInfo::init()
{
    QMutexLocker locker( &m_mutex );

    return initPrivate();
}

/* public */
void DatInfo::wait()
{
    QMutexLocker locker( &m_mutex );
}

/* Init. If loadCache = TRUE, load data from cache. */ /* private */
void DatInfo::initPrivate( bool loadCache )
{

    /* stop & delete dat loader */
    deleteAccessJob();

    /* init variables */
    m_rawData = QString::null;
    m_broken = FALSE;
    m_lock = 0;
    m_nowLoading = FALSE;
    m_lastLine = QString::null;

    /* reset readNum  */
    m_thread->setReadNum( 0 ); 
    
    /* clear ResDatVec */
    m_resDatVec.clear();
    increaseResDatVec( RESDAT_DEFAULTSIZE );

    /* reset Abone */
    resetAbonePrivate();

    /* create dat loader */
    m_access = new Kita::Access( m_datURL );

    connect( m_access, SIGNAL( receiveData( const QStringList& ) ),
             SLOT( slotReceiveData( const QStringList& ) ) );
    connect( m_access, SIGNAL( finishLoad() ), SLOT( slotFinishLoad() ) );

    if( !loadCache ) return;
    
    /* get dat from cahce   */
    /* slotReceiveData() is called from Access::getcache() */
    m_access->getcache();

    /* save thread information */
    if( m_thread->readNum() ) ThreadIndex::saveIndex( m_thread, m_datURL );
}


/* private */
void DatInfo::resetResDat( RESDAT& resdat )
{

    resdat.set = FALSE;
    resdat.parsed = FALSE;
    resdat.broken = FALSE;
    resdat.setAnclist = FALSE;
    resdat.anclist.clear();
    resdat.checkAbone = FALSE;
    resdat.abone = FALSE;
    resdat.marked = FALSE;
}


/* private */
void DatInfo::increaseResDatVec( int delta )
{
    int size = m_resDatVec.size();
    RESDAT resdat;
    resetResDat( resdat );
    m_resDatVec.resize( size + delta, resdat );
}


/* delete dat loader */ /* private */
void DatInfo::deleteAccessJob()
{
    if ( m_access ) {
        m_access->killJob();
        delete m_access;
        m_access = NULL;
    }
    if ( m_access2 ) {
        m_access2->killJob();
        delete m_access2;
        m_access2 = NULL;
    }
}


/* copy one line to resdat.            */ /* private */
bool DatInfo::copyOneLineToResDat( const QString& line, int num )
{

    if ( num <= 0 ) return FALSE;
    if ( line == QString::null ) return FALSE;

    /* If resdat vector is short, then resize the vector. */
    while ( ( int ) m_resDatVec.size() <= num ) increaseResDatVec( RESDAT_DELTA );

    /* reset and  set new data */
    RESDAT& resdat = m_resDatVec[ num ];
    resetResDat( resdat );

    resdat.set = TRUE;
    resdat.linestr = line;
    if ( m_thread->readNum() < num ) m_thread->setReadNum( num );
	
    m_rawData += line + '\n';

    if ( num == 1 ) parseDat ( num ); /* to get subject */

    /* is this dat file broken ? */
    if ( line.contains( "<>" ) != 4 ) {
        resdat.broken = TRUE;
        m_broken = TRUE;
    }

    return TRUE;
}


/* public */
const KURL& DatInfo::url()
{
    QMutexLocker locker( &m_mutex );

    return m_datURL;
}



/*--------------------------------------*/
/* cache handling functions             */

/* Update cache  */

/* When Kita::Access received new data,
   slotReceiveData is called.           */

/* When Kita::Access fineshed loading,
   slotFinishLoad is called, and
   DatInfo emits the finishLoad signal to the parent object  */ /* public */
bool DatInfo::updateCache( const QObject* parent )
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return FALSE;
    if ( m_nowLoading ) return FALSE;

    m_nowLoading = TRUE;
    m_lock++; /* By locking, DatManager can't delete this while loading. */

    connect( this, SIGNAL( receiveData() ),
             parent, SLOT( slotReceiveData() ) );

    connect( this, SIGNAL( finishLoad() ),
             parent, SLOT( slotFinishLoad() ) );
    
    m_access->getupdate( m_thread->readNum() );

    return TRUE;
}


/* slot called when Kita::Access
   received new data              */      /* private  slot */
void DatInfo::slotReceiveData( const QStringList& lineList )
{
    int rescode = m_access->responseCode();
    if ( m_access2 ) {
        rescode = m_access2->responseCode();
    }

    if ( rescode != 200 && rescode != 206 ) return ;

    /* copy lines to buffer */
    int count = lineList.count();
    for ( int i = 0; i < count ; ++i ) copyOneLineToResDat( lineList[ i ], m_thread->readNum() +1 );

    emit receiveData();
}


/* slot called when Kita::Access
   finished loading new dat */      /* private  slot */
void DatInfo::slotFinishLoad()
{
    /* save thread information */
    ThreadIndex::saveIndex( m_thread, m_datURL );
    
    if ( m_thread->readNum() == 0 && m_access2 == NULL && BoardManager::type( m_datURL ) == Board_2ch ) {
        if ( Account::isLogged() ) {
            initPrivate();
            m_access2 = new OfflawAccess( m_datURL );
            connect( m_access2, SIGNAL( receiveData( const QString& ) ),
                      SLOT( slotReceiveData( const QString& ) ) );
            connect( m_access2, SIGNAL( finishLoad() ), SLOT( slotFinishLoad() ) );
            m_access2->get();
            return;
        }
    }
    /* finish loading session & emit signal to the parent object */
    m_nowLoading = FALSE;
    emit finishLoad();

    /* disconnect signals */
    disconnect( SIGNAL( receiveData() ) );
    disconnect( SIGNAL( finishLoad() ) );

    if ( m_lock ) m_lock--;
}


/* public */
int DatInfo::getResponseCode()
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return 0;

    return m_access->responseCode();
}


/* public */
int DatInfo::getServerTime()
{
    QMutexLocker locker( &m_mutex );
    if ( m_access == NULL ) return 0;

    return m_access->serverTime();
}


/* public */
bool DatInfo::deleteCache()
{
    QMutexLocker locker( &m_mutex );
    if ( m_nowLoading ) return FALSE;

    initPrivate( FALSE );

    return TRUE;
}


/* public */
bool DatInfo::isLoadingNow()
{

    QMutexLocker locker( &m_mutex );

    return m_nowLoading;
}



/* public */
void DatInfo::stopLoading()
{

    /* Don't lock the mutex here !!!
       It will cause deadlock , because
       Kita::Access::stopJob() calls KitaHTMLPart::slotFinishLoad() back,
       then KitaHTMLPart::slotFinishLoad() calls another functions in DatInfo. */
    if ( m_access == NULL ) return ;
    if ( ! m_nowLoading ) return ;

    m_access->stopJob();
}




/*---------------------------------------------------*/
/* locking , unlocking functions                     */
/*                                                   */
/* If m_lock is not 0, DatManager can't delete this. */
/* Don't forget to call unlock() after locking this. */

/* They are public */

void DatInfo::lock ()
{
    QMutexLocker locker( &m_mutex );

    m_lock++;
}

void DatInfo::unlock()
{
    QMutexLocker locker( &m_mutex );

    if ( m_lock ) m_lock--;
}

int DatInfo::isLocked()
{
    QMutexLocker locker( &m_mutex );

    return m_lock;
}


/*------------------------------------------------------*/
/* get subject, linedata,  id, body, name, HTML, etc.   */

/* They are public */

const QString& DatInfo::getRawDat()
{
    QMutexLocker locker( &m_mutex );

    return m_rawData;
}

const QString& DatInfo::getDat( int num )
{
    QMutexLocker locker( &m_mutex );

    if ( num <= 0 || ( int ) m_resDatVec.size() <= num ) return QString::null;
    if ( ! m_resDatVec[ num ].set ) return QString::null;

    return m_resDatVec[ num ].linestr;
}

const QString& DatInfo::getId( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].id;
}

const QString& DatInfo::getBody( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].body;
}

const QString& DatInfo::getName( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].name;
}

/* plain (parsed) strings of name  */
QString DatInfo::getPlainName( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    return m_resDatVec[ num ].parsedName;
}


/* plain (parsed) strings of body  */
QString DatInfo::getPlainBody( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    QString bodytext;
    QString line = m_resDatVec[ num ].body;

    Kita::parseBODYdatText( PARSEMODE_TEXT, line, bodytext );

    return bodytext;
}


/* plain (parsed) strings of title */
QString DatInfo::getPlainTitle( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return QString::null;

    bool showAddr = KitaConfig::showMailAddress();
    QString titletext;
    Kita::parseTITLEdatText( PARSEMODE_TEXT, num, showAddr, m_resDatVec[ num ], titletext );

    return titletext;
}


/*-----------------------------------------*/
/* HTML data                               */

/* HTML strings */
/* Note that this function checks Abone internally. */ /* public */
QString DatInfo::getHtml( int startnum, int endnum )
{
    QMutexLocker locker( &m_mutex );

    QString retstr = QString::null;
    bool showAddr = KitaConfig::showMailAddress();

    for ( int num = startnum; num <= endnum; num++ ) {

        if ( !parseDat( num ) ) continue;
        if ( checkAbonePrivate( num ) ) retstr += aboneHTML( num );
        else if ( m_resDatVec[ num ].broken ) retstr += brokenHTML( num );
        else retstr += Kita::ResDatToHtml( m_resDatVec[ num ], num, showAddr );
    }

    return retstr;
}


/* return HTML strings that have ID = strid. */
/* Note that this function checks Abone internally. */ /* public */
QString DatInfo::getHtmlByID( const QString& strid, int &count )
{
    QMutexLocker locker( &m_mutex );

    QString retstr = QString::null;
    bool showAddr = KitaConfig::showMailAddress();

    count = 0;

    for ( int i = 1; i <= m_thread->readNum(); i++ ) {

        if ( !parseDat( i ) ) continue;

        if ( m_resDatVec[ i ].id == strid ) {
            count ++;
            if ( checkAbonePrivate( i ) ) retstr += aboneHTML( i );
            else if ( m_resDatVec[ i ].broken ) retstr += brokenHTML( i );
            else retstr += Kita::ResDatToHtml( m_resDatVec[ i ], i, showAddr );
        }
    }

    return retstr;
}


/* private */
QString DatInfo::aboneHTML( int num )
{

    QString tmpstr = QString( "<dl><dt>" );
    tmpstr += QString().setNum( num ) + " " + Kita::utf8ToUnicode( KITAUTF8_ABONE );
    tmpstr += "</dt><dd>";
    tmpstr += Kita::utf8ToUnicode( KITAUTF8_ABONE );;
    tmpstr += "<br/><br/></dd></dl>";

    return tmpstr;
}


/* private */
QString DatInfo::brokenHTML( int num )
{

    QString tmpstr = QString( "<dl><dt>" );
    tmpstr += QString().setNum( num ) + " " + Kita::utf8ToUnicode( KITAUTF8_NAME );
    tmpstr += "</dt><dd>";
    tmpstr += Kita::utf8ToUnicode( KITAUTF8_KOWARE );;
    tmpstr += "<br/><br/></dd></dl>";

    return tmpstr;
}


/*-------------------------------*/
/* Get HTML document of res tree.*/
/* For example, when rootnum = 1,
 
>>1 
|-->>4
|  |--->>10
|
|-->>20, and return count = 3.  */

/* Note that this function checks Abone internally. */ /* public */
QString DatInfo::getTreeByRes( const int rootnum, int& count )
{
    QMutexLocker locker( &m_mutex );
    return getTreeByResPrivate( rootnum, FALSE, count );
}

/*---------------------------------------*/
/* Get HTML document of reverse res tree.*/
/* For example, when rootnum = 10,
 
>>10 
|-->>5
|  |--->>2
|
|-->>6, and returns count = 3.  */

/* Note that this function checks Abone internally. */ /* public */
QString DatInfo::getTreeByResReverse( const int rootnum, int& count )
{
    QMutexLocker locker( &m_mutex );
    return getTreeByResPrivate( rootnum, TRUE, count );
}


/* private */
QString DatInfo::getTreeByResPrivate(
    const int rootnum,
    bool reverse,   /* reverse search */
    int& count )
{

    QString tmp = QString().setNum( rootnum );
    QString retstr = "<a href=\"#" + tmp + "\">&gt;&gt;" + tmp + "</a><br>";

    retstr += getTreeByResCore( rootnum, reverse, count, "" );

    return retstr;
}

/* private */
QString DatInfo::getTreeByResCore(
    const int rootnum,
    bool reverse,   /* reverse search */
    int& count, QString prestr )
{
    if ( !parseDat( rootnum ) ) return QString::null;
    if ( checkAbonePrivate( rootnum ) ) return QString::null;

    QString retstr = QString::null ;
    count = 0;
    QStringList strlists;

    if ( !reverse ) {

        /* collect responses that have anchor to rootnum */
        for ( int i = rootnum + 1; i <= m_thread->readNum(); i++ ) {
            if ( checkAbonePrivate( i ) ) continue;
            if ( checkRes( i, rootnum ) ) {
                count ++;
                strlists += QString().setNum( i );
            }
        }

    } else { /* collect responses for which rootnum has anchors */

        setAncList( rootnum );
        AncList& anclist = m_resDatVec[ rootnum ].anclist;
        for ( AncList::iterator it = anclist.begin(); it != anclist.end(); ++it ) {
            for ( int i = ( *it ).from; i <= QMIN( rootnum - 1, ( *it ).to ) ; i++ ) {
                if ( checkAbonePrivate( i ) ) continue;
                count ++;
                strlists += QString().setNum( i );
            }
        }
    }

    /* make HTML document */
    if ( count ) {

        for ( QStringList::iterator it = strlists.begin(); it != strlists.end(); ++it ) {
            QString tmpstr;
            if ( ( *it ) == strlists.last() ) tmpstr = m_framestr3;  /* 'L' */
            else tmpstr = m_framestr2;  /* '|-' */

            retstr += prestr + tmpstr + "<a href=\"#" + ( *it ) + "\">&gt;&gt;" + ( *it ) + "</a><br>";

            /* call myself recursively */
            int tmpnum;
            tmpstr = prestr;
            if ( ( *it ) == strlists.last() ) tmpstr += m_spacestr + m_spacestr + m_spacestr; /* "   " */
            else tmpstr += m_framestr1 + m_spacestr; /* "| " */
            retstr += getTreeByResCore( ( *it ).toInt(), reverse, tmpnum, tmpstr );
            count += tmpnum;
        }
    }

    return retstr;
}



/*----------------------------------------------*/
/* Check if No.num has anchors to No.target     */
/* For exsample, if target = 4, and No.num have
   an anchor >>4, or >>2-6, etc.,
   then return TRUE.                            */  /* private */
bool DatInfo::checkRes( const int num, const int target )
{
    const int range = 20;
    if ( ! parseDat( num ) ) return FALSE;

    setAncList( num );
    AncList& anclist = m_resDatVec[ num ].anclist;

    for ( AncList::iterator it = anclist.begin(); it != anclist.end(); ++it ) {
        if( ( *it ).to - ( *it ).from > range ) continue;
        if ( target >= ( *it ).from && target <= ( *it ).to ) return TRUE;
    }

    return FALSE;

}


/*--------------------------*/
/* set AncList              */ /* private */

/* This function sets anchor list.
   For example, a res has anchors >>2-3 and >>4, then
   
   anclist[0].from = 2,
   anclist[0].to = 3,
   anclist[1].from = 4,
   anclist[1].to = 4.   
*/
void DatInfo::setAncList( int num )
{

    if ( ! parseDat( num ) ) return ;
    if ( m_resDatVec[ num ].setAnclist ) return ;

    m_resDatVec[ num ].setAnclist = TRUE;

    QString linkstr;
    int refNum[ 2 ];
    unsigned int pos;
    ANCNUM anctmp;

    AncList& anclist = m_resDatVec[ num ].anclist;
    anclist.clear();

    QString line = m_resDatVec[ num ].body;

    /* remove HTML tags */
    line.replace( "<br>", "\n" );
    QRegExp rex( "<[^>]*>" );
    line.remove( rex );

    const QChar *chpt = line.unicode();
    unsigned int i;
    unsigned int length = line.length();

    /* parse body */
    for ( i = 0 ; i < length ; i++ ) {

        if ( chpt[ i ].unicode() == UTF16_BRACKET ||  /* > */
                ( chpt[ i ] == '&' && chpt[ i + 1 ] == 'g' && chpt[ i + 2 ] == 't' && chpt[ i + 3 ] == ';' )  /* "&gt;" */
           ) {
            while ( Kita::parseResAnchor( chpt + i, length - i, linkstr, refNum, pos ) ) {
                if ( refNum[ 1 ] < refNum[ 0 ] ) refNum[ 1 ] = refNum[ 0 ];
                anctmp.from = refNum[ 0 ];
                anctmp.to = refNum[ 1 ];
                anclist += anctmp;
                i += pos;
            }

            i += ( pos - 1 );
        }
    }

    /* parse name */
    line = m_resDatVec[ num ].name;
    chpt = line.unicode();
    i = 0;
    length = line.length();

    while ( Kita::parseResAnchor( chpt + i, length - i, linkstr, refNum, pos ) ) {
        if ( refNum[ 1 ] < refNum[ 0 ] ) refNum[ 1 ] = refNum[ 0 ];
        anctmp.from = refNum[ 0 ];
        anctmp.to = refNum[ 1 ];
        anclist += anctmp;
        i += pos;
    }
}



/*----------------------------------*/
/*  Get DOM element                 */

/* This function returns the element
   of DOM.
 
   This function checks Abone
   internally. So, if the res is
   aboned,the output is "abone" node.
   
   If the res is broken, the output
   is "broken" node.
                                    */ 
/* public */
bool DatInfo::getDomElement(

    /* input */
    int num,
    DOM::HTMLDocument& hdoc,  /* root of HTML document */

    /* output */
    DOM::Element& retelm )
{

    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    QString tmpstr;
    QString status = "normal";
    DOM::Element bodynode, titlenode, tmpelm;
    DOM::HTMLTableElement table;
    bool showAddr = KitaConfig::showMailAddress();
    bool showAA = KitaConfig::showAA();
    bool abone = checkAbonePrivate( num );

    RESDAT& resdat = m_resDatVec[ num ];

    if ( !resdat.broken && !abone ) {

        /* title node */
        table = hdoc.createElement( "TABLE" );
        titlenode = table;
        tmpelm = table.insertRow( 0 );
        Kita::parseTITLEdat( PARSEMODE_DOM, hdoc, num, showAddr, resdat, tmpelm, tmpstr );

        /* body node */
        bodynode = hdoc.createElement( "DIV" );
        {
            /* put the span node at the head of each line */
            if ( showAA ) {

                tmpelm = bodynode.appendChild( hdoc.createElement( "SPAN" ) );
                {
                    tmpelm.setAttribute( "style", "color: white" );
                    tmpelm.appendChild( hdoc.createTextNode( "" ) );
                }
            }

            Kita::parseBODYdat( PARSEMODE_DOM, resdat.body, hdoc, showAA, bodynode, tmpstr );
        }

    } else { /* abone or data is broken */

        QString namestr;
        QString bodystr;

        if ( abone ) { /* "abone" node */
            namestr = Kita::utf8ToUnicode( KITAUTF8_ABONE );
            bodystr = Kita::utf8ToUnicode( KITAUTF8_ABONE );
            status = "abone";
        } else { /* "broken" node */
            namestr = Kita::utf8ToUnicode( KITAUTF8_NAME );
            bodystr = Kita::utf8ToUnicode( KITAUTF8_KOWARE );
            status = "broken";
        }

        /* title node */
        table = hdoc.createElement( "TABLE" );
        titlenode = table;
        tmpelm = table.insertRow( 0 );
        tmpelm = tmpelm.appendChild( hdoc.createElement( "TD" ) );
        tmpelm.appendChild( hdoc.createTextNode(
                                QString().setNum( num ) + " " + namestr ) );

        /* body node */
        bodynode = hdoc.createElement( "DIV" );
        tmpelm = bodynode.appendChild( hdoc.createElement( "SPAN" ) );
        {
            tmpelm.setAttribute( "style", "color: red" );
            tmpelm.appendChild( hdoc.createTextNode( bodystr ) );
        }

    }


    /*-----------------------------*/

    retelm = hdoc.createElement( "DIV" );
    retelm.setAttribute( "class", "res_block" );
    {
        retelm.setAttribute( "kita_type", "res" );
        retelm.setAttribute( "kita_status", status );
        retelm.setAttribute( "id", QString().setNum( num ) );
        retelm.setAttribute( "kita_rname", resdat.name );
        retelm.setAttribute( "kita_rid", resdat.id );

        titlenode.setAttribute( "class", "res_title" );
        bodynode.setAttribute( "class", "res_body" );
        retelm.appendChild( titlenode );
        retelm.appendChild( bodynode );
    }

    return TRUE;
}



/*-----------------------*/
/* several information */

int DatInfo::getResNum()
{
    return m_thread->resNum();
}

/* public */
int DatInfo::getReadNum()
{
    return m_thread->readNum();
}

/* public */
int DatInfo::getViewPos()
{
    return m_thread->viewPos();
}


/* return number of responses that have ID = strid. */
/* Note that this function checks Abone internally. */ /* public */
int DatInfo::getNumByID( const QString& strid )
{
    QMutexLocker locker( &m_mutex );

    int count = 0;

    for ( int i = 1; i <= m_thread->readNum(); i++ ) {

        if ( !parseDat( i ) ) continue;
        if ( checkAbonePrivate( i ) ) continue;

        if ( m_resDatVec[ i ].id == strid ) count++;
    }

    return count;
}


/* public */
int DatInfo::getDatSize()
{
    QMutexLocker locker( &m_mutex );

    return m_rawData.length();
}


/* public */
bool DatInfo::isResValid( int num )
{
    QMutexLocker locker( &m_mutex );

    return parseDat( num );
}

/* public */
bool DatInfo::isBroken()
{
    QMutexLocker locker( &m_mutex );

    return m_broken;
}

/* public */
bool DatInfo::isResBroken( int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    return m_resDatVec[ num ].broken;
}

/* ID = strid ? */ /* public */
bool DatInfo::checkID( const QString& strid, int num )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    if ( m_resDatVec[ num ].id == strid ) return TRUE;

    return FALSE;
}


/* Are keywords included ? */ /* public */
bool DatInfo::checkWord( QStringList& stlist,  /* list of keywords */
                         int num,
                         bool checkOR /* AND or OR search */
                       )
{
    QMutexLocker locker( &m_mutex );
    if ( !parseDat( num ) ) return FALSE;

    QString str_text = m_resDatVec[ num ].body;

    for ( QStringList::iterator it = stlist.begin(); it != stlist.end(); ++it ) {

        QRegExp regexp( ( *it ) );
        regexp.setCaseSensitive( FALSE );

        if ( checkOR ) { /* OR */
            if ( str_text.find( regexp, 0 ) != -1 ) {
                return TRUE;
            }
        } else { /* AND */
            if ( str_text.find( regexp, 0 ) == -1 ) return FALSE;
        }
    }

    if ( checkOR ) return FALSE;

    return TRUE;
}

/* public */
bool DatInfo::isMarked( int num )
{
    QMutexLocker locker( &m_mutex );

    return m_resDatVec[ num ].marked;
}


/* public */
void DatInfo::setMark( int num, bool mark )
{
    QMutexLocker locker( &m_mutex );

    m_resDatVec[ num ].marked = mark;
}




/*--------------------------------*/
/* abone functions                */


/*-----------------------*/
/* reset abone.          */

/* call this when config
   of abone changed.     */  /* public */

void DatInfo::resetAbone()
{
    QMutexLocker locker( &m_mutex );

    return resetAbonePrivate();
}

/* private */
void DatInfo::resetAbonePrivate()
{
    for ( int i = 1; i < ( int ) m_resDatVec.size(); i++ ) m_resDatVec[ i ].checkAbone = FALSE;

    m_aboneByID = ( !KitaConfig::aboneIDList().empty() );
    m_aboneByName = ( !KitaConfig::aboneNameList().empty() );
    m_aboneByBody = ( !KitaConfig::aboneWordList().empty() );
    m_aboneByRes = m_aboneByID | m_aboneByName | m_aboneByBody;
}


/*--------------*/
/* check abone  */ /* public */

bool DatInfo::checkAbone( int num )
{
    QMutexLocker locker( &m_mutex );

    return checkAbonePrivate( num );
}


/* private */
bool DatInfo::checkAbonePrivate( int num )
{
    if ( !parseDat( num ) ) return FALSE;

    if ( m_resDatVec[ num ].checkAbone ) return m_resDatVec[ num ].abone;

    m_resDatVec[ num ].checkAbone = TRUE;
    bool checktmp = FALSE;

    if ( m_aboneByID )
        checktmp = checkAboneCore( m_resDatVec[ num ].id, KitaConfig::aboneIDList() );

    if ( !checktmp && m_aboneByName )
        checktmp = checkAboneCore( m_resDatVec[ num ].parsedName, KitaConfig::aboneNameList() );

    if ( !checktmp && m_aboneByBody )
        checktmp = checkAboneCore( m_resDatVec[ num ].body, KitaConfig::aboneWordList() );

    if ( !checktmp && m_aboneByRes ) {
        setAncList( num );
        AncList& anclist = m_resDatVec[ num ].anclist;

        for ( AncList::iterator it = anclist.begin();
                it != anclist.end() && !checktmp ; ++it ) {

            int refNum = ( *it ).from;
            int refNum2 = ( *it ).to;

            /* I don't want to enter loop... */
            if ( refNum >= num ) continue;
            if ( refNum2 >= num ) refNum2 = num - 1;

            for ( int i = refNum; i <= refNum2; i++ ) {
                if ( checkAbonePrivate( i ) ) {
                    checktmp = TRUE;
                    break;
                }
            }
        }
    }

    m_resDatVec[ num ].abone = checktmp;

    return m_resDatVec[ num ].abone;
}

/* private */
bool DatInfo::checkAboneCore( const QString& str, QStringList& strlist )
{
    if ( strlist.count() ) {

        int i;
        for ( QStringList::iterator it = strlist.begin();
                it != strlist.end(); ++it ) {
            i = str.find( ( *it ) );
            if ( i != -1 ) {
                return TRUE;
            }
        }
    }

    return FALSE;
}



/*---------------------------------------*/
/* simple parsing function               */

/* This function parses struct RESDAT by
   Kita::parseResDat. In short, this
   splits the raw date into name, id,
   date, and bodytext, etc.
 
   Note that this function keeps data as
   the raw data. So, you need to parse them
   later by getPlainBody, getHtml,
   getDomElement, etc. 
                                         */ /* private */
bool DatInfo::parseDat( int num )
{

    if ( num <= 0 || ( int ) m_resDatVec.size() <= num ) return FALSE;
    if ( m_resDatVec[ num ].parsed ) return TRUE;

    QString subject = QString::null;
    if ( !Kita::parseResDat( m_resDatVec[ num ], subject ) ) return FALSE;
    if ( subject != QString::null && num == 1 ) {
        m_thread->setThreadName( subject );
    }

    return TRUE;
}
