/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#ifndef KITABOARDMANAGER_H
#define KITABOARDMANAGER_H

#include <kapplication.h>
#include <kurl.h>

class QCp932Codec;
class QEucJpCodec;

namespace Kita
{
    class Thread;
    
    /* type of board */
    enum {
        Board_MachiBBS,
        Board_JBBS,
        Board_2ch,
        Board_FlashCGI,
	Board_Unknown
    };


    /* return value of BoardManager::enrollBoard */
    enum {
      Board_enrollNew,
      Board_enrollEnrolled,
      Board_enrollMoved,
      Board_enrollFailed
    };

    /*---------------------------------*/

    /* Data Base of board */
    class BoardData
    {
	QString m_boardName;
	bool m_readIdx; /* If TRUE, idx file has been read. */

	QString m_hostname;    /* latest host name */
	QString m_rootPath;
	QString m_delimiter;   /* "/test/read.cgi", "/bbs/read.pl", etc. */
	QString m_bbsPath;
	QString m_ext;         /* ".dat", ".cgi", etc.  */
	int m_type;

	QString m_basePath;  /* = (hostname)/(rootPath)/(bbsPath)/ */
	QString m_cgiBasePath;  /* = (hostname)/(rootPath)/(delimiter)/(bbsPath)/ */
	
	/* SETTING.TXT */
	bool m_settingLoaded;
	QString m_defaultName;
	int m_linenum;
	int m_msgCount;
	KURL m_titleImgURL;
	
	/* keys */
	QStringList m_keyHostList; /* list of host names. They are keys of DB. */		
	QStringList m_keyBasePathList; 
	QStringList m_keyCgiBasePathList;
	
    public:
	BoardData( const QString& boardName, const QString& hostname,
		   const QString& rootPath, const QString& delimiter,
		   const QString& bbsPath, const QString& ext, int boardtype );
	~BoardData();

	void setHostName( const QString& hostname );
	
	/* information */
	const bool readIdx() const;
	void setReadIdx( bool idx );	
	const QString& boardName() const;
	const QString& hostName() const;		
	const QString& rootPath() const;
	const QString& delimiter() const;
	const QString& bbsPath() const;
	const QString& ext() const;	
	const int type() const;

	const QString& basePath() const;
	const QString& cgiBasePath() const;

	/* SETTING.TXT */
	const QString settingURL() const;
	const bool settingLoaded() const;
	const QString& defaultName() const;
	const int lineNum() const;
	const int msgCount() const;
	const KURL& titleImgURL() const;
	void setSettingLoaded( bool set );
	void setDefaultName( const QString& newName );
	void setLineNum( int newLine );
	void setMsgCount( int msgCount );
	void setTitleImgURL( const KURL& url );

	/* keys */
	void createKeys( const QStringList& keyHostList );
	const QStringList& keyHostList() const;	
	const QStringList& keyBasePathList() const;
	const QStringList& keyCgiBasePathList() const;
    };


    /*--------------------------------------*/
    
    typedef QValueList< BoardData* > BoardDataList;
    
    /**
    @author Hideki Ikemoto
    */
    class BoardManager
    {
      static BoardDataList m_boardDataList;
      static BoardData* m_previousBoardData; /* used in getBoardData() */
      static QString m_previousBoardURL; /* used in getBoardData() */      
      static QCp932Codec* m_cp932Codec;
      static QEucJpCodec* m_eucJpCodec;
 	
    public:
        BoardManager();
        ~BoardManager();

      static const QString boardURL( const KURL& url );
      static const QStringList allBoardURLList();
      static const QString boardRoot( const KURL& url );      
      static const QString boardPath( const KURL& url );
      static const QString ext( const KURL& url );      
      static const QString boardID( const KURL& url );      
      static const QString subjectURL( const KURL& url );	
      static const QString boardName( const KURL& url );
      static const int type( const KURL& url );

      /* ThreadList */
      static void getThreadList( const KURL& url, bool oldLogs, bool online,
				 QPtrList< Thread >& threadList, QPtrList< Thread >& oldLogList );
      
      /* BoardData */
      static void clearBoardData();
      static int enrollBoard( const KURL& url, const QString& boardName, QString& oldURL,
			      int type = Board_Unknown, bool test = FALSE );
      static bool isEnrolled( const KURL& url );
      static BoardData* getBoardData( const KURL& url );
      
      /* BBSHISTORY */
      static bool loadBBSHistory( const KURL& url );
      static bool moveBoard( const KURL& fromURL, const KURL& toURL );

      /* SETTING.TXT */
      static bool loadBBSSetting( const KURL& url, bool reload = FALSE );
      static QString getBBSDefaultName( const KURL& url );
      static int getBBSMaxLine( const KURL& url );
      static int getBBSMsgCount( const KURL& url );
      static const KURL titleImgURL( const KURL& url );            

      private:

      /* BoardData */
      static int parseBoardURL( const KURL& url,  int type, QString& hostname,
			 QString& rootPath, QString& delimiter,
			 QString& bbsPath,  QString& ext);


      /* ThreadList */
      static void getCachedThreadList( const KURL& url, QPtrList< Thread >& threadList );
      static bool readSubjectTxt( BoardData* bdata, const KURL& url, QPtrList< Thread >& threadList );
      
      
      /* SETTING.TXT */
      static BoardData* openSettingTxt( const KURL& url );
    };
}

#endif
