/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "kitathreadtabwidget.h"
#include "kitathreadview.h"

#include "libkita/kita_misc.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"
#include "libkita/kita-utf8.h"

#include <kdebug.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <klocale.h>
#include <krun.h>
#include <kpopupmenu.h>

#include <qapplication.h>
#include <qregexp.h>
#include <qclipboard.h>
#include <qmessagebox.h>


/*--------------------------------------------------------------------------------*/


KitaThreadTabWidget::KitaThreadTabWidget( int mode, QWidget* parent, const char* name, WFlags f )
        : KitaTabWidgetBase( parent, name, f )
{
    m_mode = mode;

    switch( m_mode ){
    case TABMODE_THREADVIEW: setXMLFile( "kitathreadtabwidgetui.rc" ); break;
    case TABMODE_KITANAVI:   setXMLFile( "kitanaviui.rc" ); break;
    }
    
    setTabBar( new KitaThreadTabBar( this ) );

    connectSignals();
    setupActions();

    /* default view */
    if( m_mode == TABMODE_THREADVIEW ) createView( "thread" );
}


KitaThreadTabWidget::~KitaThreadTabWidget(){}


/* show "Main thread" view */  /* public slots */
void KitaThreadTabWidget::slotShowMainThread( const KURL& url, bool withNewTab )
{
    QString refstr;
    KURL datURL = Kita::ParseMisc::parseURL( url, refstr);
    QString threadName = Kita::DatManager::threadName( datURL );
    int jumpNum = 0;
    
    int viewMode = VIEWMODE_MAINVIEW;
    KitaThreadView* currentView = isThreadView( currentPage() );
    if( currentView ) viewMode = currentView->getViewMode();

    if( refstr != QString::null ){
	int i = refstr.find( "-" );
	if( i != -1 ) jumpNum = refstr.left( i ).toInt();
	else jumpNum = refstr.toInt();
    }

    KitaThreadView* view = findMainView( datURL );

    if ( view ) {
	
        setCurrentPage( indexOf( view ) );

	/* Show on the default view */
        if ( view->threadURL().isEmpty() ) view->showThread( datURL, jumpNum );

	/* The view is already shown */
	/* TODO: jump to jumpNum after reloading */
	else  view->slotReloadButton( jumpNum );
	
    }
    else if ( withNewTab
	      || currentView == NULL           /* Current page is not KitaThreadView. */
	      || viewMode != VIEWMODE_MAINVIEW /* Current page is KitaThreadView, but not Main View. */
	){

        KitaThreadView * newView = createView( threadName );

        if ( newView ) {
            newView->showThread( datURL, jumpNum );
            showPage( newView );
        }
    }
    else static_cast<KitaThreadView *>( currentPage() )->showThread( datURL, jumpNum );

    slotUpdateThreadTab( datURL );
}



/* show "KitaNavi" view  */  /* public slot */
void KitaThreadTabWidget::slotShowKitaNavi(int mode,
		      const KURL& datURL,
		      const QString& query,
		      int startnum,
		      int endnum)
{
    const int margin = 5;
    int count = 0,totalNum = 0;
    QString headstr = QString::null;
    QString footstr = QString::null;
    QString anchor = QString::null;
    QString cmbstr = QString::null;
    QString str, urlstr;
    QStringList queries;
    QRegExp qrx(" +");
    bool ext_OR;
    QString labelstr = QString::null;
    QString tabstr = QString::null;    
			  
    /* set title and label */
    switch(mode){

    case KITANAVI_SHOWRES:
	
	if( startnum == endnum ) tabstr = QString().setNum( startnum );
	else tabstr = QString( "%1-%2" ).arg(startnum).arg( endnum );
	break;
	
    case KITANAVI_EXTBYID:
	tabstr = "ID";
	break;
    case KITANAVI_EXTBYWORD:
	tabstr = Kita::ParseMisc::utf8ToUnicode( KITAUTF8_EXTRACT );
	break;
    case KITANAVI_EXTBYNAME:
	tabstr = "Name";
	break;

    case KITANAVI_SHOWRESTREE:
    case KITANAVI_SHOWREVRESTREE:	
	tabstr = "Tree";
	break;

    }

    /* create KitaThreadView and setup it */
    KitaThreadView *view = createView( tabstr );
    if( view == NULL) return;

    QString toolstr = tabstr + " : "+QString( "[%1] %2" )
	.arg( Kita::BoardManager::boardName( datURL ) )
	.arg( Kita::DatManager::threadName( datURL ) ); 
    
    view->setup( datURL, VIEWMODE_KITANAVI );
    setTabToolTip( view , toolstr );
    totalNum = Kita::DatManager::getResNum( datURL );

    labelstr = Kita::DatManager::threadName( datURL ) + " : " + tabstr;
    
    /*---------------------------------------------*/    
    /* copy data from DatManager to KitaThreadView */

    switch(mode){

    case KITANAVI_SHOWRES:

	for(int i = QMAX(1,startnum-margin); i <= QMIN(totalNum,endnum+margin); i++){
	    if(i == startnum) view->domAppendBelt("belt1");
	    str = Kita::DatManager::getDat( datURL ,i);
	    view->domAppendData(str,i,i);
	    if(i == endnum) view->domAppendBelt("belt2");
	}
	if(startnum == endnum) headstr = QString("No. %1").arg(startnum);
	else headstr = QString("No. %1 - %2").arg(startnum).arg(endnum);
	footstr = headstr;

	anchor = QString().setNum(startnum);
	
	break;
	
	/*--------------------------------*/
	
    case KITANAVI_EXTBYID:

	for(int i = 1; i<= totalNum; i++){
	    if(Kita::DatManager::checkID( datURL, query, i )){
		str = Kita::DatManager::getDat( datURL, i );
		view->domAppendData(str,i,i);
		count++;
	    }
	}
	headstr = QString("ID:%1 : [%2]").arg(query).arg(count);
	footstr = headstr;
	anchor = "header";
	break;

	/*--------------------------------*/
	
    case KITANAVI_EXTBYNAME:

	for(int i = 1; i<= totalNum; i++){
	    if( Kita::DatManager::getPlainName( datURL, i ) == query ){
		str = Kita::DatManager::getDat( datURL, i );
		view->domAppendData( str, i, i );
		count++;
	    }
	}
	headstr = Kita::ParseMisc::utf8ToUnicode(KITAUTF8_EXTRACT)
	    +QString(": %1 : [%2]").arg( query ).arg( count );
	footstr = headstr;
	anchor = "header";
	break;
	
	/*--------------------------------*/

    case KITANAVI_EXTBYWORD:

	if(query.at(0) == '|'){ /* OR */
	    ext_OR = TRUE;
	    str = query.mid(1);
	}
	else{ /* AND */
	    ext_OR = FALSE;
	    str = query;
	}
	str.replace(Kita::ParseMisc::utf8ToUnicode(KITAUTF8_ZENSPACE)," ");
	str.replace(qrx," "); /* pack space */
	queries  = QStringList::split(" ", str);

	for(int i = 1; i<= totalNum; i++){
	    if(Kita::DatManager::checkWord( datURL, queries, i, ext_OR )){
		str = Kita::DatManager::getDat( datURL, i );
		view->domAppendData(str,i,i);
		count++;
	    }
	}
	headstr = Kita::ParseMisc::utf8ToUnicode(KITAUTF8_EXTRACT)
	    +QString(": %1 : [%2]").arg(query).arg(count);
	footstr = headstr;
	anchor = "header";
	cmbstr = ":?"+query;
	break;

	/*--------------------------------*/
	
    case KITANAVI_SHOWRESTREE:
    case KITANAVI_SHOWREVRESTREE:	

	str = QString("[%1] %2 : No.%3 [%4]<br>")
	    .arg( Kita::BoardManager::boardName( datURL ) )
	    .arg( Kita::DatManager::threadName( datURL ) )
	    .arg(startnum ).arg( endnum );
	urlstr = Kita::DatManager::threadURL( datURL ) + QString().setNum( startnum );
	str += "<a href=\""+urlstr+"\">"+urlstr+"</a><br>";
	str += Kita::DatManager::getHtml( datURL, startnum, startnum );
	if( endnum ){
	str += "<div style=\"background-color: #CCCCCC; text-align: center\"><br></div>";
	str += query;
	}
	view->domSetInnerHTML(str);
	
	break;

    default:
	break;
    }
    
    showPage(view);
    view->domApplyChange(labelstr,headstr,footstr,anchor,cmbstr);
}


/* close "all" views which URL is url. */ /* public slot */
void KitaThreadTabWidget::slotCloseThreadTab( const KURL& url )
{
    int max = count();
    if( max == 0 ) return;
    KURL datURL = Kita::ParseMisc::parseURLonly( url );
    
    int i, i2;
    i = i2 = 0;
    while( i < max ){
	KitaThreadView* view = isThreadView( page ( i ) );
	if( view ){
	    if( view->datURL() == datURL ){
		slotCloseTab( i2 );
		i2--;
	    }
	}
	i++; i2++;
    }
}



/* create KitaThreadView */   /* private */
KitaThreadView* KitaThreadTabWidget::createView( QString label )
{
    KitaThreadView* view = new KitaThreadView( this );
    if( view ){
	addTab( view, label );
	connect( view, SIGNAL( closeCurrentTab() ), SLOT( slotCloseCurrentTab() ) );
    }
    
    return view;
}


/* private */
void KitaThreadTabWidget::connectSignals()
{
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();

    connect( signalCollection, SIGNAL( updateThreadTab( const KURL& ) ),
             this, SLOT( slotUpdateThreadTab( const KURL& ) ) );
    connect( signalCollection, SIGNAL( closeThreadTab( const KURL& ) ),
             SLOT( slotCloseThreadTab( const KURL& )));
}


/* private */
KitaThreadView* KitaThreadTabWidget::findMainView( const KURL& url )
{
    KURL datURL = Kita::ParseMisc::parseURLonly( url );

    int max = count();
    if( max == 0 ) return NULL;
    int i = 0;

    while( i < max ){
	KitaThreadView* view = isThreadView( page ( i ) );
	if( view ){
	    if( view->getViewMode() == VIEWMODE_MAINVIEW ){
		
		if ( view->datURL() == datURL
		     || view->datURL().isEmpty() /* default view */
		    ) return view;
	    }
	}
	i++;
    }

    return NULL;
}


/* private */
KitaThreadView* KitaThreadTabWidget::isThreadView( QWidget* w )
{
    KitaThreadView* view = NULL;
    if( w ){
	if( w->isA( "KitaThreadView" ) ) view = static_cast< KitaThreadView* >( w );
    }

    return view;
}


/* private slots */
void KitaThreadTabWidget::slotUpdateThreadTab( const KURL& url )
{
    KURL datURL = Kita::ParseMisc::parseURLonly( url );

    KitaThreadView * view = findMainView( datURL );
    if ( view ){
	QString threadName = Kita::DatManager::threadName( datURL );

	setTabLabel( view, threadName );
	setTabToolTip( view, threadName );
    }
}



/* protected */ /* virtual */
void KitaThreadTabWidget::deleteWidget( QWidget* w )
{
    KitaTabWidgetBase::deleteWidget( w );
    
    if ( count() == 0 ){

	if( m_mode == TABMODE_THREADVIEW ){

	    emit setMainCaption( QString::null );
	    emit setMainStatusbar( QString::null ); 
	    emit setMainURLLine( QString::null );
	    
	    /* default view */
	    KitaThreadView * threadView = createView( "thread" );

	    if ( threadView ) {
		showPage( threadView );
		emit switchToSubject();
	    }
	}
	else if( m_mode == TABMODE_KITANAVI ){
	    	static_cast< KitaDockWidgetBase* >( parentWidget() )->slotHideDock();
	}
    }
}



/*--------------------------------*/
/* KitaThreadView actions */


/* private */
void KitaThreadTabWidget::setupActions()
{
    KStdAction::copy( this, SLOT( slotCopyText() ), actionCollection() );
    
    new KAction( i18n( "Open with Web Browser" ),
                 0,
                 this,
                 SLOT( slotOpenBrowser() ),
                 actionCollection(),
                 "threadview_openbrowser" );

    new KAction( i18n( "copy URL" ),
                 0,
                 this,
		 SLOT( slotCopyURL() ),
                 actionCollection(),
                 "threadview_copyurl" );   
    
    new KAction( i18n( "Copy title and URL" ),
                 0,
                 this,
		 SLOT( slotCopyTitleURL() ),
                 actionCollection(),
                 "threadview_copytitleurl" );   
    
    new KAction( i18n( "Find" ),
                 "find",
                 KStdAccel::find(),
                 this,
                 SLOT( slotFocusSearchCombo() ),
                 actionCollection(),
                 "threadview_find" );
    
    KStdAction::findNext( this,
		      SLOT( slotSearchNext() ),
		      actionCollection(),
		      "threadview_findnext");

    KStdAction::findPrev( this,
		      SLOT( slotSearchPrev() ),
		      actionCollection(),
		      "threadview_findprev");

    new KAction( i18n( "go back anchor" ),
		 "2leftarrow",
		 Key_Backspace,
		 this,
		 SLOT( slotGobackAnchor() ),
		 actionCollection(),
		 "threadview_goback" );

    new KAction( i18n( "Start" ),
		 "up",
		 Key_Home,
		 this,
		 SLOT( slotGotoHeader() ),
		 actionCollection(),
		 "threadview_gotoheader" );    
    
    new KAction( i18n( "End" ),
		 "down",
		 Key_End,
		 this,
		 SLOT( slotGotoFooter() ),
		 actionCollection(),
		 "threadview_gotofooter" );    

    new KAction( i18n( "Reload" ),
                 "reload",
                 KStdAccel::reload(),
                 this,
                 SLOT( slotReloadButton() ),
                 actionCollection(),
                 "threadview_reload" );

    new KAction(i18n("Stop"),
                "stop",
                Key_Escape,
                this,
                SLOT(slotStopLoading()),
                actionCollection(),
                "threadview_stop");

    new KAction(i18n("Delete"),
                "edittrash",
                Key_Delete,
                this,
                SLOT(slotDeleteButtonClicked()),
                actionCollection(),
                "threadview_delete");
}



/* KitaThreadView actions */

/* copy selected text (Ctrl+C) */ /* public slot */
void KitaThreadTabWidget::slotCopyText()
{
    QWidget* w = currentPage();
    KitaThreadView * view = isThreadView( w );
    if( view ){
	QClipboard * clipboard = QApplication::clipboard();
	QString text = view->selectedText();
	clipboard->setText( text, QClipboard::Clipboard );
    }
}

/* public slot */
void KitaThreadTabWidget::slotOpenBrowser( int idx )
{
    QWidget* w = currentPage();
    if( idx != -1 ) w = page( idx );
    KURL url;
    
    if( w ){

	KitaThreadView * view = isThreadView( w );
	if( view ){
	    new KRun( view->threadURL() );
	    return;
	}

	KParts::Part* part = findPartFromWidget( w );
	if( part && part->inherits( "KParts::ReadOnlyPart" ) ){
	    url = static_cast<KParts::ReadOnlyPart*>( part )->url();
	    new KRun( url );
	    return;
	}
    }
}


/* public slot */
void KitaThreadTabWidget::slotCopyURL( int idx )
{
    QWidget* w = currentPage();
    if( idx != -1 ) w = page( idx );
    
    QClipboard* clipboard = QApplication::clipboard();

    KitaThreadView * view = isThreadView( w );    
    if( view ){
	KURL datURL = view->datURL();
    
	QString cliptxt = Kita::DatManager::threadURL( datURL );
	clipboard->setText( cliptxt , QClipboard::Clipboard );
	clipboard->setText( cliptxt , QClipboard::Selection );
    }
}


/* public slot */
void KitaThreadTabWidget::slotCopyTitleURL( int idx )
{
    QWidget* w = currentPage();
    if( idx != -1 ) w = page( idx );
    
    QClipboard* clipboard = QApplication::clipboard();

    KitaThreadView * view = isThreadView( w );    
    if( view ){
	KURL datURL = view->datURL();
    
	QString cliptxt = Kita::DatManager::threadName( datURL )
	    + "\n" + Kita::DatManager::threadURL( datURL );
	clipboard->setText( cliptxt , QClipboard::Clipboard );
	clipboard->setText( cliptxt , QClipboard::Selection );
    }
}


/* public slot  */
void KitaThreadTabWidget::slotFocusSearchCombo()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->focusSearchCombo();
}


/* public slot  */
void KitaThreadTabWidget::slotSearchNext()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotSearchNext();
}


/* public slot  */
void KitaThreadTabWidget::slotSearchPrev()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotSearchPrev();
}


/* public slot  */
void KitaThreadTabWidget::slotGobackAnchor()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotGobackAnchor();
}


/* public slot  */
void KitaThreadTabWidget::slotGotoHeader()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotGotoHeader();
}


/* public slot  */
void KitaThreadTabWidget::slotGotoFooter()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotGotoFooter();
}


/* public slot  */
void KitaThreadTabWidget::slotReloadButton()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotReloadButton();
}


/* public slot  */
void KitaThreadTabWidget::slotStopLoading()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotStopLoading();
}        


/* public slot  */
void KitaThreadTabWidget::slotDeleteButtonClicked()
{
    KitaThreadView * view = isThreadView( currentPage() );    
    if( view ) view->slotDeleteButtonClicked();
}


/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/



KitaThreadTabBar::KitaThreadTabBar( QWidget* parent, const char* name ) : KitaTabBarBase( parent, name ){}

KitaThreadTabBar::~KitaThreadTabBar(){}


/* private */ /* virtual */
void KitaThreadTabBar::showPopupMenu( int idx, QPoint global )
{
    enum{
    	  MENU_CLOSE,
        MENU_CLOSEOTHER,
        MENU_CLOSELEFT,
        MENU_CLOSERIGHT,	      
        MENU_OPENBROWSER,
        MENU_COPYURL,
        MENU_COPYTITLE	      
    };

    KitaThreadTabWidget* tabwidget = static_cast<KitaThreadTabWidget*>( parentWidget() );
    KActionCollection * collection = tabwidget->actionCollection();

    KPopupMenu* popup = new KPopupMenu( this );
    popup->clear();
    
    popup->insertItem( i18n( "Close this tab" ) , MENU_CLOSE );
    collection->action( "tab_prevtab" )->plug( popup );
    collection->action( "tab_nexttab" )->plug( popup );
    popup->insertSeparator();

    popup->insertItem( i18n( "Close Other Tabs" ) , MENU_CLOSEOTHER );    
    popup->insertItem( i18n( "Close right tabs" ) , MENU_CLOSERIGHT );
    popup->insertItem( i18n( "Close left tabs" ) , MENU_CLOSELEFT );
    collection->action( "tab_closealltab" )->plug( popup );        

    popup->insertSeparator();
    
    popup->insertItem( i18n( "Open with Web Browser" ) , MENU_OPENBROWSER );
    popup->insertItem( i18n( "Copy URL" ) , MENU_COPYURL );
    popup->insertItem( i18n( "Copy title and URL" ) , MENU_COPYTITLE );
    
    popup->insertSeparator();
    collection->action( "tab_configkeys" )->plug( popup );       

    
    int ret = popup->exec( global );
    delete popup;
    
    switch( ret ){
    case MENU_CLOSE: tabwidget->slotCloseTab( idx ); break;
    case MENU_CLOSEOTHER: tabwidget->slotCloseOtherTab( idx ); break;
    case MENU_CLOSERIGHT: tabwidget->slotCloseRightTab( idx ); break;
    case MENU_CLOSELEFT: tabwidget->slotCloseLeftTab( idx ); break;
    case MENU_OPENBROWSER: tabwidget->slotOpenBrowser( idx ); break;
    case MENU_COPYURL: tabwidget->slotCopyURL( idx ); break;
    case MENU_COPYTITLE: tabwidget->slotCopyTitleURL( idx ); break;	
    }

}



/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/


/* Don't forget to call setup later ! */
KitaThreadDock::KitaThreadDock( KDockManager* dockManager,
		    const char* name,
		    const QPixmap &pixmap,
		    QWidget* parent,
		    const QString& strCaption,
		    const QString& strTabPageLabel,
		    WFlags f )
    :KitaDockWidgetBase( dockManager, name, pixmap, parent, strCaption, strTabPageLabel, f ){}


KitaThreadDock::~KitaThreadDock(){}


/*-----------------------------*/
/* Main Thread View */

/* setup thig dock as main thread view */ /* public */
KitaThreadTabWidget* KitaThreadDock::setupAsMainView()
{
    m_threadTab = new KitaThreadTabWidget( TABMODE_THREADVIEW, this, "Thread View" );
    setWidget( m_threadTab );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( switchToThread() ),
	     SLOT( slotShowThreadDock() ) );
    
    return m_threadTab;
}

/* public slot */
void KitaThreadDock::slotShowThread( const KURL& url, bool withNewTab )
{
    showDock();

    if( m_threadTab ) m_threadTab->slotShowMainThread( url, withNewTab );
}



/*-----------------------------*/
/* Kita Navi */

/* setup this dock as kita navi */ /* public */
KitaThreadTabWidget* KitaThreadDock::setupAsKitaNavi()
{
    /* create KitaNaviTabWidget */
    m_threadTab = new KitaThreadTabWidget( TABMODE_KITANAVI, this, "Kita Navi");
    setWidget( m_threadTab );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( switchToKitanavi() ),
	     SLOT( slotShowThreadDock() ) );
    
    return m_threadTab;
}


/* public */
void KitaThreadDock::connectNaviSignals()
{
    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();

    connect( signalCollection, SIGNAL( showKitaNavi( const KURL& , int , int )),
	     SLOT( slotShowKitaNavi( const KURL& , int ,  int )));
    connect( signalCollection, SIGNAL( showKitaNaviByID( const KURL& , QString )),
	     SLOT( slotShowKitaNaviByID( const KURL& , QString )));	     
    connect( signalCollection, SIGNAL( showKitaNaviByWord( const KURL& , QString)),
	     SLOT( slotShowKitaNaviByWord( const KURL& , QString)));
    connect( signalCollection, SIGNAL( showKitaNaviByName( const KURL& , QString)),
	     SLOT( slotShowKitaNaviByName( const KURL& , QString)));
    connect( signalCollection, SIGNAL( showKitaNaviResTree( const KURL&,  int )),
	     SLOT( slotShowKitaNaviResTree( const KURL&,  int )));
    connect( signalCollection, SIGNAL( showKitaNaviRevResTree( const KURL&,  int )),
	     SLOT( slotShowKitaNaviRevResTree( const KURL&,  int )));
}


/* show Thread Dock */ /* private slot */
void KitaThreadDock::slotShowThreadDock()
{
    QWidget* topWidget = topLevelWidget();
    if ( topWidget->isMinimized() ) topWidget->showNormal();
    topWidget->raise();
    topWidget->setActiveWindow();

    slotShowDock();
}

/* show res */  /* private slot */
void KitaThreadDock::slotShowKitaNavi( const KURL& datURL, int startnum, int endnum )
{
    showDock();
	
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_threadTab->slotShowKitaNavi( KITANAVI_SHOWRES, parsedURL, QString::null, startnum, endnum );
}


/* extract by ID */ /* private slot */
void KitaThreadDock::slotShowKitaNaviByID( const KURL& datURL, QString strid )
{
    showDock();
    
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_threadTab->slotShowKitaNavi( KITANAVI_EXTBYID, parsedURL, strid, 0, 0 );
}



/* extract by keywords */ /* private slot */
void KitaThreadDock::slotShowKitaNaviByWord( const KURL& datURL, QString query )
{
    showDock();
    
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_threadTab->slotShowKitaNavi( KITANAVI_EXTBYWORD, parsedURL, query, 0, 0 );    
}


/* extract by name */ /* private slot */
void KitaThreadDock::slotShowKitaNaviByName( const KURL& datURL, QString query )
{
    showDock();
    
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    m_threadTab->slotShowKitaNavi( KITANAVI_EXTBYNAME, parsedURL, query, 0, 0 );    
}



/* res tree */ /* private slot */
void KitaThreadDock::slotShowKitaNaviResTree( const KURL& datURL, int resNum )
{
    showDock();
    
    int num;
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    QString htmlstr = Kita::DatManager::getTreeByRes( parsedURL, resNum, num);
    m_threadTab->slotShowKitaNavi( KITANAVI_SHOWRESTREE, parsedURL, htmlstr, resNum, num);
}


/* reverse res tree */ /* private slot */
void KitaThreadDock::slotShowKitaNaviRevResTree( const KURL& datURL, int resNum )
{
    showDock();

    int num;
    KURL parsedURL = Kita::ParseMisc::parseURLonly( datURL );
    QString htmlstr = Kita::DatManager::getTreeByResReverse( parsedURL, resNum, num );
    m_threadTab->slotShowKitaNavi( KITANAVI_SHOWREVRESTREE, parsedURL, htmlstr, resNum, num );
}
