/***************************************************************************
*   Copyright (C) 2004 by Hideki Ikemoto , (c) 2004 by 421                *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

/* Basic class of Tab widget, Tab bar, and Dock widget. */

#include "kitatabwidgetbase.h"

#include "libkita/kita_misc.h"
#include "libkita/parsemisc.h"
#include "libkita/signalcollection.h"
#include "libkita/datmanager.h"

#include <klibloader.h>
#include <kpopupmenu.h>
#include <kdebug.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <klocale.h>
#include <kkeydialog.h>
#include <kparts/dockmainwindow.h>
#include <kparts/factory.h>
#include <kdeversion.h>
#include <kglobal.h>
#include <kconfig.h>
#include <kstandarddirs.h>

#include <qapplication.h>
#include <qmessagebox.h>
#include <qptrlist.h>
#include <qfontmetrics.h>


/*--------------------------------------------------------------*/


KitaTabWidgetBase::KitaTabWidgetBase( QWidget* parent, const char* name, WFlags f )
        : QTabWidget( parent, name, f )
{
    connectSignals();
    setupActions();

    /* setup part manager */
    m_manager = new KParts::PartManager( parent, this, "KitaPartManager" );
    m_manager->addManagedTopLevelWidget( parent );
}


KitaTabWidgetBase::~KitaTabWidgetBase()
{
    /* romove parts */
    if ( m_manager && !( m_manager->parts() ->isEmpty() ) ) {
        KParts::Part * part;
        while ( ( part = m_manager->parts() ->getFirst() ) != NULL ) {
            m_manager->removePart( part );
            removePage( part->widget() );
            delete part;
        }
    }
    if ( m_manager ) delete m_manager;
    m_manager = NULL;

    /* remove widgets which don't belong to parts */
    QWidget* view = currentPage();
    while ( count() > 0 && view ) {
        removePage( view );
        delete view;
        view = currentPage();
    }
}


/* public */ /* virtual */
void KitaTabWidgetBase::addTab( QWidget* w, const QString& label )
{
    KitaTabBase * tb = new KitaTabBase( label );  /* KitaTabBase is deleted automatically. */
    static_cast< KitaTabBarBase* >( tabBar() ) ->adjustTabWidth( tb );
    QTabWidget::addTab( w, tb );
}


/* public */ /* virtual */
void KitaTabWidgetBase::addTab( QWidget* w, const QIconSet & iconset, const QString& label )
{
    KitaTabBase * tb = new KitaTabBase( label );  /* KitaTabBase is deleted automatically. */
    tb->setIconSet( iconset );
    static_cast< KitaTabBarBase* >( tabBar() ) ->adjustTabWidth( tb );
    QTabWidget::addTab( w, tb );
}


/* public */ /* virtual */
void KitaTabWidgetBase::insertTab( QWidget* w, const QString& label, int index )
{
    KitaTabBase * tb = new KitaTabBase( label );  /* KitaTabBase is deleted automatically. */
    static_cast< KitaTabBarBase* >( tabBar() ) ->adjustTabWidth( tb );
    QTabWidget::insertTab( w, tb, index );
}


/* public */ /* virtual */
void KitaTabWidgetBase::removePage( QWidget* w )
{
    QTabWidget::removePage( w );
    static_cast< KitaTabBarBase* >( tabBar() ) ->adjustTabWidth( NULL );
}


/* public */ /* virtual */
void KitaTabWidgetBase::setTabLabel( QWidget * w, const QString & label )
{
    int idx = indexOf( w );
    KitaTabBase* ktb = static_cast < KitaTabBase* >( tabBar() ->tabAt( idx ) );

    if ( ktb->getFullText() != label ) {
        ktb->setFullText( label );
        QTabWidget::setTabLabel( w, label );
        static_cast< KitaTabBarBase* >( tabBar() ) ->adjustTabWidth( NULL );
    }
}


/* public slot  */
void KitaTabWidgetBase::slotCurrentChanged( QWidget * w )
{
    if ( m_manager == NULL ) return ;
    if ( w == NULL ) return ;
    w->setActiveWindow();
    w->setFocus();

    KParts::Part* part = findPartFromWidget( w );
    if ( part ) {
        m_manager->setActivePart( part );
    }
}


/* show url with part. */ /* public slot */
void KitaTabWidgetBase::slotShowPart( const KURL& url, const QString& libName, const QString& mimetype )
{
    if ( m_manager == NULL ) return ;

    KLibFactory *factory = KLibLoader::self() ->factory( libName );
    if ( !factory ) {
        QMessageBox::critical( parentWidget(), i18n( " Load Error" ),
                               QString( i18n( "can't load %1." ) ).arg( libName ) );
        return ;
    }

    if ( factory->inherits( "KParts::Factory" ) ) {
        KParts::Part * part
        = static_cast< KParts::Factory* >( factory ) ->createPart( this );
        m_manager->addPart( part );
        addTab( part->widget(), url.url() );
        showPage( part->widget() );
        setTabToolTip( currentPage(), url.url() );

        KParts::BrowserExtension *ext = KParts::BrowserExtension::childObject( part );
        if ( ext ) {
            KParts::URLArgs arg( false, 0, 0, mimetype );
            ext->setURLArgs( arg );
        }

        static_cast<KParts::ReadOnlyPart*>( part ) ->openURL( url );
    }
}


/* close num-th tab       */
/* see also customEvent   */ /* public slot */
void KitaTabWidgetBase::slotCloseTab( int num )
{
    CloseTabEvent * e = new CloseTabEvent( num );
    QApplication::postEvent( this, e );  // Qt will delete it when done
}


/* Calling deleteWidget in the child part will be the
   cause of crash.  So you need to call deleteWidget
   via custom event.                                   */ /* protected */ /* virtual */
void KitaTabWidgetBase::customEvent( QCustomEvent * e )
{
    if ( e->type() == EVENT_CloseTab ) {
        deleteWidget ( page( static_cast< CloseTabEvent* >( e ) ->getIndex() ) );
    }
}


/* protected */ /* virtual */
void KitaTabWidgetBase::deleteWidget( QWidget* w )
{
    if ( w == NULL ) return ;

    removePage( w );
    KParts::Part* part = findPartFromWidget( w );
    if ( part ) m_manager->removePart( part );
    delete w;
}


/* protected */
KParts::Part* KitaTabWidgetBase::findPartFromWidget( QWidget* w )
{
    if ( w == NULL ) return NULL;
    if ( m_manager == NULL ) return NULL;
    if ( m_manager->parts() ->isEmpty() ) return NULL;

    KParts::Part *part;
    QPtrListIterator<KParts::Part> it( *( m_manager->parts() ) );
    while ( ( part = ( *it ) ) != NULL ) {
        if ( part->widget() == w ) return part;
        ++it;
    }

    return NULL;
}


/* private */
void KitaTabWidgetBase::connectSignals()
{
    /* connect signals */
    Kita::SignalCollection * signalCollection = Kita::SignalCollection::getInstance();

    connect( this, SIGNAL( switchToBoard() ),
             signalCollection, SIGNAL( switchToBoard() ) );

    connect( this, SIGNAL( switchToSubject() ),
             signalCollection, SIGNAL( switchToSubject() ) );

    connect( this, SIGNAL( switchToThread() ),
             signalCollection, SIGNAL( switchToThread() ) );

    connect( this, SIGNAL( switchToKitanavi() ),
             signalCollection, SIGNAL( switchToKitanavi() ) );

    connect( this, SIGNAL( switchToImgview() ),
             signalCollection, SIGNAL( switchToImgview() ) );

    connect( this, SIGNAL( switchToWritedock() ),
             signalCollection, SIGNAL( switchToWritedock() ) );

    connect( this, SIGNAL( currentChanged ( QWidget * ) ),
             SLOT( slotCurrentChanged ( QWidget * ) ) );

    connect( this, SIGNAL( setMainStatusbar( const QString& ) ),
             signalCollection, SIGNAL( setMainStatusbar ( const QString& ) ) );

    connect( this, SIGNAL( setMainURLLine( const KURL& ) ),
             signalCollection, SIGNAL( setMainURLLine( const KURL& ) ) );

    connect( this, SIGNAL( setMainCaption( const QString& ) ),
             signalCollection, SIGNAL( setMainCaption( const QString& ) ) );

    connect( this, SIGNAL( openURLRequestExt(
                               const KURL&, const KParts::URLArgs&, QString, int, int,
                               const KURL&, const KURL&, const QString&, const QString& ) ),
             signalCollection, SIGNAL( openURLRequestExt(
                                           const KURL& , const KParts::URLArgs&, QString, int, int,
                                           const KURL&, const KURL&, const QString&, const QString& ) ) );
}


/*------------------------------------*/
/* common tab actions */


/* private */
void KitaTabWidgetBase::setupActions()
{
    actionCollection() ->setWidget( this );


    QString str = i18n( "Configure S&hortcuts..." ) + "(" + QString( name() ) + ")";
    new KAction( str,
                 0,
                 this,
                 SLOT( slotConfigureKeys() ),
                 actionCollection(),
                 "tab_configkeys" );

    new KAction( i18n( "Activate Next Tab" ),
#if KDE_IS_VERSION( 3, 2, 0 )
                 KStdAccel::tabNext(),
#else
                 KShortcut( Qt::CTRL + Qt::Key_Period ),
#endif
                 this,
                 SLOT( slotNextTab() ),
                 actionCollection(),
                 "tab_nexttab" );

    new KAction( i18n( "Activate Previous Tab" ),
#if KDE_IS_VERSION( 3, 2, 0 )
                 KStdAccel::tabPrev(),
#else
                 KShortcut( Qt::CTRL + Qt::Key_Comma ),
#endif
                 this,
                 SLOT( slotPrevTab() ),
                 actionCollection(),
                 "tab_prevtab" );

    new KAction( i18n( "Close this tab" ),
                 KStdAccel::close(),
                 this,
                 SLOT( slotCloseCurrentTab() ),
                 actionCollection(),
                 "tab_closetab" );

    new KAction( i18n( "Close Other Tabs" ),
                 0,
                 this,
                 SLOT( slotCloseOtherTab() ),
                 actionCollection(),
                 "tab_closeothertab" );

    new KAction( i18n( "Close right tabs" ),
                 0,
                 this,
                 SLOT( slotCloseRightTab() ),
                 actionCollection(),
                 "tab_closerighttab" );

    new KAction( i18n( "Close left tabs" ),
                 0,
                 this,
                 SLOT( slotCloseLeftTab() ),
                 actionCollection(),
                 "tab_closelefttab" );

    new KAction( i18n( "Close all tabs" ),
                 0,
                 this,
                 SLOT( slotCloseAllTab() ),
                 actionCollection(),
                 "tab_closealltab" );

    new KAction( i18n( "Switch to Board" ),
                 Key_1,
                 this,
                 SIGNAL( switchToBoard() ),
                 actionCollection(),
                 "tab_toboard" );

    new KAction( i18n( "Switch to Subject" ),
                 Key_2,
                 this,
                 SIGNAL( switchToSubject() ),
                 actionCollection(),
                 "tab_tosubject" );

    new KAction( i18n( "Switch to Thread" ),
                 Key_3,
                 this,
                 SIGNAL( switchToThread() ),
                 actionCollection(),
                 "tab_tothread" );

    new KAction( i18n( "Switch to KitaNavi" ),
                 Key_4,
                 this,
                 SIGNAL( switchToKitanavi() ),
                 actionCollection(),
                 "tab_tokitanavi" );

    new KAction( i18n( "Switch to Imgviewer" ),
                 Key_5,
                 this,
                 SIGNAL( switchToImgview() ),
                 actionCollection(),
                 "tab_toimgview" );

    new KAction( i18n( "Switch to writedock" ),
                 Key_6,
                 this,
                 SIGNAL( switchToWritedock() ),
                 actionCollection(),
                 "tab_towritedock" );
}



/* public slot */
void KitaTabWidgetBase::slotConfigureKeys()
{
    QString str = "Tab Actions (" + QString( name() ) + ")";
    KKeyDialog dlg( TRUE, this );
    dlg.insert( actionCollection(), str );
    dlg.configure();
}


/* public slot */
void KitaTabWidgetBase::slotPrevTab()
{
    int max = count();
    int curpage = currentPageIndex();
    if ( max <= 1 ) return ;

    if ( curpage == 0 ) setCurrentPage( max - 1 );
    else setCurrentPage( curpage - 1 );
}

/* public slot */
void KitaTabWidgetBase::slotNextTab()
{
    int max = count();
    int curpage = currentPageIndex();
    if ( max <= 1 ) return ;

    if ( curpage == max - 1 ) setCurrentPage( 0 );
    else setCurrentPage( curpage + 1 );
}


/* see also customEvent */ /* public slot */
void KitaTabWidgetBase::slotCloseCurrentTab()
{
    slotCloseTab( currentPageIndex() );
}


/* see also customEvent */ /* public slot */
void KitaTabWidgetBase::slotCloseOtherTab( int idx )
{
    int max = count();
    if ( max == 0 ) return ;
    if ( idx == -1 ) idx = currentPageIndex();

    int i = 0;

    while ( i < max && i != idx ) {
        slotCloseTab( 0 );
        i++;
    }

    i++;
    while ( i < max ) {
        slotCloseTab( 1 );
        i++;
    }
}


/* see also customEvent */ /* public slot */
void KitaTabWidgetBase::slotCloseRightTab( int idx )
{
    int max = count();
    if ( max == 0 ) return ;
    if ( idx == -1 ) idx = currentPageIndex();

    int i, i2;

    i = i2 = idx + 1;
    while ( i < max ) {
        slotCloseTab( i2 );
        i++;
    }
}


/* see also customEvent */ /* public slot */
void KitaTabWidgetBase::slotCloseLeftTab( int idx )
{
    int max = count();
    if ( max == 0 ) return ;
    if ( idx == -1 ) idx = currentPageIndex();

    int i = 0;

    while ( i < max && i != idx ) {
        slotCloseTab( 0 );
        i++;
    }
}


/* see also customEvent */ /* public slot */
void KitaTabWidgetBase::slotCloseAllTab()
{
    int max = count();
    if ( max == 0 ) return ;

    int i = 0;

    while ( i < max ) {
        slotCloseTab( 0 );
        i++;
    }
}





/*--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*/




KitaTabBarBase::KitaTabBarBase( QWidget* parent, const char* name ) : QTabBar( parent, name ) {}

KitaTabBarBase::~KitaTabBarBase() {}


/* protected */ /* virtual */
void KitaTabBarBase::mousePressEvent( QMouseEvent *e )
{
    if ( e->button() == RightButton ) {
        QTab * tab = selectTab( e->pos() );
        if ( tab != NULL ) {
            int idx = indexOf( tab->identifier() );
            showPopupMenu( idx, mapToGlobal( e->pos() ) );
            return ;
        }
    }
    QTabBar::mousePressEvent( e );
}



/* private */ /* virtual */
void KitaTabBarBase::showPopupMenu( int idx, QPoint global )
{
    enum{
        MENU_CLOSE,
        MENU_CLOSEOTHER,
        MENU_CLOSELEFT,
        MENU_CLOSERIGHT
    };

    KitaTabWidgetBase* tabwidget = static_cast<KitaTabWidgetBase*>( parentWidget() );
    KActionCollection * collection = tabwidget->actionCollection();

    KPopupMenu* popup = new KPopupMenu( this );
    popup->clear();

    popup->insertItem( i18n( "Close this tab" ) , MENU_CLOSE );
    collection->action( "tab_prevtab" ) ->plug( popup );
    collection->action( "tab_nexttab" ) ->plug( popup );
    popup->insertSeparator();

    popup->insertItem( i18n( "Close Other Tabs" ) , MENU_CLOSEOTHER );
    popup->insertItem( i18n( "Close right tabs" ) , MENU_CLOSERIGHT );
    popup->insertItem( i18n( "Close left tabs" ) , MENU_CLOSELEFT );
    collection->action( "tab_closealltab" ) ->plug( popup );

    popup->insertSeparator();
    collection->action( "tab_configkeys" )->plug( popup );       
    
    int ret = popup->exec( global );
    delete popup;

    switch ( ret ) {
    case MENU_CLOSE: tabwidget->slotCloseTab( idx ); break;
    case MENU_CLOSEOTHER: tabwidget->slotCloseOtherTab( idx ); break;
    case MENU_CLOSERIGHT: tabwidget->slotCloseRightTab( idx ); break;
    case MENU_CLOSELEFT: tabwidget->slotCloseLeftTab( idx ); break;
    }

}


/* public */
void KitaTabBarBase::adjustTabWidth( KitaTabBase* newTab )
{
    if ( count() == 0 ) return ;
    if ( tabAt( 0 ) ->text().length() == 0 ) return ; /* maybe this is image tab */

    if ( !shrinkTab( newTab ) ) expandTab();
}


/* private */
bool KitaTabBarBase::shrinkTab( KitaTabBase* newTab )
{
    const int minwidth = 240;
    const int mrg = 32;

    int beforewd = 0;
    int afterwd = 0;
    bool chgWidth = FALSE;
    int tabnum = count();
    if ( newTab ) ++tabnum;
    unsigned int* lng = new unsigned int[ tabnum ];
    int* tabwd = new int[ tabnum ];
    KitaTabBase** ktb = new KitaTabBase * [ tabnum ];
    int parentWidth = QMAX( minwidth, parentWidget() ->width() - mrg );
    QFontMetrics fm( parentWidget() ->font() );
    int tabmrg = tabAt( 0 ) ->rect().width() - fm.width( tabAt( 0 ) ->text() );

    for ( int i = 0; i < count(); ++i ) ktb[ i ] = static_cast< KitaTabBase* >( tabAt( i ) );
    if ( newTab ) ktb[ tabnum - 1 ] = newTab;

    /* get the current width */
    for ( int i = 0; i < tabnum; ++i ) {
        lng[ i ] = ktb[ i ] ->text().length();
        if ( lng[ i ] != ktb[ i ] ->getTextLng() ) lng[ i ] -= 2; /* remove the length of ".." */
        tabwd[ i ] = fm.width( ktb[ i ] ->text() );
        beforewd += ( tabwd[ i ] + tabmrg );
    }
    if ( beforewd <= parentWidth + mrg / 2 ) return FALSE;

    afterwd = beforewd;
    for ( ;; ) {

        int maxwd;
        int idx;

        /* find the tab which has the largest width. */
        maxwd = 0;
        idx = -1;
        for ( int i = 0; i < tabnum; ++i ) {
            if ( lng[ i ] > 2 && tabwd[ i ] > maxwd ) {
                maxwd = tabwd[ i ];
                idx = i;
            }
        }
        if ( idx == -1 ) break;

        chgWidth = TRUE;
        if ( lng[ idx ] == ktb[ idx ] ->getTextLng() ) lng[ idx ] = ktb[ idx ] ->getTextLng() - 3;
        else --lng[ idx ];
        afterwd -= tabwd[ idx ];
        QString text = ktb[ idx ] ->getFullText().left( lng[ idx ] ) + "..";
        tabwd[ idx ] = fm.width( text );
        afterwd += tabwd[ idx ];

        if ( afterwd <= parentWidth ) break;
    }

    /* set new texts */
    if ( beforewd != afterwd && chgWidth ) {

        for ( int i = 0; i < tabnum; ++i ) {

            if ( lng[ i ] != ktb[ i ] ->text().length() ) {

                QString text = ktb[ i ] ->getFullText().left( lng[ i ] ) + "..";
                ktb[ i ] ->setText( text );
            }
        }
    }

    delete lng;
    delete ktb;
    delete tabwd;
    return chgWidth;
}




/* private */
bool KitaTabBarBase::expandTab()
{
    const int minwidth = 240;
    const int mrg = 32;

    int beforewd = 0;
    int afterwd = 0;
    bool chgWidth = FALSE;
    int tabnum = count();
    unsigned int* lng = new unsigned int[ tabnum ];
    int* tabwd = new int[ tabnum ];
    KitaTabBase** ktb = new KitaTabBase * [ tabnum ];
    int parentWidth = QMAX( minwidth, parentWidget() ->width() - mrg );
    QFontMetrics fm( parentWidget() ->font() );
    int tabmrg = tabAt( 0 ) ->rect().width() - fm.width( tabAt( 0 ) ->text() );

    for ( int i = 0; i < count(); ++i ) ktb[ i ] = static_cast< KitaTabBase* >( tabAt( i ) );

    /* get the current width */
    for ( int i = 0; i < tabnum; ++i ) {
        lng[ i ] = ktb[ i ] ->text().length();
        if ( lng[ i ] != ktb[ i ] ->getTextLng() ) lng[ i ] -= 2; /* remove the length of ".." */
        tabwd[ i ] = fm.width( ktb[ i ] ->text() );
        beforewd += ( tabwd[ i ] + tabmrg );
    }
    if ( parentWidth - mrg / 2 <= beforewd ) return FALSE;

    afterwd = beforewd;
    for ( ;; ) {

        int minwd;
        int idx;

        /* find the tab which has the smallest width. */
        minwd = parentWidth;
        idx = -1;
        for ( int i = 0; i < tabnum; ++i ) {
            if ( lng[ i ] != ktb[ i ] ->getTextLng() && tabwd[ i ] < minwd ) {
                minwd = tabwd[ i ];
                idx = i;
            }
        }
        if ( idx == -1 ) break;

        chgWidth = TRUE;
        if ( lng[ idx ] >= ktb[ idx ] ->getTextLng() - 3 ) lng[ idx ] = ktb[ idx ] ->getTextLng();
        else ++lng[ idx ];
        afterwd -= tabwd[ idx ];
        QString text = ktb[ idx ] ->getFullText().left( lng[ idx ] );
        if ( lng[ idx ] != ktb[ idx ] ->getTextLng() ) text += "..";
        tabwd[ idx ] = fm.width( text );
        afterwd += tabwd[ idx ];

        if ( afterwd >= parentWidth ) break;
    }

    /* set new texts */
    if ( beforewd != afterwd && chgWidth ) {

        for ( int i = 0; i < tabnum; ++i ) {

            if ( lng[ i ] != ktb[ i ] ->text().length() ) {

                QString fullText = ktb[ i ] ->getFullText();
                if ( lng[ i ] < ktb[ i ] ->getTextLng() ) ktb[ i ] ->setText( fullText.left( lng[ i ] ) + ".." );
                else ktb[ i ] ->setText( fullText );
            }
        }
    }

    delete lng;
    delete ktb;
    delete tabwd;
    return chgWidth;
}


/* private */
int KitaTabBarBase::getWidthOfTabs()
{
    int wd = 0;
    for ( int i = 0; i < count(); ++i ) {
        QTab* tb = tabAt( i );
        wd += tb->rect().width();
    }

    return wd;
}



/*--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*/


KitaTabBase::KitaTabBase( const QString & text ) : QTab( text )
{
    setFullText( text );
}

KitaTabBase::~KitaTabBase() {}


/* public */
QString KitaTabBase::getFullText() const
{
    return m_fullText;
}


/* public */
const unsigned int KitaTabBase::getTextLng() const
{
    return m_textlng;
}


/* public */
void KitaTabBase::setFullText( const QString& text )
{
    m_fullText = text;
    m_textlng = text.length();
}



/*--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*/



KitaDockWidgetBase::KitaDockWidgetBase( KDockManager* dockManager,
                                        const char* name,
                                        const QPixmap &pixmap,
                                        QWidget* parent,
                                        const QString& strCaption,
                                        const QString& strTabPageLabel,
                                        WFlags f )
        : KDockWidget( dockManager, name, pixmap, parent, strCaption, strTabPageLabel, f )
{
    setDockSite( KDockWidget::DockNone );
    m_docked = TRUE;
    m_parentDock = static_cast< KParts::DockMainWindow* >( parent );
    m_closeButtonClicked = FALSE;

    /* connect signals */
    connect( this, SIGNAL( headerCloseButtonClicked() ), SLOT( slotHeaderCloseButtonClicked() ) );
    connect( this, SIGNAL( iMBeingClosed() ), SLOT( slotSaveDocStatus() ) );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();

    /* emit when this widget is deactivated */
    connect( this, SIGNAL( windowDeactivated() ),
             signalCollection, SIGNAL( windowDeactivated() ) );

    /* If this widget is active and receives
       signal isKitaActive, then emit signal kitaIsActive. */
    /* see also KitaHTMLPart::slotOnURL                    */
    connect( signalCollection, SIGNAL( isKitaActive() ),
             this, SLOT( slotIsKitaActive() ) );

    connect( this, SIGNAL( kitaIsActive() ),
             signalCollection, SIGNAL( kitaIsActive() ) );
}


KitaDockWidgetBase::~KitaDockWidgetBase() {}


/* public slot */
void KitaDockWidgetBase::slotShowPart( const KURL& url, const QString& libName, const QString& mimetype )
{
    slotShowDock( TRUE, TRUE );

    static_cast<KitaTabWidgetBase*> ( getWidget() ) ->slotShowPart( url, libName, mimetype );
}


/* see also KitaMainWindow::KitaMainWindow() */ /* public */
void KitaDockWidgetBase::loadSession()
{
    QString cfgPath = locateLocal( "appdata", "session.conf" );
    KConfig cfg( cfgPath );
    m_tabbed = cfg.readBoolEntry( name() + QString( "_Tabbed" ), FALSE );
    m_docked = cfg.readBoolEntry( name() + QString( "_Docked" ), FALSE );
}


/* see also KitaMainWindow::~KitaMainWindow() */ /* public */
void KitaDockWidgetBase::saveSession()
{
    /* copied from slotSaveDocStatus */
    if ( isVisible() ) {
        m_docked = FALSE;
        m_tabbed = FALSE;
        if ( parent() ) {
            m_docked = TRUE;
            if ( parentDockTabGroup() ) m_tabbed = TRUE;
        }
    }

    QString cfgPath = locateLocal( "appdata", "session.conf" );
    KConfig cfg( cfgPath );
    cfg.writeEntry( name() + QString( "_Tabbed" ), m_tabbed );
    cfg.writeEntry( name() + QString( "_Docked" ), m_docked );
}


/* see also KitaMainWindow::resetWindows() */ /* public */
void KitaDockWidgetBase::setSession( bool docked, bool tabbed )
{
    m_docked = docked;
    m_tabbed = tabbed;
}


/* save the current dock status.                      */
/* This slot is called before this widget is closed.  */
/* See also slotHeaderCloseButtonClicked(),           */
/* slotHideDock(), and closeEvent().                  */ /* private slot */
void KitaDockWidgetBase::slotSaveDocStatus()
{
    if ( isVisible() ) {
        m_docked = FALSE;
        m_tabbed = FALSE;
        /* This widget is docked.*/
        if ( parent() ) {
            m_docked = TRUE;
            if ( parentDockTabGroup() ) m_tabbed = TRUE; /* This widget is in the tab group.*/
        }

        emit checkToggleAction( FALSE ); /* to KitaMainWindow. see also KitaMainWindow::setupView() */
    }
}

/* This slot is called when user clicks the close button.      */
/* m_closeButtonClicked is set to prevent from re-openning     */
/* the dock. To reset m_closeButtonClicked, call showDock      */
/* with force = TRUE.   See also slotShowDock() and showDock() */ /* private slot */
void KitaDockWidgetBase::slotHeaderCloseButtonClicked()
{
    m_closeButtonClicked = TRUE;
    slotSaveDocStatus();
}


/* show this dock widget later.                   */
/* Usually, call this slot instead of showDock(). */
/* See also showDock() and customEvent().         */ /* public slot */
void KitaDockWidgetBase::slotShowDock( bool activate, bool force )
{
    ShowDockEvent * e = new ShowDockEvent( activate, force );
    QApplication::postEvent( this, e );  // Qt will delete it when done
}


/* show this dock widget immediately.          */
/* If activate is TRUE, activate this dock.    */
/* If force is TRUE, show this anyway.         */
/* If force is FALSE and tab has no children,  */
/* this dock is not shown.                     */ /* public */
void KitaDockWidgetBase::showDock( bool activate, bool force )
{
    if ( force ) m_closeButtonClicked = FALSE; /* see also slotHeaderCloseButtonClicked() */
    if ( m_closeButtonClicked ) return ;

    QWidget* wd = getWidget();
    KitaTabWidgetBase* w = NULL;
    if ( wd && wd->inherits( "KitaTabWidgetBase" ) ) w = static_cast<KitaTabWidgetBase*> ( wd );
    if ( !force && w && w->count() == 0 ) return ;

    if ( !isVisible() ) {

        if ( m_docked ) { /* This dock was docked to the other dock. */

#if KDE_IS_VERSION( 3, 2, 0 )

            /* Sometimes the dock widget loses the former brother DockWidget.
               because the brogher widget had transformed. So, find and set new brogher widget here. */

            /* find new brother dock widget */
            KDockWidget * newBrsDock = dockManager() ->getDockWidgetFromName( "Thread" );

            if ( newBrsDock ) {

                if ( !m_tabbed ) { /* This dock was not tabbed */

                    /* Is thread dock on the tab? */
                    QWidget * tmpwidget = newBrsDock->parentDockTabGroup();
                    if ( tmpwidget ) {
                        tmpwidget = tmpwidget->parentWidget();
                        if ( tmpwidget->inherits( "KDockWidget" ) )
                            newBrsDock = static_cast< KDockWidget* >( tmpwidget );
                    }
                }

                setFormerBrotherDockWidget( newBrsDock );
            }
#endif
            makeDockVisible();
        } else show();
    }

    if ( !activate ) return ;

    if ( isMinimized() ) showNormal();
    topLevelWidget() ->raise();
    raise();
    setActiveWindow();
    emit checkToggleAction( TRUE ); /* to KitaMainWindow */

    /* activate child Part */
    if ( w ) w->slotCurrentChanged( w->currentPage() );
    else if ( wd ) {
        wd->setActiveWindow();
        wd->setFocus();
    }
}


/* public slot */
void KitaDockWidgetBase::slotHideDock()
{
    if ( isVisible() ) {
        slotSaveDocStatus();
        m_parentDock->makeDockInvisible( this );
    }
}


/* public slot */
void KitaDockWidgetBase::slotToggleShowHide()
{
    if ( !isVisible() ) slotShowDock( TRUE, TRUE );
    else slotHideDock();
}


/* when this widget is deactivated, emit signal
   to popup vie SignalCollection.               */  /* protected */
void KitaDockWidgetBase::windowActivationChange ( bool )
{
    if ( !isActiveWindow() ) emit windowDeactivated();
}


/* protected */
void KitaDockWidgetBase::closeEvent( QCloseEvent* e )
{
    slotSaveDocStatus();
    KDockWidget::closeEvent( e );
}


/* protected */ /* virtual */
void KitaDockWidgetBase::customEvent( QCustomEvent * e )
{
    if ( e->type() == EVENT_ShowDock ) {
        ShowDockEvent * sde = static_cast< ShowDockEvent* >( e );
        showDock( sde->getActivate(), sde->getForce() );
    }
}


/* see also KitaHTMLPart::slotOnURL */  /* private slot */
void KitaDockWidgetBase::slotIsKitaActive()
{
    if ( isActiveWindow() ) emit kitaIsActive();
}

