/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <kio/netaccess.h>
#include <klistview.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kconfig.h>
#include <kdeversion.h>

#include <qfile.h>
#include <qtextcodec.h>
#include <qstringlist.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <qheader.h>
#include <qclipboard.h>
#include <qmessagebox.h>

#include "kitaboardview.h"
#include "kita.h"
#include "listviewitem.h"
#include "libkita/qcp932codec.h"
#include "libkita/favoriteboards.h"
#include "libkita/kitaconfig.h"
#include "libkita/signalcollection.h"
#include "libkita/boardmanager.h"


namespace Kita
{
    struct Category
    {
        QString category_name;
        QStringList boardNameList;
        QStringList boardURLList;
    };
}


/*--------------------------------------*/

QCp932Codec* KitaBoardView::m_cp932Codec = NULL;

KitaBoardView::KitaBoardView( QWidget *parent, const char *name )
        : KitaBoardViewBase( parent, name )
        , m_favorites( 0 )
{
    m_boardList->setSorting( -1 );
    m_boardList->addColumn( i18n( "board name" ) );
    m_boardList->header() ->setClickEnabled( FALSE );

    /* default colors */
    QColorGroup colors = m_boardList->viewport()->colorGroup();
    m_textColor = colors.text();
    m_baseColor = colors.base();
    m_backColor = m_boardList->viewport()->paletteBackgroundColor();
    m_altColor = m_boardList->alternateBackground();

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( m_boardList, SIGNAL( mouseButtonClicked( int, QListViewItem*, const QPoint&, int ) ),
             SLOT( slotMouseButtonClicked( int, QListViewItem* ) ) );
    connect( m_boardList, SIGNAL( returnPressed( QListViewItem* ) ), SLOT( loadBoard( QListViewItem* ) ) );
    connect( m_boardList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( Kita::FavoriteBoards::getInstance(), SIGNAL( changed() ), SLOT( refreshFavoriteBoards() ) );
    connect( this, SIGNAL( openURLRequestExt(
                               const KURL&, const KParts::URLArgs&, QString, int, int,
                               const KURL&, const KURL&, const QString&, const QString& ) ),
             signalCollection, SIGNAL( openURLRequestExt(
                                           const KURL& , const KParts::URLArgs&, QString, int, int,
                                           const KURL&, const KURL&, const QString&, const QString& ) ) );
}

KitaBoardView::~KitaBoardView()
{
    saveOpened();
}

QString KitaBoardView::getCategory( const QString& line ) const
{
    QRegExp regexp( "<BR><BR><B>(.*)</B><BR>", false );
    if ( regexp.search( line ) != -1 ) {
        return regexp.cap( 1 );
    } else {
        return QString::null;
    }
}

QString KitaBoardView::getBoardURL( const QString& line ) const
{
    QRegExp regexp( "<A HREF=([^ ]*).*>(.*)</A>", false );
    if ( regexp.search( line ) != -1 ) {
        QString board_url = regexp.cap( 1 );
        QString board_title = regexp.cap( 2 );
        return board_url;
    } else {
        return QString::null;
    }
}

bool KitaBoardView::isBoardURL( const QString& url ) const
{
    QRegExp url_2ch( "http://.*\\.2ch\\.net/.*" );
    QRegExp url_bbspink( "http://.*\\.bbspink\\.com/.*" );
    QRegExp url_www_2ch( "http://www\\.2ch\\.net/.*" );
    QRegExp url_machibbs( "http://.*\\.machi\\.to/.*" );

    if ( url.isEmpty() ) return false;

    if ( url_2ch.search( url ) == -1 && url_bbspink.search( url ) == -1
            && url_machibbs.search( url ) == -1 ) return false;
    if ( url_www_2ch.search( url ) != -1 ) return false;

    return true;
}

bool KitaBoardView::downloadBoardList()
{
    QString tmpFile;
    QString url = KitaConfig::boardListURL();
    if ( !
#if KDE_IS_VERSION( 3, 2, 0 )
	 KIO::NetAccess::download( url, tmpFile, NULL )
#else
	 KIO::NetAccess::download( url, tmpFile )
#endif	 
	) {
        return FALSE;
    }

    QFile file( tmpFile );
    if ( ! file.open( IO_ReadOnly ) ) {
        return FALSE;
    }

    QTextStream stream( &file );
    if ( !m_cp932Codec ) m_cp932Codec = new QCp932Codec();
    stream.setCodec( m_cp932Codec );
    QString html = stream.read();

    // parse
    QStringList list;
    QValueList<Kita::Category> categoryList = getCategoryList( html );
    QValueList<Kita::Category>::iterator it;

    /* Are there new boards or moved boards ? */
    QString newBoards = QString::null;    
    QString oldBoards = QString::null;
    for ( it = categoryList.begin(); it != categoryList.end(); ++it ) {
        Kita::Category category = ( *it );
        QValueList<QString> board_url_list = category.boardURLList;
        if ( board_url_list.isEmpty() ) continue;

        int count = 0;
        for ( QValueList<QString>::iterator it2 = board_url_list.begin();
	      it2 != board_url_list.end(); ++it2 ) {
	    
            QString boardURL = *it2;
            QString boardName = category.boardNameList[ count ];
            QString oldURL;
            int ret = Kita::BoardManager::enrollBoard( boardURL, boardName, oldURL, Kita::Board_Unknown,
						       TRUE /* test only */
		);
            if( ret == Kita::Board_enrollNew ) newBoards += boardName + "  ( " + category.category_name + " )\n";
            if( ret == Kita::Board_enrollMoved ) oldBoards += boardName + "  ( " + category.category_name + " )\n";
            count++;
        }
    }

    const int maxNewBoard = 16;

    /* show new board names */
    if( newBoards != QString::null && newBoards.contains( "\n" ) < maxNewBoard ){

	QString msg = i18n( "New boards:\n\n" ) + newBoards;
	QMessageBox mb( "Kita", msg,  QMessageBox::Information,
			QMessageBox::Ok,
			QMessageBox::No | QMessageBox::Default | QMessageBox::Escape,
			QMessageBox::NoButton, this );
	mb.setButtonText( QMessageBox::No, i18n( "Copy" ) );

	while( mb.exec() == QMessageBox::No ){
	    QApplication::clipboard()->setText( msg , QClipboard::Clipboard );
	    QApplication::clipboard()->setText( msg , QClipboard::Selection );
	}
    }

    /* show moved board names */    
    if( oldBoards != QString::null ){

	QString msg = i18n( "These boards were moved:\n\n" )
	    + oldBoards
	    + i18n( "\nPlease create the backup of those caches.\n" );

	QMessageBox mb( "kita", msg, QMessageBox::Information,
			QMessageBox::Ok,
			QMessageBox::No | QMessageBox::Default | QMessageBox::Escape,
			QMessageBox::NoButton, this );
	mb.setButtonText( QMessageBox::No, i18n( "Copy" ) );
	
	while( mb.exec() == QMessageBox::No ){
	    QApplication::clipboard()->setText( msg , QClipboard::Clipboard );
	    QApplication::clipboard()->setText( msg , QClipboard::Selection );
	}
    }

    if( ( newBoards != QString::null && newBoards.contains( "\n" ) < maxNewBoard )
	|| oldBoards != QString::null ){
	
	if ( QMessageBox::information( this, i18n( "kita" ),
				       i18n( "Do you really want to update board list?" ),
				       QMessageBox::Yes,
				       QMessageBox::No | QMessageBox::Default ) 
	     == QMessageBox::No ) return FALSE; 
    }
    else if( newBoards == QString::null && oldBoards == QString::null ){
	
	QMessageBox::information( this, "Kita", i18n( "no new boards" ) );
	return FALSE;
    }

    /*------------------------------------------------*/
    
    /* save config */
    QString configPath = locateLocal( "appdata", "board_list" );
    KConfig config( configPath );
    for ( it = categoryList.begin(); it != categoryList.end(); ++it ) {
        Kita::Category category = ( *it );
        QValueList<QString> board_url_list = category.boardURLList;

        if ( board_url_list.isEmpty() ) {
            continue;
        }

        list << category.category_name;
        config.setGroup( category.category_name );

        QValueList<QString>::iterator it2;
        int count = 0;
        for ( it2 = board_url_list.begin(); it2 != board_url_list.end(); ++it2 ) {
            QString key = QString( "item%1").arg( count );
            QString boardURL = *it2;
            QString boardName = category.boardNameList[ count ];
            QStringList tmpList;
            tmpList << boardURL;
            tmpList << boardName;
            config.writeEntry( key, tmpList );
            count++;
        }
    }
    config.setGroup( QString::null );
    config.writeEntry( "Categories", list );

    return TRUE;
}

void KitaBoardView::updateBoardList()
{
    if( downloadBoardList() ) showBoardList();
}

void KitaBoardView::showBoardList()
{
    /* color setting */
/*
    m_textColor = QColor( "white" );
    m_baseColor = QColor( "red" );
    m_backColor = QColor( "yellow" );
    m_altColor = QColor( "blue" );    
*/
    /*-------------------------------------------------*/

    /* set background color */
    m_boardList->viewport()->setPaletteBackgroundColor( m_backColor );
    m_boardList->setAlternateBackground( m_altColor );
    
    // clear list
    m_boardList->clear();
    m_favorites = NULL;

    QString configPath = locateLocal( "appdata", "board_list" );
    KConfig config( configPath );
    QStringList category_list = config.readListEntry( "Categories" );

    Kita::ListViewItem* previous = 0;
    QStringList::iterator it;
    for ( it = category_list.begin(); it != category_list.end(); ++it ) {
        QString category = ( *it );

        config.setGroup( category );
        Kita::ListViewItem* categoryItem = new Kita::ListViewItem( m_boardList, previous, category );
        categoryItem->setColor( m_textColor, m_baseColor );
        Kita::ListViewItem* previousBoard = 0;

        for ( int count = 0; ; count++ ) {
            QString key = QString( "item%1" ).arg( count );
            QStringList value = config.readListEntry( key );
            if ( value.count() != 2 ) {
                break;
            }
            QString boardURL = value[0];
            QString boardName = value[1];
            QString oldURL;
            int ret = Kita::BoardManager::enrollBoard( boardURL, boardName, oldURL );
            if( ret == Kita::Board_enrollMoved ){ /* board is moved. */
		
               if( !Kita::BoardManager::moveBoard( oldURL, boardURL ) ) break;
               QString msg = QString( i18n( "%1 is moved.\n\n" ) ).arg( boardName );
               msg += oldURL +"\n->\n" + boardURL;
               QMessageBox::warning( this, "Kita", msg );
            }
            Kita::BoardManager::loadBBSHistory( boardURL );
            previousBoard = new Kita::ListViewItem( categoryItem, previousBoard, boardName, boardURL );
            previousBoard->setColor( m_textColor, m_baseColor );
        }
        previous = categoryItem;
    }

    loadExtBoard();
    refreshFavoriteBoards();
}


/* private */
void KitaBoardView::loadExtBoard()
{
    QString configPath = locateLocal( "appdata", "extbrd.conf" );
    QFile file( configPath );
    if ( file.open( IO_ReadOnly ) ) {

        QTextStream stream( &file );
        stream.setEncoding( QTextStream::UnicodeUTF8 );

        QStringList list;
        QString str;

        Kita::ListViewItem* categoryItem = new Kita::ListViewItem( m_boardList, "extboard" );
        categoryItem->setColor( m_textColor, m_baseColor );

        while ( ( str = stream.readLine() ) != QString::null ) {
            if ( ! str.isEmpty() ) {
                QStringList list = QStringList::split( "<>", str );

                if ( list.size() == 2 || list.size() == 3 ) {
                    QString board_title = list[ 0 ];
                    QString board_url = list[ 1 ];

                    Kita::ListViewItem* tmpitem = new Kita::ListViewItem( categoryItem, board_title, board_url );
		        tmpitem->setColor( m_textColor, m_baseColor );
                    QString oldURL;
                    int type = Kita::Board_Unknown;
                    if( list.size() == 3 ) type = list[ 2 ].toInt();
                    Kita::BoardManager::enrollBoard( board_url, board_title, oldURL, type );	    
                }
            }
        }

        file.close();
    }
}


QValueList<Kita::Category> KitaBoardView::getCategoryList( const QString& html ) const
{
    QValueList<Kita::Category> result;

    QStringList lines = QStringList::split( "\n", html );
    QStringList::iterator it;

    Kita::Category current_category;
    current_category.category_name = QString::null;
    current_category.boardNameList.clear();
    current_category.boardURLList.clear();   
    for ( it = lines.begin(); it != lines.end(); ++it ) {
        QString category_name = getCategory( *it );
        if ( category_name != QString::null ) {
            if ( current_category.category_name != QString::null ) {
                result.append( current_category );
            }
            current_category.category_name = category_name;
            current_category.boardNameList.clear();
            current_category.boardURLList.clear();   
        } else {
	    QRegExp regexp( "<A HREF=([^ ]*).*>(.*)</A>", false );
	    if ( regexp.search( ( *it ) ) != -1 ) {
		QString url = regexp.cap( 1 );
		QString title = regexp.cap( 2 );
		if( isBoardURL( url ) && current_category.category_name != QString::null ){
		    current_category.boardNameList.append( title );
		    current_category.boardURLList.append( url );
		}
	    }
        }
    }
    return result;
}

void KitaBoardView::refreshFavoriteBoards()
{
    if ( ! m_favorites ) {
        m_favorites = new Kita::ListViewItem( m_boardList, 0, i18n( "Favorites" ) );
    }
    m_favorites->setColor( m_textColor, m_baseColor );	    

    do {
        QListViewItem* child = m_favorites->firstChild();
        delete child;
    } while ( m_favorites->childCount() != 0 );

    QValueList<KURL> boards = Kita::FavoriteBoards::boards();
    QValueList<KURL>::iterator it;

    for ( it = boards.begin(); it != boards.end(); ++it ) {
        QString boardURL = Kita::BoardManager::boardURL( ( *it ) ); /* get up-to-date URL */
        QString boardName = Kita::BoardManager::boardName( boardURL );
        Kita::ListViewItem* tmpitem = new Kita::ListViewItem( m_favorites, 0, boardName, boardURL );
        tmpitem->setColor( m_textColor, m_baseColor );
    }
}

void KitaBoardView::loadBoard( QListViewItem* item )
{
    if ( ! item ) return ;

    QString boardName = item->text( 0 );
    QString boardURL = item->text( 1 );

    if ( boardURL.isEmpty() ) {
        return ;
    }

    if ( KitaConfig::alwaysUseTab() ) {
        emit openURLRequestExt( boardURL, KParts::URLArgs(), "kita_open_2chboard", 1 );
    } else {
        emit openURLRequestExt( boardURL, KParts::URLArgs(), "kita_open_2chboard", 0 );
    }
}

void KitaBoardView::setFont( const QFont& font )
{
    m_boardList->setFont( font );
}

/* public slot :*/ /* virtual */
void KitaBoardView::setFocus()
{
    m_boardList->setFocus();
}


void KitaBoardView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    enum {
        Menu_OpenWithBrowser,
        Menu_OpenWithNewTab,
        Menu_CopyURL,
        Menu_CopyTitleAndURL,
        Menu_AddToFavorites,
        Menu_RemoveFromFavorites
    };

    if ( item == 0 ) {
        return ;
    }

    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), Menu_OpenWithBrowser );
    // FIXME: open with "new tab" is meaningless.
    popup.insertItem( i18n( "Open with New Tab" ), Menu_OpenWithNewTab );
    popup.insertItem( i18n( "Copy URL" ), Menu_CopyURL );
    popup.insertItem( i18n( "Copy title and URL" ), Menu_CopyTitleAndURL );
    if ( item->parent() == m_favorites ) {
        popup.insertItem( i18n( "Remove from Favorites" ), Menu_RemoveFromFavorites );
    } else {
        popup.insertItem( i18n( "Add to Favorites" ), Menu_AddToFavorites );
    }

    QString boardName = item->text( 0 );
    KURL boardURL = item->text( 1 );
    QClipboard* clipboard = QApplication::clipboard();

    switch ( popup.exec( point ) ) {
    case Menu_OpenWithBrowser:
        emit openURLRequestExt( boardURL, KParts::URLArgs(), "text/html" );
        break;
    case Menu_OpenWithNewTab:
        emit openURLRequestExt( boardURL, KParts::URLArgs(), "kita_open_2chboard", 1 );
        break;
    case Menu_CopyURL:
        clipboard->setText( boardURL.prettyURL(), QClipboard::Clipboard );
        clipboard->setText( boardURL.prettyURL(), QClipboard::Selection );
        break;
    case Menu_CopyTitleAndURL:
        clipboard->setText( boardName + "\n" + boardURL.prettyURL(), QClipboard::Clipboard );
        clipboard->setText( boardName + "\n" + boardURL.prettyURL(), QClipboard::Selection );	
        break;
    case Menu_AddToFavorites:
        Kita::FavoriteBoards::append( boardURL );
        break;
    case Menu_RemoveFromFavorites:
        Kita::FavoriteBoards::remove( boardURL );
        break;
    default:
        break;
    }
}

void KitaBoardView::slotMouseButtonClicked( int button, QListViewItem* item )
{
    if ( ! item ) return ;

    QString boardName = item->text( 0 );
    QString boardURL = item->text( 1 );
    if ( boardURL.isEmpty() ) {
        m_boardList->setOpen( item, !item->isOpen() );
        return ;
    }

    emit sigShowSubject();

    switch ( button ) {
    case MidButton:
        emit openURLRequestExt( boardURL, KParts::URLArgs(), "kita_open_2chboard", 1 );
        break;
    case LeftButton:
        if ( KitaConfig::alwaysUseTab() ) {
            emit openURLRequestExt( boardURL, KParts::URLArgs(), "kita_open_2chboard", 1 );
        } else {
            emit openURLRequestExt( boardURL, KParts::URLArgs(), "kita_open_2chboard", 0 );
        }
        break;
    }
}

void KitaBoardView::loadOpened()
{
    QString configPath = locateLocal( "appdata", "board_state.conf" );
    KConfig config( configPath );

    QStringList openedList = config.readListEntry( "Opened" );

    QListViewItem* item;
    for ( item = m_boardList->firstChild(); item; item = item->nextSibling() ) {
        QString categoryName = item->text( 0 );
        if ( openedList.findIndex( categoryName ) != -1 ) {
            item->setOpen( true );
        }
    }
}

void KitaBoardView::saveOpened()
{
    QStringList openedList;
    QListViewItem* item;

    for ( item = m_boardList->firstChild(); item; item = item->nextSibling() ) {
        QString categoryName = item->text( 0 );
        if ( item->isOpen() ) {
            openedList << categoryName;
        }
    }

    QString configPath = locateLocal( "appdata", "board_state.conf" );
    KConfig config( configPath );

    config.writeEntry( "Opened", openedList );
}
