/***************************************************************************
*   Copyright (C) 2003 by Hideki Ikemoto                                  *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "favoritelistview.h"

#include "libkita/favoritethreads.h"
#include "libkita/kita_misc.h"
#include "libkita/signalcollection.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"

#include <qmap.h>
#include <qapplication.h>
#include <qclipboard.h>
#include <qdatetime.h>
#include <qtoolbutton.h>
#include <qlabel.h>

#include <klocale.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kdebug.h>
#include <klistview.h>

FavoriteListView::FavoriteListView( QWidget* parent, const char* name )
        : Kita::ThreadListView( parent, name )
{
    KindLabel->hide();

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( subjectList, SIGNAL( returnPressed( QListViewItem* ) ),
             SLOT( loadThread( QListViewItem* ) ) );
    connect( subjectList, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&, int ) ),
             SLOT( slotContextMenuRequested( QListViewItem*, const QPoint&, int ) ) );
    connect( ReloadButton, SIGNAL( clicked() ),
             SLOT( slotReloadButton() ) );
    connect( signalCollection, SIGNAL( favoritesUpdated() ),
             SLOT( update() ) );

    connect( signalCollection, SIGNAL( updateSubjectTab( const KURL& ) ),
             SLOT( slotUpdateSubject( const KURL& ) ) );
}

FavoriteListView::~FavoriteListView()
{}

void FavoriteListView::update()
{
    // FIXME: KitaSubjectView::loadBoard()˥ԡ
    {
        m_hitList.clear();
        m_nextHitIndex = 0;
        m_prevquery = "";
    }

    const QValueList<QString> threadList = FavoriteThreads::getInstance() -> threadList();

    subjectList->clear();

    QValueList<QString>::const_iterator it;
    for ( it = threadList.begin(); it != threadList.end(); ++it ) {
        QString datURL = ( *it );

        QDateTime since;
        since.setTime_t( Kita::datToSince( datURL ) );

        int viewPos = Kita::DatManager::getViewPos( datURL );
        int resNum = Kita::DatManager::getResNum( datURL );

        KListViewItem* item = new KListViewItem( subjectList );
        item->setText( Col_Board, Kita::BoardManager::boardName( datURL ) );
        item->setText( Col_Subject, Kita::DatManager::threadName( datURL ) );
        item->setText( Col_ReadNum, QString( "%1" ).arg( viewPos, 4 ) );
        if ( resNum > 0 ) {
            item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
            if ( resNum != viewPos ) {
                item->setText( Col_Unread, QString( "%1" ).arg( resNum - viewPos, 4 ) );
            }
        }
        item->setText( Col_Since, since.toString( "yy/MM/dd hh:mm" ) );
        item->setText( Col_DatURL, datURL );
    }
    subjectList->setSorting( Col_Board );
}

void FavoriteListView::loadThread( QListViewItem* item )
{
    if ( ! item ) return ;

    const QValueList<QString> threadList = FavoriteThreads::getInstance() -> threadList();

    QValueList<QString>::const_iterator it;
    for ( it = threadList.begin(); it != threadList.end(); ++it ) {
        QString datURL = ( *it );

        if ( datURL == item->text( Col_DatURL ) ) {
            emit openURLRequestExt( datURL, KParts::URLArgs(), "kita_open_2chthread", 0 );
        }
    }
}

void FavoriteListView::slotUpdateSubject( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    for ( QListViewItem * item = subjectList->firstChild(); item; item = item->nextSibling() ) {
        if ( item->text( Col_DatURL ) == datURL.prettyURL() ) {
            int resNum = Kita::DatManager::getResNum( datURL );
            int viewPos = Kita::DatManager::getViewPos( datURL );
            item->setText( Col_ResNum, QString( "%1" ).arg( resNum, 4 ) );
            item->setText( Col_ReadNum, QString( "%1" ).arg( viewPos, 4 ) );
        }
    }
}

// TODO: KitaSubjectView饹Ʊ᥽åɤȤۤȤƱ
void FavoriteListView::slotContextMenuRequested( QListViewItem* item, const QPoint& point, int )
{
    enum {
        Menu_OpenWithBrowser,
        Menu_OpenWithNewTab,
        Menu_CopyURL,
        Menu_CopyTitleAndURL,
        Menu_RemoveFromFavorites
    };

    if ( item == 0 ) {
        return ;
    }

    KPopupMenu popup( 0 );
    popup.insertItem( i18n( "Open with Web Browser" ), Menu_OpenWithBrowser );
    popup.insertItem( i18n( "Open with new tab" ), Menu_OpenWithNewTab );
    popup.insertItem( i18n( "Copy URL" ), Menu_CopyURL );
    popup.insertItem( i18n( "Copy title and URL" ), Menu_CopyTitleAndURL );
    popup.insertItem( i18n( "Remove from Favorites" ), Menu_RemoveFromFavorites );

    QString datURL = item->text( Col_DatURL );
    QString threadURL = Kita::DatManager::threadURL( datURL );

    QClipboard* clipboard = QApplication::clipboard();
    QString cliptxt;

    switch ( popup.exec( point ) ) {
    case Menu_OpenWithBrowser:
        KRun::runURL( Kita::DatManager::threadURL( datURL ), "text/html" );
        break;
    case Menu_OpenWithNewTab:
        emit openURLRequestExt( datURL, KParts::URLArgs(), "kita_open_2chthread", 1 );
        break;
    case Menu_CopyURL:
        clipboard->setText( threadURL );
        break;
    case Menu_CopyTitleAndURL:
	cliptxt = Kita::DatManager::threadName( datURL ) + "\n" + threadURL;
	clipboard->setText( cliptxt , QClipboard::Clipboard );
	clipboard->setText( cliptxt , QClipboard::Selection );
        break;
    case Menu_RemoveFromFavorites:
        emit bookmarked( datURL, false );
        break;
    default:
        break;
    }
}

void FavoriteListView::slotReloadButton()
{
    QValueList<QString> boardList;

    const QValueList<QString> threadList = FavoriteThreads::getInstance() -> threadList();
    QValueList<QString>::const_iterator it;
    for ( it = threadList.begin(); it != threadList.end(); ++it ) {
        QString datURL = ( *it );
        QString boardURL = Kita::datToBoard( datURL );
        if ( boardList.contains( boardURL ) == 0 ) {
            boardList.append( boardURL );
        }
    }

    QValueList<QString>::const_iterator it2;
    for ( it2 = boardList.begin(); it2 != boardList.end(); ++it2 ) {

	bool online = TRUE;
	QPtrList<Kita::Thread> threadList;
	QPtrList<Kita::Thread> tmpList;
	Kita::BoardManager::getThreadList( ( *it2 ), FALSE, online, threadList, tmpList );
    }
}

