/* $Id: keitai.c,v 1.73 2006/02/12 19:00:25 ichiro Exp $ */
/*
 * Copyright (c) 2004, 2005, 2006
 *	Ichiro FUKUHARA <ichiro@ichiro.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Ichiro FUKUHARA.
 * 4. The name of the company nor the name of the author may be used to
 *    endorse or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ICHIRO FUKUHARA ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL ICHIRO FUKUHARA OR THE VOICES IN HIS HEAD BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <sys/param.h>

extern int debug;

#include "kircd.h"

#ifdef HAVE_ICONV
#include <iconv.h>
#endif

#ifndef _LIBICONV_VERSION
#define _LIBICONV_VERSION 0x0
#endif

int
keitai_start(struct tparam *th)
{
	/* debug */
	DPRINTF(2, ("RECV_DATA:(%s)\r\n", th->l_buff));

	/* skip NOTICE messages */
	if (strstr(th->l_buff, "NOTICE")) {
	}

	/* catch PING messages */
	if (strstr(th->l_buff, "PING"))
		ping_recv(th);

	/* catch NICK name */
	if (strstr(th->l_buff, "NICK"))
		nick_recv(th);

	/* detect JOIN channel */
	if (strstr(th->l_buff, "JOIN"))
		join_recv(th);

	if (strstr(th->l_buff, "PART"))
		part_recv(th);

	/* catch PRIVMSGs */
	if (strstr(th->l_buff, "PRIVMSG"))
		priv_recv(th);

	/* catch QUIT */
	if (strstr(th->l_buff, "QUIT")) {
		DPRINTF(0, ("QUIT:%s\n", th->l_buff));
	}

	/* detect TOPIC */
	if (strstr(th->l_buff, "332") || strstr(th->l_buff, "TOPIC"))
		topic_recv(th);

	/* MODE */
	if (strstr(th->l_buff, "MODE")) {
		DPRINTF(0, ("MODE:%s\n", th->l_buff));
	}

	/* NAMES */
	if (strstr(th->l_buff, "353"))
		names_recv(th);

	return (0);
}

int priv_recv(struct tparam *th)
{
	time_t t = time(NULL);
	struct tm* tm;
	char id[IRC_MAX];
	char channel[C_NAME], channel2[C_NAME];
	char message[IRC_MAX * 2];
	char message2[IRC_MAX * 2]; /* include &lt,&gt charactor */

	int n = 0;
	int ch_number = 255; /* 255 means "not found" */
	tm = localtime(&t);

	/* zero clear */
	bzero(channel,  sizeof(channel));
	bzero(channel2, sizeof(channel2));
	bzero(message,  sizeof(message));
	bzero(message2, sizeof(message2));

	/* get PRIVMSG */
	n = sscanf(th->l_buff, ":%[^!]!%*s PRIVMSG %s :%[^\r\n]",
			id, channel, message);
	if ((n == 2) && (strcmp(message, "\0") == 0)) { /* type 0 */
		strcpy(message, "0");
	} else if ((n == 1) && (strcmp(message, "\0") == 0)) {
		/* for madoka4.2 */
		n = sscanf(th->l_buff, ":%s PRIVMSG %s :%[^\r\n]",
			id, channel, message);
	}

	/* CTCP */
	if ((strcmp2(th->real_nickname, channel) == 0) &&
	    (message[0] == 0x01) &&
	    (message[strlen(message)-1] == 0x01)) {
		ctcp_recv(th, id, message);
		return (0);
	}

	/* message; '<','>' ---> '&lt;','&gt;' */
        ltgt_encode(message2, message);

	/* conv charset JIS to CP932 */
	decode_jis2cp932(message2);

	/* emoji decode */
	emoji_decode(message2, th->phone_carrier);

	/* channel;  #hoge:*.jp -> %hoge(CP932) */
	channel_encode(channel2, channel);

	/* add href tag */
	href_conv(message2);

	for (n = 0; n <= th->ch_count; n++) {
		if (strcmp2(th->name[n], channel2) == 0) {
					/* message to channel */
			ch_number = n;
			break;
		} else if (strcmp2(th->name[n], id) == 0) { 
					/* message from another person */
			ch_number = n;
			break;
		}
	}
	/* can not found out channel, maybe conversation
	   with another person ? */
	if (ch_number == 255) {
		if (strcmp2(th->real_nickname, channel) == 0) {
			ch_number = channel_make(th, id);
			DPRINTF(0, ("make new channel with %s\n", id));
		} else {
			DPRINTF(0, ("channel not found(%s)\n", channel2));
			return (1);
		}
	} 

	/* check history count and roll up */
	if (th->history_count[ch_number] == th->max_line) {
		for (n = 0; n <= th->max_line; n++) {
			strcpy(th->data[ch_number][n],
			       th->data[ch_number][n + 1]);
			bzero(th->data[ch_number][n + 1],
			       sizeof(th->data[ch_number][n + 1]));
		}
		th->history_count[ch_number]--;
		if (th->read_count[ch_number] > 0)
			th->read_count[ch_number]--;
	}

	/* add message with time into historical buffer */
	if (strcmp2(th->real_nickname, id) == 0) { /* own messages */
		sprintf(th->
			data[ch_number][th->history_count[ch_number]],
			"%02d:%02d &gt;%s&lt; %s <br>",
			tm->tm_hour, tm->tm_min,id, message2);
		th->read_count[ch_number]++;
	} else {
		sprintf(th->
			data[ch_number][th->history_count[ch_number]],
			"%02d:%02d &lt;%s&gt; %s <br>",
			tm->tm_hour, tm->tm_min,id, message2);
	}
	th->history_count[ch_number]++;

	/* debug */
	DPRINTF(1, ("(%2d): <CH:%s> | %s\n",
		(th->history_count[ch_number] - 1), channel2,
		th->data[ch_number][th->history_count[ch_number] - 1]));

	return (0);
}

int ctcp_recv(struct tparam *th, char *id, char *message)
{
	char *ctcp_mes, data[IRC_MAX];
	int len;

	len = strlen(message);

	ctcp_mes = (char*)malloc(len - 2 + 1);
        if (ctcp_mes == NULL) return (0);
        *ctcp_mes = '\0';
        strncpy(ctcp_mes, &message[1], len - 2);

	DPRINTF(0, ("CTCP message:%s\n", ctcp_mes));

	if (strcmp2(ctcp_mes, "VERSION") == 0) {
		/* 0x01VERSION #1:#2:#3 0x01
		   #1: client name 
		   #2: client version
		   #3: env of client */
		sprintf(data,
		    ":%s NOTICE %s :%cVERSION %s %s+ICONV %d.%d %c\r\n",
		    th->real_nickname, id, 0x01, PACKAGE,
		    VERSION, (_LIBICONV_VERSION >> 8),
		    (_LIBICONV_VERSION & 0x00ff), 0x01);
	}

	/* transfer message */
	send_postdata(th->socket_number, data);
	DPRINTF(0, ("CTCP send: %s\n", data));

	free(ctcp_mes);
	return (0);
}

int names_recv(struct tparam *th)
{
	int n = 0;
	int count = 0;
	char id[IRC_MAX];
	char channel[C_NAME], channel2[C_NAME];
	char names[IRC_MAX];

	/* get NAMES */
	if ((sscanf(th->l_buff, ":%*s 353 %s %*s %s :%[^\r\n]",
		id, channel, names)) != 3)
		return (1);

	if (strcmp2(th->real_nickname, id) != 0)
		return (0);

	/* channel;  #hoge:*.jp -> %hoge(CP932) */
	channel_encode(channel2, channel);

	/* count user number */
	count = count_user(names);

	DPRINTF(0, ("NAMES:channel=%s, names(%d)=%s\n",
			channel2, count, names));

	for (n = 0; n <= th->ch_count; n++) {
		if (strcmp2(th->name[n], channel2) == 0) {
                        sprintf(th->member_names[n], " %s", names);
			th->member_count[n] = count;
		}
        }

	return (0);
}

int topic_recv(struct tparam *th)
{
	int n = 0;
	char topic[IRC_MAX * 2], buf[IRC_MAX * 2];
	char channel[C_NAME], channel2[C_NAME];

	/* zero clear */
	bzero(buf, sizeof(buf));
	bzero(channel, sizeof(channel));
	bzero(channel2, sizeof(channel2));

	/* get TOPIC */
	if (strstr(th->l_buff, "332")) {
		if ((sscanf(th->l_buff, ":%*s 332 %*s %s :%[^\r\n]",
			channel, topic)) != 2)
			return (1);
	} else if (strstr(th->l_buff, "TOPIC")) {
		if ((sscanf(th->l_buff, ":%*s TOPIC %s :%[^\r\n]",
			channel, topic)) != 2)
			return (1);
	}

	/* '<','>' ---> '&lt;','&gt;' */
	ltgt_encode(buf, topic);

	/* conv charset JIS to CP932 */
	decode_jis2cp932(buf);

	/* channel;  #hoge:*.jp -> %hoge(CP932) */
	channel_encode(channel2, channel);

	DPRINTF(0, ("TOPIC:channel=%s, topic=%s\n", channel2, topic));

	/* add href tag */
	href_conv(buf);

	for (n = 0; n <= th->ch_count; n++) {
		if (strcmp2(th->name[n], channel2) == 0)
			sprintf(th->topic[n], " %s", buf);
	}
	return (0);
}

int part_recv(struct tparam *th)
{
	char id[IRC_MAX];
	char channel[C_NAME], channel2[C_NAME];

	/* zero clear */
	bzero(channel,  sizeof(channel));
	bzero(channel2, sizeof(channel2));

	/* get PART */
	if (sscanf(th->l_buff, ":%[^!]!%*s PART %s :%*[^\r\n]",
		id, channel) == 2) {
		if (strcmp2(th->real_nickname, id) != 0) {
		   /* send NAMES command */
		   send_names(th, channel);
		   return (0);
		}
	} else if (sscanf(th->l_buff, ":%s PART %s :%*[^\r\n]",
		id, channel) == 2) {
		if (strcmp2(th->real_nickname, id) != 0) {
			return (0);
		}
	}

	/* channel;  #hoge:*.jp -> %hoge(CP932) */
	channel_encode(channel2, channel);

	/* delete channel */
	channel_delete(th, channel2);

	return (0);
}

int channel_delete(struct tparam *th, char *channel)
{
	int i, j, match = 255;

	DPRINTF(0, ("channel_delete()\n"));

	 /* search exist channel */
	for (i = 0; i <= th->ch_count; i++) {
		if (strcmp2(th->name[i], channel) == 0) {
			match = i;
			break;
		}
	}

	if (match > th->ch_count)
		return (0);

	if (match == th->ch_count) {
		bzero(th->name[match], sizeof(th->name[match]));
		bzero(th->topic[match], sizeof(th->topic[match]));
		bzero(th->member_names[match], sizeof(th->member_names[match]));
		th->member_count[match] = 0;
		bzero(th->data[match], sizeof(th->data[match]));
		th->history_count[match] = 0;
		th->read_count[match] = 0;
	} else {
	    for (i = match; i < th->ch_count; i++) {
		bzero(th->name[i], sizeof(th->name[i]));
		bzero(th->topic[i], sizeof(th->topic[i]));
		bzero(th->member_names[i], sizeof(th->member_names[i]));
		th->member_count[i] = 0;
		bzero(th->data[i], sizeof(th->data[i]));

		strcpy(th->name[i], th->name[i + 1]);
		strcpy(th->topic[i], th->topic[i + 1]);
		strcpy(th->member_names[i], th->member_names[i + 1]);
		th->member_count[i] = th->member_count[i + 1];

		for (j = 0; j < th->history_count[i + 1]; j++) {
			strcpy(th->data[i][j], th->data[i + 1][j]);
		}
		th->history_count[i] = th->history_count[i + 1];
		th->read_count[i] = th->read_count[i + 1];
	    }
	    bzero(th->name[th->ch_count],
		   sizeof(th->name[th->ch_count]));
	    bzero(th->topic[th->ch_count],
		   sizeof(th->topic[th->ch_count]));
	    bzero(th->member_names[th->ch_count],
		   sizeof(th->member_names[th->ch_count]));
	    th->member_count[th->ch_count] = 0;
	    bzero(th->data[th->ch_count],
		   sizeof(th->data[match]));
	    th->history_count[th->ch_count] = 0;
	    th->read_count[th->ch_count] = 0;
	}
	th->ch_count--;

	DPRINTF(0, ("part channel name(%d) = %s\n", th->ch_count, channel));

	return (0);
}

int join_recv(struct tparam *th)
{
	int i;
	char id[IRC_MAX];
	char channel[C_NAME], channel2[C_NAME];

	/* zero clear */
	bzero(channel, sizeof(channel));
	bzero(channel2, sizeof(channel2));

	/* get JOIN */
	if (strstr(th->l_buff, "JOIN")) {
		if (sscanf(th->l_buff, ":%[^!]!%*s JOIN :%[^\r\n]",
			   id, channel) == 2);
		else if (sscanf(th->l_buff, ":%[^!]!%*s JOIN %[^\r\n]",
				id, channel) == 2);
		else if (sscanf(th->l_buff, ":%s JOIN %[^\r\n]",
				id, channel) == 2);
	/* madoka hack (delete space(0x20) charactor) */
	if (channel[strlen(channel) - 1] == 0x20)
		channel[strlen(channel) - 1] = 0x0;

		if (strcmp2(th->real_nickname, id) == 0) {
			/* channel;  #hoge:*.jp -> %hoge(CP932) */
			channel_encode(channel2, channel);
			/* search exist channel */
			for (i = 0; i <= th->ch_count; i++) {
				if (strcmp2(th->name[i], channel) == 0) {
					DPRINTF(0, ("channel exist!\n"));
					return (0);
				}
			}
			/* make new channel */
			channel_make(th, channel2);
                } else {
			/* send NAMES command */
                        send_names(th, channel);
                }
        }

	return (0);
}

int channel_make(struct tparam *th, char *channel)
{
	int n = 0;

	/* make new channel */
	while (th->name[n][0] != '\0')
		n++;
	if (n > th->ch_count)
		th->ch_count = n; /* largest channel number */

	strcpy(th->name[n], channel);

	DPRINTF(0, ("channel_make(%d) = %s\n", n, th->name[n]));

	return (n);
}

int ping_recv (struct tparam *th)
{
	char server[IRC_MAX], data[IRC_MAX];

	/* zero clear */
	bzero(server,  sizeof(server));

	/* get PING */
	sscanf(th->l_buff, "PING :%[^\r\n]", server);

	/* transfer message */
	sprintf(data, "PONG %s\r\n", server);

	DPRINTF(0, ("PING:sent data\n"));

	send_postdata(th->socket_number, data);

	return (0);
}

int nick_recv(struct tparam *th)
{
	char old_nickname[NAME_LEN];
	char new_nickname[NAME_LEN];

	sscanf(th->l_buff, ":%[^!]!%*s NICK :%[^\r\n]",
		old_nickname, new_nickname);

	if (strcmp2(th->real_nickname, old_nickname) == 0) {
		strcpy(th->real_nickname, new_nickname);
	DPRINTF(0, ("NICK:%s -> %s\n",
		old_nickname, th->real_nickname));

	}

	return (0);
}

int send_names(struct tparam *th, char *channel)
{
	char data[IRC_MAX];

	sprintf(data, "NAMES %s\n", channel);

	DPRINTF(0, ("NAMES:sent data(%s)(ch_len:%d)\n",
			data, strlen(channel)));

	send_postdata(th->socket_number, data);

	return (0);
}
