#include "dicthiragana.h"
#include "inputmode.h"
#include "config.h"
#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtextstream.h>
#include <qregexp.h>


DictHiragana::DictHiragana() : 
  _dicthira(QDict<QString>(17, FALSE)),
  _dictkata(QDict<QString>(17, FALSE)),
  _dictalphbt(QDict<QString>()),
  _dictsymbol(QDict<QString>()),
  _dictkanainp(QDict<QString>(17, FALSE)),
  _dictdakuten(QDict<QString>())
{
  _dicthira.setAutoDelete(TRUE);
  _dictkata.setAutoDelete(TRUE);
  _dictalphbt.setAutoDelete(TRUE);
  _dictsymbol.setAutoDelete(TRUE);
  _dictkanainp.setAutoDelete(TRUE);
  _dictdakuten.setAutoDelete(TRUE);
}


DictHiragana::~DictHiragana() { }


void
DictHiragana::init()
{
  _dicthira.clear();
  _dictkata.clear();
  _dictalphbt.clear();
  _dictsymbol.clear();
  _dictkanainp.clear();
  _dictdakuten.clear();

  writeDictHira();
  writeDictKata();
  writeDictAlphbt();
  writeDictSymbol();
  writeDictKanaInp();
  writeDictDakuten();

  QString dictdir = QString(getenv("HOME")) + "/.kimera/dic/";
  initDict(_dicthira, dictdir + "hiragana.dic");          // Ҥ餬ʼ
  initDict(_dictkata, dictdir + "katakana.dic");          // ʼ
  initDict(_dictalphbt, dictdir + "zenkakualphabet.dic"); // ѱѻ
  initDict(_dictsymbol, dictdir + "numeralsymbols.dic");  // ѿ漭
  initDict(_dictkanainp, dictdir + "kanainput.dic");      // ϼ
  initDict(_dictdakuten, dictdir + "dakuten.dic");        // 

  // 桦
  _dictsymbol.insert(",", new QString(Config::readEntry("_cmbtouten", ","))); 
  _dictsymbol.insert(".", new QString(Config::readEntry("_cmbkuten", ".")));
  _dictsymbol.insert("/", new QString(Config::readEntry("_cmbsymbol", "/")));
  _dictsymbol.insert("[", new QString(Config::readEntry("_cmbbracket", "[").left(1)));
  _dictsymbol.insert("]", new QString(Config::readEntry("_cmbbracket", "]").right(1)));
}


void
DictHiragana::initDict(QDict<QString>& dict, const QString& filename)
{
  if ( !dict.isEmpty() ) return;
  
  QFile file(filename);
  
  if ( !file.open(IO_ReadOnly) ) {
    QMessageBox::critical(0, "File open error", "Cannot open file: " + filename,
    			  QMessageBox::Ok | QMessageBox::Default, 0);
    qApp->quit();
    return;
  }
    
  QTextStream  stream(&file);
  stream.setEncoding(QTextStream::Latin1);

  while ( !stream.eof() ) {

    QString str;
    str = QString::fromLocal8Bit(stream.readLine());
    
    if ( !str.isEmpty() && !str.contains(QRegExp("^[ \t]")) && 
	 !str.contains(QRegExp("^##")) ) {
      QString keystr, itemstr;
      int idx = str.find(QRegExp("[ \t]"));
      if (idx >= 0) {
	keystr = str.left(idx);
	
	str.remove(0, keystr.length());
	itemstr = str.replace(QRegExp("[ \t]"), "");
	
	dict.insert(keystr, new QString(itemstr));
      }
    }
  }
  
  file.close();
  return;
}


// ʿ̾Ѵ (ʸ˹碌ƣʸˤ :->  )
// parameter:    Roma string (only ASCII characters) or Hiragana string
// return value: Hiragana string
QString 
DictHiragana::convertToHira(const QString& src)
{
  // check parameter
  if (src.isEmpty()) {
    return src;

  } else if (src.local8Bit().length() != src.length()) {
    // Ѵ
    Q_ASSERT(!_dictdakuten.isEmpty());
    
    if (src.length() == 1) return src;

    QString dest;
    int i = 0;
    while(i < src.length()) {
      QString* pstr;
      pstr = _dictdakuten.find( src.mid(i, 2) );
      
      if ( pstr ) {
	dest += *pstr;
	i++;
	
      } else {
	dest += src.mid(i, 1);
      }
      i++;
    }
    
    return dest;
  }
  
  Q_ASSERT(!_dicthira.isEmpty());
  Q_ASSERT(!_dictsymbol.isEmpty());
  
  QString str_roma(src);    // ΰ
  QString dest;
  
  while ( !str_roma.isEmpty() ) {   
    uint i;
    i = (4 > str_roma.length()) ? str_roma.length() : 4;
    
    QString*  pstr;
    while (1) {
      // Ҥ餬ʸ
      pstr = _dicthira.find(str_roma.left(i));
      if (pstr) {
	dest += *pstr;
	break;
      }

      // ѿ渡
      pstr = _dictsymbol.find(str_roma.left(i));
      if (pstr) {
	dest += *pstr;
	break;
      }

      if (i == 1) {
	dest += str_roma.left(i);
	break;
      }

      i--;
    }
    
    QRegExp rxnn("nn", FALSE);
    if (i == 2 && dest.right(1) == QString::fromLocal8Bit("") 
	&& !rxnn.exactMatch(str_roma.left(i))) {
      // 'nm'ʤɤ ''ѴȤν
      str_roma.remove(0, 1);     // Σʸ
    } else if (i == 2 && str_roma[0] == str_roma[1] 
	       && !rxnn.exactMatch(str_roma.left(i))) { 
      // 'tt'ʤɤ ''Ѵʸν     
      str_roma.remove(0, 1);     // Σʸ
    } else {
      str_roma.remove(0, i);    // Ѵʬ
    }
  }
  
  return dest;
}


// convertToHira ƤӽФ塤Ǹʸ'n'ä
// '' Ѵ
QString 
DictHiragana::forceConvtToHira(const QString& src)
{
  if (src.isEmpty()) return src;

  QString dest = convertToHira(src);
  QRegExp rxn("n", FALSE);
  if ( rxn.exactMatch(dest.right(1))) {
    dest = dest.replace(dest.length() - 1, 1, QString::fromLocal8Bit("")); 
  }
  
  return dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString 
DictHiragana::convertToKata(const QString& src)
{
  Q_ASSERT(!_dictkata.isEmpty());

  QString dest;
  QString srchira;
  // check parameter
  if (src.local8Bit().length() == src.length()) {
    srchira = convertToHira(src);
  } else {
    srchira = src;
  }

  for (int i = 0; i < srchira.length(); i++) {
    QString* pstr;
    pstr = _dictkata.find( srchira.mid(i, 1) );
    if ( pstr ) {
      dest += *pstr;
    } else {
      dest += srchira.mid(i, 1);
    }
  }

  return  dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString 
DictHiragana::forceConvtToKata(const QString& src)
{
  return convertToKata(forceConvtToHira(src));
}


// parameter: Roma string (only ASCII characters)
QString
DictHiragana::convertToZenkakuEisu(const QString& src)
{  
  Q_ASSERT(!_dictalphbt.isEmpty());
  Q_ASSERT(!_dictsymbol.isEmpty());

  QString dest;
  for (int i = 0; i < src.length(); i++) {
    QString* pstr;
    if ( pstr = _dictalphbt.find(src.mid(i, 1)) ) {  // if the key exists 
      dest += *pstr;
      
    } else if ( pstr = _dictsymbol.find(src.mid(i, 1)) ) {  // if the key exists 
      dest += *pstr;

    } else {
      dest += src.mid(i, 1);
    }
  }
  
  return dest;
}


QString 
DictHiragana::convertYomi(const QString& src, const InputMode& mode)
{
  switch (mode.id() & InputMode::ModeMask) {
  case InputMode::Hiragana:
    return convertToHira(src);
    break;
    
  case InputMode::Katakana:
    return convertToKata(src);
    break;
    
  case InputMode::ZenkakuEisu:
    if (src.local8Bit().length() != src.length()) {
      Q_ASSERT(0);
      break;
    }
    return convertToZenkakuEisu(src);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


QString
DictHiragana::forceConvtYomi(const QString& src, const InputMode& mode)
{
  switch (mode.id() & InputMode::ModeMask) {
  case InputMode::Hiragana:
    return forceConvtToHira(src);
    break;
    
  case InputMode::Katakana:
    return forceConvtToKata(src);
    break;
    
  case InputMode::ZenkakuEisu:
    if (src.local8Bit().length() != src.length()) {
      Q_ASSERT(0);
      break;
    }
    return convertToZenkakuEisu(src);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


//  Ѵ
//  params: QKeyEvent
//  params: Input mode
//  return: Hiragana string
QString 
DictHiragana::convertYomi(const QKeyEvent& key, const InputMode& mode)
{
  if ( !(mode.id() & InputMode::KanaInput) ) {
    return QString();

  } else if (mode.id() & InputMode::ZenkakuEisu) {
    return convertToZenkakuEisu(key.text());
  }

  Q_ASSERT(!_dictkanainp.isEmpty());
  QString keystr;
  if (key.state() == Qt::NoButton) {
    keystr = key.text();
    
  } else if (key.state() == Qt::ShiftButton) { 
    keystr = QString("Shift+") + key.text();
    
  } else if (key.state() == Qt::Keypad) {
    keystr = QString("Keypad+") + key.text();
    
  } else {
    return QString();
  }
  
  QString* pres = _dictkanainp[keystr];
  if ( !pres ) return QString();

  switch (mode.id() & InputMode::ModeMask) {
  case InputMode::Hiragana:
    return *pres;
    break;
    
  case InputMode::Katakana:
    return convertToKata(*pres);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


void
DictHiragana::writeDictHira() const
{
  QString s = QObject::tr("## ޻ Ҥ餬Ѵ

a	  
i         
u         
e         
o         

ka        
ki        
ku        
ke        
ko        

sa        
si        
su        
se        
so        

ta        
ti        
tu        
te        
to        

na        
ni        
nu        
ne        
no        

ha        
hi        
hu        
he        
ho        

ma        
mi        
mu        
me        
mo        

ya        
yi        
yu        
ye 	  
yo        

ra        
ri        
ru        
re        
ro        

wa        
wi        
wu        
we        
wo        

ga        
gi        
gu        
ge        
go        

za        
zi        
zu        
ze        
zo        

da        
di        
du        
de        
do        

ba        
bi        
bu        
be        
bo        

pa        
pi        
pu        
pe        
po        

fa        դ
fi        դ
fu        
fe        դ
fo        դ

ja        
ji        
ju        
je        
jo        

ca	  
cu        
co        

la        
li        
lu        
le        
lo        

va	  
vi	  
vu	  
ve	  
vo	  

kya       
kyi       
kyu       
kye       
kyo       

gya       
gyi       
gyu       
gye       
gyo       

sya     
syi     
syu     
sye     
syo     

zya     
zyi     
zyu     
zye     
zyo     

jya     
jyi     
jyu     
jye     
jyo     

tya     
tyi     
tyu     
tye     
tyo     

cya     
cyi     
cyu     
cye     
cyo     

dya     ¤
dyi     ¤
dyu     ¤
dye     ¤
dyo     ¤

nya     ˤ
nyi     ˤ
nyu     ˤ
nye     ˤ
nyo     ˤ

hya     Ҥ
hyi     Ҥ
hyu     Ҥ
hye     Ҥ
hyo     Ҥ

bya     Ӥ
byi     Ӥ
byu     Ӥ
bye     Ӥ
byo     Ӥ

pya     Ԥ
pyi     Ԥ
pyu     Ԥ
pye     Ԥ
pyo     Ԥ

mya     ߤ
myi     ߤ
myu     ߤ
mye     ߤ
myo     ߤ

rya     
ryi     ꤣ
ryu     
rye     ꤧ
ryo     

tsa     Ĥ 
tsi     Ĥ
tsu     
tse     Ĥ
tso     Ĥ

lya     
lyi     ꤣ
lyu     
lye     ꤧ
lyo     

sha     
shi       
shu     
she     
sho     

tha     Ƥ
thi     Ƥ
thu     Ƥ
the     Ƥ
tho     Ƥ

dha     Ǥ
dhi     Ǥ
dhu     Ǥ
dhe     Ǥ
dho     Ǥ

cha     
chi       
chu     
che     
cho     

gwa	
gwi	
gwu	
gwe	
gwo	

xa        
xi        
xu        
xe        
xo        

xwa       
xtu       
xtsu      
xya       
xyu       
xyo       

nb       
nc       
nd       
nf       
ng       
nh       
nj       
nk       
nl       
nm       
nn       
np       
nr       
ns       
nt       
nv       
nw       
nx       
ny       
nz       

bb	
cc	
dd	
ff	
gg	
hh	
jj	
kk	
ll      
mm	
pp	
rr	
ss	
tt	
vv	
ww	
yy	
zz	
tch	
");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("hiragana.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictKata() const
{
  QString s = QObject::tr("## Ҥ餬 Ѵ

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    

    
    
    
    
    

    
    
    
    

    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    
    
    

    
    
    

    
    
");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("katakana.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictAlphbt() const
{
  QString s = QObject::tr("## ѱѻѴ

a    
b    
c    
d    
e    
f    
g    
h    
i    
j    
k    
l    
m    
n    
o    
p    
q    
r    
s    
t    
u    
v    
w    
x    
y    
z    

A    
B    
C    
D    
E    
F    
G    
H    
I    
J    
K    
L    
M    
N    
O    
P    
Q    
R    
S    
T    
U    
V    
W    
X    
Y    
Z    
");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("zenkakualphabet.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictSymbol() const
{
  QString s = QObject::tr("## ѿѴ 

1       
2       
3       
4       
5       
6       
7       
8       
9       
0       

{       
}       
^       
<       
>       
(       
)       
-       
{       
}       

?       
!       
#       
$       
\"       
%       
&       
'       
`       
+       
*       
=       
|       
;       
:       
~       
_       
/       
\\       

@/      
@\\     
@       
@~      
@||     
@3      
@2      
");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("numeralsymbols.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  }
}


void
DictHiragana::writeDictKanaInp() const
{
QString s = QObject::tr("##  

1         
Shift+!   
2         
Shift+\"   
3         
Shift+#   
4         
Shift+$   
5         
Shift+%   
6         
Shift+&   
7         
Shift+'   
8         
Shift+(   
9         
Shift+)   
0         
Shift+~   
-         
Shift+=   
^         
\\         
Shift+|   
Q         
Shift+Q   
W         
Shift+W   
E         
Shift+E   
R         
Shift+R   
T         
Shift+T   
Y         
Shift+Y   
U         
Shift+U   
I         
Shift+I   
O         
Shift+O   
P         
Shift+P   
@         
Shift+`       
[         
Shift+{   
A         
Shift+A   
S         
Shift+S   
D         
Shift+D   
F         
Shift+F   
G         
Shift+G   
H         
Shift+H   
J         
Shift+J   
K         
Shift+K   
L         
Shift+L   
;         
Shift++   
:         
Shift+*   
]         
Shift+}   
Z         
Shift+Z   
X         
Shift+X   
C         
Shift+C   
V         
Shift+V   
B         
Shift+B   
N         
Shift+N   
M         
Shift+M   
,         
Shift+<   
.         
Shift+>   
/         
Shift+?   
\\         
Shift+_   
Keypad+0     
Keypad+1  
Keypad+2  
Keypad+3  
Keypad+4  
Keypad+5  
Keypad+6  
Keypad+7  
Keypad+8  
Keypad+9  
Keypad+/  
Keypad+*  
Keypad+-  
Keypad++  
Keypad+.  
");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("kanainput.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictDakuten() const
{
  QString s = QObject::tr("##  ʸѴ

    
    
    
    
    

    
    
    
    
    

    
    
ġ    
ơ    
ȡ    

ϡ    
ҡ    
ա    
ء    
ۡ    

ϡ    
ҡ    
ա    
ء    
ۡ    

    
    
    
    
    

    
    
    
    
    

    
    
ġ    
ơ    
ȡ    

ϡ    
ҡ    
ա    
ء    
ۡ    

ϡ      
ҡ    
ա    
ء    
ۡ    
");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("dakuten.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}
