#include "dicthiragana.h"
#include "inputmode.h"
#include <qapplication.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtextstream.h>
#include <qregexp.h>


DictHiragana::DictHiragana() : 
  _dicthira(QDict<QString>(17, FALSE)),
  _dictkata(QDict<QString>(17, FALSE)),
  _dictalphbt(QDict<QString>()),
  _dictsymbol(QDict<QString>()),
  _dictkanainp(QDict<QString>(17, FALSE))
{
  _dicthira.setAutoDelete(TRUE);
  _dictkata.setAutoDelete(TRUE);
  _dictalphbt.setAutoDelete(TRUE);
  _dictsymbol.setAutoDelete(TRUE);
  _dictkanainp.setAutoDelete(TRUE);
}


DictHiragana::~DictHiragana() { }


void
DictHiragana::initDict(const QString& filename, QDict<QString>& dict)
{
  if ( !dict.isEmpty() ) return;
  
  QFile file(filename);
  
  if ( !file.open(IO_ReadOnly) ) {
    QMessageBox::critical(0, "File open error", "Cannot open file: " + filename,
    			  QMessageBox::Ok | QMessageBox::Default, 0);
    qApp->quit();
    return;
  }
    
  QTextStream  stream(&file);
  stream.setEncoding(QTextStream::Latin1);

  while ( !stream.eof() ) {

    QString str;
    str = QString::fromLocal8Bit(stream.readLine());
    
    if ( !str.isEmpty() && !str.contains(QRegExp("^[ \t]")) && 
	 !str.contains(QRegExp("^##")) ) {
      QString keystr, itemstr;
      int idx = str.find(QRegExp("[ \t]"));
      if (idx >= 0) {
	keystr = str.left(idx);
	
	str.remove(0, keystr.length());
	itemstr = str.replace(QRegExp("[ \t]"), "");
	
	dict.insert(keystr, new QString(itemstr));
      }
    }
  }
  
  file.close();
  return;
}


// ʿ̾Ѵ (ʸ˹碌ƣʸˤ :->  )
// parameter:    Roma string (only ASCII characters) or Hiragana string
// return value: Hiragana string
QString 
DictHiragana::convertToHira(const QString& src)
{
  // check parameter
  if (src.isEmpty()) {
    return src;

  } else if (src.local8Bit().length() != src.length()) {
    // Ѵ
    if (_dictdakuten.isEmpty()) {
      QString dictname("dic/dakuten.dic");
      initDict(dictname, _dictdakuten);
    }
    
    if (src.length() == 1) return src;

    QString dest;
    int i = 0;
    while(i < src.length()) {
      QString* pstr;
      pstr = _dictdakuten.find( src.mid(i, 2) );
      
      if ( pstr ) {
	dest += *pstr;
	i++;
	
      } else {
	dest += src.mid(i, 1);
      }
      i++;
    }
    
    return dest;
  }
  
  // Ҥ餬ʼ񥻥å
  if (_dicthira.isEmpty()) {
    QString dictname("dic/hiragana.dic");
    initDict(dictname, _dicthira);
  }

  // ѿ漭񥻥å
  if (_dictsymbol.isEmpty()) {
    QString dictname("dic/numeralsymbols.dic");
    initDict(dictname, _dictsymbol);
  }
  
  QString str_roma(src);    // ΰ
  QString dest;
  
  while ( !str_roma.isEmpty() ) {   
    uint i;
    i = (4 > str_roma.length()) ? str_roma.length() : 4;
    
    QString*  pstr;
    while (1) {
      // Ҥ餬ʸ
      pstr = _dicthira.find(str_roma.left(i));
      if (pstr) {
	dest += *pstr;
	break;
      }

      // ѿ渡
      pstr = _dictsymbol.find(str_roma.left(i));
      if (pstr) {
	dest += *pstr;
	break;
      }

      if (i == 1) {
	dest += str_roma.left(i);
	break;
      }

      i--;
    }
    
    QRegExp rxnn("nn", FALSE);
    if (i == 2 && dest.right(1) == QString::fromLocal8Bit("") 
	&& !rxnn.exactMatch(str_roma.left(i))) {
      // 'nm'ʤɤ ''ѴȤν
      str_roma.remove(0, 1);     // Σʸ
    } else if (i == 2 && str_roma[0] == str_roma[1] 
	       && !rxnn.exactMatch(str_roma.left(i))) { 
      // 'tt'ʤɤ ''Ѵʸν     
      str_roma.remove(0, 1);     // Σʸ
    } else {
      str_roma.remove(0, i);    // Ѵʬ
    }
  }
  
  return dest;
}


// convertToHira ƤӽФ塤Ǹʸ'n'ä
// '' Ѵ
QString 
DictHiragana::forceConvtToHira(const QString& src)
{
  if (src.isEmpty()) return src;

  QString dest = convertToHira(src);
  QRegExp rxn("n", FALSE);
  if ( rxn.exactMatch(dest.right(1))) {
    dest = dest.replace(dest.length() - 1, 1, QString::fromLocal8Bit("")); 
  }
  
  return dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString 
DictHiragana::convertToKata(const QString& src)
{
  if (_dictkata.isEmpty()) {
    QString dictname("dic/katakana.dic");
    initDict(dictname, _dictkata);
  }

  QString dest;
  QString srchira;
  // check parameter
  if (src.local8Bit().length() == src.length()) {
    srchira = convertToHira(src);
  } else {
    srchira = src;
  }

  for (int i = 0; i < srchira.length(); i++) {
    QString* pstr;
    pstr = _dictkata.find( srchira.mid(i, 1) );
    if ( pstr ) {
      dest += *pstr;
    } else {
      dest += srchira.mid(i, 1);
    }
  }

  return  dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString 
DictHiragana::forceConvtToKata(const QString& src)
{
  return convertToKata(forceConvtToHira(src));
}


// parameter: Roma string (only ASCII characters)
QString
DictHiragana::convertToZenkakuEisu(const QString& src)
{  
  // ѱѻ
  if (_dictalphbt.isEmpty()) {
    QString dictname("dic/zenkakualphabet.dic");
    initDict(dictname, _dictalphbt);
  }

  // ѿ漭񥻥å
  if (_dictsymbol.isEmpty()) {
    QString dictname("dic/numeralsymbols.dic");
    initDict(dictname, _dictsymbol);
  }

  QString dest;
  for (int i = 0; i < src.length(); i++) {
    QString* pstr;
    if ( pstr = _dictalphbt.find(src.mid(i, 1)) ) {  // if the key exists 
      dest += *pstr;
      
    } else if ( pstr = _dictsymbol.find(src.mid(i, 1)) ) {  // if the key exists 
      dest += *pstr;

    } else {
      dest += src.mid(i, 1);
    }
  }
  
  return dest;
}


QString 
DictHiragana::convertYomi(const QString& src, const InputMode& mode)
{
  switch (mode.id() & InputMode::ModeMask) {
  case InputMode::Hiragana:
    return convertToHira(src);
    break;
    
  case InputMode::Katakana:
    return convertToKata(src);
    break;
    
  case InputMode::ZenkakuEisu:
    if (src.local8Bit().length() != src.length()) {
      Q_ASSERT(0);
      break;
    }
    return convertToZenkakuEisu(src);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


QString
DictHiragana::forceConvtYomi(const QString& src, const InputMode& mode)
{
  switch (mode.id() & InputMode::ModeMask) {
  case InputMode::Hiragana:
    return forceConvtToHira(src);
    break;
    
  case InputMode::Katakana:
    return forceConvtToKata(src);
    break;
    
  case InputMode::ZenkakuEisu:
    if (src.local8Bit().length() != src.length()) {
      Q_ASSERT(0);
      break;
    }
    return convertToZenkakuEisu(src);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


//  Ѵ
//  params: QKeyEvent
//  params: Input mode
//  return: Hiragana string
QString 
DictHiragana::convertYomi(const QKeyEvent& key, const InputMode& mode)
{
  if ( !(mode.id() & InputMode::KanaInput) ) {
    return QString();

  } else if (mode.id() &InputMode::ZenkakuEisu) {
    return convertToZenkakuEisu(key.text());
  }

  if (_dictkanainp.isEmpty()) {
    QString dictname("dic/kanainput.dic");
    initDict(dictname, _dictkanainp);
  }
  
  QString keystr;
  if (key.state() & Qt::ShiftButton) { 
    keystr = QString("Shift+") + key.text();

  } else if (key.state() == Qt::NoButton) {
    keystr = key.text();

  } else {
    return QString();
  }
  
  QString* pres = _dictkanainp[keystr];
  if ( !pres ) return QString();

  switch (mode.id() & InputMode::ModeMask) {
  case InputMode::Hiragana:
    return *pres;
    break;
    
  case InputMode::Katakana:
    return convertToKata(*pres);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}
