#include "preeditarea.h"
#include "kimeraapp.h"
#include "inputmethod.h"
#include "ximattribute.h"
#include "config.h"
#include <qlabel.h>
#include <qrect.h>
#include <qtimer.h>
#include <qmessagebox.h>

const QString  KeyColor = "_colorsettingcolor";
const QString  KeyUnderline = "_colorsettingunderline";

PreeditArea::PreeditArea() 
  : QWidget((QWidget*)0, "preedit", WStyle_StaysOnTop | WStyle_Customize | WStyle_NoBorder | WStyle_Tool | WX11BypassWM), _state(Input)
{
  _attentseg  = new QLabel(this, "attention_segment");
  _backseg    = new QLabel(this, "back_segment");
  _forwardseg = new QLabel(this, "forward_segment");

  resize(0, 0);
}


void 
PreeditArea::polish()
{
  init();

  _attentseg->setMargin( 1 );
  _attentseg->setIndent( 0 );
  _attentseg->setAutoMask( FALSE );
  _attentseg->setFrameStyle( QFrame::Plain | QFrame::Box );
  _attentseg->setLineWidth( 0 );
  _attentseg->setAlignment( AlignAuto | AlignTop );

  _backseg->setMargin( 1 );
  _backseg->setIndent( 0 );
  _backseg->setAutoMask( FALSE );
  _backseg->setFrameStyle( QFrame::Plain | QFrame::Box );
  _backseg->setLineWidth( 0 );
  _backseg->setAlignment( AlignAuto | AlignTop );
  _backseg->move(0, 0);

  _forwardseg->setMargin( 1 );
  _forwardseg->setIndent( 0 );
  _forwardseg->setAutoMask( FALSE );
  _forwardseg->setFrameStyle( QFrame::Plain | QFrame::Box );
  _forwardseg->setLineWidth( 0 );
  _forwardseg->setAlignment( AlignAuto | AlignTop );  
}


void 
PreeditArea::init()
{
  debug("PreeditArea init");
  // Loads colors
  for (int i = 0; i < NumColors; i++) {
    _colorlist[i] = QColor( Config::readEntry(KeyColor + QString::number(i), "black") );
  }
}


void 
PreeditArea::setFont(const QFont & f)
{
  for (int i = 0; i < NumStates; i++) {
      _font[i] = f;
      _font[i].setUnderline(Config::readEntry(KeyUnderline + QString::number(i), "0") == QString("1"));
  }

  _attentseg->setFont( f );
  _backseg->setFont( f );
  _forwardseg->setFont( f );
}


void
PreeditArea::showText(const QStringList& strlist, uint attention)
{
  if (attention >= strlist.count())
    return;
  
  // Ѵʸ
  if ( strlist.isEmpty() ) {
    _attentseg->setText(QString::null);
    _backseg->setText(QString::null);
    _forwardseg->setText(QString::null);

  } else {
    // Sets back_segment
    QStringList sl(strlist);
    QStringList  backsegment;
    for (uint i = 0; i < attention; i++) {
      backsegment << sl.first();
      sl.pop_front();
    }
    backsegment.isEmpty() ? _backseg->setText(QString::null) : _backseg->setText(backsegment.join(""));

    // Set text of preedit area
    _attentseg->setText(sl.first());
    sl.pop_front();
    
    // Sets forward_segment
    sl.isEmpty() ? _forwardseg->setText(QString::null) : _forwardseg->setText(sl.join(""));
  }

  // Show
  switch ( KimeraApp::inputmethod()->currentXIMStyle() ) {
  case OVER_THE_SPOT_STYLE:
    if (_attentseg->text().isEmpty()) {
      hide();
      break;
    }

    _backseg->text().isEmpty() ? _backseg->hide() : _backseg->show();
    _attentseg->show();
    _forwardseg->text().isEmpty() ? _forwardseg->hide() : _forwardseg->show();
    show();
    adjustSize();
    break;

  case ON_THE_SPOT_STYLE:
    KimeraApp::inputmethod()->clearPreedit();
    if (_attentseg->text().isEmpty()) {
      break;
    }

    if (_state == Input) {
      KimeraApp::inputmethod()->appendPreeditString(_attentseg->text(), InputMethod::FT_UNDERLINE);
    } else {
      if ( !_backseg->text().isEmpty())
	KimeraApp::inputmethod()->appendPreeditString(_backseg->text(), InputMethod::FT_UNDERLINE);
      
      KimeraApp::inputmethod()->appendPreeditString(_attentseg->text(), InputMethod::FT_REVERSE);
      if ( !_forwardseg->text().isEmpty() )
	KimeraApp::inputmethod()->appendPreeditString(_forwardseg->text(), InputMethod::FT_UNDERLINE);
    }
    break;

  default:
    qWarning("Bad XIMStyle");
    Q_ASSERT(0);
    return;
    break;
  }

  qDebug("showText  back: %s  attention: %s  forward: %s", _backseg->text().local8Bit().data(),
	 _attentseg->text().local8Bit().data(), _forwardseg->text().local8Bit().data());
}


void 
PreeditArea::showInputingString(const QString& str)
{
  Q_ASSERT( !str.isEmpty() );
  _state = Input;
  _attentseg->setFont( _font[Input] );
  _attentseg->setPaletteForegroundColor( _colorlist[FgInput] );
  _attentseg->setPaletteBackgroundColor( _colorlist[BgInput] );

  showText(QStringList(str));
}


void 
PreeditArea::showConvertingSegments(const QStringList& strlist, uint attention)
{
  Q_ASSERT( !strlist.empty() );
  _state = Attention;
  _attentseg->setFont( _font[Attention] ); 
  _attentseg->setPaletteForegroundColor( _colorlist[FgAttention] );
  _attentseg->setPaletteBackgroundColor( _colorlist[BgAttention] );

  _backseg->setFont( _font[Converted] ); 
  _backseg->setPaletteForegroundColor( _colorlist[FgConverted] );
  _backseg->setPaletteBackgroundColor( _colorlist[BgConverted] );

  _forwardseg->setFont( _font[Converted] ); 
  _forwardseg->setPaletteForegroundColor( _colorlist[FgConverted] );
  _forwardseg->setPaletteBackgroundColor( _colorlist[BgConverted] );

  showText(strlist, attention);
}


void 
PreeditArea::showChangingSegmentLength(const QStringList& strlist, uint attention)
{
  Q_ASSERT( !strlist.empty() );
  _state = Changing;
  _attentseg->setFont( _font[Changing] ); 
  _attentseg->setPaletteForegroundColor( _colorlist[FgChanging] );
  _attentseg->setPaletteBackgroundColor( _colorlist[BgChanging] );

  _backseg->setFont( _font[Converted] ); 
  _backseg->setPaletteForegroundColor( _colorlist[FgConverted] );
  _backseg->setPaletteBackgroundColor( _colorlist[BgConverted] );

  _forwardseg->setFont( _font[Converted] ); 
  _forwardseg->setPaletteForegroundColor( _colorlist[FgConverted] );
  _forwardseg->setPaletteBackgroundColor( _colorlist[BgConverted] );

  showText(strlist, attention);
}


void
PreeditArea::adjustSize()
{
  if ( !_attentseg->text().isEmpty() ) {
    _backseg->resize( _backseg->sizeHint() );

    QPoint p(0, 0);  
    if ( !_backseg->text().isEmpty() )
      p += QPoint(_backseg->size().width(), 0);
  
    _attentseg->move( p );
    _attentseg->resize( _attentseg->sizeHint() );
    emit listPointChanged( mapToGlobal(p + QPoint(0, _attentseg->height()) + QPoint(-4, 1)) );   // Current segment pos
    p += QPoint(_attentseg->size().width(), 0);
    _forwardseg->move( p );
    _forwardseg->resize( _forwardseg->sizeHint() );

    if ( !_forwardseg->text().isEmpty() )
      p += QPoint(_forwardseg->size().width(), 0);
  
    resize( QSize(p.x(), _attentseg->sizeHint().height()) );
    qDebug("back: %s  attention: %s  forward: %s", _backseg->text().local8Bit().data(),
	   _attentseg->text().local8Bit().data(), _forwardseg->text().local8Bit().data());
  }

  QWidget::adjustSize();
}


void
PreeditArea::movePreeditPos(const QPoint& pos)
{
  int y_offset = (font().pointSize() > 0) ? font().pointSize() + 3 : font().pixelSize();
  move(pos - QPoint(0, y_offset));

  if (KimeraApp::inputmethod()->currentXIMStyle() == ON_THE_SPOT_STYLE)
    emit listPointChanged(pos);   // Current segment pos
}


QString
PreeditArea::text() const
{
  return _backseg->text() + _attentseg->text() + _forwardseg->text();
}


QString
PreeditArea::backwardText() const
{
  return _backseg->text();
}


QString
PreeditArea::attentionText() const
{
  return _attentseg->text();
}


QString
PreeditArea::forwardText() const
{
  return _forwardseg->text();
}


void
PreeditArea::hide()
{
  KimeraApp::inputmethod()->clearPreedit();
  _attentseg->setText(QString::null);
  _backseg->setText(QString::null);
  _forwardseg->setText(QString::null);
  
  QWidget::hide();
}
