#include "kanjiconvert.h"
#include "kanjiengine.h"
#include "dicthiragana.h"
#include "preeditarea.h"
#include "inputmode.h"
#include "config.h"
#include "keyassigner.h"
#include "kimeraglobal.h"
#include "debug.h"
#include <qapplication.h>
#include <qkeysequence.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <X11/keysym.h>
using namespace Kimera;

QIntDict<FuncPtr>  KanjiConvert::funcptrdict = QIntDict<FuncPtr>();

KanjiConvert::KanjiConvert() : _crntsegidx(0),
			       _minsegidx(0),
			       _inputkeys(QString::null),
			       _yomiseglist(QStringList()),
			       _convtkanjilist(QStringList()),
			       _stat(NONE_YOMI),
			       _mode(), 
			       _dict(DictHiragana()),
			       _eng(0),
			       _preedit(0),
			       _candlist(0)
{
  // Create PreeditArea instance
  _preedit = new PreeditArea();

  // Create CandidacyListBox instance
  _candlist = new CandidacyListBox();

  connect(_candlist, SIGNAL(clicked(QListBoxItem*)), this, SLOT(slotDecideSegment(QListBoxItem*)));
  connect(_preedit, SIGNAL(listPointChanged(const QPoint&)), _candlist, SLOT(setPos(const QPoint&)));
}
 

KanjiConvert::~KanjiConvert()
{
  if (_eng)
    _eng->cleanup();
  delete  _preedit;
  delete  _candlist;
}


void 
KanjiConvert::init()
{
  DEBUG_TRACEFUNC();

  if (_eng)
    _eng->cleanup();

  QString keng = Config::readEntry("_cmbkanjieng", "Canna");
  _eng = KanjiEngine::kanjiEngine(keng);
  if (!_eng) {
    QMessageBox::critical(0, "Kanji Engine Error", keng + ": No such kanji engine\nAssert!",
			  QMessageBox::Ok | QMessageBox::Default, 0);
    qApp->quit();
    return;
  }
  
  // initialize kanji engine
  bool ok = _eng->init();
  clear();
  if ( !ok ) {
    QMessageBox::critical(0, "Kanji Engine Error",
			  "Error Initializing " + keng + "!\nChange Kanji Engine.",
			  QMessageBox::Ok | QMessageBox::Default, 0);
  }

  // Dictionary Initialization
  _dict.init();

  // Initiate preedit
  _preedit->init();

  // Initialize  dictionary of function pointer
  init_funcptrdict();
}


void 
KanjiConvert::init_funcptrdict()
{
  DEBUG_TRACEFUNC();
  if ( ! funcptrdict.isEmpty() ) 
    return;
  
  funcptrdict.setAutoDelete(TRUE);
  
  funcptrdict.insert(FID_NextCandidacy,          new FuncPtr(&KanjiConvert::nextCandidacy));
  funcptrdict.insert(FID_NextCandidacyGroup,     new FuncPtr(&KanjiConvert::nextCandidacyGroup));
  funcptrdict.insert(FID_PreviousCandidacy,      new FuncPtr(&KanjiConvert::previousCandidacy));
  funcptrdict.insert(FID_PreviousCandidacyGroup, new FuncPtr(&KanjiConvert::previousCandidacyGroup));
  funcptrdict.insert(FID_ConvertAllSegments,     new FuncPtr(&KanjiConvert::convertAllSegments));
  funcptrdict.insert(FID_ConvertCrntSegment,     new FuncPtr(&KanjiConvert::convertCrntSegment));
  funcptrdict.insert(FID_DeleteBackward,         new FuncPtr(&KanjiConvert::deleteBackward));
  funcptrdict.insert(FID_DecideAllSegments,      new FuncPtr(&KanjiConvert::decideAllSegments));
  funcptrdict.insert(FID_DecideCrntSegment,      new FuncPtr(&KanjiConvert::decideCrntSegment));
  funcptrdict.insert(FID_CancelConversion,       new FuncPtr(&KanjiConvert::cancelConversion));
  funcptrdict.insert(FID_LengthenSegment,        new FuncPtr(&KanjiConvert::lengthenSegment));
  funcptrdict.insert(FID_ShortenSegment,         new FuncPtr(&KanjiConvert::shortenSegment));
  funcptrdict.insert(FID_InputSpaceChar,         new FuncPtr(&KanjiConvert::inputSpaceChar));
  funcptrdict.insert(FID_ConvertToHira,          new FuncPtr(&KanjiConvert::convertToHira));
  funcptrdict.insert(FID_ConvertToKana,          new FuncPtr(&KanjiConvert::convertToKata));
  funcptrdict.insert(FID_ConvertToHankakuKana,   new FuncPtr(&KanjiConvert::convertToHankakuKana));
  funcptrdict.insert(FID_ConvertToHankakuEisu,   new FuncPtr(&KanjiConvert::convertToHankakuEisu));
  funcptrdict.insert(FID_ConvertToZenkakuEisu,   new FuncPtr(&KanjiConvert::convertToZenkakuEisu));
  funcptrdict.insert(FID_ForwardSegment,         new FuncPtr(&KanjiConvert::forwardSegment));
  funcptrdict.insert(FID_BackwardSegment,        new FuncPtr(&KanjiConvert::backwardSegment));
  funcptrdict.insert(FID_ToFirstSegment,         new FuncPtr(&KanjiConvert::toFirstSegment));
  funcptrdict.insert(FID_ToLastSegment,          new FuncPtr(&KanjiConvert::toLastSegment));
  funcptrdict.insert(FID_SwitchZenkakuEisu,      new FuncPtr(&KanjiConvert::switchZenkakuEisu));
}


void
KanjiConvert::setInputMode(const InputMode& mode)
{
  DEBUG_TRACEFUNC("mode: 0x%x", mode.id());
  _mode.merge(mode);
  clear();
}


void    
KanjiConvert::clear()
{
  DEBUG_TRACEFUNC();
  _crntsegidx = 0;
  _minsegidx = 0;
  _inputkeys = QString::null;
  _yomiseglist.clear();
  _convtkanjilist.clear();
  if (_eng)
    _eng->endConvert();
  _stat = NONE_YOMI;
  hideToolBox();
}


void
KanjiConvert::setFont(QFont ft) 
{
  DEBUG_TRACEFUNC();
  _preedit->setFont(ft);        // Set font to preedit area
  _candlist->setFont(ft);       // Set font to candidacy list box
}


bool
KanjiConvert::convertAllSegments()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  switch (_stat) {
  case NONE_YOMI:
    // Do nothing
    return TRUE;
    break;

  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _eng->beginConvert(forceConvtYomigana(Mode_Hiragana), _convtkanjilist, _yomiseglist);
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::convertCrntSegment()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  QStringList convstr, yomiseg;
  switch (_stat) {
  case NONE_YOMI:
    // Do nothing
    return TRUE;
    break;

  case INPUTING_YOMI:
    convertAllSegments();
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
    break;
    
  case CHANGING_LENGTH:
    _eng->resizeSegment(_crntsegidx, _convtkanjilist[_crntsegidx].length(), convstr, yomiseg);
    setConvertSegments(_crntsegidx, convstr);
    setYomigana(_crntsegidx, yomiseg);
    break;    
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool 
KanjiConvert::nextCandidacy()
{
  DEBUG_TRACEFUNC();
  return  moveCrntItemCandidacyListBox(1);
}


bool 
KanjiConvert::nextCandidacyGroup()
{
  DEBUG_TRACEFUNC();
  return  moveCrntItemCandidacyListBox( CandidacyListBox::MAX_VISIBLE_ITEMS );
}


bool 
KanjiConvert::previousCandidacy()
{
  DEBUG_TRACEFUNC();
  return  moveCrntItemCandidacyListBox(-1);
}


bool 
KanjiConvert::previousCandidacyGroup()
{
  DEBUG_TRACEFUNC();
  return  moveCrntItemCandidacyListBox( -CandidacyListBox::MAX_VISIBLE_ITEMS );
}


bool
KanjiConvert::moveCrntItemCandidacyListBox(int d)
{
  DEBUG_TRACEFUNC("d: %d", d);
  if (!_eng)
    return FALSE;

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
    if ( _candlist->isVisible() ) {
      // It means CandidacyListBox shows
      if (d > 0) {
	_candlist->showNextCandidacy(d);
      } else if (d < 0) {
	_candlist->showPreviousCandidacy(-d);
      }

      // Replace string of preeidt area
      setConvertSegments(_crntsegidx, _candlist->currentText());

    } else if (_eng->isConverting()) {
      // Get Candidacies
      QStringList cand;
      if ( !_eng->getCandidate(_crntsegidx, cand) ) {
	break;
      }
      if (cand.isEmpty()) {
	cand << _convtkanjilist[_crntsegidx];
      }      
      if (cand.count() > 1 && cand.first() == _convtkanjilist[_crntsegidx]) {
	_candlist->showList(cand, 1);
      } else {
	_candlist->showList(cand);
      }
      // Replace string of converted kanji string list
      setConvertSegments(_crntsegidx, _candlist->currentText());

    } else {
      convertAllSegments();
    }
    break;
    
  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  // show preedit
  _stat = SHOWING_LIST;
  showPreeditArea();  
  return TRUE;
}


// Append a character to roma string
bool
KanjiConvert::appendChar(const QKeyEvent& ke)
{
  DEBUG_TRACEFUNC();
  if (ke.count() != 1 || ke.text().isEmpty()) {
    clear();
    return FALSE;
  }  

  switch (_stat) {
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:  
    decideAllSegments();
    // no break

  case NONE_YOMI:
  case INPUTING_YOMI:
    if ((_mode.id() & Mode_KanaInput) && !(_mode.id() & Mode_ZenkakuEisu)) {
      // case Kana-Input
      _inputkeys = QString::null;
      if (ke.state() == Keypad && Config::readEntry("_cmbtenkey", "") == tr("Ⱦ")) {
        // case Hanaku Ten key
        QString str = _yomiseglist.first() + ke.text();
        _yomiseglist.clear();
        _yomiseglist << str;
      } else {
        QString  yomi = _yomiseglist.first() + _dict.convert( ke );
        yomi = _dict.convert(yomi, InputMode(Mode_Hiragana));  // ʸѴ
        _yomiseglist = _dict.convert(yomi, _mode);
      }

    } else {
      // case Zenkaku Eisu or Roma-Input
      _inputkeys += ke.text();
      if (ke.state() == Keypad && Config::readEntry("_cmbtenkey", "") == tr("Ⱦ")) {
        // case Hanaku Ten key
        QString str = _yomiseglist.first() + ke.text();
        _yomiseglist.clear();
        _yomiseglist << str;
      } else {
        _yomiseglist = _dict.convert(_inputkeys, _mode);
      }
    }
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = INPUTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::deleteBackward()
{
  DEBUG_TRACEFUNC();
  qDebug("current Yomigana:%s", _yomiseglist.join(" ").local8Bit().data());

  QString str;
  switch (_stat) {
  case NONE_YOMI:
    // Do noting
    return TRUE;
    break;
    
  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    str = forceConvtYomigana(_mode);
    str.truncate(str.length() - 1);
    _yomiseglist.clear();
    if ( !str.isEmpty() ) {
      _yomiseglist = str;
    }
    syncInputKeysBackward( str );    
    qDebug("current Yomigana:%s", _yomiseglist.join("").local8Bit().data());
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  // Sets status
  _stat = (_yomiseglist.isEmpty()) ? NONE_YOMI : INPUTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::decideCrntSegment()
{
  DEBUG_TRACEFUNC();
 
  QString str;
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // Emit segment decided
    emit  decideSegments(_preedit->backwardText() + _preedit->attentionText());
    _minsegidx = ++_crntsegidx;
    if ( _convtkanjilist[_crntsegidx].isEmpty() ) {
      qDebug("no more kanji to to converted");
      if (_eng)
	_eng->endConvert(_convtkanjilist);
      clear();
      return TRUE;
    }

    qDebug("Segment minimum index: %d", _minsegidx);
    qDebug("Number of segments remaining: %d : %s", _convtkanjilist.count() - _minsegidx, _convtkanjilist.join(" ").local8Bit().data());
    
    // Sync input-keys
    for (int i = _crntsegidx; i < (int)_yomiseglist.count(); ++i) {
      str += _yomiseglist[i];
    }
    syncInputKeysForward(str);
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::decideAllSegments()
{
  DEBUG_TRACEFUNC();
  emit decideSegments(_preedit->text());
  if (_eng)
    _eng->endConvert(_convtkanjilist);
  clear();
  return TRUE;
}


bool
KanjiConvert::cancelConversion()
{
  DEBUG_TRACEFUNC();

  QString yomi;  
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if ((_mode.id() & Mode_KanaInput) && !(_mode.id() & Mode_ZenkakuEisu)) {
      // case Kana-Input
      _yomiseglist = _yomiseglist.join("");

    } else {
      // case Roma-Input
      for (int i = _minsegidx; i < (int)_yomiseglist.count(); ++i) {
	yomi += _yomiseglist[i];
      }
      
      // Reset input-keys
      if (_minsegidx > 0) {
	_inputkeys = _dict.convert(yomi, Mode_HankakuEisu);
      }
      
      _yomiseglist = _dict.convert(_inputkeys, _mode);
    }

    // send End Convert message
    if (_eng)
      _eng->endConvert(); 
    hideToolBox();
    _crntsegidx = _minsegidx = 0;
    _convtkanjilist.clear();
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = INPUTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool    
KanjiConvert::lengthenSegment()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_yomiseglist[_crntsegidx + 1].isEmpty()) {
      // Do nothing
      return TRUE;

    } else {
      QStringList convstr, yomiseg;
      _eng->resizeSegment(_crntsegidx, KanjiEngine::LENGTHEN_1CHAR, convstr, yomiseg);
      setConvertSegments(_crntsegidx, yomiseg);
      setYomigana(_crntsegidx, yomiseg);
    }
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  _candlist->hide();
  _stat = CHANGING_LENGTH;
  showPreeditArea();
  return TRUE;
}


bool    
KanjiConvert::shortenSegment()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_yomiseglist[_crntsegidx].length() <= 1) {
      // Do nothing
      return TRUE;

    } else {
      QStringList convstr, yomiseg;
      _eng->resizeSegment(_crntsegidx, KanjiEngine::SHORTEN_1CHAR, convstr, yomiseg);
      setConvertSegments(_crntsegidx, yomiseg);
      setYomigana(_crntsegidx, yomiseg);
    }
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CHANGING_LENGTH;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::forwardSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_crntsegidx < _convtkanjilist.count() - 1)
      _crntsegidx++;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::backwardSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_crntsegidx > _minsegidx)
      _crntsegidx--;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::toFirstSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _crntsegidx = _minsegidx;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::toLastSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _crntsegidx = _yomiseglist.count() - 1;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::inputSpaceChar()
{
  DEBUG_TRACEFUNC();

  bool res = TRUE;
  QString s;
  switch (_stat) {
  case NONE_YOMI:
    s = Config::readEntry("_cmbspacekey", "");
    if (s == tr("Ⱦ")) {
      res = FALSE;        // Forward key-event for xemacs bug?
    } else if (s == tr("")) {
      emit decideSegments(tr(""));  // Multibyte space
    }
    break;

  case INPUTING_YOMI:    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  clear();
  return res;
}


bool
KanjiConvert::convertToHira()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  QStringList dummy;
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _convtkanjilist = forceConvtYomigana(Mode_Hiragana);
    _eng->beginConvert(_convtkanjilist.join(""), dummy, _yomiseglist);
    _eng->resizeSegment(0, _yomiseglist.join("").length(), dummy, _yomiseglist);   // Resize max length
    Q_ASSERT(_yomiseglist.count() == 1);
    break;
 
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // convert first segment to Hiragara
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());
    // Replace string
    setConvertSegments(_crntsegidx, _yomiseglist[_crntsegidx]);
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::convertToKata()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  QStringList dummy;
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _convtkanjilist = forceConvtYomigana(Mode_Katakana);
    _eng->beginConvert(forceConvtYomigana(Mode_Hiragana), dummy, _yomiseglist);
    _eng->resizeSegment(0, _yomiseglist.join("").length(), dummy, _yomiseglist);   // Resize max length
    Q_ASSERT(_yomiseglist.count() == 1);
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // convert first segment to Katakana
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());
    // Replace string
    setConvertSegments(_crntsegidx, _dict.forceConvt(_yomiseglist[_crntsegidx], InputMode(Mode_Katakana)));
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::convertToHankakuKana()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  QStringList dummy;
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _convtkanjilist = forceConvtYomigana(Mode_HankakuKana);
    _eng->beginConvert(forceConvtYomigana(Mode_Hiragana), dummy, _yomiseglist);
    _eng->resizeSegment(0, _yomiseglist.join("").length(), dummy, _yomiseglist);   // Resize max length
    Q_ASSERT(_yomiseglist.count() == 1);
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // convert first segment to Katakana
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());
    // Replace string
    setConvertSegments(_crntsegidx, _dict.forceConvt(_yomiseglist[_crntsegidx], InputMode(Mode_HankakuKana)));
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::convertToHankakuEisu()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  QStringList dummy;
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _convtkanjilist = forceConvtYomigana(Mode_HankakuEisu);
    _eng->beginConvert(forceConvtYomigana(Mode_Hiragana), dummy, _yomiseglist);
    _eng->resizeSegment(0, _yomiseglist.join("").length(), dummy, _yomiseglist);   // Resize max length
    Q_ASSERT(_yomiseglist.count() == 1);
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());
    if (!_crntsegidx && _convtkanjilist.count() == 1 && !_inputkeys.isEmpty()) {
      setConvertSegments(_crntsegidx, _inputkeys);
    } else {
      setConvertSegments(_crntsegidx, _dict.convert(_yomiseglist[_crntsegidx], Mode_HankakuEisu));
    }
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::convertToZenkakuEisu()
{
  DEBUG_TRACEFUNC();
  if (!_eng)
    return FALSE;

  QStringList dummy;
  //  QStringList::iterator it;
  QString str;
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;
    
  case INPUTING_YOMI:
    _convtkanjilist = forceConvtYomigana(Mode_ZenkakuEisu);
    _eng->beginConvert(forceConvtYomigana(Mode_Hiragana), dummy, _yomiseglist);
    _eng->resizeSegment(0, _yomiseglist.join("").length(), dummy, _yomiseglist);   // Resize max length
    Q_ASSERT(_yomiseglist.count() == 1);
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    if (!_crntsegidx && _convtkanjilist.count() == 1 && !(_mode.id() & Mode_KanaInput)) { 
      str = _inputkeys;
    } else {
      str = _dict.convert(_yomiseglist[_crntsegidx], Mode_HankakuEisu);
    }

    // Replace string of preeidt area
    setConvertSegments(_crntsegidx, _dict.forceConvt(str, InputMode(Mode_ZenkakuEisu)));
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI; 
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::switchZenkakuEisu()
{
  DEBUG_TRACEFUNC();
  
  QString s;
  switch (_stat) {
  case NONE_YOMI:
    if (_mode.id() & Kimera::Mode_ZenkakuEisu) {
      s = Config::readEntry("_cmbinputstyle", "");
      if (s == tr("޻")) {
	setInputMode(Kimera::Mode_RomaInput);
	
      } else if (s == tr("")) {
	setInputMode(Kimera::Mode_KanaInput);
      }
      
    } else {
      setInputMode(Kimera::Mode_ZenkakuEisu);
    }
    break;

  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = NONE_YOMI;
  clear();
  return TRUE;
}


void
KanjiConvert::setConvertSegments(int idx, const QString& str)
{
  DEBUG_TRACEFUNC("list: %s  idx: %d", str.local8Bit().data(), idx);
  CHECK_NOCHANGE(_stat, int);
  
  if (str.isEmpty() || idx < 0 || idx >= (int)_convtkanjilist.count()) {
    Q_ASSERT(0);
    return;
  }
  
  QStringList::iterator it = _convtkanjilist.begin();
  it += idx;
  *it = str;
}


void
KanjiConvert::setConvertSegments(int idx, const QStringList& convseg)
{
  DEBUG_TRACEFUNC("list: %s  idx: %d", convseg.join(" ").local8Bit().data(), idx);
  CHECK_NOCHANGE(_stat, int);

  qDebug("candidacies(%d) :%s  (crnt:%d  min:%d)", convseg.count(), convseg.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);  
  if (convseg.isEmpty() || idx < 0 || idx >= (int)_convtkanjilist.count()) {
    Q_ASSERT(0);
    return;
  }

  // Sets converted kanji list
  QStringList::iterator it = _convtkanjilist.begin();
  it += idx;
  _convtkanjilist.erase(it, _convtkanjilist.end());  // Erase
  for (int i = 0; i < (int)convseg.count(); ++i) {
    if ( !convseg[i].isEmpty() )
      _convtkanjilist << convseg[i];   // insert str
  }
}


void
KanjiConvert::setYomigana(int idx, const QStringList& yomiseg)
{
  DEBUG_TRACEFUNC("yomiseg:%s  idx:%d", yomiseg.join(" ").local8Bit().data(), idx);
  CHECK_NOCHANGE(_crntsegidx, int);
  CHECK_NOCHANGE(_convtkanjilist, QStringList);

  if (yomiseg.isEmpty() || idx < 0 || idx >= (int)_yomiseglist.count()) {
    Q_ASSERT(0);
    return;
  }

  // Sets converted kanji list
  QStringList::iterator it = _yomiseglist.begin();
  it += idx;
  _yomiseglist.erase(it, _yomiseglist.end());  // Erase
  for (int i = 0; i < (int)yomiseg.count(); ++i)
    if ( !yomiseg[i].isEmpty() )
      _yomiseglist << yomiseg[i];    // append str
  
  // Show segments
  qDebug("yomi: %s  convt:%s (crnt:%d min:%d)", _yomiseglist.join(" ").local8Bit().data(),
	 _convtkanjilist.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
}


void 
KanjiConvert::slotDecideSegment(QListBoxItem* item)
{
  Q_CHECK_PTR( item );
  DEBUG_TRACEFUNC("item: %s", item->text().local8Bit().data());
  CHECK_NOCHANGE(_yomiseglist, QStringList);

  if (_stat != SHOWING_LIST)
    return;

  // Replace string
  setConvertSegments(_crntsegidx, item->text());
  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
}


void
KanjiConvert::hideToolBox()
{
  DEBUG_TRACEFUNC();
  _preedit->hide();
  _candlist->hide();
}


bool
KanjiConvert::processKeyEvent(const QKeyEvent& key)
{
  DEBUG_TRACEFUNC("The key code: 0x%x   key count: %d  ascii: %s", key.key(), key.count(), key.text().data());

  int k = key.key();
  k |= (key.state() & Qt::ControlButton) ? Qt::CTRL : 0;
  k |= (key.state() & Qt::ShiftButton) ? Qt::SHIFT : 0;
  k |= (key.state() & Qt::AltButton) ? Qt::ALT : 0;

  bool res = TRUE;
  FuncID  fid = KeyAssigner::functionID( k, _stat );
  switch ( fid ) {
  case FID_AppendChar:
    res = appendChar( key );
    break;
    
  case FID_None:
    // Do nothing
    break;

  case FID_ForwardKeyEvent:
    res = (_stat == NONE_YOMI) ? FALSE : TRUE;
    break;

  default:
    pmfunc  pf = funcptrdict[fid]->ptr();  // get pointer
    if ( !pf ) {
      qFatal("Incorrect processing  %s:%d", __FILE__, __LINE__);
      clear();
      res = FALSE;
    } else {
      qDebug("FuncName: %s", KeyAssigner::functionName( fid ).local8Bit().data());
      res = (this->*pf)();      // execute function
    }
    break;
  }
  
  return res;
}


void 
KanjiConvert::setPreeditPoint(QPoint p)
{
  DEBUG_TRACEFUNC("x:%d y:%d", p.x(), p.y());
  _preedit->movePreeditPos( p );
}


void
KanjiConvert::showPreeditArea()
{
  DEBUG_TRACEFUNC();
  CHECK_NOCHANGE(_convtkanjilist, QStringList);
  CHECK_NOCHANGE(_yomiseglist, QStringList);

  QStringList list;
  switch (_stat) {
  case NONE_YOMI:
    clear();
    break;

  case INPUTING_YOMI:
    _preedit->showInputingString( _yomiseglist.join("") );
    qDebug("Yomigana:%s  (%s)", _yomiseglist.join("").local8Bit().data(), _inputkeys.local8Bit().data());
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
    if ( !_convtkanjilist.count() ) {
      qDebug("no converted kanji list");
      return;
    } else if (_minsegidx >= _convtkanjilist.count()) {
      qFatal("Bad parameter  _minsegidx:%d  _convtkanjilist.count():%d  %s:%d",
	     _minsegidx, _convtkanjilist.count(), __FILE__, __LINE__);
      return;
    }    

    for (int i = _minsegidx; i < (int)_convtkanjilist.count(); i++)
	list << _convtkanjilist[i];

    qDebug("showConvertingSegments : %s  (%d:%d)", list.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
    _preedit->showConvertingSegments(list, _crntsegidx - _minsegidx);
    break;

  case CHANGING_LENGTH:
    if ( !_convtkanjilist.count() ) {
      qDebug("no converted kanji list");
      return;
    } else if (_minsegidx >= _yomiseglist.count()) {
      qFatal("Bad parameter  _minsegidx:%d  _convtkanjilist.count():%d  %s:%d",
	     _minsegidx, _convtkanjilist.count(), __FILE__, __LINE__);
      return;
    }
    
    for (int i = _minsegidx; i < (int)_crntsegidx; i++)
      list << _convtkanjilist[i];
    
    for (int i = _crntsegidx; i < (int)_yomiseglist.count(); i++)
      list << _yomiseglist[i];
    
    qDebug("showChangingSegmentLength : %s  (%d:%d)", list.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
    _preedit->showChangingSegmentLength(list, _crntsegidx - _minsegidx);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }
}


//
// @param: str: yomogana string
//
void
KanjiConvert::syncInputKeysForward(const QString& str)
{
  DEBUG_TRACEFUNC("str: %s", str.local8Bit().data());

  if (_inputkeys.isEmpty()) {
    return;
  }
  if (str.isEmpty()) {
    _inputkeys = QString::null;
    return;
  }

  int len;
  QString convt;
  for (len = _inputkeys.length(); len > 0; --len) {
    convt = _dict.convert(_inputkeys.right(len), _mode.id());
    if (convt == str) {
      _inputkeys = _inputkeys.right(len);
      break;
    }
  }

  if (convt != str) {
    _inputkeys = _dict.convert(str, Mode_HankakuEisu);
  }
  qDebug("%s  str:%s  input-keys:%s", __func__, str.local8Bit().data(), _inputkeys.local8Bit().data());
}


void
KanjiConvert::syncInputKeysBackward(const QString& str)
{
  DEBUG_TRACEFUNC("str: %s", str.local8Bit().data());

  if (_inputkeys.isEmpty()) {
    return;
  }
  if (str.isEmpty()) {
    _inputkeys = QString::null;
    return;
  }

  int len;
  QString convt;
  for (len = _inputkeys.length(); len > 0; --len) {
    convt = _dict.convert(_inputkeys.left(len), _mode.id());
    if (convt == str) {
      _inputkeys = _inputkeys.left(len);
      break;
    }
  }

  if (convt != str) {
    _inputkeys = _dict.convert(str, Mode_HankakuEisu);
    Q_ASSERT(_inputkeys.local8Bit().length() == _inputkeys.length());
  }
  qDebug("syncInputKeys : %s  _inputkeys: %s", str.local8Bit().data(), _inputkeys.local8Bit().data());
  qDebug("%s  str:%s  input-keys:%s", __func__, str.local8Bit().data(), _inputkeys.local8Bit().data());
}


QString
KanjiConvert::forceConvtYomigana(const InputMode& m) const
{
  QString res;
  if ((_mode.id() & Mode_KanaInput) && !(_mode.id() & Mode_ZenkakuEisu)) {
    res = (m.id() & Mode_Hiragana) ? _yomiseglist.join("") : _dict.convert(_yomiseglist.join(""), m);
  } else {
    res = _dict.forceConvt(_inputkeys, m.id());
  }
  return res;
}
