#include "dicthiragana.h"
#include "inputmode.h"
#include "config.h"
#include "debug.h"
#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtextstream.h>
#include <qregexp.h>
using namespace Kimera;

DictHiragana::DictHiragana() : _dicthira(QDict<QString>(17, FALSE)),
			       _dictkata(QDict<QString>()),
			       _dicthankana(QDict<QString>()),
			       _dictalphbt(QDict<QString>()),
			       _dictsymbol(QDict<QString>()),
			       _dictkanainp(QDict<QString>(17, FALSE)),
			       _dictdakuten(QDict<QString>()),
			       _reversedict(QDict<QString>())
{
  _dicthira.setAutoDelete(TRUE);
  _dictkata.setAutoDelete(TRUE);
  _dicthankana.setAutoDelete(TRUE);
  _dictalphbt.setAutoDelete(TRUE);
  _dictsymbol.setAutoDelete(TRUE);
  _dictkanainp.setAutoDelete(TRUE);
  _dictdakuten.setAutoDelete(TRUE);
  _reversedict.setAutoDelete(TRUE);
}


DictHiragana::~DictHiragana() { }


void
DictHiragana::init()
{
  _dicthira.clear();
  _dictkata.clear();
  _dicthankana.clear();
  _dictalphbt.clear();
  _dictsymbol.clear();
  _dictkanainp.clear();
  _dictdakuten.clear();

  // Copy dictionary files
  copyDictFile("hiragana.dic");
  copyDictFile("katakana.dic");
  copyDictFile("hankakukana.dic");
  copyDictFile("zenkakualphabet.dic");
  copyDictFile("numeralsymbols.dic");
  copyDictFile("kanainput.dic");
  copyDictFile("dakuten.dic");

  QString dictdir = Config::dictionaryPath();
  initDict(_dicthira, dictdir + "/hiragana.dic");          // Ҥ餬ʼ
  initDict(_dictkata, dictdir + "/katakana.dic");          // ʼ
  initDict(_dicthankana, dictdir + "/hankakukana.dic");    // Ⱦѥʼ
  initDict(_dictalphbt, dictdir + "/zenkakualphabet.dic"); // ѱѻ
  initDict(_dictsymbol, dictdir + "/numeralsymbols.dic");  // ѿ漭
  initDict(_dictkanainp, dictdir + "/kanainput.dic");      // ϼ
  initDict(_dictdakuten, dictdir + "/dakuten.dic");        // 

  // 桦
  _dictsymbol.insert(",", new QString(Config::readEntry("_cmbtouten", ","))); 
  _dictsymbol.insert(".", new QString(Config::readEntry("_cmbkuten", ".")));
  _dictsymbol.insert("/", new QString(Config::readEntry("_cmbsymbol", "/")));
  _dictsymbol.insert("[", new QString(Config::readEntry("_cmbbracket", "[").left(1)));
  _dictsymbol.insert("]", new QString(Config::readEntry("_cmbbracket", "]").right(1)));

  // հ
  initDict(_reversedict, dictdir + "/hiragana.dic", TRUE);
  initDict(_reversedict, dictdir + "/zenkakualphabet.dic", TRUE);
  initDict(_reversedict, dictdir + "/numeralsymbols.dic", TRUE);
  initDict(_reversedict, dictdir + "/hankakukana.dic", TRUE);
  initDict(_reversedict, dictdir + "/katakana.dic", TRUE);
  _reversedict.replace(Config::readEntry("_cmbtouten", ","), new QString(","));
  _reversedict.replace(Config::readEntry("_cmbkuten", "."), new QString("."));
  _reversedict.replace(Config::readEntry("_cmbsymbol", "/"), new QString("/")); 
  _reversedict.replace(Config::readEntry("_cmbbracket", "[").left(1), new QString("["));
  _reversedict.replace(Config::readEntry("_cmbbracket", "]").right(1), new QString("]")); 
  _reversedict.replace(QObject::tr(""), new QString("@"));
  _reversedict.replace(QObject::tr(""), new QString("["));
  _reversedict.replace(QObject::tr(""), new QString("@"));
  _reversedict.replace(QObject::tr(""), new QString("["));
}


void
DictHiragana::initDict(QDict<QString>& dict, const QString& dictfile, bool reverse_dict)
{
  QFile file(dictfile);
  if ( !file.open(IO_ReadOnly) ) {
    QMessageBox::critical(0, "File open error", "Cannot open file: " + dictfile,
    			  QMessageBox::Ok | QMessageBox::Default, 0);
    qApp->quit();
    return;
  }
    
  QTextStream  stream(&file);
  stream.setEncoding(QTextStream::Latin1);

  while ( !stream.eof() ) {
    QString line;
    line = QString::fromLocal8Bit(stream.readLine());
    
    if ( !line.isEmpty() && !line.contains(QRegExp("^[ \t]")) && 
	 !line.contains(QRegExp("^##")) ) {
      QString keystr, itemstr;
      int idx = line.find(QRegExp("[ \t]"));
      if (idx > 0) {
	keystr = line.left(idx);	
	line.remove(0, keystr.length());
	itemstr = line.replace(QRegExp("[ \t]"), "");
	if ( !reverse_dict ) {
	  if ( !dict.find(keystr) )
	    dict.replace(keystr, new QString(itemstr));  
	
	} else {
	  // reverse-dictionary
	  if ( !dict.find(itemstr) )
	    dict.insert(itemstr, new QString(keystr));
	}
      }
    }
  }
  
  file.close();
}


// ʿ̾Ѵ (ʸ˹碌ƣʸˤ :->  )
// parameter:    Roma string (only ASCII characters) or Hiragana string
// return value: Hiragana string
QString
DictHiragana::convertToHira(const QString& src) const
{
  // check parameter
  if ( src.isEmpty() )
    return QString::null;

  QString dest;
  if (src.local8Bit().length() != src.length()) {
    // Ѵ
    if (src.length() == 1) 
      return src;

    uint i = 0;
    while (i < src.length()) {
      QString* pstr;
      pstr = _dictdakuten.find( src.mid(i, 2) );
      
      if ( pstr ) {
	dest += *pstr;
	i++;
      } else {
	dest += src.constref(i);
      }
      i++;
    }
    
    return dest;

  }  
  
  uint index = 0;
  while (index < src.length()) {
    for (int len = 4; len > 0; --len) {
      QString*  pstr;
      // Ҥ餬ʸ
      pstr = _dicthira.find( src.mid(index, len) );
      if (pstr) {
	dest += *pstr;
	index += len;
	break;
      } 

      // ѿ渡
      pstr = _dictsymbol.find( src.mid(index, len) );
      if (pstr) {
	dest += *pstr;
	index += len;
	break;
      }
      
      if (len == 1) {
	if (src.constref(index) == src.constref(index + 1)
	    && src.mid(index, 1).find( QRegExp("[^aiueo]", FALSE) ) == 0
	    && src.mid(index + 2, 1).find( QRegExp("[aiueoyhsw]", FALSE) ) == 0 ) {
	  // ''Ѵ
	  pstr = _dicthira.find( src.mid(index + 1, 2) );
	  if (pstr) {
	    dest += QObject::tr("") + *pstr;
	    index += 3;
	  } else {
	    dest += QObject::tr("");
	    ++index;
	  }

	} else if ((src.constref(index) == "n" || src.constref(index) == "N")
		   && src.mid(index + 1, 1).find( QRegExp("[^aiueoy]", FALSE) ) == 0) {
	  // ''Ѵ
	  dest += QObject::tr("");
	  ++index;
	  
	} else {
	  // ̵Ѵ
	  dest += src.constref(index);
	  ++index;
	}
	break;
      }
    }
  }
  
  return dest;
}


// convertToHira ƤӽФ塤Ǹʸ'n'ä '' Ѵ
//
QString
DictHiragana::forceConvtToHira(const QString& src) const
{
  if ( src.isEmpty() )
    return QString::null;

  QString dest = convertToHira(src);
  if (dest.right(1) == "n" || dest.right(1) == "N") {
    dest.remove(dest.length() - 1, 1);
    dest += QString::fromLocal8Bit("");
  }
  
  return dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString
DictHiragana::convertToKata(const QString& src) const
{
  Q_ASSERT(!_dictkata.isEmpty());

  QString dest;
  // check parameter
  if (src.local8Bit().length() == src.length()) {
    QString hira = convertToHira(src);
    for (int j = 0; j < (int)hira.length(); j++) {
      QString* pstr = _dictkata.find( hira.constref(j) );
      if ( pstr ) {
	dest += *pstr;
      } else {
	dest += hira.constref(j);
      }
    }
    
  } else {
    // Hiragana -> Katakana
    for (int i = 0; i < (int)src.length(); i++) {
      QString* pstr = _dictkata.find( src.constref(i) );
      if ( pstr ) {
	dest += *pstr;
      } else {
	dest += src.constref(i);
      }
    }
  }
  
  return  dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString
DictHiragana::forceConvtToKata(const QString& src) const
{
  return convertToKata( forceConvtToHira(src) );
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString
DictHiragana::convertToHankakuKana(const QString& src) const
{
  Q_ASSERT(!_dicthankana.isEmpty());

  QString dest;
  // check parameter
  if (src.local8Bit().length() == src.length()) {
    QString hira = convertToHira(src);
    for (int j = 0; j < (int)hira.length(); j++) {
      QString* pstr = _dicthankana.find( hira.constref(j) );
      if ( pstr ) {
	dest += *pstr;
      } else {
	dest += hira.constref(j);
      }
    }
    
  } else {
    // Hiragana -> HankakuKana
    for (int i = 0; i < (int)src.length(); i++) {
      QString* pstr = _dicthankana.find( src.constref(i) );
      if ( pstr ) {
	dest += *pstr;
      } else {
	dest += src.constref(i);
      }
    }
  }
  
  return  dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString
DictHiragana::forceConvtToHankakuKana(const QString& src) const
{
  return convertToHankakuKana( forceConvtToHira(src) );
}


// parameter: Roma string (only ASCII characters)
QString
DictHiragana::convertToZenkakuEisu(const QString& src) const
{  
  Q_ASSERT(!_dictalphbt.isEmpty());
  Q_ASSERT(!_dictsymbol.isEmpty());

  QString dest;
  for (int i = 0; i < (int)src.length(); i++) {
    QString* pstr;
    if ( (pstr = _dictalphbt.find(src.constref(i))) ) {  // if the key exists
      dest += *pstr;
      
    } else if ( (pstr = _dictsymbol.find(src.constref(i))) ) {  // if the key exists 
      dest += *pstr;
      
    } else {
      dest += src.constref(i);
    }
  }
  
  return dest;
}


QString
DictHiragana::convert(const QString& roma, const InputMode& mode) const
{
  QString res;
  switch (mode.id() & Mode_ModeMask) {
  case Mode_Hiragana:
    res = convertToHira(roma);
    break;
    
  case Mode_Katakana:
    res = convertToKata(roma);
    break;

  case Mode_HankakuKana:
    res = convertToHankakuKana(roma);
    break;

  case Mode_ZenkakuEisu:
    if (roma.local8Bit().length() != roma.length()) {
      // Multibyte chars
      res = convertToZenkakuEisu( reverseConvt(roma) );
    } else {
      res = convertToZenkakuEisu(roma);
    }
    break;
    
  case Mode_HankakuEisu:
    res = reverseConvt(roma);
    break;

  default:
    Q_ASSERT(0);
    break;
  }
  
  return res;
}


QString
DictHiragana::forceConvt(const QString& roma, const InputMode& mode) const
{
  QString res;
  switch (mode.id() & Mode_ModeMask) {
  case Mode_Hiragana:
    res = forceConvtToHira(roma);
    break;
    
  case Mode_Katakana:
    res = forceConvtToKata(roma);
    break;

  case Mode_HankakuKana:
    res = forceConvtToHankakuKana(roma);
    break;
    
  case Mode_ZenkakuEisu:
    if (roma.local8Bit().length() != roma.length()) {
      // Multibyte chars
      res = convertToZenkakuEisu( reverseConvt(roma) );     
    } else {
      res = convertToZenkakuEisu(roma);
    }
    break;
    
  case Mode_HankakuEisu:
    res = reverseConvt(roma);
    break;

  default:
    Q_ASSERT(0);
    break;
  }

  return res;
}


//  Ѵ
//  params: QKeyEvent
//  params: Input mode
//  return: Hiragana string
QString
DictHiragana::convert(const QKeyEvent& key, const InputMode& mode) const
{
  if ( !(mode.id() & Mode_KanaInput) ) {
    return QString::null;

  } else if (mode.id() & Mode_ZenkakuEisu) {
    return convertToZenkakuEisu(key.text());
  }

  Q_ASSERT(!_dictkanainp.isEmpty());
  QString keystr;
  if (key.state() == Qt::NoButton) {
    keystr = key.text();
    
  } else if (key.state() == Qt::ShiftButton) { 
    keystr = QString("Shift+") + key.text();
    
  } else if (key.state() == Qt::Keypad) {
    keystr = QString("Keypad+") + key.text();
    
  } else {
    return QString::null;
  }
  
  QString* pres = _dictkanainp[keystr];
  if ( !pres )
    return QString::null;

  QString res;
  switch (mode.id() & Mode_ModeMask) {
  case Mode_Hiragana:
    res = *pres;
    break;
    
  case Mode_Katakana:
    res = convertToKata(*pres);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return res;
}


QString
DictHiragana::reverseConvt(const QString& hira) const
{
  QString res;
  int i = 0;
  while (i < (int)hira.length()) {
    QChar c = hira[i];
    if ( c.latin1() ) {   // Check ascii or multibyte-char
      // Ascii char
      res += c;
      ++i;
    } else {
      // Multibyte char
      QString* pstr;
      int len;
      for (len = 2; len > 0; --len)
	if ( (pstr = _reversedict.find(hira.mid(i, len))) )
	  break;
      
      if (pstr) {
	res += *pstr;
	i += len;
      } else {
	res += c;
	++i;
      }
    }
  }
  
  return (res == hira) ? res : reverseConvt(res);
}


void
DictHiragana::copyDictFile(const QString dicname)
{
  QFile src(QString(INSTALL_PATH "/dic/") + dicname);
  QDir  dir(Config::dictionaryPath());
  QFile dst( dir.filePath(dicname) );

  if( !dst.exists() ) {  
    if ( !src.exists() || !src.open(IO_ReadOnly) ) {
      qFatal("can not open read-mode file: %s", dicname.ascii());
      return;
    }
    
    if ( !dst.open(IO_WriteOnly) ) {
      qFatal("can not open write-mode file: %s", dicname.ascii());
      return;
    }
    
    QTextStream tssrc(&src);
    QTextStream tsdst(&dst);
    tssrc.setEncoding(QTextStream::Latin1);
    tsdst.setEncoding(QTextStream::Latin1);
    tsdst << tssrc.read();
  }
}
