#include "sockkanjiclient.h"
#include "config.h"
#include <qapplication.h>
#include <qmessagebox.h>
#include <qcstring.h>
#include <qdatastream.h>
#include <qtextcodec.h>
#include <qsocket.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdlib.h>

const QString CANNA_AF_UNIX_PATH = "/tmp/.iroha_unix/IROHA";


SockKanjiClient::SockKanjiClient() 
  : _contxt(0), _crntidx(0), _convertflag(FALSE), _strlist(QStringList()), _diclist(QStringList())
{
  _sock = new QSocket();
  connect(_sock, SIGNAL(connected()), this, SLOT(slotConnected()));
  connect(_sock, SIGNAL(error(int)), this, SLOT(slotDetectError()));
  connect(_sock, SIGNAL(connectionClosed()), this, SLOT(slotConnectionClosed()));
  connect(_sock, SIGNAL(readyRead()), this, SLOT(slotDispatch()));
}


SockKanjiClient::~SockKanjiClient()
{
  delete _sock;
}


// connect to Kanji server
void 
SockKanjiClient::init() 
{
  _sock->close();

  if ( Config::readBoolEntry("_chkremote", FALSE) ) {
    // INET Domain
    qDebug("INET Domain socket");
    _sock->connectToHost(Config::readEntry("_edtsvrname", "localhost"), 
			 Config::readEntry("_edtport", "0").toUInt());

  } else {
    // UNIX Domain
    qDebug("UNIX Domain socket");

    int    fd;
    struct sockaddr_un  addr;
    
    if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
      QMessageBox::critical(0, "Kanji Server Error", 
			    "Socket Create Error",
			    QMessageBox::Ok | QMessageBox::Default, 0);
      return;
    }
    
    memset((char *)&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    strcpy(addr.sun_path, CANNA_AF_UNIX_PATH.latin1());
 
    // connect
    if (::connect(fd, (sockaddr *)&addr, sizeof(addr)) < 0){
      QMessageBox::critical(0, "Kanji Server Error", 
			    "Socket Connect Error",
			    QMessageBox::Ok | QMessageBox::Default, 0);
      ::close(fd);
      return;
    }
    
    _sock->setSocket(fd);  // UNIX Domain socket set
    slotConnected();
  }
}


void 
SockKanjiClient::slotConnected() 
{
  qDebug("socket: %d", _sock->socket());
  sendInitialize();
}


void 
SockKanjiClient::slotDetectError() 
{
  QMessageBox::critical(0, "Kanji Server Error", 
			"Socket Error!\nConnects to localhost by UNIX domain.",
			QMessageBox::Ok | QMessageBox::Default, 0);
  
  Config::writeEntry("_chkremote", FALSE);
  init();
}


void
SockKanjiClient::slotConnectionClosed()
{
  QMessageBox::critical(0, "Kanji Server Connection Closed",
			"Socket Connection Closed!\nConnects to localhost by UNIX domain.",
			QMessageBox::Ok | QMessageBox::Default, 0);
  
  Config::writeEntry("_chkremote", FALSE);
  init();
}


void 
SockKanjiClient::sendInitialize()
{
  QByteArray  pack;
  QDataStream ds(pack, IO_WriteOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  QCString user("3.1:");
  user += QCString(getenv("USER"));
  ds << 1L << 0L;
  ds.writeRawBytes(user.data(), user.length());

  _sock->writeBlock(pack.data(), pack.size());
}


void 
SockKanjiClient::recvInitializeReply()
{
  uint len;
  if ((len = _sock->size()) != 4) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
  } 
  
  QByteArray respack(len);
  QDataStream ds(respack, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  // response packet recv
  if (_sock->readBlock(respack.data(), len) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
  }
  
  int res;
  ds >> res;
  if (res == -1 || res == -2) {
    int ret = QMessageBox::critical(0, "Protocol version error", 
				    "Protoc0l version error",
				    QMessageBox::Ok | QMessageBox::Default, 0);
    if (ret) {
      qApp->quit();
    }
  } 

  QDataStream dst(respack, IO_ReadOnly);
  dst.setByteOrder(QDataStream::BigEndian);
  u_int16_t  minor;
  dst >> minor >> _contxt;
  qDebug("minor version: %d  context: %d", minor, _contxt);

  getDictionaryList();  
}


void
SockKanjiClient::slotDispatch()
{
  if (_sock->bytesAvailable() == 4) {
    recvInitializeReply();
    return;
  }

  QByteArray pack(2);
  if (_sock->readBlock(pack.data(), 2) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
  }
  
  switch (pack.at(0)) {
  case 0x06:
    recvGetDictionaryListReply();
    break;

  case 0x08:
    recvMountDictionaryReply();
    break;

  case 0x0f:
    recvBeginConvertReply();
    break;

  case 0x10:
    recvEndConvertReply();
    break;

  case 0x11:
    recvGetCandidacyListReply();
    break;
    
  case 0x12:
    recvGetYomiReply();
    break;

  case 0x1a:
    recvResizePause();
    break;
    
  default:
    qDebug("unsupported:%d", pack.at(0)); 
    break;
  }
}



void 
SockKanjiClient::getDictionaryList()
{
  QByteArray  pack;
  QDataStream ds(pack, IO_ReadWrite);
  ds.setByteOrder(QDataStream::BigEndian);
  
  ds << (uchar)0x06 << (uchar)0 << (u_int16_t)4 << _contxt << (u_int16_t)4096;
  _sock->writeBlock(pack.data(), pack.size());
}


void 
SockKanjiClient::recvGetDictionaryListReply()
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
  }

  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  int16_t len, ndic;
  ds >> len >> ndic;
  
  if (len != datalen - 2) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
  } 
  
  qDebug("number of dictionary: %d", ndic);
  
  QString dic;
  for (int i = 0; i < ndic; i++) {
    dic = QString();
    int8_t s;
    while (1) {
      ds >> s;
      if ( !s ) break;
      dic += s;
    }
    
    if (!dic.isEmpty())
      _diclist << dic;
  }

  mountDictionary();
}


void 
SockKanjiClient::mountDictionary()
{
  if ( _diclist.isEmpty() ) return;

  QByteArray  pack;
  QDataStream ds(pack, IO_ReadWrite);
  ds.setByteOrder(QDataStream::BigEndian);
  
  QString dic = _diclist.first();
  _diclist.pop_front();
  
  if (dic == "pub") {    // Dosen't mount "pub" dictionary
    if (_diclist.isEmpty()) return;
    dic = _diclist.first();
    _diclist.pop_front();
  }
  
  qDebug("request mount dictionary name: %s", dic.data());    
  ds << (uchar)0x08 << (uchar)0 << (u_int16_t)(dic.length()+7) << 0L << _contxt;
  ds.writeRawBytes(dic.latin1(), dic.length());
  ds << (uchar)0;
  
  _sock->writeBlock(pack.data(), pack.size());
}


void 
SockKanjiClient::recvMountDictionaryReply()
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
  }

  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  int16_t len;
  ds >> len;
  
  if (len != datalen - 2) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
  } 
  
  int8_t  stat;
  ds >> stat;
  if (stat) {
    int ret = QMessageBox::critical(0, "mount dictionary error",
				    "mount dictionary error",
				    QMessageBox::Ok | QMessageBox::Default, 0);
    if (ret) qApp->quit();
  }  
  qDebug("mount dictionary successful.");

  mountDictionary();
}



void
SockKanjiClient::beginConvert(const QString& hira)
{
  if ( _convertflag ) {
    qWarning("%s:_convertflag incorrect", __FUNCTION__);
    return;
  }

  QByteArray  pack;
  QDataStream ds(pack, IO_WriteOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  qDebug("Yomigana size(QCString): %d", hira.local8Bit().length());
  QByteArray str = eucToUint16(hira.local8Bit());
  qDebug("Yomigana size(uint16 array): %d", str.size());

  ds << (uchar)0x0f << (uchar)0 << (u_int16_t)(str.size()+6) << 0L << _contxt;
  ds.writeRawBytes(str.data(), str.size());
  
  _sock->writeBlock(pack.data(), pack.size());

  _crntidx = 0;
  _convertflag = TRUE;  // start converting
  qDebug("beginConvert in progress ...");
}


void 
SockKanjiClient::recvBeginConvertReply()
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
  }

  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  int16_t len;
  ds >> len;
  
  if (len != datalen - 2) {
    qWarning("beginConvert: server response incorrect. (%s)", __FUNCTION__);
  } 

  int16_t nbunsetu;
  ds >> nbunsetu;
  if (nbunsetu <= 0) {
    qDebug("beginConvert: convert error.  %s", __FUNCTION__);
    return;
  }

  _strlist.clear();
  for (int i = 0; i < nbunsetu; i++) {
    QByteArray  ba;
    QDataStream dsba(ba, IO_WriteOnly);
    dsba.setByteOrder(QDataStream::BigEndian);
    u_int16_t us;

    ds >> us;
    while (us) {
      dsba << us;
      ds >> us;
    }

    QString str = (tr(uint16ToEuc(ba)));
    if (!str.isEmpty())
      _strlist << str;
  }

  qDebug("beginConvert successful");
  qDebug("[%s]  %s", __FUNCTION__, _strlist.join(" ").local8Bit().data()); 
  emit converted(_strlist, _crntidx);
}


void 
SockKanjiClient::endConvert()
{
  if ( !_convertflag ) {
    qDebug("%s: unnecessary to end convert", __FUNCTION__);
    return;
  }

  QByteArray  pack;
  QDataStream ds(pack, IO_WriteOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  ds << (uchar)0x10 << (uchar)0 << (u_int16_t)8 << _contxt 
     << (int16_t)0 << 0L;

  _sock->writeBlock(pack.data(), pack.size());

  _strlist.clear();
  _convertflag = FALSE;  // conversion done
  qDebug("%s: done", __FUNCTION__);
}


void 
SockKanjiClient::recvEndConvertReply() const
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
    return;
  }
  
  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  int16_t len;
  ds >> len;
  
  if (len != datalen - 2) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
    return;
  }
  
  int8_t res;
  ds >> res;
  if ( res ) {
    qFatal("End Convert error.  %s", __FUNCTION__);
    return;
  } 
    
  qDebug("endConvert successful");
}


void 
SockKanjiClient::getCandidacyList(int idx)
{
  if ( !_convertflag ) return;
 
  QByteArray  pack;
  QDataStream ds(pack, IO_WriteOnly);
  ds.setByteOrder(QDataStream::BigEndian);

  ds << (uchar)0x11 << (uchar)0 << (u_int16_t)6 << _contxt 
     << (int16_t)idx << (u_int16_t)4096;
  
  _sock->writeBlock(pack.data(), pack.size());
  _crntidx = idx;
}


void 
SockKanjiClient::getYomi(int idx)
{
  if ( !_convertflag ) return;

  QByteArray  pack;
  QDataStream ds(pack, IO_WriteOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  ds << (uchar)0x12 << (uchar)0 << (u_int16_t)6 << _contxt 
     << (int16_t)idx << (u_int16_t)4096;
  
  _sock->writeBlock(pack.data(), pack.size());
  _crntidx = idx;
}


void 
SockKanjiClient::recvGetYomiReply()
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
    return;
  }
  
  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  int16_t len;
  ds >> len;
  
  if (len != datalen - 2) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
    emit yomigana(QString::null, _crntidx);
    return;
  }
  
  int16_t n;
  ds >> n;
  
  QByteArray  ba;
  QDataStream dsba(ba, IO_WriteOnly);
  dsba.setByteOrder(QDataStream::BigEndian);
  u_int16_t us;
  
  ds >> us;
  while (us) {
    dsba << us;
    ds >> us;
  }    
  
  QString str = (tr(uint16ToEuc(ba)));
  emit yomigana(str, _crntidx); 
}


void 
SockKanjiClient::recvGetCandidacyListReply()
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
    return;
  }
  
  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  int16_t len;
  ds >> len;
  
  if (len != datalen - 2) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
    return;
  }

  int16_t ncand;
  ds >> ncand;
  
  if (ncand < 0) {
    qFatal("GetCandidacyList error.  %s", __FUNCTION__);
    return;
  }
  qDebug("Number of Candidates: %d", ncand);
  
  QStringList  listcand;
  for (int i = 0; i < ncand; i++) {
    QByteArray  ba;
    QDataStream dsba(ba, IO_WriteOnly);
    dsba.setByteOrder(QDataStream::BigEndian);
    u_int16_t us;
    
    ds >> us;
    while (us) {
      dsba << us;
      ds >> us;
    }    
    
    QString str = (tr(uint16ToEuc(ba)));
    if ( !str.isEmpty() )
      listcand << str;
  }

  qDebug("[%s]  %s", __FUNCTION__, _strlist.join(" ").local8Bit().data());
  emit candidate(listcand, _crntidx);
}


void
SockKanjiClient::resizePause(int idx, int len)
{
  QByteArray  pack;
  QDataStream ds(pack, IO_WriteOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  ds << (uchar)0x1a << (uchar)0 << (u_int16_t)6 << _contxt 
     << (int16_t)idx << (u_int16_t)len;
  
  _sock->writeBlock(pack.data(), pack.size());
  _crntidx = idx;
}


void 
SockKanjiClient::recvResizePause()
{
  uint datalen = _sock->size();
  QByteArray data(datalen);
  if (_sock->readBlock(data.data(), datalen) < 0) {
    qFatal("Recv error.  %s", __FUNCTION__);
    return;
  }
  
  QDataStream ds(data, IO_ReadOnly);
  ds.setByteOrder(QDataStream::BigEndian);
  
  int16_t len;
  ds >> len;
  
  if (len != datalen - 2) {
    qWarning("Server response incorrect.  %s", __FUNCTION__);
    return;
  }
  
  int16_t nbunsetu;
  ds >> nbunsetu;
  if (nbunsetu <= 0) {
    qDebug("Convert error.  %s", __FUNCTION__);
    return;
  }
  
  qDebug("%s Number of Bunsetu: %d", __FUNCTION__, nbunsetu);

  _strlist.clear();
  for (int i = 0; i < nbunsetu; i++) {
    QByteArray  ba;
    QDataStream dsba(ba, IO_WriteOnly);
    dsba.setByteOrder(QDataStream::BigEndian);
    u_int16_t us;

    ds >> us;
    while (us && !ds.atEnd()) {
      dsba << us;
      ds >> us;
    }
    
    QString str = (tr(uint16ToEuc(ba)));
    if ( !str.isEmpty() )
      _strlist << str;
  }

  qDebug("[%s]  %s", __FUNCTION__, _strlist.join(" ").local8Bit().data()); 
  emit converted(_strlist, _crntidx);
}


// ֵͤϥͥåȥХȥѴѤʤΤǡľǽ
//  (u_int16_t)0 ɲ
QByteArray 
SockKanjiClient::eucToUint16(const QCString& src)
{
  QByteArray dest;
  QDataStream dsdest(dest, IO_ReadWrite);
  dsdest.setByteOrder(QDataStream::BigEndian);

  QCString::ConstIterator it = src.begin();
  while (it != src.end()) {

    if (!(*it & 0x80)) {
      // ASCII
      dsdest << (u_int16_t)*it;

    } else if (it + 1 != src.end()) {
      switch ((uchar)*it) {
      case 0x8e:   // Ⱦѥ	
	dsdest << (u_int16_t)(0x0080 | (*(++it) & 0x7f));
	break;

      case 0x8f:   // 
	if (it + 2 == src.end())
	  return 0;
	
	dsdest << (u_int16_t)(0x8000 | ((*(++it) & 0x7f) << 8) | (*(++it) & 0x7f));
	break;

      default:  // Ѥ
	dsdest << (u_int16_t)(0x8080 | ((*it & 0x7f) << 8) | (*(++it) & 0x7f));
	break;
      }

    } else {
      // error
      return 0;
    }

    ++it;
  }
  
  return dest;
}


// ϥͥåȥХȥ
QCString 
SockKanjiClient::uint16ToEuc(const QByteArray& src)
{
#if 0
  QDataStream dssrc(src, IO_ReadOnly);
  dssrc.setByteOrder(QDataStream::BigEndian);
  QCString dest;
  u_int8_t b; 
  while (!dssrc.atEnd()) {
    u_int16_t us;
    dssrc >> us;
 
    switch (us & 0x8080) {
    case 0:  /* ASCII */
      dest += (uchar)(us & 0x007f);
      break;

    case 0x0080:  /* Ⱦѥ */
      dest += (uchar)0x8e;
      dest += (uchar)((us & 0x007f) | 0x80);
      break;

    case 0x8000:  /*  */
      dest += (uchar)0x8f;
      dest += (uchar)(((us & 0x7f00) >> 8) | 0x80);
      dest += (uchar)((us & 0x007f) | 0x80);
      break;

    case 0x8080:  /* Ѥ */
      b = (us & 0xff00) >> 8;
      if (b >= 0xa1 && b <= 0xa8 || b >= 0xb0 && b <= 0xf4) {  // EUCɤξ
	dest += (uchar)(((us & 0x7f00) >> 8) | 0x80);
	dest += (uchar)((us & 0x007f) | 0x80);
	
      } else {
	qWarning("no such EUC code  us:0x%x", us);
      }
      break;

    default:
      Q_ASSERT(0);
      break;
    }
  }
  
  return dest;
#else

  QDataStream dssrc(src, IO_ReadOnly);
  dssrc.setByteOrder(QDataStream::BigEndian);
  QCString dst;
  QDataStream  dsdst(dst, IO_WriteOnly);

  u_int8_t b;
  QCString str;
  while (!dssrc.atEnd()) {
    u_int16_t us;
    dssrc >> us;
 
    switch (us & 0x8080) {
    case 0:  // ASCII
      dsdst << (u_int8_t)(us & 0x007f);
      break;

    case 0x0080:  // Ⱦѥ
      dsdst << (0x8e80 | (us & 0x007f));
      break;

    case 0x8000:  // 
      dsdst << (u_int8_t)0x8f;
      dsdst << us;
      break;

    case 0x8080:  // Ѥ
      b = (us & 0xff00) >> 8;
      if (b >= 0xa1 && b <= 0xa8 || b >= 0xb0 && b <= 0xf4) {  // EUC code
	dsdst << us;
	
      } else if (us >= 0xada1 && us <= 0xadb4) {
	str.sprintf("(%u)", us - 0xada0);
	dst += str;

      } else {
	qWarning("No such EUC code!!!!!  us:0x%x", us);
      }
      break;

    default:
      Q_ASSERT(0);
      break;
    }
  }
  
  dsdst << (u_int8_t)0;
  return dst;
#endif
}



