#include "kanjiconvert.h"
#include "sockkanjiclient.h"
#include "dicthiragana.h"
#include "preeditarea.h"
#include "inputmode.h"
#include "config.h"
#include "keyassigner.h"
#include "kimeranamespace.h"
#include "debug.h"
#include <qkeysequence.h>
#include <qregexp.h>
#include <X11/keysym.h>
using namespace Kimera;

QIntDict<FuncPtr>  KanjiConvert::funcptrdict = QIntDict<FuncPtr>();


KanjiConvert::KanjiConvert() : _crntsegidx(0),
			       _minsegidx(0),
			       _inputkeys(QString()),
			       _yomiseglist(QStringList()),
			       _convtkanjilist(QStringList()),
			       _stat(NONE_YOMI),
			       _mode(), 
			       _dict(DictHiragana())
{
  // Create socket
  _sockkanjiclt = new SockKanjiClient();

  // Create PreeditArea instance
  _preedit = new PreeditArea();

  // Create CandidacyListBox instance
  _candlist = new CandidacyListBox();

  connect(_sockkanjiclt, SIGNAL(converted(const QStringList&, int)), this, SLOT(slotConvertSegments(const QStringList&, int)));
  connect(_sockkanjiclt, SIGNAL(candidate(const QStringList&, int)), this, SLOT(slotCandidacyList(const QStringList&, int)));
  connect(_sockkanjiclt, SIGNAL(yomigana(const QString&, int)), this, SLOT(slotYomigana(const QString&, int))); 
  connect(_candlist, SIGNAL(clicked(QListBoxItem*)), this, SLOT(slotDecideSegment(QListBoxItem*)));
  connect(_preedit, SIGNAL(listPointChanged(const QPoint&)),  _candlist, SLOT(move(const QPoint&)));
  
  clear();
}
 

KanjiConvert::~KanjiConvert()
{
  delete  _sockkanjiclt;
  delete  _preedit;
  delete  _candlist;
}


void 
KanjiConvert::init()
{
  DEBUG_TRACEFUNC();
  // initialize socket
  _sockkanjiclt->init();
  clear();

  // Dictionary Initialization
  _dict.init();

  // Initiate preedit
  _preedit->init();

  // Initialize  dictionary of function pointer
  init_funcptrdict();
}


void 
KanjiConvert::init_funcptrdict()
{
  DEBUG_TRACEFUNC();
  if ( ! funcptrdict.isEmpty() ) 
    return;
  
  funcptrdict.setAutoDelete(TRUE);
  
  funcptrdict.insert(FID_NextCandidacy,         new FuncPtr(&KanjiConvert::nextCandidacy));
  funcptrdict.insert(FID_PreviousCandidacy,     new FuncPtr(&KanjiConvert::previousCandidacy));
  funcptrdict.insert(FID_ConvertAllSegments,    new FuncPtr(&KanjiConvert::convertAllSegments));
  funcptrdict.insert(FID_ConvertCrntSegment,    new FuncPtr(&KanjiConvert::convertCrntSegment));
  funcptrdict.insert(FID_DeleteBackward,        new FuncPtr(&KanjiConvert::deleteBackward));
  funcptrdict.insert(FID_DecideAllSegments,     new FuncPtr(&KanjiConvert::decideAllSegments));
  funcptrdict.insert(FID_DecideCrntSegment,     new FuncPtr(&KanjiConvert::decideCrntSegment));
  funcptrdict.insert(FID_CancelConversion,      new FuncPtr(&KanjiConvert::cancelConversion));
  funcptrdict.insert(FID_LengthenSegment,       new FuncPtr(&KanjiConvert::lengthenSegment));
  funcptrdict.insert(FID_ShortenSegment,        new FuncPtr(&KanjiConvert::shortenSegment));
  funcptrdict.insert(FID_InputSpaceChar,        new FuncPtr(&KanjiConvert::inputSpaceChar));
  funcptrdict.insert(FID_ConvertToHira,         new FuncPtr(&KanjiConvert::convertToHira));
  funcptrdict.insert(FID_ConvertToKana,         new FuncPtr(&KanjiConvert::convertToKata));
  funcptrdict.insert(FID_ConvertToHankakuEisu,  new FuncPtr(&KanjiConvert::convertToHankakuEisu));
  funcptrdict.insert(FID_ConvertToZenkakuEisu,  new FuncPtr(&KanjiConvert::convertToZenkakuEisu));
  funcptrdict.insert(FID_ForwardSegment,        new FuncPtr(&KanjiConvert::forwardSegment));
  funcptrdict.insert(FID_BackwardSegment,       new FuncPtr(&KanjiConvert::backwardSegment));
  funcptrdict.insert(FID_ToFirstSegment,        new FuncPtr(&KanjiConvert::toFirstSegment));
  funcptrdict.insert(FID_ToLastSegment,         new FuncPtr(&KanjiConvert::toLastSegment));
  funcptrdict.insert(FID_SwitchZenkakuEisu,     new FuncPtr(&KanjiConvert::switchZenkakuEisu));
}


void
KanjiConvert::setInputMode(const InputMode& mode)
{
  DEBUG_TRACEFUNC("mode: 0x%x", mode.id());
  _mode.merge(mode);
  clear();
}


void    
KanjiConvert::clear()
{
  DEBUG_TRACEFUNC();
  _crntsegidx = 0;
  _minsegidx = 0;
  _inputkeys.truncate(0);
  _yomiseglist.clear();
  _convtkanjilist.clear();
  _sockkanjiclt->endConvert();
  _stat = NONE_YOMI;
  hideToolBox();
}


void
KanjiConvert::setFont(QFont ft) 
{
  DEBUG_TRACEFUNC();
  _preedit->setFont(ft);        // Set font to preedit area
  _candlist->setFont(ft);       // Set font to candidacy list box
}


bool
KanjiConvert::convertAllSegments()
{
  DEBUG_TRACEFUNC();

  switch (_stat) {
  case NONE_YOMI:
    // Do nothing
    return TRUE;
    break;

  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _convtkanjilist = _yomiseglist = _dict.forceConvt(_inputkeys, InputMode(Mode_Hiragana));
    _sockkanjiclt->beginConvert( _yomiseglist.join("") );
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = CONVERTING_YOMI;
  return TRUE;
}


bool
KanjiConvert::convertCrntSegment()
{
  DEBUG_TRACEFUNC();
  
  switch (_stat) {
  case NONE_YOMI:
    // Do nothing
    return TRUE;
    break;

  case INPUTING_YOMI:
    convertAllSegments();
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
    //_sockkanjiclt->getCandidacyList(_crntsegidx);
    break;
    
  case CHANGING_LENGTH:
    _sockkanjiclt->resizePause(_crntsegidx, _convtkanjilist[_crntsegidx].length());
    break;    
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = CONVERTING_YOMI;
  return TRUE;
}


bool 
KanjiConvert::nextCandidacy()
{
  DEBUG_TRACEFUNC();
  CHECK_NOCHANGE(_yomiseglist, QStringList);

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
    if ( _candlist->isVisible() ) {  
      // Means CandidacyListBox shows
      _candlist->showNextCandidacy();
      
      // Replace string of preeidt area
      QStringList::iterator it = _convtkanjilist.begin();
      it += _crntsegidx;
      it = _convtkanjilist.erase(it);
      _convtkanjilist.insert(it, _candlist->currentText());
      
    } else if (_sockkanjiclt->isConverting()) {
      // Get Candidacies
      _sockkanjiclt->getCandidacyList(_crntsegidx);
      
    } else {
      return TRUE;
    }
    break;

  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  // show preedit
  _stat = SHOWING_LIST;
  showPreeditArea();
  return TRUE;
}


bool 
KanjiConvert::previousCandidacy()
{
  DEBUG_TRACEFUNC();
  CHECK_NOCHANGE(_yomiseglist, QStringList);

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
    if ( _candlist->isVisible() ) {
      // It means CandidacyListBox shows
      _candlist->showPreviousCandidacy();
      
      // Replace string of preeidt area
      QStringList::iterator it = _convtkanjilist.begin();
      it += _crntsegidx;
      it = _convtkanjilist.erase(it);
      _convtkanjilist.insert(it, _candlist->currentText());

    } else {
      return TRUE;
    }
    break;
    
  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  // show preedit
  _stat = SHOWING_LIST;
  showPreeditArea();  
  return TRUE;
}


// Append a character to roma string
bool
KanjiConvert::appendChar(const QKeyEvent& ke)
{
  DEBUG_TRACEFUNC();
  if (ke.count() != 1 || ke.text().isEmpty()) {
    clear();
    return FALSE;
  }  

  switch (_stat) {
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:  
    decideAllSegments();
    // no break

  case NONE_YOMI:
  case INPUTING_YOMI:
    if (ke.state() == Keypad && Config::readEntry("_cmbtenkey", "") == tr("Ⱦ")) {
      // case Tenkey
      _inputkeys += ke.text();
      _yomiseglist << ke.text();
    
    } else if ((_mode.id() & Mode_KanaInput) && !(_mode.id() & Mode_ZenkakuEisu)) {
      // case Kana-Input
      _inputkeys += _dict.inverseConvt( _dict.convert(ke) );
      _yomiseglist[0] += _dict.convert( ke );
      _yomiseglist = _dict.convert(_yomiseglist[0], InputMode(Mode_Hiragana));  // ʸѴ
      _yomiseglist = _dict.convert(_yomiseglist.join(""), _mode);

    } else {
      // case Zenkaku Eisu or Roma-Input 
      _inputkeys += ke.text();
      _yomiseglist = _dict.convert(_inputkeys, _mode);
    }
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = INPUTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::deleteBackward()
{
  DEBUG_TRACEFUNC();
  qDebug("current Yomigana:%s", _yomiseglist.join(" ").local8Bit().data());

  QString str;
  switch (_stat) {
  case NONE_YOMI:
    // Do noting
    return TRUE;
    break;
    
  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _yomiseglist.clear();
    str = _dict.forceConvt(_inputkeys, _mode.id());
    str.truncate(str.length() - 1);
    if ( !str.isEmpty() ) {
      _yomiseglist << str;
    }
    syncInputKeysBackward( str );    
    qDebug("current Yomigana:%s", _yomiseglist.join("").local8Bit().data());
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  // Sets status
  _stat = (_yomiseglist.isEmpty()) ? NONE_YOMI : INPUTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::decideCrntSegment()
{
  DEBUG_TRACEFUNC();
 
  QString str;
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // Emit segment decided
    emit  decideSegments(_preedit->backwardText() + _preedit->attentionText());
    _minsegidx = ++_crntsegidx;
    if ( _convtkanjilist[_crntsegidx].isEmpty() ) {
      qDebug("no more kanji to to converted");
      clear();
      return TRUE;
    }

    qDebug("Segment minimum index: %d", _minsegidx);
    qDebug("Number of segments remaining: %d : %s", _convtkanjilist.count() - _minsegidx, _convtkanjilist.join(" ").local8Bit().data());
    
    // Sync input-keys
    for (int i = _crntsegidx; i < _yomiseglist.count(); ++i) {
      str += _yomiseglist[i];
    }
    syncInputKeysForward( str );
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::decideAllSegments()
{
  DEBUG_TRACEFUNC();

  if (_preedit->isVisible())
    emit decideSegments(_preedit->text());

  clear();
  return TRUE;
}


bool
KanjiConvert::cancelConversion()
{
  DEBUG_TRACEFUNC();

  QString yomi;  
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // send End Convert message
    _sockkanjiclt->endConvert(); 
    hideToolBox();    
    
    // case Roma-Input
    for (int i = _minsegidx; i < _yomiseglist.count(); ++i) {
      yomi += _yomiseglist[i];
    }
   
    // Reset input-keys
    if (_minsegidx > 0) {
      _inputkeys = _dict.inverseConvt( yomi );
    }

    _yomiseglist = _dict.convert(_inputkeys, _mode);
    _crntsegidx = _minsegidx = 0;
    _convtkanjilist.clear();
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = INPUTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool    
KanjiConvert::lengthenSegment()
{
  DEBUG_TRACEFUNC();

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_yomiseglist[_crntsegidx + 1].isEmpty()) {
      // Do nothing
      return TRUE;

    } else {
      _yomiseglist[_crntsegidx] += _yomiseglist[_crntsegidx + 1].left(1);
      _yomiseglist[_crntsegidx + 1].remove(0, 1);
      joinYomiList(_crntsegidx + 1);
      _sockkanjiclt->resizePause(_crntsegidx, _yomiseglist[_crntsegidx].length());
    }
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  _candlist->hide();
  _stat = CHANGING_LENGTH;
  showPreeditArea();
  return TRUE;
}


bool    
KanjiConvert::shortenSegment()
{
  DEBUG_TRACEFUNC();

  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_yomiseglist[_crntsegidx].length() <= 1) {
      // Do nothing
      return TRUE;

    } else {
      QStringList::iterator it;
      it = _yomiseglist.begin();
      it += _crntsegidx + 1;
      it = _yomiseglist.insert(it, _yomiseglist[_crntsegidx].right(1));
      _yomiseglist[_crntsegidx].truncate(_yomiseglist[_crntsegidx].length() - 1);
      
      joinYomiList(_crntsegidx + 1);
      _sockkanjiclt->resizePause(_crntsegidx, _yomiseglist[_crntsegidx].length());
    }
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CHANGING_LENGTH;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::forwardSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_crntsegidx < _convtkanjilist.count() - 1)
      _crntsegidx++;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::backwardSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    if (_crntsegidx > _minsegidx)
      _crntsegidx--;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::toFirstSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _crntsegidx = _minsegidx;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::toLastSegment()
{
  DEBUG_TRACEFUNC();
  switch (_stat) {
  case NONE_YOMI:
  case INPUTING_YOMI:
    // Do nothing
    return TRUE;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    _crntsegidx = _yomiseglist.count() - 1;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::inputSpaceChar()
{
  DEBUG_TRACEFUNC();

  bool res = TRUE;
  QString s;
  switch (_stat) {
  case NONE_YOMI:
    s = Config::readEntry("_cmbspacekey", "");
    if (s == tr("Ⱦ")) {
      res = FALSE;        // Forward key-event for xemacs bug?
      
    } else if (s == tr("")) {
      emit decideSegments(tr(""));  // Multibyte space
    }
    break;

  case INPUTING_YOMI:    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;
    
  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  clear();
  return res;
}


bool
KanjiConvert::convertToHira()
{
  DEBUG_TRACEFUNC();

  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _convtkanjilist = _yomiseglist = _dict.forceConvt(_inputkeys);
    break;
 
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // convert first segment to Hiragara
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());
    _convtkanjilist[_crntsegidx] = _yomiseglist[_crntsegidx];
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::convertToKata()
{
  DEBUG_TRACEFUNC();
 
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _yomiseglist = _dict.forceConvt(_inputkeys);
    _convtkanjilist = _dict.convert(_yomiseglist.join(""), InputMode(Mode_Katakana));
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // convert first segment to Katakana
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());  
    _convtkanjilist[_crntsegidx] = _dict.forceConvt(_yomiseglist[_crntsegidx], InputMode(Mode_Katakana));
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
  return TRUE;
}


bool
KanjiConvert::convertToHankakuEisu()
{
  DEBUG_TRACEFUNC();

  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;

  case INPUTING_YOMI:
    _yomiseglist = _dict.forceConvt(_inputkeys);
    _convtkanjilist = _inputkeys;
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    Q_ASSERT(!_convtkanjilist[_crntsegidx].isEmpty());
    if (!_crntsegidx && _convtkanjilist.count() == 1) { 
      _convtkanjilist[_crntsegidx] =  _inputkeys;
    } else {
      _convtkanjilist[_crntsegidx] = _dict.inverseConvt(_yomiseglist[_crntsegidx]);
    }
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::convertToZenkakuEisu()
{
  DEBUG_TRACEFUNC();

  QString str;
  switch (_stat) {
  case NONE_YOMI:
    return TRUE;
    break;
    
  case INPUTING_YOMI:
    _yomiseglist = _dict.forceConvt(_inputkeys);
    _convtkanjilist = _dict.forceConvt(_inputkeys, InputMode(Mode_ZenkakuEisu));
    break;
    
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    Q_ASSERT(!_yomiseglist[_crntsegidx].isEmpty());
    if (!_crntsegidx && _convtkanjilist.count() == 1) { 
      str = _inputkeys;
    } else {
      str = _dict.inverseConvt(_yomiseglist[_crntsegidx]);
    }
    _convtkanjilist[_crntsegidx] = _dict.forceConvt(str, InputMode(Mode_ZenkakuEisu));
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }
  
  _stat = CONVERTING_YOMI; 
  showPreeditArea();  
  return TRUE;
}


bool
KanjiConvert::switchZenkakuEisu()
{
  DEBUG_TRACEFUNC();
  
  QString s;
  switch (_stat) {
  case NONE_YOMI:
    if (_mode.id() & Kimera::Mode_ZenkakuEisu) {
      s = Config::readEntry("_cmbinputstyle", "");
      if (s == tr("޻")) {
	setInputMode(Kimera::Mode_RomaInput);
	
      } else if (s == tr("")) {
	setInputMode(Kimera::Mode_KanaInput);
      }
      
    } else {
      setInputMode(Kimera::Mode_ZenkakuEisu);
    }
    break;

  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
  case CHANGING_LENGTH:
    // Do nothing
    return TRUE;
    break;

  default:
    Q_ASSERT(0);
    return FALSE;
    break;
  }

  _stat = NONE_YOMI;
  clear();
  return TRUE;
}


void
KanjiConvert::slotConvertSegments(const QStringList& strs, int idx)
{
  DEBUG_TRACEFUNC("list: %s  idx: %d", strs.join(" ").local8Bit().data(), idx);
  CHECK_NOCHANGE(_stat, int);

  QStringList::iterator it;
  switch (_stat) {
  case NONE_YOMI:
    clear();
    break;

  case INPUTING_YOMI:
  case CONVERTING_YOMI:
  case SHOWING_LIST:
    qDebug("candidacies(%d) :%s  (crnt:%d  min:%d)", strs.count(), strs.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
    
    if (!strs.isEmpty() && idx >= 0 && idx < _convtkanjilist.count()) {
      // Sets converted kanji list
      it = _convtkanjilist.begin();
      it += idx;
      _convtkanjilist.erase(it, _convtkanjilist.end());  // Erase
      for (int i = 0; i < strs.count(); ++i) {
	if ( !strs[i].isEmpty() )
	  _convtkanjilist << strs[i];   // append str
      }
    }

    // Gets yomigana
    joinYomiList(_crntsegidx);
    _sockkanjiclt->getYomi(_crntsegidx);
    showPreeditArea();
    break;

  case CHANGING_LENGTH:
    // Gets yomigana
    joinYomiList(_crntsegidx);
    _sockkanjiclt->getYomi(_crntsegidx);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }
}


// curidx: current selected candidacy index
// strs:   candidacy list
void
KanjiConvert::slotCandidacyList(const QStringList& strs, int idx)
{
  DEBUG_TRACEFUNC("candidacy: %s  idx: %d", strs.join(" ").local8Bit().data(), idx); 
  CHECK_NOCHANGE(_stat, int);
  CHECK_NOCHANGE(_crntsegidx, int);

  if (idx !=_crntsegidx) {
    qWarning("mismatch index"); 
    return;
  }

  QStringList  strlist(strs);
  if (strlist.isEmpty()) {
    strlist << _convtkanjilist[_crntsegidx];
  }
  
  if (strlist.count() > 1 && strlist.first() == _convtkanjilist[_crntsegidx]) {
    _candlist->showList(strlist, 1);
    
  } else {
    _candlist->showList(strlist);
  }
  
  // Replace string of converted kanji string list
  _convtkanjilist[_crntsegidx] = _candlist->currentText();
  showPreeditArea();
}


void
KanjiConvert::slotYomigana(const QString& str, int index)
{
  DEBUG_TRACEFUNC("str:%s  index:%d", str.local8Bit().data(), index);
  CHECK_NOCHANGE(_crntsegidx, int);

  if (str.isEmpty() || index < 0 || index >= _yomiseglist.count()) {
    // Do nothing
    qFatal("Bad param  index:%d  _yomiseglist.count()", index, _yomiseglist.count());
    return;

  } else if (_yomiseglist[index].find(str) < 0) {
    qFatal("Cannot find string");
    return;
  }
  
  // Replace yomigana of yomi-segment-list
  qDebug("yomi-list: %s", _yomiseglist.join(" ").local8Bit().data());

  // Sets yomi list
  QString s = _yomiseglist[index];
  QStringList::iterator it = _yomiseglist.begin();
  it += index;
  _yomiseglist.erase(it, _yomiseglist.end());
  _yomiseglist << str;
  if (s.length() > str.length()) {
    _yomiseglist << s.mid(str.length());
    
    // Get yomigana of next segment
    _sockkanjiclt->getYomi(index + 1);
  }

  if (_stat == CHANGING_LENGTH) {
    // Sets converted-kanji-list
    it = _convtkanjilist.begin();
    it += index;
    _convtkanjilist.erase(it, _convtkanjilist.end());
    _convtkanjilist << str;
    if (s.length() > str.length())
      _convtkanjilist << s.mid(str.length());
  }

  // Show segments
  qDebug("yomi: %s  convt:%s (crnt:%d min:%d)", _yomiseglist.join(" ").local8Bit().data(),
	 _convtkanjilist.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
}


void 
KanjiConvert::slotDecideSegment(QListBoxItem* item)
{
  Q_CHECK_PTR( item );
  DEBUG_TRACEFUNC("item: %s", item->text().local8Bit().data());
  CHECK_NOCHANGE(_yomiseglist, QStringList);

  if (_stat != SHOWING_LIST)
    return;

  // Replace string
  QStringList::iterator it = _convtkanjilist.begin();
  it += _crntsegidx;
  it = _convtkanjilist.erase(it);
  _convtkanjilist.insert(it, item->text());

  _candlist->hide();
  _stat = CONVERTING_YOMI;
  showPreeditArea();
}


void
KanjiConvert::hideToolBox()
{
  _preedit->hide();
  _candlist->hide();
}


bool
KanjiConvert::processKeyEvent(const QKeyEvent& key)
{
  DEBUG_TRACEFUNC("The key code: 0x%x   key count: %d  ascii: %s", key.key(), key.count(), key.text().data());

  int k = key.key();
  k |= (key.state() & Qt::ControlButton) ? Qt::CTRL : 0;
  k |= (key.state() & Qt::ShiftButton) ? Qt::SHIFT : 0;
  k |= (key.state() & Qt::AltButton) ? Qt::ALT : 0;

  bool res = TRUE;
  FuncID  fid = KeyAssigner::functionID( k, _stat );
  switch ( fid ) {
  case FID_AppendChar:
    res = appendChar( key );
    break;
    
  case FID_None:
    // Do nothing
    break;

  case FID_ForwardKeyEvent:
    res = (_stat == NONE_YOMI) ? FALSE : TRUE;
    break;

  default:
    pmfunc  pf = funcptrdict[fid]->ptr();  // get pointer
    if ( !pf ) {
      qWarning("%s(%d): incorrect processing?", __FUNCTION__, __LINE__);
      clear();
      res = FALSE;
    
    } else {
      qDebug("FuncName: %s", KeyAssigner::functionName( fid ).local8Bit().data());
      res = (this->*pf)();      // execute function
    }
    break;
  }
  
  return res;
}


void 
KanjiConvert::setPreeditPoint(QPoint p)
{
  DEBUG_TRACEFUNC("x:%d y:%d", p.x(), p.y());
  _preedit->movePreeditPos( p );
}


void
KanjiConvert::showPreeditArea()
{
  DEBUG_TRACEFUNC();
  CHECK_NOCHANGE(_convtkanjilist, QStringList);
  CHECK_NOCHANGE(_yomiseglist, QStringList);

  QStringList list;
  switch (_stat) {
  case NONE_YOMI:
    clear();
    break;

  case INPUTING_YOMI:
    _preedit->showInputingString( _yomiseglist.join("") );
    qDebug("Yomigana:%s  (%s)", _yomiseglist.join("").local8Bit().data(), _inputkeys.local8Bit().data());
    break;

  case CONVERTING_YOMI:
  case SHOWING_LIST:
    if (_minsegidx >= _convtkanjilist.count()) {
      qFatal("Bad parameter  _minsegidx:%d  _convtkanjilist.count():%d",  _minsegidx, _convtkanjilist.count());
      return;
    }    

    for (int i = _minsegidx; i < _convtkanjilist.count(); i++)
	list << _convtkanjilist[i];

    qDebug("showConvertingSegments : %s  (%d:%d)", list.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
    _preedit->showConvertingSegments(list, _crntsegidx - _minsegidx);
    break;

  case CHANGING_LENGTH:
    if (_minsegidx >= _yomiseglist.count()) {
      qFatal("Bad parameter  _minsegidx:%d  _convtkanjilist.count():%d",  _minsegidx, _convtkanjilist.count());
      return;
    }    
    
    for (int i = _minsegidx; i < _crntsegidx; i++)
      list << _convtkanjilist[i];
    
    for (int i = _crntsegidx; i < _yomiseglist.count(); i++)
      list << _yomiseglist[i];
    
    qDebug("showChangingSegmentLength : %s  (%d:%d)", list.join(" ").local8Bit().data(), _crntsegidx, _minsegidx);
    _preedit->showChangingSegmentLength(list, _crntsegidx - _minsegidx);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }
}


//
// @param: str: yomogana string
//
void
KanjiConvert::syncInputKeysForward(const QString& str)
{
  DEBUG_TRACEFUNC("str: %s", str.local8Bit().data());

  if (str.isEmpty()) {
    _inputkeys.truncate(0);
    return;
  }

  int len;
  QString convt;
  for (len = _inputkeys.length(); len > 0; --len) {
    convt = _dict.convert(_inputkeys.right(len), _mode.id());
    if (convt == str) {
      _inputkeys = _inputkeys.right(len);
      break;
    }
  }

  if (convt != str) {
    _inputkeys = _dict.inverseConvt(str);
  }
  qDebug("%s  str:%s  input-keys:%s", __func__, str.local8Bit().data(), _inputkeys.local8Bit().data());
}


void
KanjiConvert::syncInputKeysBackward(const QString& str)
{
  DEBUG_TRACEFUNC("str: %s", str.local8Bit().data());

  if (str.isEmpty()) {
    _inputkeys.truncate(0);
    return;
  }

  int len;
  QString convt;
  for (len = _inputkeys.length(); len > 0; --len) {
    convt = _dict.convert(_inputkeys.left(len), _mode.id());
    if (convt == str) {
      _inputkeys = _inputkeys.left(len);
      break;
    }
  }

  if (convt != str) {
    _inputkeys = _dict.inverseConvt(str);
  }
  qDebug("syncInputKeys : %s  _inputkeys: %s", str.local8Bit().data(), _inputkeys.local8Bit().data());
  qDebug("%s  str:%s  input-keys:%s", __func__, str.local8Bit().data(), _inputkeys.local8Bit().data());
}


void
KanjiConvert::joinYomiList(int index)
{
  DEBUG_TRACEFUNC("index: %d", index);
  QString str;
  for (int i = index; i < _yomiseglist.count(); ++i) {
    str += _yomiseglist[i];
  }

  QStringList::iterator it = _yomiseglist.begin();
  it += index;
  _yomiseglist.erase(it, _yomiseglist.end());
  if ( !str.isEmpty() )
    _yomiseglist << str;

  qDebug("Yomigana:%s  (%s)", _yomiseglist.join(" ").local8Bit().data(), _inputkeys.local8Bit().data());
}

