#include "dicthiragana.h"
#include "inputmode.h"
#include "config.h"
#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qtextstream.h>
#include <qregexp.h>
using namespace Kimera;

DictHiragana::DictHiragana() : _dicthira(QDict<QString>(17, FALSE)),
			       _dictkata(QDict<QString>(17, FALSE)),
			       _dictalphbt(QDict<QString>()),
			       _dictsymbol(QDict<QString>()),
			       _dictkanainp(QDict<QString>(17, FALSE)),
			       _dictdakuten(QDict<QString>()),
			       _reversedict(QDict<QString>())
{
  _dicthira.setAutoDelete(TRUE);
  _dictkata.setAutoDelete(TRUE);
  _dictalphbt.setAutoDelete(TRUE);
  _dictsymbol.setAutoDelete(TRUE);
  _dictkanainp.setAutoDelete(TRUE);
  _dictdakuten.setAutoDelete(TRUE);
  _reversedict.setAutoDelete(TRUE);
}


DictHiragana::~DictHiragana() { }


void
DictHiragana::init()
{
  _dicthira.clear();
  _dictkata.clear();
  _dictalphbt.clear();
  _dictsymbol.clear();
  _dictkanainp.clear();
  _dictdakuten.clear();

  writeDictHira();
  writeDictKata();
  writeDictAlphbt();
  writeDictSymbol();
  writeDictKanaInp();
  writeDictDakuten();

  QString dictdir = QString(getenv("HOME")) + "/.kimera/dic/";
  initDict(_dicthira, dictdir + "hiragana.dic");          // Ҥ餬ʼ
  initDict(_dictkata, dictdir + "katakana.dic");          // ʼ
  initDict(_dictalphbt, dictdir + "zenkakualphabet.dic"); // ѱѻ
  initDict(_dictsymbol, dictdir + "numeralsymbols.dic");  // ѿ漭
  initDict(_dictkanainp, dictdir + "kanainput.dic");      // ϼ
  initDict(_dictdakuten, dictdir + "dakuten.dic");        // 

  // 桦
  _dictsymbol.insert(",", new QString(Config::readEntry("_cmbtouten", ","))); 
  _dictsymbol.insert(".", new QString(Config::readEntry("_cmbkuten", ".")));
  _dictsymbol.insert("/", new QString(Config::readEntry("_cmbsymbol", "/")));
  _dictsymbol.insert("[", new QString(Config::readEntry("_cmbbracket", "[").left(1)));
  _dictsymbol.insert("]", new QString(Config::readEntry("_cmbbracket", "]").right(1)));

  // հ
  initReverseDict(_dicthira);
  initReverseDict(_dictalphbt);
  initReverseDict(_dictsymbol);
}


void
DictHiragana::initDict(QDict<QString>& dict, const QString& dictfile)
{
  if ( !dict.isEmpty() )
    return;
  
  QFile file(dictfile);
  
  if ( !file.open(IO_ReadOnly) ) {
    QMessageBox::critical(0, "File open error", "Cannot open file: " + dictfile,
    			  QMessageBox::Ok | QMessageBox::Default, 0);
    qApp->quit();
    return;
  }
    
  QTextStream  stream(&file);
  stream.setEncoding(QTextStream::Latin1);

  while ( !stream.eof() ) {
    QString str;
    str = QString::fromLocal8Bit(stream.readLine());
    
    if ( !str.isEmpty() && !str.contains(QRegExp("^[ \t]")) && 
	 !str.contains(QRegExp("^##")) ) {
      QString keystr, itemstr;
      int idx = str.find(QRegExp("[ \t]"));
      if (idx >= 0) {
	keystr = str.left(idx);
	
	str.remove(0, keystr.length());
	itemstr = str.replace(QRegExp("[ \t]"), "");
	
	dict.insert(keystr, new QString(itemstr));
      }
    }
  }
  
  file.close();
}


void
DictHiragana::initReverseDict(const QDict<QString>& src)
{
  QDict<QString> dict(src);
  for (QDictIterator<QString> it(dict); it.current(); ++it) {
    if ( !_reversedict.find(*it.current()) ) {
      _reversedict.insert(*it.current(), new QString(it.currentKey()));
    }
  }
}


// ʿ̾Ѵ (ʸ˹碌ƣʸˤ :->  )
// parameter:    Roma string (only ASCII characters) or Hiragana string
// return value: Hiragana string
QString
DictHiragana::convertToHira(const QString& src)
{
  // check parameter
  if ( src.isEmpty() )
    return QString();

  QString dest;
  if (src.local8Bit().length() != src.length()) {
    // Ѵ
    if (src.length() == 1) 
      return src;

    int i = 0;
    while(i < src.length()) {
      QString* pstr;
      pstr = _dictdakuten.find( src.mid(i, 2) );
      
      if ( pstr ) {
	dest += *pstr;
	i++;
	
      } else {
	dest += src.constref(i);
      }
      i++;
    }
    
    return dest;

  }  
  
  int index = 0;
  while (index < src.length()) {
    for (int len = 1; len <= 4; len++) {
      QString*  pstr;
      // Ҥ餬ʸ
      pstr = _dicthira.find( src.mid(index, len) );
      if (pstr) {
	dest += *pstr;
	index += len;
	break;
      } 

      // ѿ渡
      pstr = _dictsymbol.find( src.mid(index, len) );
      if (pstr) {
	dest += *pstr;
	index += len;
	break;
      }
      
      if (len == 4) {
	if (src.constref(index) == src.constref(index + 1)
	    && src.mid(index, 1).find( QRegExp("[^aiueo]", FALSE) ) == 0
	    && src.mid(index + 2, 1).find( QRegExp("[aiueoyhsw]", FALSE) ) == 0 ) {
	  // ''Ѵ
	  pstr = _dicthira.find( src.mid(index + 1, 2) );
	  if (pstr) {
	    dest += QObject::tr("") + *pstr;
	    index += 3;

	  } else {
	    dest += QObject::tr("");
	    ++index;
	  }

	} else if (src.constref(index) == "n"
		   && src.mid(index + 1, 1).find( QRegExp("[^aiueoy]", FALSE) ) == 0) {
	  // ''Ѵ
	  dest += QObject::tr("");
	  ++index;
	  
	} else {
	  // ̵Ѵ
	  dest += src.constref(index);
	  ++index;
	}
	break;
      }
    }
  }
  
  return dest;
}


// convertToHira ƤӽФ塤Ǹʸ'n'ä '' Ѵ
//
QString
DictHiragana::forceConvtToHira(const QString& src)
{
  if ( src.isEmpty() )
    return QString();

  QString dest = convertToHira(src);
  if (dest.right(1) == "n" || dest.right(1) == "N") {
    dest.remove(dest.length() - 1, 1);
    dest += QString::fromLocal8Bit("");
  }
  
  return dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString
DictHiragana::convertToKata(const QString& src)
{
  Q_ASSERT(!_dictkata.isEmpty());

  QString dest;
  // check parameter
  if (src.local8Bit().length() == src.length()) {
    QString hira = convertToHira(src);
    for (int j = 0; j < hira.length(); j++) {
      QString* pstr = _dictkata.find( hira.constref(j) );
      if ( pstr ) {
	dest += *pstr;
      } else {
	dest += hira.constref(j);
      }
    }
    
  } else {
    // Hiragana -> Katakana
    for (int i = 0; i < src.length(); i++) {
      QString* pstr = _dictkata.find( src.constref(i) );
      if ( pstr ) {
	dest += *pstr;
      } else {
	dest += src.constref(i);
      }
    }
  }
  
  return  dest;
}


// parameter: Roma string (only ASCII characters) or Hiragana string
QString
DictHiragana::forceConvtToKata(const QString& src)
{
  return convertToKata( forceConvtToHira(src) );
}


// parameter: Roma string (only ASCII characters)
QString
DictHiragana::convertToZenkakuEisu(const QString& src)
{  
  Q_ASSERT(!_dictalphbt.isEmpty());
  Q_ASSERT(!_dictsymbol.isEmpty());

  QString dest;
  for (int i = 0; i < src.length(); i++) {
    QString* pstr;
    if ( pstr = _dictalphbt.find(src.constref(i)) ) {  // if the key exists
      dest += *pstr;
      
    } else if ( pstr = _dictsymbol.find(src.constref(i)) ) {  // if the key exists 
      dest += *pstr;
      
    } else {
      dest += src.constref(i);
    }
  }
  
  return dest;
}


QString
DictHiragana::convert(const QString& roma, const InputMode& mode)
{
  switch (mode.id() & Mode_ModeMask) {
  case Mode_Hiragana:
    return convertToHira(roma);
    break;
    
  case Mode_Katakana:
    return convertToKata(roma);
    break;
    
  case Mode_ZenkakuEisu:
    if (roma.local8Bit().length() != roma.length()) {
      Q_ASSERT(0);
      break;
    }
    return convertToZenkakuEisu(roma);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


QString
DictHiragana::forceConvt(const QString& roma, const InputMode& mode)
{
  switch (mode.id() & Mode_ModeMask) {
  case Mode_Hiragana:
    return forceConvtToHira(roma);
    break;
    
  case Mode_Katakana:
    return forceConvtToKata(roma);
    break;
    
  case Mode_ZenkakuEisu:
    if (roma.local8Bit().length() != roma.length()) {
      Q_ASSERT(0);
      break;
    }
    return convertToZenkakuEisu(roma);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


//  Ѵ
//  params: QKeyEvent
//  params: Input mode
//  return: Hiragana string
QString
DictHiragana::convert(const QKeyEvent& key, const InputMode& mode)
{
  if ( !(mode.id() & Mode_KanaInput) ) {
    return QString();

  } else if (mode.id() & Mode_ZenkakuEisu) {
    return convertToZenkakuEisu(key.text());
  }

  Q_ASSERT(!_dictkanainp.isEmpty());
  QString keystr;
  if (key.state() == Qt::NoButton) {
    keystr = key.text();
    
  } else if (key.state() == Qt::ShiftButton) { 
    keystr = QString("Shift+") + key.text();
    
  } else if (key.state() == Qt::Keypad) {
    keystr = QString("Keypad+") + key.text();
    
  } else {
    return QString();
  }
  
  QString* pres = _dictkanainp[keystr];
  if ( !pres ) return QString();

  switch (mode.id() & Mode_ModeMask) {
  case Mode_Hiragana:
    return *pres;
    break;
    
  case Mode_Katakana:
    
    return convertToKata(*pres);
    break;
    
  default:
    Q_ASSERT(0);
    break;
  }

  return QString();
}


QString
DictHiragana::inverseConvt(const QString& hira)
{
  QString res;
  for (int i = 0; i < hira.length(); ++i) {
    QString* pstr = _reversedict.find(hira.mid(i, 1));
    if ( pstr ) {
      res += *pstr;
    } else {
      res += hira.mid(i, 1);
    }
  }
  
  return res;
}


void
DictHiragana::writeDictHira() const
{
  QString s = QObject::tr("dict_hiragana");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("hiragana.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictKata() const
{
  QString s = QObject::tr("dict_katakana");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("katakana.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictAlphbt() const
{
  QString s = QObject::tr("dict_alphabet");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("zenkakualphabet.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictSymbol() const
{
  QString s = QObject::tr("dict_symbol");
  
  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("numeralsymbols.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  }
}


void
DictHiragana::writeDictKanaInp() const
{
  QString s = QObject::tr("dict_kanainput");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("kanainput.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}


void
DictHiragana::writeDictDakuten() const
{
  QString s = QObject::tr("dict_dakuten");

  QString d = QString(getenv("HOME")) + "/.kimera/dic";
  QDir dir( d );
  if ( !dir.exists() ) {
    dir.mkdir(dir.path());
  }
  
  QFile f( dir.filePath("dakuten.dic") );  
  if ( !f.exists() && f.open( IO_WriteOnly ) ) {
    QTextStream stream( &f );
    stream << s;
    f.close();
  } 
}
