/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-root-bookmark.c,v 1.4 2004/03/21 13:47:37 makeinu Exp $
 */

#include "kz-root-bookmark.h"

#include "gobject-utils.h"
#include "intl.h"


static void     kz_root_bookmark_class_init  (KzRootBookmarkClass *klass);
static void     kz_root_bookmark_init        (KzRootBookmark *bookmark);
static GObject *kz_root_bookmark_constructor (GType           type,
					      guint           n_props,
					      GObjectConstructParam *props);
static void     kz_root_bookmark_dispose     (GObject *object);


static KzBookmarkClass *parent_class;


KZ_OBJECT_GET_TYPE(kz_root_bookmark, "KzRootBookmark", KzRootBookmark,
		   kz_root_bookmark_class_init, kz_root_bookmark_init,
		   KZ_TYPE_BOOKMARK)


static void
kz_root_bookmark_class_init (KzRootBookmarkClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);
	object_class = (GObjectClass *) klass;

	object_class->constructor = kz_root_bookmark_constructor;
	object_class->dispose     = kz_root_bookmark_dispose;
}


static void
kz_root_bookmark_init (KzRootBookmark *root)
{
	root->menu          = kz_bookmark_file_new(NULL, NULL, NULL);
	root->home          = kz_bookmark_file_new(NULL, NULL, NULL);
	root->scrap         = kz_bookmark_file_new(NULL, NULL, NULL);
	root->bookmark_bars = kz_bookmark_pure_folder_new();
	root->history       = kz_history_new(NULL);
	root->sessions      = kz_bookmark_pure_folder_new();
	root->windows       = kz_bookmark_pure_folder_new();
	root->tab_groups    = kz_bookmark_pure_folder_new();

	kz_bookmark_set_title(root->menu,                 "Bookmark Menu");
	kz_bookmark_set_title(root->home,                 "Home & Lairs");
	kz_bookmark_set_title(root->bookmark_bars,        "Bookmark Bars");
	kz_bookmark_set_title(KZ_BOOKMARK(root->history), "History");
	kz_bookmark_set_title(root->sessions,             "Sessions");
	kz_bookmark_set_title(root->windows,              "Windows");
	kz_bookmark_set_title(root->tab_groups,           "Tab Groups");
}


static GObject *
kz_root_bookmark_constructor (GType type, guint n_props,
			      GObjectConstructParam *props)
{
	KzRootBookmark *root;

	GObject *object;
	GObjectClass *klass = G_OBJECT_CLASS(parent_class);

	object = klass->constructor(type, n_props, props);

	root = KZ_ROOT_BOOKMARK(object);

	return object;
}


static void
kz_root_bookmark_dispose (GObject *object)
{
	KzRootBookmark *root;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(object));

	root = KZ_ROOT_BOOKMARK(object);

	if (root->menu)
	{
		g_object_unref(G_OBJECT(root->menu));
		root->menu = NULL;
	}
	if (root->home)
	{
		g_object_unref(G_OBJECT(root->home));
		root->home = NULL;
	}
	if (root->scrap)
	{
		g_object_unref(G_OBJECT(root->scrap));
		root->scrap = NULL;
	}
	if (root->bookmark_bars)
	{
		g_object_unref(G_OBJECT(root->bookmark_bars));
		root->bookmark_bars = NULL;
	}
	if (root->history)
	{
		g_object_unref(G_OBJECT(root->history));
		root->history = NULL;
	}
	if (root->sessions)
	{
		g_object_unref(G_OBJECT(root->sessions));
		root->sessions = NULL;
	}
	if (root->windows)
	{
		g_object_unref(G_OBJECT(root->windows));
		root->windows = NULL;
	}
	if (root->tab_groups)
	{
		g_object_unref(G_OBJECT(root->tab_groups));
		root->tab_groups = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


KzRootBookmark *
kz_root_bookmark_new (const gchar *menu_file,  const gchar *sys_menu_file,
		      const gchar *home_file,  const gchar *sys_home_file,
		      const gchar *scrap_file, const gchar *sys_scrap_file)
{
	KzRootBookmark *root;

	root = KZ_ROOT_BOOKMARK(g_object_new(KZ_TYPE_ROOT_BOOKMARK,
					     "type",  KZ_BOOKMARK_PURE_FOLDER,
					     "title", _("Root"),
					     NULL));

	if (root->menu)
		kz_bookmark_append(KZ_BOOKMARK(root), root->menu);
	if (root->home)
		kz_bookmark_append(KZ_BOOKMARK(root), root->home);
	if (root->scrap)
		kz_bookmark_append(KZ_BOOKMARK(root), root->scrap);
	if (root->bookmark_bars)
		kz_bookmark_append(KZ_BOOKMARK(root), root->bookmark_bars);
	/*
	kz_bookmark_append(KZ_BOOKMARK(root), KZ_BOOKMARK(root->history));
	kz_bookmark_append(KZ_BOOKMARK(root), root->sessions);
	kz_bookmark_append(KZ_BOOKMARK(root), root->windows);
	kz_bookmark_append(KZ_BOOKMARK(root), root->tab_groups);
	*/

	kz_root_bookmark_set_menu_file(root, menu_file, sys_menu_file);
	kz_root_bookmark_set_home_file(root, home_file, sys_home_file);
	kz_root_bookmark_set_scrap_file(root, scrap_file, sys_scrap_file);

	return root;
}


static KzBookmark *
ensure_bookmark_file (const gchar *file, const gchar *subfile,
		      const gchar *title, const gchar *type)
{
	KzBookmark *bookmark;

	g_return_val_if_fail(file && *file, NULL);
	g_return_val_if_fail(type && *type, NULL);

	if (g_file_test(file, G_FILE_TEST_EXISTS))
	{
		bookmark = kz_bookmark_file_new(file, title, type);
		kz_bookmark_load_in_memory_start(bookmark);
	}
	else if (subfile && *subfile &&
		 g_file_test(subfile, G_FILE_TEST_EXISTS))
	{
		bookmark = kz_bookmark_file_new(subfile, title, type);
		kz_bookmark_load_in_memory_start(bookmark);
	}
	else
	{
		bookmark = kz_bookmark_file_create_new(file, title, type);
	}

	kz_bookmark_set_location(bookmark, file);

	return bookmark;
}


void
kz_root_bookmark_set_menu_file (KzRootBookmark *root,
				const gchar *file, const gchar *subfile)
{
	KzBookmark *menu;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	/* FIXME! Check permission */

	menu = ensure_bookmark_file(file, subfile, _("Bookmarks"), "XBEL");

	if (root->menu)
	{
		kz_bookmark_remove(KZ_BOOKMARK(root), root->menu);
		g_object_unref(G_OBJECT(root->menu));
		root->menu = NULL;
	}

	if (menu)
	{
		root->menu = menu;
		kz_bookmark_insert_before(KZ_BOOKMARK(root),
					  menu, root->home);
	}
}


void
kz_root_bookmark_set_home_file (KzRootBookmark *root,
				const gchar *file, const gchar *subfile)
{
	KzBookmark *home;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	/* FIXME! Check permission */

	home = ensure_bookmark_file(file, subfile, _("Home & Lairs"), "XBEL");

	if (root->home)
	{
		kz_bookmark_remove(KZ_BOOKMARK(root), root->home);
		g_object_unref(G_OBJECT(root->home));
		root->home = NULL;
	}

	if (home)
	{
		root->home = home;
		kz_bookmark_insert_before(KZ_BOOKMARK(root),
					  home, root->scrap);
	}
}


void
kz_root_bookmark_set_scrap_file (KzRootBookmark *root,
				 const gchar *file, const gchar *subfile)
{
	KzBookmark *scrap;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	/* FIXME! Check permission */

	scrap = ensure_bookmark_file(file, subfile, _("Scrap Bookmarks"), "SCRAPBOOKMARK");

	if (root->scrap)
	{
		kz_bookmark_remove(KZ_BOOKMARK(root), root->scrap);
		g_object_unref(G_OBJECT(root->scrap));
		root->scrap = NULL;
	}

	if (scrap)
	{
		root->scrap = scrap;
		kz_bookmark_insert_before(KZ_BOOKMARK(root),
					  scrap, root->bookmark_bars);
	}
}


void
kz_root_bookmark_add_bookmark_bar_file (KzRootBookmark *root,
					const gchar *file, const gchar *subfile)
{
	KzBookmark *bar;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);
	g_return_if_fail(root->bookmark_bars);

	/* FIXME! Check permission */

	if (g_file_test(file, G_FILE_TEST_EXISTS))
	{
		bar = kz_bookmark_file_new(file,
					   _("Bookmark Bar"),
					   "XBEL");
	}
	else if (subfile && *subfile &&
		 g_file_test(subfile, G_FILE_TEST_EXISTS))
	{
		bar = kz_bookmark_file_new(subfile,
					   _("Bookmark Bar"),
					   "XBEL");
	}
	else
	{
		bar = kz_bookmark_file_create_new(file,
						  _("Bookmark Bar"),
						  "XBEL");
	}

	kz_bookmark_load_in_memory_start(bar);
	kz_bookmark_set_location(bar, file);

	kz_bookmark_append(root->bookmark_bars, bar);
	g_object_unref(bar);
}


void
kz_root_bookmark_save_all (KzRootBookmark *root)
{
	GList *node;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));

	if (root->menu)
		kz_bookmark_save(root->menu);
	if (root->home)
		kz_bookmark_save(root->home);
	if (root->scrap)
		kz_bookmark_save(root->scrap);

	if (root->bookmark_bars)
	{
		GList *children;

		children = kz_bookmark_get_children(root->bookmark_bars);
		for (node = children; node; node = g_list_next(node))
		{
			KzBookmark *bar = node->data;

			if (!KZ_IS_BOOKMARK(bar) || !kz_bookmark_is_file(bar))
				g_warning("Invalid bookmark bar file!");
			kz_bookmark_save(bar);
		}
		g_list_free(children);
	}
}
