/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmark-edit.c,v 1.12 2004/03/14 11:43:01 makeinu Exp $
 */

#include "kz-bookmark-edit.h"
#include "gobject-utils.h"
#include "intl.h"

static GtkVBoxClass *parent_class = NULL;


static void kz_bookmark_edit_class_init (KzBookmarkEditClass *klass);
static void kz_bookmark_edit_init       (KzBookmarkEdit *edit);
static void kz_bookmark_edit_dispose    (GObject        *object);

static void kz_bookmark_edit_set_sensitive   (KzBookmarkEdit *edit);
static void kz_bookmark_edit_sync_properties (KzBookmarkEdit *edit,
					      KzBookmark *bookmark);

static void cb_bookmark_notify          (GObject *object,
					 GParamSpec *pspec,
					 KzBookmarkEdit *edit);
static void cb_title_entry_changed      (GtkEditable    *editable,
					 KzBookmarkEdit *edit);
static void cb_uri_entry_changed        (GtkEditable    *editable,
					 KzBookmarkEdit *edit);
static void cb_location_entry_changed   (GtkEditable    *editable,
					 KzBookmarkEdit *edit);
static void cb_interval_value_changed   (GtkSpinButton  *button,
					 KzBookmarkEdit *edit);
static void cb_description_changed      (GtkTextBuffer  *textbuf,
					 KzBookmarkEdit *edit);


KZ_OBJECT_GET_TYPE(kz_bookmark_edit, "KzBookmarkEdit", KzBookmarkEdit,
		   kz_bookmark_edit_class_init, kz_bookmark_edit_init,
		   GTK_TYPE_VBOX)

static void
kz_bookmark_edit_class_init (KzBookmarkEditClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
	object_class  = (GtkObjectClass *) klass;
	widget_class  = (GtkWidgetClass *) klass;

	gobject_class->dispose = kz_bookmark_edit_dispose;
}


static void
kz_bookmark_edit_init (KzBookmarkEdit *edit)
{
	GtkVBox *main_vbox;
	GtkWidget *hbox, *table, *scrwin;
	GtkWidget *title_label, *title_entry;
	GtkWidget *uri_label, *uri_entry;
	GtkWidget *location_label, *location_entry;
	GtkWidget *interval_label, *interval_spin;
	GtkWidget *memo_label, *textview;
	GtkTextBuffer *textbuf;

	main_vbox = GTK_VBOX(edit);

	table = gtk_table_new(5, 2, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), table, TRUE, TRUE, 0);
	gtk_widget_show(table);

	/* Title */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 0, 1,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	title_label = gtk_label_new(_("Title:"));
	gtk_box_pack_start(GTK_BOX(hbox), title_label, FALSE, FALSE, 0);
	gtk_widget_show(title_label);

	title_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), title_entry,
			 1, 2, 0, 1,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(G_OBJECT(title_entry), "changed",
			 G_CALLBACK(cb_title_entry_changed), edit);
	gtk_widget_show(title_entry);

	/* URI */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 1, 2,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	uri_label = gtk_label_new(_("URI:"));
	gtk_box_pack_start(GTK_BOX(hbox), uri_label, FALSE, FALSE, 0);
	gtk_widget_show(uri_label);

	uri_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), uri_entry,
			 1, 2, 1, 2,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(G_OBJECT(uri_entry), "changed",
			 G_CALLBACK(cb_uri_entry_changed), edit);
	gtk_widget_show(uri_entry);

	/* Location */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 2, 3,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	location_label = gtk_label_new(_("Location:"));
	gtk_box_pack_start(GTK_BOX(hbox), location_label, FALSE, FALSE, 0);
	gtk_widget_show(location_label);

	location_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), location_entry,
			 1, 2, 2, 3,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(G_OBJECT(location_entry), "changed",
			 G_CALLBACK(cb_location_entry_changed), edit);
	gtk_widget_show(location_entry);

	/* Update interval */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 2, 3, 4,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	interval_label = gtk_label_new(_("Update interval (min):"));
	gtk_box_pack_start(GTK_BOX(hbox), interval_label, FALSE, FALSE, 0);
	gtk_widget_show(interval_label);

	interval_spin = gtk_spin_button_new_with_range(0, 60 * 24 * 7, 1);
	gtk_box_pack_start(GTK_BOX(hbox), interval_spin, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(interval_spin), "value-changed",
			 G_CALLBACK(cb_interval_value_changed), edit);
	gtk_widget_show(interval_spin);

	/* Memo */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 4, 5,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	memo_label = gtk_label_new(_("Memo:"));
	gtk_box_pack_start(GTK_BOX(hbox), memo_label, FALSE, FALSE, 0);
	gtk_widget_show(memo_label);

	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(table), scrwin,
			 1, 2, 4, 5,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(scrwin);

	textview = gtk_text_view_new();
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
	gtk_container_add(GTK_CONTAINER(scrwin), textview);
	textbuf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	g_signal_connect(G_OBJECT(textbuf), "changed",
			 G_CALLBACK(cb_description_changed), edit);
	gtk_widget_show(textview);

	/* initialize */
	edit->title_label    = title_label;
	edit->title_entry    = title_entry;
	edit->uri_label      = uri_label;
	edit->uri_entry      = uri_entry;
	edit->location_label = location_label;
	edit->location_entry = location_entry;
	edit->interval_label = interval_label;
	edit->interval_spin  = interval_spin;
	edit->memo_label     = memo_label;
	edit->memo_textview  = textview;
	edit->memo_scrwin    = scrwin;
	edit->bookmark       = NULL;
	edit->changing       = FALSE;
}


static void
kz_bookmark_edit_dispose (GObject *object)
{
	KzBookmarkEdit *edit = KZ_BOOKMARK_EDIT(object);

	if (edit->bookmark)
	{
		g_object_unref(edit->bookmark);
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(edit->bookmark),
			 G_CALLBACK(cb_bookmark_notify),
			 edit);
	}
	edit->bookmark = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


GtkWidget *
kz_bookmark_edit_new (void)
{
	KzBookmarkEdit *edit;

	edit = KZ_BOOKMARK_EDIT(g_object_new(KZ_TYPE_BOOKMARK_EDIT, NULL));

	return GTK_WIDGET(edit);
}


void
kz_bookmark_edit_set (KzBookmarkEdit *edit, KzBookmark *bookmark)
{
	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));

	kz_bookmark_edit_clear(edit);

	edit->bookmark = bookmark;
	if (!edit->bookmark) return;
	g_object_ref(edit->bookmark);

	g_signal_connect(G_OBJECT(edit->bookmark), "notify",
			 G_CALLBACK(cb_bookmark_notify), edit);

	kz_bookmark_edit_sync_properties (edit, bookmark);
}


void
kz_bookmark_edit_clear (KzBookmarkEdit *edit)
{
	GtkTextBuffer *textbuf;
	GtkTextIter start_iter, end_iter;
	GtkAdjustment *adj;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->bookmark)
	{
		g_object_unref(edit->bookmark);
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(edit->bookmark),
			 G_CALLBACK(cb_bookmark_notify),
			 edit);
	}
	edit->bookmark = NULL;

	gtk_entry_set_text(GTK_ENTRY(edit->title_entry),    "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->uri_entry),      "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->location_entry), "\0");
	adj = gtk_spin_button_get_adjustment
		(GTK_SPIN_BUTTON(edit->interval_spin));
	gtk_adjustment_set_value(adj, 0.0);

	textbuf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(edit->memo_textview));
	gtk_text_buffer_get_start_iter(textbuf, &start_iter);
	gtk_text_buffer_get_end_iter(textbuf, &end_iter);
	gtk_text_buffer_delete(textbuf, &start_iter, &end_iter);

	kz_bookmark_edit_set_sensitive(edit);
}


static void
kz_bookmark_edit_set_sensitive (KzBookmarkEdit *edit)
{
	gboolean can_edit_title = TRUE;
	gboolean can_edit_uri = TRUE;
	gboolean can_edit_desc = TRUE;
	gboolean can_edit_location = FALSE;
	gboolean can_edit_interval = FALSE;

	g_return_if_fail(KZ_BOOKMARK_EDIT(edit));

	/* set sensitive */
	if (!edit->bookmark ||
	    !kz_bookmark_is_editable(edit->bookmark) ||
	    kz_bookmark_is_separator(edit->bookmark))
	{
		can_edit_title    = FALSE;
		can_edit_uri      = FALSE;
		can_edit_desc     = FALSE;
		can_edit_location = FALSE;
		can_edit_interval = FALSE;
	}
	else if (edit->bookmark && kz_bookmark_is_pure_folder(edit->bookmark))
	{
		can_edit_uri = FALSE;
	}
	else if (edit->bookmark && kz_bookmark_is_file(edit->bookmark))
	{
		can_edit_location = TRUE;
		can_edit_interval = TRUE;
	}

	gtk_editable_set_editable(GTK_EDITABLE(edit->title_entry),
				  can_edit_title);
	gtk_editable_set_editable(GTK_EDITABLE(edit->uri_entry),
				  can_edit_uri);
	gtk_editable_set_editable(GTK_EDITABLE(edit->location_entry),
				  can_edit_location);
	gtk_widget_set_sensitive(edit->interval_spin, can_edit_interval);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(edit->memo_textview),
				   can_edit_desc);

	/* set visible */
	if (edit->bookmark && kz_bookmark_is_file(edit->bookmark))
	{
		gtk_widget_hide(edit->uri_label);
		gtk_widget_hide(edit->uri_entry);
		gtk_widget_hide(edit->memo_label);
		gtk_widget_hide(edit->memo_scrwin);

		gtk_widget_show(edit->location_label);
		gtk_widget_show(edit->location_entry);
		gtk_widget_show(edit->interval_label);
		gtk_widget_show(edit->interval_spin);
	}
	else
	{
		if (edit->bookmark &&
		    kz_bookmark_is_pure_folder(edit->bookmark))
		{
			gtk_widget_hide(edit->uri_label);
			gtk_widget_hide(edit->uri_entry);
		}
		else
		{
			gtk_widget_show(edit->uri_label);
			gtk_widget_show(edit->uri_entry);
		}
		gtk_widget_show(edit->memo_label);
		gtk_widget_show(edit->memo_scrwin);

		gtk_widget_hide(edit->location_label);
		gtk_widget_hide(edit->location_entry);
		gtk_widget_hide(edit->interval_label);
		gtk_widget_hide(edit->interval_spin);
	}
}


static void
kz_bookmark_edit_sync_properties (KzBookmarkEdit *edit, KzBookmark *bookmark)
{
	GtkTextBuffer *buffer;
	GtkAdjustment *adj;
	const gchar *title, *uri, *location, *desc;
	guint interval;

	edit->changing = TRUE;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));

	title    = kz_bookmark_get_title(bookmark);
	uri      = kz_bookmark_get_link(bookmark);
	location = kz_bookmark_get_location(bookmark);
	interval = kz_bookmark_get_interval(bookmark);
	desc     = kz_bookmark_get_description(bookmark);

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(edit->memo_textview));

	if (title)
		gtk_entry_set_text(GTK_ENTRY(edit->title_entry), title);
	if (uri)
		gtk_entry_set_text(GTK_ENTRY(edit->uri_entry), uri);
	if (location)
		gtk_entry_set_text(GTK_ENTRY(edit->location_entry), location);
	adj = gtk_spin_button_get_adjustment
		(GTK_SPIN_BUTTON(edit->interval_spin));
	gtk_adjustment_set_value(adj, interval);

	if (desc)
	{
		GtkTextView *textview = GTK_TEXT_VIEW(edit->memo_textview);
		GtkTextIter iter, end_iter;

		buffer = gtk_text_view_get_buffer(textview);
		gtk_text_buffer_get_start_iter(buffer, &iter);
		gtk_text_buffer_get_end_iter(buffer, &end_iter);
		gtk_text_buffer_delete(buffer, &iter, &end_iter);
		gtk_text_buffer_insert(buffer, &iter, desc, -1);
	}

	kz_bookmark_edit_set_sensitive(edit);

	edit->changing = FALSE;
}


static void
cb_bookmark_notify (GObject *object, GParamSpec *pspec,
		    KzBookmarkEdit *edit)
{
	KzBookmark *bookmark;

	g_return_if_fail(KZ_IS_BOOKMARK(object));
	bookmark = KZ_BOOKMARK(object);

	kz_bookmark_edit_sync_properties(edit, bookmark);
}


static void
cb_title_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *title;

	g_return_if_fail(GTK_IS_EDITABLE(editable));
	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

	title = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_set_title(edit->bookmark, title);
}


static void
cb_uri_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *uri;

	g_return_if_fail(GTK_IS_EDITABLE(editable));
	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

	uri = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_set_link(edit->bookmark, uri);
}


static void
cb_location_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *uri;

	g_return_if_fail(GTK_IS_EDITABLE(editable));
	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

        uri = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_set_location(edit->bookmark, uri);
}


static void
cb_interval_value_changed (GtkSpinButton *button, KzBookmarkEdit *edit)
{
	GtkAdjustment *adj;
	gint interval;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (!KZ_IS_BOOKMARK(edit->bookmark)) return;

	adj = gtk_spin_button_get_adjustment
		(GTK_SPIN_BUTTON(edit->interval_spin));
	interval = gtk_adjustment_get_value(adj);
	kz_bookmark_set_interval(edit->bookmark, interval);
}


static void
cb_description_changed (GtkTextBuffer *textbuf, KzBookmarkEdit *edit)
{
	GtkTextIter start_iter, end_iter;
	gchar *desc;

	g_return_if_fail(GTK_IS_TEXT_BUFFER(textbuf));
	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

	gtk_text_buffer_get_start_iter(textbuf, &start_iter);
	gtk_text_buffer_get_end_iter(textbuf, &end_iter);

	desc = gtk_text_buffer_get_text(GTK_TEXT_BUFFER(textbuf),
					&start_iter, &end_iter,
					FALSE);

	kz_bookmark_set_description(edit->bookmark, desc);

	g_free(desc);
}
