/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2007 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <glib/gi18n.h>

#define __USE_XOPEN
#include <time.h>
#include "kazehakase.h"
#include "kz-ie.h"
#include "kz-embed-prefs.h"
#include "kz-embed.h"
#include "kz-prompt-dialog.h"
#include "kz-proxy-item.h"
#include "gtk-utils.h"
#include "kz-missing.h"
#include "kz-bookmark.h"

typedef struct _KzIEPrivate	KzIEPrivate;
struct _KzIEPrivate
{
	glong current_progress;
	glong max_progress;
	gchar *location;
	gchar *title;
	gboolean lock;
};
#define KZ_IE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_IE, KzIEPrivate))

static GType           kz_type_ie = 0;
static GtkIEEmbedClass *kz_ie_parent_class;

GtkWidget  *kz_ie_new          (void);
static void kz_ie_class_init   (KzIEClass *klass);
static void kz_ie_iface_init   (KzEmbedIFace *iface);
static void kz_ie_init         (KzIE *ie);
static void kz_ie_dispose      (GObject      *object);

/* GtkIEEmbed signals */
static void	    kz_ie_embed_location	 (GtkIEEmbed   *ieembed,
						  const gchar  *location);
static void	    kz_ie_embed_title		 (GtkIEEmbed   *ieembed,
						  const gchar  *title);
static void	    kz_ie_embed_progress	 (GtkIEEmbed   *ieembed,
						  glong		current_progress,
						  glong		max_progress);
static void	    kz_ie_embed_net_start	 (GtkIEEmbed   *ieembed);
static void	    kz_ie_embed_net_stop	 (GtkIEEmbed   *ieembed);
static gboolean	    kz_ie_embed_dom_mouse_down   (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_dom_mouse_up     (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_dom_mouse_move   (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_dom_mouse_click  (GtkIEEmbed   *embed,
						  GtkIEEmbedDOMMouseEvent *event);
static gboolean	    kz_ie_embed_new_window	 (GtkIEEmbed   *embed, GtkIEEmbed **newembed);
static gboolean	    kz_ie_embed_close_window	 (GtkIEEmbed   *embed);
static void	    kz_ie_embed_selection_changed(GtkIEEmbed   *embed);
/* embed iface */
static void         kz_ie_load_url         (KzEmbed      *kzembed,
					    const gchar  *url);
static const gchar *kz_ie_get_title        (KzEmbed      *kzembed);
static const gchar *kz_ie_get_location     (KzEmbed      *kzembed);
static gdouble	    kz_ie_get_progress	   (KzEmbed	 *kzembed);
static void         kz_ie_reload           (KzEmbed      *kzembed,
					    KzEmbedReloadFlag flags);
static void         kz_ie_stop_load        (KzEmbed      *kzembed);
static gboolean	    kz_ie_is_loading	   (KzEmbed	 *kzembed);
static void         kz_ie_go_back          (KzEmbed      *kzembed);
static void         kz_ie_go_forward       (KzEmbed      *kzembed);
static gboolean     kz_ie_can_go_back      (KzEmbed      *kzembed);
static gboolean     kz_ie_can_go_forward   (KzEmbed      *kzembed);
static gboolean     kz_ie_can_cut_selection
					   (KzEmbed      *kzembed);
static gboolean     kz_ie_can_copy_selection
					   (KzEmbed      *kzembed);
static gboolean     kz_ie_can_paste        (KzEmbed      *kzembed);
static void         kz_ie_cut_selection    (KzEmbed      *kzembed);
static void         kz_ie_copy_selection   (KzEmbed      *kzembed);
static void         kz_ie_paste            (KzEmbed      *kzembed);
static void         kz_ie_select_all       (KzEmbed      *kzembed);
static void         kz_ie_print            (KzEmbed      *kzembed);
static void         kz_ie_print_preview    (KzEmbed      *kzembed);
static void	    kz_ie_set_text_size	   (KzEmbed      *kzembed, 
					    gint          zoom, 
					    gboolean      reflow);
static gint	    kz_ie_get_text_size	   (KzEmbed      *kzembed);
static void	    kz_ie_set_encoding	   (KzEmbed      *kzembed,
					    const gchar  *encoding);
static void	    kz_ie_get_encoding	   (KzEmbed      *kzembed,
					    gchar       **encoding,
					    gboolean     *forced);
static guint	    kz_ie_get_last_modified
					   (KzEmbed	 *kzembed);
static gchar       *kz_ie_get_selection_string 
					   (KzEmbed      *kzembed);
static gboolean	    kz_ie_find		   (KzEmbed	 *kzembed,
					    const gchar  *keyword,
					    gboolean      backward);
static gboolean     kz_ie_get_lock         (KzEmbed      *kzembed);
static void         kz_ie_set_lock         (KzEmbed      *kzembed,
					    gboolean      lock);
static void	    get_history            (KzEmbed      *kzembed,
					    KzBookmarkFolder   *history);
static void	    set_history            (KzEmbed      *kzembed,
					    KzBookmarkFolder   *history);
static void         copy_page              (KzEmbed      *src_embed,
					    KzEmbed      *dest_embed);
static gboolean     shistory_get_pos       (KzEmbed      *kzembed,
					    gint         *pos,
					    gint         *count);
static void         shistory_get_nth       (KzEmbed      *kzembed, 
					    gint          nth,
					    gboolean      is_relative,
					    gchar       **uri,
					    gchar       **title);
static void         go_history_index       (KzEmbed      *kzembed,
					    gint          index);

/* KzEmbedPrefs interfaces */
/* no functions is implemented yet */
static void kz_ie_prefs_iface_init (KzEmbedPrefsIFace *iface);

void
kz_ie_register_type (GTypeModule *module)
{
	static const GTypeInfo kz_ie_info =
	{
		sizeof (KzIEClass),
		NULL,		/* base_init */
		NULL,		/* base_finalize */
		(GClassInitFunc) kz_ie_class_init,
		NULL,		/* class_finalize */
		NULL,		/* class_data */
		sizeof (KzIE),
		0,		/* n_preallocs */
		(GInstanceInitFunc) kz_ie_init,
	};

	const GInterfaceInfo kz_embed_info =
	{
		(GInterfaceInitFunc) kz_ie_iface_init,
		NULL,
		NULL
	};

	const GInterfaceInfo kz_embed_prefs_info =
	{
		(GInterfaceInitFunc) kz_ie_prefs_iface_init,
		NULL,
		NULL
	};

	kz_type_ie = g_type_module_register_type(module,
						 GTK_TYPE_IE_EMBED,
						 "KzIE",
						 &kz_ie_info,
						 (GTypeFlags)0);

	g_type_module_add_interface(module,
				    KZ_TYPE_IE,
				    KZ_TYPE_EMBED,
				    &kz_embed_info);

	g_type_module_add_interface(module,
				    KZ_TYPE_IE,
				    KZ_TYPE_EMBED_PREFS,
				    &kz_embed_prefs_info);
}

GType
kz_ie_get_type (void)
{
       return kz_type_ie;
}

static void
kz_ie_class_init (KzIEClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;
	GtkIEEmbedClass *ie_class;

	kz_ie_parent_class = GTK_IE_EMBED_CLASS(g_type_class_peek_parent(klass));
	object_class = G_OBJECT_CLASS(klass);
	widget_class = GTK_WIDGET_CLASS(klass);
	ie_class = GTK_IE_EMBED_CLASS(klass);

	object_class->dispose = kz_ie_dispose;

	ie_class->location          = kz_ie_embed_location;
	ie_class->title             = kz_ie_embed_title;
	ie_class->net_start         = kz_ie_embed_net_start;
	ie_class->net_stop          = kz_ie_embed_net_stop;
	ie_class->progress          = kz_ie_embed_progress;
	ie_class->dom_mouse_down    = kz_ie_embed_dom_mouse_down;
	ie_class->dom_mouse_move    = kz_ie_embed_dom_mouse_move;
	ie_class->dom_mouse_up      = kz_ie_embed_dom_mouse_up;
	ie_class->dom_mouse_click   = kz_ie_embed_dom_mouse_click;
	ie_class->new_window	    = kz_ie_embed_new_window;
	ie_class->close_window	    = kz_ie_embed_close_window;
	ie_class->selection_changed = kz_ie_embed_selection_changed;

	g_type_class_add_private(object_class, sizeof(KzIEPrivate));
}


static void
kz_ie_iface_init (KzEmbedIFace *iface)
{
	iface->load_url               = kz_ie_load_url;
	iface->view_source            = NULL;
	iface->view_current_page_source_in_new
				      = NULL;
	iface->is_loading             = kz_ie_is_loading;
	iface->get_title              = kz_ie_get_title;
	iface->get_location           = kz_ie_get_location;
	iface->get_link_message       = NULL;
	iface->get_progress           = kz_ie_get_progress;
	iface->can_cut_selection      = kz_ie_can_cut_selection;
	iface->can_copy_selection     = kz_ie_can_copy_selection;
	iface->can_paste              = kz_ie_can_paste;
	iface->cut_selection          = kz_ie_cut_selection;
	iface->copy_selection         = kz_ie_copy_selection;
	iface->paste                  = kz_ie_paste;
	iface->select_all             = kz_ie_select_all;
	iface->get_selection_string   = kz_ie_get_selection_string;
	iface->find                   = kz_ie_find;
	iface->incremental_search     = NULL;
	iface->selection_is_collapsed = NULL;
	iface->get_links              = NULL;
	iface->copy_page              = copy_page;
	iface->shistory_get_pos       = shistory_get_pos;
	iface->shistory_get_nth       = shistory_get_nth;
	iface->reload                 = kz_ie_reload;
	iface->stop_load              = kz_ie_stop_load;
	iface->go_back                = kz_ie_go_back;
	iface->go_forward             = kz_ie_go_forward;
	iface->can_go_back            = kz_ie_can_go_back;
	iface->can_go_forward         = kz_ie_can_go_forward;
	iface->can_go_nav_link        = NULL;
	iface->go_nav_link            = NULL;
	iface->append_nav_link        = NULL;
	iface->set_nav_link           = NULL;
	iface->set_nth_nav_link       = NULL;
	iface->get_nav_link           = NULL;
	iface->get_nth_nav_link       = NULL;
	iface->get_nav_links          = NULL;
	iface->go_history_index       = go_history_index;
	iface->do_command             = NULL;
	iface->can_do_command         = NULL;
	iface->get_lock               = kz_ie_get_lock;
	iface->set_lock               = kz_ie_set_lock;
	iface->get_body_text          = NULL;
#if 0
	iface->get_selection_source   = NULL;
#endif
	iface->set_encoding           = kz_ie_set_encoding;
	iface->get_encoding           = kz_ie_get_encoding;
	iface->print                  = kz_ie_print;
	iface->print_preview          = kz_ie_print_preview;
	iface->get_printer_list       = NULL;
	iface->create_thumbnail       = NULL;
	iface->save_with_content      = NULL;
	iface->set_text_into_textarea = NULL;
	iface->get_text_from_textarea = NULL;
	iface->zoom_set               = kz_ie_set_text_size;
	iface->zoom_get               = kz_ie_get_text_size;
	iface->set_text_size          = kz_ie_set_text_size;
	iface->get_text_size          = kz_ie_get_text_size;
	iface->get_html_with_contents = NULL;
	iface->set_history            = set_history;
	iface->get_history            = get_history;
	iface->get_last_modified      = kz_ie_get_last_modified;
	iface->fine_scroll            = NULL;
	iface->page_up                = NULL;
	iface->page_down              = NULL; 
	iface->get_allow_javascript   = NULL;
	iface->set_allow_javascript   = NULL; 
	iface->get_allow_images       = NULL;
	iface->set_allow_images       = NULL;
#if 0
	iface->set_edit_mode          = NULL;
	iface->set_view_mode          = NULL;
#endif

	iface->link_message           = NULL;
	iface->js_status              = NULL;
	iface->location               = NULL;
	iface->title                  = NULL;
	iface->progress               = NULL;
	iface->net_start              = NULL;
	iface->net_stop               = NULL;
	iface->new_window             = NULL;
	iface->open_uri               = NULL;
	iface->size_to                = NULL;
	iface->dom_key_down           = NULL;
	iface->dom_key_press          = NULL;
	iface->dom_key_up             = NULL;
	iface->dom_mouse_down         = NULL;
	iface->dom_mouse_up           = NULL;
	iface->dom_mouse_click        = NULL;
	iface->dom_mouse_dbl_click    = NULL;
	iface->dom_mouse_over         = NULL;
	iface->dom_mouse_out          = NULL;
	iface->security_change        = NULL;
	iface->status_change          = NULL;
}


static void
kz_ie_prefs_iface_init (KzEmbedPrefsIFace *iface)
{
	iface->init             = NULL; /* kz_ie_prefs_init;*/
	iface->get_font_list    = NULL; /* kz_ie_prefs_get_font_list; */
	iface->get_passwords    = NULL; /* kz_ie_prefs_get_passwords; */
	iface->remove_passwords = NULL; /* kz_ie_prefs_remove_passwords; */
}

static void
kz_ie_init (KzIE *ie)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE (ie);

	priv->location = NULL;
	priv->title = NULL;

	priv->current_progress = 0;
	priv->max_progress =0;

	priv->lock = FALSE;

	gtk_ie_embed_set_use_context_menu (GTK_IE_EMBED (ie), FALSE);
}

static void
kz_ie_dispose (GObject *object)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(object);

	if (priv->location)
		g_free(priv->location);
	if (priv->title)
		g_free(priv->title);
	priv->location = NULL;
	priv->title = NULL;

	if (G_OBJECT_CLASS(kz_ie_parent_class)->dispose)
		G_OBJECT_CLASS(kz_ie_parent_class)->dispose(object);
}

GtkWidget *
kz_ie_new (void)
{
	return GTK_WIDGET(g_object_new(KZ_TYPE_IE,
				       "use-context-menu", FALSE,
				       NULL));
}

static void
kz_ie_embed_location (GtkIEEmbed *ieembed, const gchar  *location)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(ieembed);

	if (priv->location)
		g_free(priv->location);
	priv->location = g_strdup(location);
	g_signal_emit_by_name(ieembed, "kz-location", location);
}

static void
kz_ie_embed_title (GtkIEEmbed *ieembed, const gchar *title)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(ieembed);

	if (priv->title)
		g_free(priv->title);
	priv->title = g_strdup(title);
	g_signal_emit_by_name(ieembed, "kz-title", title);
}

static void
kz_ie_embed_progress (GtkIEEmbed *ieembed,
		      glong current_progress,
		      glong max_progress)
{
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE(ieembed);
	gdouble progress = 0.0;

	priv->current_progress = current_progress;
	priv->max_progress = max_progress;

	if (priv->max_progress > 0 ||
	    priv->current_progress >= 0)
	{
		progress = (gdouble) priv->current_progress
			/ (gdouble) priv->max_progress;
	}


	g_signal_emit_by_name(ieembed, "kz-progress", progress);
}

static void
kz_ie_embed_net_start (GtkIEEmbed *ieembed)
{
	g_signal_emit_by_name(ieembed, "kz-net-start");
}

static void
kz_ie_embed_net_stop (GtkIEEmbed *ieembed)
{
	if (GTK_IE_EMBED_CLASS(kz_ie_parent_class)->net_stop)
		GTK_IE_EMBED_CLASS(kz_ie_parent_class)->net_stop(ieembed);

	g_signal_emit_by_name(ieembed, "kz-net-stop");
}

static KzEmbedEventMouse *
create_kz_embed_event_mouse (GtkIEEmbedDOMMouseEvent *event)
{
	KzEmbedEventMouse *kzevent;

	kzevent = (KzEmbedEventMouse *) kz_embed_event_new(KZ_EMBED_EVENT_MOUSE);
	if (event->shift_key)
		kzevent->modifier |= KZ_SHIFT_KEY;
	if (event->control_key)
		kzevent->modifier |= KZ_CTRL_KEY;
	if (event->alt_key)
		kzevent->modifier |= KZ_ALT_KEY;
	if (event->meta_key)
		kzevent->modifier |= KZ_META_KEY;
	kzevent->x = event->x;
	kzevent->y = event->y;
	kzevent->button = event->button;
	kzevent->cinfo.context = KZ_CONTEXT_NONE;

	if (event->target)
	{
		const gchar *name;
		name = gtk_ie_embed_dom_event_target_get_name(event->target);
		if (!g_ascii_strcasecmp("a", name))
		{
			const gchar *value;
			kzevent->cinfo.context = KZ_CONTEXT_LINK;
			value = gtk_ie_embed_dom_event_target_get_attribute_value(event->target,
									     	  "href");
			kzevent->cinfo.link = g_strdup(value);
		}
	}
	return kzevent;
}

static gboolean
emit_mouse_signal (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event,
		   const gchar *signal_name)
{
	KzEmbedEventMouse *kzevent;
	gint ret = FALSE;

	kzevent = create_kz_embed_event_mouse(event);
	g_signal_emit_by_name(embed, signal_name, kzevent, &ret);
	kz_embed_event_free((KzEmbedEvent *) kzevent);

	return ret;
}

static gboolean
kz_ie_embed_dom_mouse_down (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-down");
}

static gboolean
kz_ie_embed_dom_mouse_up (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-up");
}

static gboolean
kz_ie_embed_dom_mouse_move (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-move");
}

static gboolean
kz_ie_embed_dom_mouse_click (GtkIEEmbed *embed, GtkIEEmbedDOMMouseEvent *event)
{
	return emit_mouse_signal(embed, event, "kz-dom-mouse-click");
}

static gboolean
kz_ie_embed_new_window (GtkIEEmbed *embed, GtkIEEmbed **newembed)
{
	g_signal_emit_by_name(embed, "kz-new-window", newembed);

	return newembed ? FALSE : TRUE;
}

static gboolean
kz_ie_embed_close_window (GtkIEEmbed *embed)
{
	gtk_widget_destroy(GTK_WIDGET(embed));
	return FALSE;
}

static void
kz_ie_embed_selection_changed (GtkIEEmbed *embed)
{
	g_signal_emit_by_name(embed, "kz-selection");
}

static void
kz_ie_load_url (KzEmbed *kzembed, const gchar  *url)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_load_url(GTK_IE_EMBED(kzembed), url);
}

static const gchar *
kz_ie_get_title (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), NULL);

	return KZ_IE_GET_PRIVATE(kzembed)->title;
}

static const gchar *
kz_ie_get_location (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), NULL);

	return KZ_IE_GET_PRIVATE(kzembed)->location;
}

static gdouble
kz_ie_get_progress (KzEmbed *kzembed)
{
	gdouble progress;
	KzIEPrivate *priv;

	g_return_val_if_fail(KZ_IS_IE(kzembed), 0.0);

	priv = KZ_IE_GET_PRIVATE(kzembed);

	if (priv->max_progress <= 0 ||
	    priv->current_progress <= 0)
	{
		return 0.0;
	}

	progress = (gdouble) priv->current_progress
		/ (gdouble) priv->max_progress;

	return (progress < 1.0) ? progress : 1.0;
}

static void
kz_ie_reload (KzEmbed *kzembed, KzEmbedReloadFlag flags)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_reload(GTK_IE_EMBED(kzembed), GTK_IE_EMBED_RELOAD_NORMAL);
}


static void
kz_ie_stop_load (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_stop(GTK_IE_EMBED(kzembed));
}


static gboolean
kz_ie_is_loading (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_is_loading(GTK_IE_EMBED(kzembed));
}


static void
kz_ie_go_back (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_go_back(GTK_IE_EMBED(kzembed));
}


static void
kz_ie_go_forward (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_go_forward(GTK_IE_EMBED(kzembed));
}


static gboolean
kz_ie_can_go_back (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_go_back(GTK_IE_EMBED(kzembed));
}


static gboolean
kz_ie_can_go_forward (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_go_forward(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_can_cut_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_cut_clipboard(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_can_copy_selection (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_copy_clipboard(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_can_paste (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_can_paste_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_cut_selection (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_cut_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_copy_selection (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_copy_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_paste (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_paste_clipboard(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_select_all (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_select_all(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_print (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_print(GTK_IE_EMBED(kzembed));
}


static void
kz_ie_print_preview (KzEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_print_preview(GTK_IE_EMBED(kzembed));
}

static void
kz_ie_set_text_size (KzEmbed *kzembed, int zoom, gboolean reflow)
{
	GtkIEEmbedFontSize size;

	g_return_if_fail(KZ_IS_IE(kzembed));

	if (zoom > 400)
		size = GTK_IE_EMBED_FONT_LARGEST;
	else if (zoom > 200)
		size = GTK_IE_EMBED_FONT_LARGE;
	else if (zoom > 80)
		size = GTK_IE_EMBED_FONT_MEDIUM;
	else if (zoom > 50)
		size = GTK_IE_EMBED_FONT_SMALL;
	else 
		size = GTK_IE_EMBED_FONT_SMALLEST;
		
	gtk_ie_embed_set_font_size(GTK_IE_EMBED(kzembed), size);
}

static int
kz_ie_get_text_size (KzEmbed *kzembed)
{
	GtkIEEmbedFontSize size;
	gint ret_size;

	g_return_val_if_fail(KZ_IS_IE(kzembed), 100);

	size = gtk_ie_embed_get_font_size(GTK_IE_EMBED(kzembed));
	switch (size)
	{
	case GTK_IE_EMBED_FONT_LARGEST:
		ret_size = 400;
		break;
	case GTK_IE_EMBED_FONT_LARGE:
		ret_size = 200;
		break;
	case GTK_IE_EMBED_FONT_MEDIUM:
		ret_size = 80;
		break;
	case GTK_IE_EMBED_FONT_SMALL:
		ret_size = 50;
		break;
	case GTK_IE_EMBED_FONT_SMALLEST:
		ret_size = 20;
		break;
	default:
		ret_size = 100;
		break;
	}

	return ret_size;
}

static void
kz_ie_set_encoding (KzEmbed *kzembed, const gchar *encoding)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	gtk_ie_embed_set_charset(GTK_IE_EMBED(kzembed), encoding);
}

static void
kz_ie_get_encoding (KzEmbed *kzembed, gchar **encoding, gboolean *forced)
{
	g_return_if_fail(KZ_IS_IE(kzembed));

	*encoding = gtk_ie_embed_get_charset(GTK_IE_EMBED(kzembed));
}

static guint
kz_ie_get_last_modified (KzEmbed *kzembed)
{
	gchar *string;
	struct tm t = {0};

	g_return_val_if_fail(KZ_IS_IE(kzembed), 0);

	string = gtk_ie_embed_get_last_modified(GTK_IE_EMBED(kzembed));
	strptime(string, "%m/%d/%C %H:%M:%S", &t);
	g_free (string);

	return (guint) mktime(&t);
}

static gchar *
kz_ie_get_selection_string (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), NULL);

	return gtk_ie_embed_get_selected_text(GTK_IE_EMBED(kzembed));
}

static gboolean
kz_ie_find (KzEmbed *kzembed, const char *keyword, gboolean backward)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);

	return gtk_ie_embed_find_string(GTK_IE_EMBED(kzembed), keyword, !backward, FALSE);
}

static gboolean
kz_ie_get_lock (KzEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_IE(kzembed), FALSE);
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE (kzembed);
	return priv->lock;
}


static void
kz_ie_set_lock (KzEmbed *kzembed, gboolean lock)
{
	g_return_if_fail(KZ_IS_IE(kzembed));
	KzIEPrivate *priv = KZ_IE_GET_PRIVATE (kzembed);
	priv->lock = lock;
}

static void
merge_to_bookmark_list (const gchar *uri, const gchar *title,
			GList **bookmark_list_node, KzBookmarkFolder *folder)
{
	if (*bookmark_list_node)
	{
		kz_bookmark_set_title(KZ_BOOKMARK((*bookmark_list_node)->data), title);
		kz_bookmark_set_link(KZ_BOOKMARK((*bookmark_list_node)->data), uri);
		*bookmark_list_node = g_list_next(*bookmark_list_node);
	}
	else
	{
		KzBookmark *bookmark;
		bookmark = kz_bookmark_new_with_attrs(title, uri, NULL);
		kz_bookmark_folder_append(folder, bookmark);
		g_object_unref(bookmark);
	}
}

static guint
merge_ie_history_list_to_bookmark_list (const GList *history_list, GList **bookmark_list_node,
					KzBookmarkFolder *folder)
{
	const GList *node;
	guint count = 0;

	for (node = history_list; node; node = g_list_next(node))
	{
		const gchar *uri, *title;
		GtkIEEmbedHistoryItem *item = GTK_IE_EMBED_HISTORY_ITEM(node->data);

		uri = gtk_ie_embed_history_item_get_uri(item);
		title = gtk_ie_embed_history_item_get_title(item);
		merge_to_bookmark_list(uri, title, bookmark_list_node, folder);
		count++;
	}
	return count;
}

static void
history_list_free (GList *list)
{
	g_list_foreach(list, (GFunc)g_object_unref, NULL);
	g_list_free(list);
}

static void
get_history (KzEmbed *kzembed, KzBookmarkFolder *history)
{
	GList *children, *bookmark_node;
	GList *backward_history, *forward_history;
	gint current_position = 0;
	const gchar *uri, *title;

	backward_history = gtk_ie_embed_get_backward_history(GTK_IE_EMBED(kzembed));
	forward_history = gtk_ie_embed_get_forward_history(GTK_IE_EMBED(kzembed));

	children = kz_bookmark_folder_get_children(history);
	bookmark_node = children;
	current_position = merge_ie_history_list_to_bookmark_list(backward_history,
							          &bookmark_node, 
								  history);

	uri = KZ_IE_GET_PRIVATE(kzembed)->location;
	title = KZ_IE_GET_PRIVATE(kzembed)->title;
	merge_to_bookmark_list(uri, title, &bookmark_node, history);

	merge_ie_history_list_to_bookmark_list(forward_history,
					       &bookmark_node,
					       history);

	if (children)
		g_list_free(children);
	if (backward_history)
		history_list_free(backward_history);
	if (forward_history)
		history_list_free(forward_history);

	kz_bookmark_folder_set_current_position(history, current_position);
}

static void
append_history_item (KzBookmark *bookmark, GList **history_item_list)
{
        GtkIEEmbedHistoryItem *item;

        item = gtk_ie_embed_history_item_new(kz_bookmark_get_link(bookmark),
					     kz_bookmark_get_title(bookmark));
        *history_item_list =
                g_list_append(*history_item_list, item);
}

static GList *
bookmark_to_history_item_list (KzBookmarkFolder *history)
{
	GList *history_item_list = NULL;

	kz_bookmark_folder_foreach_child(history, 
					 (GFunc)append_history_item, &history_item_list);

	return history_item_list;
}

static void
set_history (KzEmbed *kzembed, KzBookmarkFolder *history)
{
	GList *history_item_list;
	gint current_position;

	current_position = kz_bookmark_folder_get_current_position(history);

	history_item_list = bookmark_to_history_item_list(history);
	if (!history_item_list)
		return;

	gtk_ie_embed_set_whole_history(GTK_IE_EMBED(kzembed),
				       history_item_list, current_position);
	history_list_free(history_item_list);
}

static void
copy_page (KzEmbed *src_embed, KzEmbed *dest_embed)
{
        GList *backward_history_list, *forward_history_list;
        GList *history_list = NULL;
        GtkIEEmbedHistoryItem *current_item;
        gint current_position;
        GtkIEEmbed *src_ie = GTK_IE_EMBED(src_embed);

        backward_history_list = gtk_ie_embed_get_backward_history(src_ie);
        forward_history_list = gtk_ie_embed_get_forward_history(src_ie);
        current_item = gtk_ie_embed_get_history_item_at_relative_position(src_ie, 0);
        current_position = backward_history_list ? g_list_length(backward_history_list) : 0;

        if (current_item)
                history_list = g_list_append(backward_history_list, current_item);
        history_list = g_list_concat(history_list, forward_history_list);

	gtk_ie_embed_set_whole_history(GTK_IE_EMBED(dest_embed),
				       history_list, current_position + 1);
        history_list_free(history_list);
        if (current_item)
                g_object_unref(current_item);
}

static gboolean 
shistory_get_pos (KzEmbed *kzembed, gint *pos, gint *count)
{
	GtkIEEmbed *ie = GTK_IE_EMBED(kzembed);

	*pos = (gint)gtk_ie_embed_get_current_position_in_history(ie);
	*count = (gint)gtk_ie_embed_get_history_count(ie);

	return TRUE;
}

static void
shistory_get_nth (KzEmbed *kzembed,
		  gint nth, gboolean is_relative,
		  gchar **uri, gchar **title)
{
        GtkIEEmbedHistoryItem *item;
	guint current_position;

	current_position = gtk_ie_embed_get_current_position_in_history(GTK_IE_EMBED(kzembed));
	if (!is_relative)
		nth -= current_position;
	item = gtk_ie_embed_get_history_item_at_relative_position(GTK_IE_EMBED(kzembed), nth);
	if (!item)
		return;

	*uri = g_strdup(gtk_ie_embed_history_item_get_uri(item));
	*title = g_strdup(gtk_ie_embed_history_item_get_title(item));
	g_object_unref(item);
}

static void
go_history_index (KzEmbed *kzembed, gint index)
{
	guint current_position;

	current_position = gtk_ie_embed_get_current_position_in_history(GTK_IE_EMBED(kzembed));
	gtk_ie_embed_go_relative_position(GTK_IE_EMBED(kzembed), index - current_position);
}


