/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <glib/gstdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include "utils.h"
#include "glib-utils.h"
#include "gtk-utils.h"
#include "config.h"
#include "kazehakase.h"
#include "kz-search.h"

#define BUFFER_SIZE 1024

#define TIME_STAMP_FORMAT "%ld,%s\n"

gboolean
kz_utils_cp (const gchar *from, const gchar *to)
{
	FILE *fp;
	gboolean success = FALSE;
	gchar *contents;
	gsize length;

	if (!g_file_get_contents(from, &contents, &length, NULL))
		return FALSE;

	fp = g_fopen(to, "wb");
	if (fp)
	{
		success = TRUE;
		if (fwrite(contents, length, 1, fp) != 1)
			success = FALSE;
		fclose(fp);

		if (!success)
			g_unlink(to);
	}

	g_free(contents);
	return success;
}

gboolean
key_seems_sequential (const gchar *key, const gchar *prefix)
{
	gint len;
	const gchar *tail;
	gint i;
	gboolean valid;

	g_return_val_if_fail(key && *key, FALSE);
	g_return_val_if_fail(prefix && *prefix, FALSE);

	len = strlen(prefix);
	tail = key + len;

	if (strncmp(key, prefix, len)) return FALSE;

	len = strlen(tail);
	if (len <= 0) return FALSE;

	valid = TRUE;
	for (i = 0; i < len; i++)
	{
		if (!isdigit(tail[i]))
		{
			valid = FALSE;
			break;
		}
	}

	return valid;
}


gchar *
remove_tag (const gchar *string, gsize len)
{
	GString *work_string;
	guint i = 0;
	
	work_string = g_string_new(NULL);
	while (string[i] != '\0' && i < len)
	{
		if (string[i] == '<')
		{
			while (string[i] != '>' && string[i] != '\0' && i < len)
			{
				i++;
			}
		}
		else
			g_string_append_c(work_string, string[i]);
		i++;
	}
	
	return g_string_free(work_string, FALSE);
}


gchar *
html_to_text (const gchar *html)
{
	GString *work_string;
	guint i=0;
	
	work_string = g_string_new(NULL);
	while (html[i] != '\0')
	{
		gboolean ignore = FALSE;
		switch (html[i])
		{
		case '<':
		{
			gchar *name = NULL;
			if (!g_ascii_strncasecmp(html+i+1, "script", 6))
			{
				name = "/script";
				ignore = TRUE;
			}
			else if (!g_ascii_strncasecmp(html+i+1, "style", 5)) 
			{
				name = "/style"; 
				ignore = TRUE;
			}
			else if (!g_ascii_strncasecmp(html+i+1, "noscript", 8)) 
			{
				name = "/noscript"; 
				ignore = TRUE;
			}
			while (html[i] != '>' && html[i] != '\0')
				i++;
			if (ignore)
			{
				/* ignore until close tag */
				while (html[i] != '<' && html[i] != '\0' &&
				       g_ascii_strncasecmp(html+i+1, name, strlen(name)))
					i++;
				while (html[i] != '>' && html[i] != '\0')
					i++;
			}
			break;
		}
		case '&':
			if (!strncmp(html+i, "&amp;", 5))
			{
				g_string_append_c(work_string, '&');
				i+=4;
			}
			else if (!strncmp(html+i, "&quot;", 6))
			{
				g_string_append_c(work_string, '"');
				i+=5;
			}
			else if (!strncmp(html+i, "&lt;", 4))
			{
				g_string_append_c(work_string, '<');
				i+=3;
			}
			else if (!strncmp(html+i, "&gt;", 4))
			{
				g_string_append_c(work_string, '>');
				i+=3;
			}
			else
			{
				g_string_append_c(work_string, html[i]);
			}
			break;
		default:
			g_string_append_c(work_string, html[i]);
			break;
		}
		i++;
	}
	
	return g_string_free(work_string, FALSE);
}


gchar *
create_filename_from_uri (const gchar *uri)
{
	gchar *filename;
	gint i = 0;
	
	filename = g_strdup(uri);

	while (filename[i] != '\0')
	{
		if (filename[i] == '/')
			filename[i] = '_';
		i++;
	}

	return filename;
}


gchar *
create_filename_with_path_from_uri (const gchar *uri)
{
	gchar *filename;
	gchar *pos;
	gchar *scheme;
	
	pos = strstr(uri, "://");
	
	if (!pos)
	{
		pos = (gchar*)uri;
		scheme = g_strdup("");
	}
	else
	{
		scheme = g_strndup(uri, pos - uri);
		pos += 3;
	}

	if (g_str_has_suffix(uri, "/"))
	{
		filename = g_strconcat(scheme, G_DIR_SEPARATOR_S, pos, "_", NULL);
	}
	else
	{
		gchar *query_pos = strchr(uri, '?');
		if (query_pos)
		{
			gchar *string = g_strndup(pos, query_pos - pos);
			filename = g_strconcat(scheme,
					       G_DIR_SEPARATOR_S,
					       string,
					       "_"G_DIR_SEPARATOR_S,
					       query_pos + 1,
					       NULL);
			g_free(string);
		}
		else
		{
			filename = g_strconcat(scheme,
					       G_DIR_SEPARATOR_S,
					       pos,
					       NULL);
		}
	}

	g_free(scheme);

	return filename;
}


gchar *
create_uri_from_filename (const gchar *filename)
{
	gchar *uri;
	gchar *pos;
	gchar *scheme;
	gchar *path;
	
	pos = strstr(filename, G_DIR_SEPARATOR_S);
	
	if (!pos)
		return g_strdup(filename);

	scheme = g_strndup(filename, pos - filename);

	if (g_str_has_suffix(filename, G_DIR_SEPARATOR_S"_"))
	{
		path = g_strndup(pos + 1, strlen(pos) - 2);
	}
	else
	{
		gchar *query_pos = g_strrstr(filename, "_"G_DIR_SEPARATOR_S);
		if (query_pos)
		{
			gchar *string = g_strndup(pos + 1, query_pos - pos - 1);
			path = g_strconcat(string,
					   "?",
					   query_pos + 2,
					   NULL);
			g_free(string);
		}
		else
		{
			path = g_strdup(pos + 1);
		}
	}

	if (!strcmp(scheme, "file"))
	{
		uri = g_strconcat(scheme, ":///", path, NULL);
	}
	else
	{
		uri = g_strconcat(scheme, "://", path, NULL);
	}

	g_free(scheme);
	g_free(path);

	return uri;
}


gchar *
url_decode(const gchar *src)
{
	GString *dest;
	gint len, i=0;

	if (!src) return NULL;
	len = strlen(src);
	dest = g_string_sized_new(len);

	while (src[i] != '\0' && i < len)
	{
		if (src[i] == '%')
		{
			if (i + 2 <= len &&
			    g_ascii_isxdigit(src[i+1]) && 
			    g_ascii_isxdigit(src[i+2]))
			{				
				g_string_append_c(dest,
						  g_ascii_xdigit_value(src[i+1]) * 16 + 
						  g_ascii_xdigit_value(src[i+2]));
				i+=2;
			}
		}
		else
		{
			g_string_append_c(dest, src[i]);
		}
		i++;
	}

	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}


gchar *
create_profile_key_from_uri (const gchar *string)
{
	gchar *key, *pos;
	gint len, i = 0;
	
	if (!string)
		return NULL;
	pos = strchr(string, '?');
	len = strlen(string);
	if (pos)
		len = pos - string;
	key = g_strndup(string, len);

	while (key[i] != '\0' && i < len)
	{
		if (key[i] == '=')
			key[i] = '_';
		i++;
	}

	return key;
}

/* return hex-encoded UTF-8 data
 * please free returned gchar* if unnecessary
 */
gchar *
url_encode(const gchar* utf8src)
{
	GString *dest;
	const gchar *ch = utf8src;
	unsigned char buf;

	if (!utf8src) return "";

	dest = g_string_sized_new(strlen(utf8src));

	while(*ch != '\0')
	{
		if (((*ch >= 'A') && (*ch <= 'Z')) ||
		    ((*ch >= 'a') && (*ch <= 'z')) ||
		    ((*ch >= '0') && (*ch <= '9')))
		{
			g_string_append_c(dest, *ch);
		}
		else if (*ch == ' ')
		{
			g_string_append_c(dest, '+');

		}
		else
		{
			g_string_append_c(dest, '%');
			buf = (*ch >> 4) & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
			buf = *ch & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
		}
		ch++;
	}
	
	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}


gchar *
complement_scheme (const gchar* url)
{
	gchar *file;

	if (g_file_test(url, G_FILE_TEST_EXISTS))
	{
		if (!g_path_is_absolute(url))
		{
			gchar *current= g_get_current_dir();
			file = g_strdup_printf("file://%s/%s",
					       current,
					       url);
			g_free(current);
		}
		else
		{
			file = g_strdup_printf("file://%s",
					       url);
		}
		return file;
	}
	else
		return g_strdup(url);
}

gchar *
xml_get_content (const gchar *buffer)
{
	gchar *pos1, *pos2, *pos3;
	gchar *name;
	gchar *content = NULL;
	
	pos1 = strchr(buffer, '>');
	pos3 = strchr(buffer, ' ');
	if (pos1)
	{
		gchar *string;
		guint len;
		if (pos3 && pos1 > pos3)
			len = pos3 - buffer - 1;
		else
			len = pos1 - buffer - 1;
		name = g_strndup(buffer + 1, len); 
		string = g_strconcat("</", name, NULL);
		pos2 = strstr(pos1, string);
		if (pos2)
			content = g_strndup(pos1 +1 , pos2 - pos1 - 1);
		g_free(string);
		g_free(name);
	}
	return content;
}

gchar *
xml_get_attr (const gchar *buffer, const gchar *attr_name)
{
	gchar *pos1, *pos2;
	gchar *string;
	gchar *attr = NULL;
	guint len;

	pos1 = strchr(buffer, '>');
	if (!pos1)
		return NULL;

	string = g_strdup_printf("%s=\"", attr_name);
	pos1 = g_strstr_len(buffer, pos1 - buffer, string); 
	if (pos1)
	{
		len = strlen(string);
		pos1 += len;
		pos2 = strchr(pos1, '"'); 
		if (pos2)
			attr = g_strndup(pos1, pos2 - pos1); 
	}
	g_free(string);
	
	return attr;
}

static void
_append_time_stamp(const gchar *path, FILE *time_stamp)
{
	struct stat st;

	if (g_stat(path, &st) == 0)
		g_fprintf(time_stamp, TIME_STAMP_FORMAT, st.st_mtime, path);
}

void
kz_utils_append_time_stamp(const gchar *target_file,
			   const gchar *time_stamp_path)
{
	FILE *fp;

	fp = g_fopen(time_stamp_path, "a");
	if (!fp)
		return;

	_append_time_stamp(target_file, fp);
	fclose(fp);
}

static void
_make_time_stamp (const gchar *dir, FILE *time_stamp)
{
	GDir *gdir;
	const gchar *entry;

	gdir = g_dir_open(dir, 0, NULL);
	if (!gdir) return;

	while ((entry = g_dir_read_name(gdir)))
	{
		gchar *file_name;
		file_name = g_build_filename(dir, entry, NULL);

		if (g_file_test(file_name, G_FILE_TEST_IS_DIR))
			_make_time_stamp(file_name, time_stamp);
		else
			_append_time_stamp(file_name, time_stamp);
		g_free(file_name);
	}
	g_dir_close(gdir);
}

void
kz_utils_make_time_stamp(const gchar *target_dir, const gchar *time_stamp_path)
{
	FILE *fp;

	fp = g_fopen(time_stamp_path, "w");
	if (!fp)
		return;

	_make_time_stamp(target_dir, fp);
	fclose(fp);
}

void
kz_utils_purge_files(const gchar *target_dir, time_t limit_seconds)
{
	GDir *dir;
	const gchar *entry;

	dir = g_dir_open(target_dir, 0, NULL);

	if (!dir) return;

	while ((entry = g_dir_read_name(dir)))
	{
		gchar *file_name;
		file_name = g_build_filename(target_dir, entry, NULL);

		if (g_file_test(file_name, G_FILE_TEST_IS_DIR))
		{
			kz_utils_purge_files(file_name, limit_seconds);
		}
		else
		{
			struct stat st;
			time_t t;
			GTimeVal now;

			g_get_current_time(&now);
			t = (time_t)now.tv_sec;

			if (g_stat(file_name, &st) == 0 &&
			    st.st_mtime < t - limit_seconds)
				g_unlink(file_name);
		}
		g_free(file_name);
	}
	g_dir_close(dir);
}

void
kz_utils_purge_files_by_time_stamp(const gchar *target_dir,
				   const gchar *time_stamp_path,
				   time_t limit_seconds)
{
	gchar buf[BUFFER_SIZE];
	gchar *tmp_file;
	gint fd, target_dir_len;
	FILE *fp;
	GTimeVal now;
	time_t t;
	KzSearch *search;

	fp = g_fopen(time_stamp_path, "r");
	if (!fp)
	{
		g_unlink(time_stamp_path);
		return;
	}


	fd = g_file_open_tmp("kzXXXXXX", &tmp_file, NULL);
	if (fd == -1)
		return;

	search = KZ_GET_SEARCH;
	g_get_current_time(&now);
	t = now.tv_sec;
	target_dir_len = strlen(target_dir);
	while (fgets(buf, sizeof(buf), fp))
	{
		gchar *pos, *time, *path, *path_end, *dir_name;
		gint time_int;
		struct stat st;
		gboolean path_exists;

		pos = strstr(buf, ",");
		time = g_strndup(buf, pos - buf);
		path = g_strdup(pos+1);

		path_end = strstr(path, "\n");
		if (path_end)
			path[path_end - path] = '\0';

		time_int = atoi(time);
		if (t - time_int < limit_seconds)
		{
			write(fd, buf, strlen(buf));
			goto CLEANUP;
		}

		path_exists = g_stat(path, &st) == 0;
		if (!path_exists)
			goto CLEANUP;

		if (t - st.st_mtime < limit_seconds)
		{
			gchar *entry;

			entry = g_strdup_printf(TIME_STAMP_FORMAT,
						st.st_mtime, path);
			write(fd, entry, strlen(entry));
			goto CLEANUP;
		}


		if (search && strlen(path) > target_dir_len)
		{
			gchar *uri;
			gchar *relative_path;

			relative_path = path + target_dir_len;
			uri = create_uri_from_filename(relative_path);
			kz_search_unregister_document(search, uri);
			g_free(uri);
		}
		/* remove file over storage period */
		g_unlink(path);

		dir_name = g_path_get_dirname(path);
		g_rmdir(dir_name); /* ignore error */
		g_free(dir_name);

	  CLEANUP:
		g_free(time);
		g_free(path);
	}
	close(fd);
	fclose(fp);

	g_unlink(time_stamp_path);
	kz_utils_cp(tmp_file, time_stamp_path);
	g_unlink(tmp_file);
}

gchar *
ensure_encode_string(const gchar *text,
		     const gchar *encode,
		     gboolean urlencode)
{
	GError *e = NULL;
	gchar *encode_string = NULL, *url_string = NULL;

	if (!text) 
	{
		encode_string = g_strdup("");
		return encode_string;
	}

	if (encode)
	{
		encode_string = 
			g_convert(text,
				  strlen(text),
				  encode, "UTF-8",
				  NULL, NULL,
				  &e);
	}

	if (e || !encode)
	{
		encode_string =
		    g_strdup(text);
		if (e)
			g_error_free(e);
	}

	if (urlencode)	
	{
		url_string = url_encode(encode_string);
	}
	else
	{
		url_string = g_strdup(encode_string);
	}

	g_free(encode_string);
	
	return url_string;
}

GTime 
thumbnail_get_last_modified (const gchar *uri)
{
	gchar *thumb_filename;
	struct stat st;
	int ret;
	
	thumb_filename = egg_pixbuf_get_thumb_filename(uri,
						       EGG_PIXBUF_THUMB_LARGE);
	
	ret = g_stat(thumb_filename, &st);
	g_free(thumb_filename);

	if (ret == 0)
		return st.st_mtime;
	else
		return 0;
}

GTime 
history_get_last_modified (const gchar *uri)
{
	gchar *history_filename, *filename;
	struct stat st;
	int ret;
	
	filename = create_filename_with_path_from_uri(uri);
	history_filename = g_build_filename(g_get_home_dir(),
					    HISTORY_DIR,
					    filename,
					    NULL);
	
	ret = g_stat(history_filename, &st);
	g_free(history_filename);
	g_free(filename);

	if (ret == 0)
		return st.st_mtime;
	else
		return 0;
}

#ifndef MOZ_NSICANVASRENDERINGCONTEXTINTERNAL_HAVE_GETINPUTSTREAM_
typedef struct _KzPixbufConverter
{
	GdkPixbufLoader *loader;
	gchar *uri;
	gchar *filename;
	time_t mtime;
	EggPixbufThumbSize size;
} KzPixbufConverter;


static gboolean
idle_dispose_converter(gpointer data)
{
	KzPixbufConverter *converter = data;

	g_free(converter->uri);
	g_free(converter->filename);
	g_free(converter);

	return FALSE;
}

static gboolean
cb_io_in(GIOChannel *iochannel, GIOCondition condition,
	 gpointer data)
{
	GIOStatus iostatus;
	gsize bytes_read;
	gchar buffer[BUFFER_SIZE];
	KzPixbufConverter *converter = data;

	if (condition & G_IO_ERR)
	{
		g_idle_add(idle_dispose_converter, converter);
		return FALSE;
	}

	/* Read the data into our buffer */
	iostatus = g_io_channel_read_chars(iochannel, buffer,
					   sizeof(buffer),
					   &bytes_read,
					   NULL);

	if (bytes_read)
	{
		gdk_pixbuf_loader_write(converter->loader,
					(const guchar*)buffer, bytes_read,
					NULL);
	}

	switch (iostatus)
	{
	 case G_IO_STATUS_EOF:
	 {
		GdkPixbuf *pixbuf;
		pixbuf = gdk_pixbuf_loader_get_pixbuf(converter->loader);
		if (pixbuf)
		{
			GdkPixbuf *rotate, *scale;

			/* scale */
			scale = gdk_pixbuf_scale_simple(pixbuf,
						        KZ_THUMB_HEIGHT,
							converter->size,
						        GDK_INTERP_HYPER);

			/* rotate image */
			rotate = gdk_pixbuf_rotate_simple(scale,
        						  GDK_PIXBUF_ROTATE_CLOCKWISE);
			g_object_unref(scale);
			/* save image */
			if (rotate)
			{
				egg_pixbuf_add_thumbnail_data(rotate,
							      converter->uri,
							      converter->mtime,
							      converter->size);
				egg_pixbuf_save_thumbnail(rotate, NULL, NULL);
				g_object_unref(rotate);
			}
		}
		gdk_pixbuf_loader_close(converter->loader, NULL);
		g_unlink(converter->filename);
		g_idle_add(idle_dispose_converter, converter);

		return FALSE;
  	 }
	 case G_IO_STATUS_NORMAL:
		return TRUE;
	 default:
		g_idle_add(idle_dispose_converter, converter);
		return FALSE;
	}
}


void
kz_utils_save_ps_as_thumbnail (const gchar *ps_file, const gchar *uri,
			       time_t mtime, EggPixbufThumbSize size)
{
	const gchar *gs =
		"gs -q -dNOPAUSE -dBATCH -r24 -sDEVICE=ppmraw -sOutputFile=- %s";
	gchar *command;
	gint argc, out;
	gchar **argv = NULL;
	GSpawnFlags flags;
	GPid pid;
	GIOChannel *io;
	GdkPixbufLoader *loader;
	KzPixbufConverter *converter;

	command = g_strdup_printf(gs, ps_file);

	g_shell_parse_argv(command,
			   &argc,
			   &argv,
			   NULL);

        /* Check whether ghostscript is installed or not */
	if (g_find_program_in_path(argv[0]) == NULL)
	{
		g_warning("Can't find %s in your PATH.\n"
			  "Currently kazehakase using gecko engine needs ghostscript to create thumbnails.\n", argv[0]);
		return;
	}

	flags = G_SPAWN_SEARCH_PATH;
	g_spawn_async_with_pipes(NULL,
		      		 argv,
				 NULL,
		   		 flags,
		      		 NULL,
		      		 NULL,
		      		 &pid,
				 NULL,
				 &out,
				 NULL,
		      		 NULL);
	g_strfreev(argv);
	g_free(command);

	io = g_io_channel_unix_new(out);
	g_io_channel_set_encoding(io, NULL, NULL);

	loader = gdk_pixbuf_loader_new_with_type("pnm", NULL);
	converter = g_new0(KzPixbufConverter, 1);
	converter->loader = loader;
	converter->uri = g_strdup(uri);
	converter->filename = g_strdup(ps_file);
	converter->mtime = mtime;
	converter->size = size;

	g_io_add_watch(io,
		       G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP,
		       cb_io_in, converter);
}
#endif

gboolean
str_isdigit (const gchar *str)
{
	int i;
	size_t len;
	gboolean is_digit = TRUE;

	if (!str) return FALSE;

	len = strlen(str);
	for (i = 0; i < len; i++)
	{
		if (!isdigit(str[i]))
		{
			is_digit = FALSE;
			break;
		}
	}
	return is_digit;
}

/* the following tow functions are picked from utils/gul-general.c in galeon-1.3.18 */
static void
find_file_recursive (const gchar *path,
		     const gchar *fname,
		     GSList **l,
		     gint depth,
		     gint maxdepth)
{
	GDir *d = g_dir_open (path, 0, NULL);
	const gchar *f;
	if (d)
	{
		while ((f = g_dir_read_name (d)))
		{
			gchar *new_path = g_build_filename (path, f, NULL);
			if (depth < maxdepth)
			{
				find_file_recursive (new_path, fname, l,
							 depth + 1, maxdepth);
			}
			if (!strcmp (f, fname))
			{
				*l = g_slist_prepend (*l, new_path);
			}
			else
			{
				g_free (new_path);
			}
		}
		g_dir_close (d);
	}
}

/*
 * Looks for a file in a directory, recursively.
 */
GSList *
find_file (const gchar *path, 
	   const gchar *fname,
	   gint maxdepth)
{
	GSList *ret = NULL;
	find_file_recursive (path, fname, &ret, 0, maxdepth);
	return ret;
}

void
kz_str_replace_char (gchar *str, gchar replace_char, gchar new_char)
{
	gint i = 0;

	while (str[i] != '\0')
	{
		if (str[i] == replace_char)
			str[i] = new_char;
		i++;
	}
}

gchar *
kz_uri_encode_last_component (const gchar *uri)
{
	gchar *last_component_encoded_uri, *last_sep_pos;

	last_sep_pos = g_strrstr(uri, "/");
	if (last_sep_pos)
	{
		gchar *pre_component, *last_component;

		pre_component = g_strndup(uri, last_sep_pos - uri + 1);
		last_component = url_encode(last_sep_pos + 1);
		last_component_encoded_uri =
			g_strconcat(pre_component, last_component, NULL);
		g_free(pre_component);
	}
	else
	{
		last_component_encoded_uri = g_strdup(uri);
	}

	return last_component_encoded_uri;
}

#ifdef G_OS_WIN32
static gchar *win32_base_path = NULL;

const gchar *
kz_win32_base_path (void)
{
	if (win32_base_path)
		return win32_base_path;

	win32_base_path = g_win32_get_package_installation_directory(PACKAGE,
								     NULL);
	return win32_base_path;
}
#endif
