unit untHttp;

interface

uses
  Classes, SysUtils,
  IdThread, IdHttp, IdComponent, gzip,
  untStreamTool, SyncObjs;

type

  THttpErrpr = (heNoError, heBrokenGzip);

  TAsyncHttp = class
  private
    FURL           : string;
    FIdHttp        : TIdHttp;
    FWriteEvent    : TMemoryStreamEx;
    FBufferReader  : TStreamReader;
    FReadPosition  : Integer;
    FReceivedLines : TStringList;
    FAddHeaders    : string;
    FResponseCode  : Integer;
    FBuffer        : TMemoryStream;
    FGzipStream    : TGzipDecompressStream;
    procedure HttpReceived(const Buff; Count : int64);
    procedure HttpStatus(ASender: TObject; const AStatus: TIdStatus; const AStatusText: string);
    function  GetReplayCode : Integer;
  public
    LastModified: string;
    UseGzip: Boolean;
    ErrorCode: THttpErrpr;
    OnStatus: TIdStatusEvent;
    OnComplete: TNotifyEvent;
    OnReceived: TNotifyEvent;
    ContentLength: Integer;
    UserAgent: string;
    ProxyHost: string;
    ProxyPort: Integer;
    StartRange: Integer;
    property ReceivedLines: TStringList read FReceivedLines;
    property ResponseCode: Integer read GetReplayCode;

    procedure AddHeader(HeaderName, Value : string);
    procedure Get(URL : string);
  end;

implementation

{ TAsyncHttp }

procedure TAsyncHttp.AddHeader(HeaderName, Value: string);
begin
  FAddHeaders := FAddHeaders + HeaderName + ': ' + Value + #13#10;
end;

procedure TAsyncHttp.Get(URL: string);
var
  headers : TStringList;
  I       : Integer;

begin
  FURL := URL;

  if StartRange = 0 then
  begin
    UseGzip := true;
  end;

  ErrorCode := heNoError;

  FReceivedLines := TStringList.Create;
  FWriteEvent := TMemoryStreamEx.Create;
  FBuffer := TMemoryStream.Create;
  FIdHttp := TIdHttp.Create(nil);
  headers := TStringList.Create;
  FBufferReader := TStreamReader.Create(FBuffer);
  FGzipStream := TGzipDecompressStream.Create(FBuffer);

  try
    FWriteEvent.OnWrite := HttpReceived;
    FReadPosition := 0;

    {*debug Config.SetReadProxy(FHttp)
    FHttp.Request.UserAgent   := gUserAgent;
    FHttp.Request.RawHeaders.Add('X-2ch-UA: ' + APP_2chUA);
    }
    with FIdHttp do
    begin
      Request.ContentRangeStart := StartRange;
      Request.Connection        := 'close';
      Request.UserAgent         := UserAgent;
      OnStatus                  := HttpStatus;
    end;

    if UseGzip then
    begin
      FIdHttp.Request.AcceptEncoding := 'gzip';
    end;

    if LastModified <> '' then
    begin
      AddHeader('If-Modified-Since', LastModified);
    end;

    headers.Text := FAddHeaders;
    for I := 0 to headers.Count - 1 do
    begin
      FIdHttp.Request.CustomHeaders.Add(headers[I]);
    end;

    if ProxyHost <> '' then
    begin
      with FIdHttp do
      begin
        ProxyParams.ProxyServer := ProxyHost;
        ProxyParams.ProxyPort   := ProxyPort;
        Request.Pragma          := 'no-cache';
      end;
    end;

    try
      FIdHttp.Get(FURL, FWriteEvent);
    except
      on EIdHTTPProtocolException do;
    end;

    //FIdHttp.Disconnect;

    with FIdHttp do
    begin
      FResponseCode := ResponseCode;
      LastModified  := Response.RawHeaders.Values['Last-Modified'];
    end;

    if FIdHttp.Response.ContentEncoding = 'gzip' then
    begin
      ContentLength := FBuffer.Size
    end else
    begin
      ContentLength := FIdHttp.Response.ContentLength;
    end;

    if Assigned(OnComplete) then
    begin
      OnComplete(self);
    end;
  finally
    FBufferReader.Free;
    FWriteEvent.Free;
    FBuffer.Clear;
    FBuffer.Free;
    FReceivedLines.Clear;
    FReceivedLines.Free;
    headers.Free;
    FIdHttp.Free;
    FGzipStream.Free;
  end;
  ErrorCode := heNoError;
end;

procedure TAsyncHttp.HttpReceived(const Buff; Count: int64);
var
  line : string;

begin
  FBuffer.Seek(0, soFromEnd);

  if FIdHttp.Response.ContentEncoding = 'gzip' then
  begin
    if ErrorCode = heNoError then
    begin
      try
        FGzipStream.Write(Buff, Count);
      except on Exception do
        begin
          ErrorCode := heBrokenGzip;
          exit;
        end;
      end;
    end;
  end else
  begin
    FBuffer.Write(Buff, Count);
  end;

  line := '';

  FBuffer.Seek(FReadPosition, soFromBeginning);
  while FBufferReader.ReadLine(line) do
  begin
    FReadPosition := FBuffer.Position;
    FReceivedLines.Add(line);
  end;
  if Assigned(OnReceived) and (FReceivedLines.Count > 0) then
  begin
    OnReceived(self);
  end;
end;

procedure TAsyncHttp.HttpStatus(ASender: TObject;
                                const AStatus: TIdStatus;
                                const AStatusText: string);
begin
  if Assigned(OnStatus) then
    OnStatus(self, AStatus, AStatusText)
end;

function TAsyncHttp.GetReplayCode: Integer;
begin
  result := FResponseCode;
end;

end.
