unit untBBSFramework;

interface

uses
  Classes,
  IdComponent, IdTCPConnection, IdTCPClient, IdHTTP,
  IdCookieManager,
  RegExpr,
  untTool, untHttp,  untBoard, untTopic;

type
  TBBSType = (btAuto, bt2ch, btDolib, btKako, btJBBS, btShitaraba,
              btMitinoku, btYahoo, btWinny, btNone);

  TFrameworkErrorType = (etNoError, etAbone, etParse, etPostArticle,
                         etDatFreezed, etStartRange, etBrokenGZip);

  TStatusTextChangeEvent = procedure(Sender: TObject;
                                     StatusText: string) of object;
  TFrameworkError        = procedure(Sender: TObject;
                                     ErrorCode: TFrameworkErrorType;
                                     ErrorString: string) of object;

  TBBSGetTopic = class
  private
    FReceivedIndex: integer;
    FNoBrowser: boolean;
    FError: boolean;
    FRetryed: boolean;
    FErrorString: string;
    procedure Download; overload;
  protected
    FTopic: TTopic;
    FGotMessageCount: integer;
    FLastModified: string;
    FArticleList: TStringList;
    FErrorCode: TFrameworkErrorType;
    procedure ChangeStatusText(StatusText: string);
    procedure SetGotMessageCount(const Value: integer);
    procedure RaiseError(ErrorCode : TFrameworkErrorType; ErrorString : string);
    procedure HttpReceived(Sender: TObject); virtual;
    procedure HttpComplete(Sender: TObject); virtual;
  public
    OnComplete: TNotifyEvent;
    OnMessageReceived: TNotifyEvent;
    OnChangeDownloadState: TNotifyEvent;
    GZiped: Boolean;
    LastModified: string;
    function Retry(tp: TTopic): Boolean;
    procedure Download(tp: TTopic); overload;
    property    ErrorCode: TFrameworkErrorType read FErrorCode;
    property    GotMessageCount: integer read FGotMessageCount write SetGotMessageCount;
    property    ArticleList: TStringList read FArticleList;
    procedure   Get(); virtual; abstract;
    constructor Create;
    destructor  Destroy; override;
  end;

  TBBSPostArticle = class
  protected
    FOnComplete         : TNotifyEvent;
    FOnStatusTextChange : TStatusTextChangeEvent;
    FOnError            : TFrameworkError;
    FHttp               : TIdHttp;
    procedure SetOnComplete(const Value: TNotifyEvent);
    procedure SetOnStatusTextChange(const Value: TStatusTextChangeEvent);
    procedure SetOnError(const Value: TFrameworkError);
    procedure HttpStatusChange(axSender: TObject;
      const axStatus: TIdStatus; const asStatusText: string);
    procedure ChangeStatusText(NewText : string);
    procedure RaiseError(ErrorCode : TFrameworkErrorType; ErrorString : string);
  public
    property    OnComplete         : TNotifyEvent read FOnComplete write SetOnComplete;
    property    OnError            : TFrameworkError read FOnError write SetOnError;
    property    OnStatusTextChange : TStatusTextChangeEvent read FOnStatusTextChange write SetOnStatusTextChange;
    procedure   Post(PostName, PostEmail, Body : string); virtual; abstract;
    constructor Create();
    destructor  Destroy; override;
  end;

  TBBSGetTopicList = class
  protected
    FOnError            : TFrameworkError;
    FOnComplete         : TNotifyEvent;
    FOnReceived         : TNotifyEvent;
    FOnStatusTextChange : TStatusTextChangeEvent;
    procedure ChangeStatusText(NewText : string);
    procedure SetOnComplete(const Value: TNotifyEvent);
    procedure SetOnError(const Value: TFrameworkError);
    procedure SetOnReceived(const Value: TNotifyEvent);
    procedure SetOnStatusTextChange(const Value: TStatusTextChangeEvent);
    procedure HttpStatusChange(axSender: TObject;
                               const axStatus: TIdStatus;
                               const asStatusText: string);
    procedure RaiseError(ErrorCode: TFrameworkErrorType;
                         ErrorString: string);
    procedure AddNewLine(line : string);
  public
    Board: TOnlineBoard;
    TopicList: TStringList;
    property    OnReceived         : TNotifyEvent read FOnReceived write SetOnReceived;
    property    OnComplete         : TNotifyEvent read FOnComplete write SetOnComplete;
    property    OnStatusTextChange : TStatusTextChangeEvent read FOnStatusTextChange write SetOnStatusTextChange;
    property    OnError : TFrameworkError read FOnError write SetOnError;
    constructor Create;
    destructor  Destroy; override;
    procedure Get(); virtual; abstract;
  end;

  function CreateBBSGetTopic    (Server, BoardId, TopicId : string; BBSType : TBBSType = btAuto) : TBBSGetTopic;
  function CreateBBSPostArticle (Server, BoardId, TopicId : string) : TBBSPostArticle;
  function CreateBBSGetTopicList(Server, BoardId : string) : TBBSGetTopicList;

implementation

uses
  SysUtils, untBBS2ch, untBBSJBBS, untBBS2chDolib,
  untBBS2chKako, untGlobal, untConfig;

function GetBBSType(Server : string) : TBBSType;
begin
  // BBSType𔻒f
  if Pos('jbbs', Server) > 0 then
    result := btJBBS
  else if Pos('machi.to', Server) > 0 then
    result := btJBBS
  else if Pos('shitaraba.com', Server) > 0 then
    result := btShitaraba
  else if Pos('www.mitinoku.jp', Server) > 0 then
    result := btMitinoku
  else
    result := bt2ch;
end;

procedure ParseUrl(Url : string; var BBSType : TBBSType; var DatUrl : string; var SubjectUrl : string);
begin
  (* sample format *)
  (* http://server.2ch.net/bbs/ *)
  (* http://server.2ch.net/test/read.cgi/bbs/?opt  *)
  (* http://server.2ch.net/test/read.cgi/bbs/dat/opt  *)
  (* http://server.2ch.net/bbs/dat/nnn.dat  *)
  (* http://server.2ch.net/bbs/kako/nnn/dat.html  *)
  (* http://server.2ch.net/bbs/kako/nnn/nnnn/dat.html  *)
  (* http://server.2ch.net/test/read.cgi?bbs=bbs&key=dat  *)
  (* http://server.2ch.net/sub/bbs/kako/nnn/dat.html  *)
  (* http://server.2ch.net/sub/test/read.cgi?bbs=bbs&key=dat  *)
  (* http://server.yahoo.co.jp/bbs?action=topics&board=bbs&sid=dat *)
end;

function CreateBBSGetTopic(Server, BoardId, TopicId: string; BBSType: TBBSType = btAuto): TBBSGetTopic;
begin
  if BBSType = btAuto then
    BBSType := GetBBSType(Server);

  case BBSType of
    bt2ch:
      result := TBBS2chGetTopic.Create     (Server, BoardId, TopicId);
    btDolib:
      result := TBBS2chDolibTopic.Create(Server, BoardId, TopicId);
    btKako:
      result := TBBS2chGetKakoTopic.Create (Server, BoardId, TopicId);
    btJBBS:
      result := TBBSJBBSGetTopic.Create    (Server, BoardId, TopicId);
    else
      result := nil;
  end;
end;

function CreateBBSPostArticle(Server, BoardId, TopicId : string) : TBBSPostArticle;
begin

  case GetBBSType(Server) of
    bt2ch:
      result := TBBS2chPostArticle.Create (Server, BoardId, TopicId);
    btJBBS:
      result := TBBSJBBSPostArticle.Create(Server, BoardId, TopicId)
    else
      result := nil;
  end;

end;

function CreateBBSGetTopicList(Server, BoardId : string) : TBBSGetTopicList;
begin

  case GetBBSType(Server) of
    bt2ch:
      result := TBBS2chGetTopicList.Create(Server, BoardId);
    btJBBS, btMitinoku:
      result := TBBSJBBSGetTopicList.Create(Server, BoardId);
    else
      result := nil;
  end;
end;


{ TBBSGetTopic }
procedure TBBSGetTopic.ChangeStatusText(StatusText: string);
begin
  FTopic.StatusText := StatusText;
  OnChangeDownloadState(self);
end;

function TBBSGetTopic.Retry(tp: TTopic): Boolean;
begin
  if FRetryed then
  begin
    result := false;
    exit;
  end;

  tp.StatusText := 'ēǂݍݒ';
  tp.EraseMessageList;
  tp.DatSize := 0;
  tp.LastModified := '';
  Download;

  FRetryed := true;

  result := true;
end;

{ --------------------------------------------------------
  pr  : gsbÑ_E[h
    : Topic - _E[hΏۂ̃gsbN
  l  : Ȃ
  ------------------------------------------------------ }
procedure TBBSGetTopic.Download(tp: TTopic);
begin
  FTopic := tp;
  LastModified    := tp.LastModified;
  GotMessageCount := tp.GotMessageCount;
  Download;
end;

procedure TBBSGetTopic.Download;
begin
  FReceivedIndex := -1;

  if FTopic.IsDownloadingTopic = false then
  begin
    FTopic.IsDownloadingTopic := true;

    FTopic.DownloadState := dsStarting;
    FTopic.CheckWriteFolder;
    FTopic.NewMessageCount := 0;
    FNoBrowser   := not Assigned(OnMessageReceived);
    FRetryed := false;
    FTopic.LoadLog;
    Get;
  end;
end;

procedure TBBSGetTopic.HttpReceived(Sender: TObject);
var
  title: string;
  i: integer;
  msg: TTopicMessage;
  newDat: string;

begin
  for i := FReceivedIndex + 1 to ArticleList.Count - 1 do
  begin
    msg := FTopic.ParseDat(ArticleList[i], Title);
    msg.IsNewMessage := true;

    FTopic.AddNewMessage(msg);
    if (msg.Index = 1) and (Title <> '') then
    begin
      FTopic.Title := Title;
    end;
    Inc(FTopic.NewMessageCount);
    FReceivedIndex := i;
    newDat := newDat + ArticleList[i] + #13#10;    
  end;
  FTopic.SaveToLog(newDat);
  if Assigned(OnMessageReceived) then
  begin
    OnMessageReceived(self);
  end;
end;

procedure TBBSGetTopic.HttpComplete(Sender: TObject);
begin
  // I
  // L^
  FTopic.LastModified    := LastModified;
  FTopic.GotMessageCount := GotMessageCount + FTopic.NewMessageCount;
  FTopic.MessageCount    := FTopic.GotMessageCount;
  FTopic.GZip            := GZiped;
  FTopic.Loaded          := true;

  if FTopic.NewMessageCount > 0 then
  begin
    FTopic.LastReadDate := DateTimeToStr(Now());
  end;

  if not FError then
  begin
    FTopic.DownloadState := dsNone;

    if FTopic.NewMessageCount = 0 then
    begin
      FTopic.StatusText := 'VȂ'
    end else
    begin
      FTopic.StatusText := IntToStr(FTopic.NewMessageCount) + '̃XM';
    end;
  end else
  begin
    if FRetryed then
    begin
      FTopic.StatusText := 'ēǂݍ݂܂'
    end else
    begin
      FTopic.StatusText := FErrorString;
    end;
  end;

  FTopic.SaveIdx();

  FTopic.DownloadState := dsNone;
  FTopic.IsDownloadingTopic := false;

  if Assigned(OnComplete) then
  begin
    OnComplete(self);
  end;
end;

constructor TBBSGetTopic.Create;
begin
  FArticleList := TStringList.Create;
end;

destructor TBBSGetTopic.Destroy;
begin

  FArticleList.Free;

  inherited;
end;

procedure TBBSGetTopic.RaiseError(ErrorCode: TFrameworkErrorType; ErrorString: string);
begin
  FError := true;
  FErrorString := ErrorString;
end;

procedure TBBSGetTopic.SetGotMessageCount(const Value: integer);
begin
  FGotMessageCount := Value;
  if FGotMessageCount < 0 then
  begin
    FGotMessageCount := 0;
  end;
end;


{ TBBSPostArticle }

constructor TBBSPostArticle.Create;
var
  proxyhost : string;
  proxyport : integer;
begin

  FHttp := nil;

  FHttp := TIdHttp.Create(nil);
  FHttp.OnStatus            := HttpStatusChange;
  FHttp.Request.UserAgent   := gConfig.UserAgent;
  FHttp.Request.RawHeaders.Add('X-2ch-UA: ' + APP_2chUA);

  // proxy
  if gConfig.WriteProxyUse then
  begin
    gConfig.ParseProxy(gConfig.WriteProxy, proxyhost, proxyport);
    FHttp.ProxyParams.ProxyServer := proxyhost;
    FHttp.ProxyParams.ProxyPort   := proxyport;
  end;

end;

destructor TBBSPostArticle.Destroy;
begin
  FHttp.Free;
  inherited;
end;

procedure TBBSPostArticle.SetOnComplete(const Value: TNotifyEvent);
begin
  FOnComplete := Value;
end;

procedure TBBSPostArticle.SetOnError(const Value: TFrameworkError);
begin
  FOnError := Value;
end;

procedure TBBSPostArticle.SetOnStatusTextChange(
  const Value: TStatusTextChangeEvent);
begin
  FOnStatusTextChange := Value;
end;

procedure TBBSPostArticle.ChangeStatusText(NewText: string);
begin
  if Assigned(FOnStatusTextChange) then
    FOnStatusTextChange(self, NewText);
end;

procedure TBBSPostArticle.RaiseError(ErrorCode: TFrameworkErrorType;
  ErrorString: string);
begin
  if Assigned(FOnError) then
    FOnError(self, ErrorCode, ErrorString);
end;

procedure TBBSPostArticle.HttpStatusChange(axSender: TObject;
  const axStatus: TIdStatus; const asStatusText: string);
begin

  case axStatus of

    hsConnecting :
    begin
      ChangeStatusText(FHttp.Host + 'ɐڑ');
    end;

    hsConnected :
    begin
      //FParent.DownloadState := dsProcessing;
      ChangeStatusText(FHttp.Host + 'ɐڑ܂');
    end;

  end;

end;

{ TBBSGetTopicList }

constructor TBBSGetTopicList.Create;
begin

  TopicList := TStringList.Create;

end;

destructor TBBSGetTopicList.Destroy;
begin

  TopicList.Free;
  inherited;
end;

procedure TBBSGetTopicList.SetOnComplete(const Value: TNotifyEvent);
begin
  FOnComplete := Value;
end;

procedure TBBSGetTopicList.SetOnError(const Value: TFrameworkError);
begin
  FOnError := Value;
end;

procedure TBBSGetTopicList.SetOnReceived(const Value: TNotifyEvent);
begin
  FOnReceived := Value;
end;

procedure TBBSGetTopicList.SetOnStatusTextChange(
  const Value: TStatusTextChangeEvent);
begin
  FOnStatusTextChange := Value;
end;

procedure TBBSGetTopicList.HttpStatusChange(axSender: TObject;
  const axStatus: TIdStatus; const asStatusText: string);
begin

  case axStatus of
    hsConnecting :
    begin
      ChangeStatusText('test server' + 'ɐڑ');
    end;

    hsConnected :
    begin
      //FParent.DownloadState := dsProcessing;
      ChangeStatusText('test server' + 'ɐڑ܂');
    end;
  end;

end;

procedure TBBSGetTopicList.ChangeStatusText(NewText: string);
begin
  if Assigned(FOnStatusTextChange) then
    FOnStatusTextChange(self, NewText);
end;

procedure TBBSGetTopicList.RaiseError(ErrorCode: TFrameworkErrorType;
                                   ErrorString: string);
begin
  if Assigned(FOnError) then
    FOnError(self, ErrorCode, ErrorString);
end;

procedure TBBSGetTopicList.AddNewLine(line: string);
begin
  TopicList.Add(line);
end;

end.
