unit untTopicBrowser;

interface

uses
  Classes, ExtCtrls, SyncObjs,
  untTopic;

type
  TNavigateUrlEvent = procedure(sender: TObject; URL: string) of object;

  TTopicBrowser = class(TPanel)
  protected
    FLoading:     boolean;
    FAutoReload:  boolean;
    FDownloading: boolean;
    FViewLimit: integer;
    FReloadTimer: TTimer;
    function  ChangePopup(const statustext: string): boolean;
    procedure SetAutoReload(const Value: Boolean);
    procedure RaiseNavigateUrlEvent(Url: string);
    procedure ChangeImageIndex;
    procedure AutoReloadTimer(Sender: TObject);
    procedure SyncInit;
    procedure SyncSetTitle;
    procedure Topic_MessageReceived(Sender: TObject);  virtual; abstract;
    procedure Topic_ChangeDownloadState(Sender: TObject); virtual;
    procedure Topic_Complete(Sender: TObject); virtual;
  public
    RangeNoList: TStringList;
    Topic: TTopic;
    Caption: string;
    StartNo:    integer;
    ImageIndex: integer;
    OnNavigateUrl: TNavigateUrlEvent;
    OnMessageReceived:     TNotifyEvent;
    OnComplete:            TNotifyEvent;
    OnChangeDownloadState: TNotifyEvent;
    CompleteEvent: TSimpleEvent;
    property AutoReload: Boolean read FAutoReload write SetAutoReload;
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure JumpMessage(const msgNo: integer); virtual; abstract;
    procedure SaveScrollPosition; virtual; abstract;
    procedure OpenTopic(tp: TTopic); virtual;
    procedure CloseTopic; virtual;
    procedure Reload; virtual;
    procedure SearchText(const str: string); virtual; abstract;
    procedure ChangeViewLimit(NewLimit: integer); virtual; abstract;
    procedure TrashTopic;
  end;

implementation

uses
  ComCtrls, Controls, windows, RegExpr, StrUtils, SysUtils, Forms, Math,
  untOnlineBoard, untGlobal, untTool, untHintWindow, untTaskManager,
  untMainForm, unt2chURLParser;

type
  TPopupManager = class(TObject)
  private
    class function GetMessages(Msgs: TList;
                               const First,
                                     Last: Integer): string;
    class function ParseMailto(const URI: string): string;
  public
    class procedure ParseJump(const URI: string;
                                var First,
                                    Last: Integer);
    class function GetPopupMessages(Topic: TTopic; const URI: string): string;
  end;

procedure TTopicBrowser.TrashTopic;
begin
  if Assigned(Topic) then
  begin
    Topic.OnComplete := MainForm.TopicBrowser_TrashComplete;
    Topic.EraseLog;
  end;
end;

procedure TTopicBrowser.RaiseNavigateUrlEvent(Url: string);
var
  p: Integer;
  urlStr: string;
  msgNo: integer;

begin
  if AnsiStartsText('Jump://goto/', Url) then
  begin
    p := Pos('-', Url);
    if p <> 0 then
    begin
      urlStr := Copy(Url, 0, p - 1)
    end else
    begin
      UrlStr := Url;
    end;
    msgNo := StrToIntNeo(CopyAfter(UrlStr, 13));
    JumpMessage(msgNo);
  end else
  begin
    if Assigned(OnNavigateUrl) then
    begin
      OnNavigateUrl(self, Url);
    end;
  end;
end;

constructor TTopicBrowser.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  self.BevelInner := bvNone;
  self.BevelOuter := bvNone;

  FLoading := false;
  ImageIndex := -1;
  FReloadTimer := TTimer.Create(self);
  FReloadTimer.Enabled  := false;
  FReloadTimer.Interval := 60000;
  FReloadTimer.OnTimer  := AutoReloadTimer;

  RangeNoList := TStringList.Create;
  CompleteEvent := TSimpleEvent.Create;
end;

procedure TTopicBrowser.AutoReloadTimer(Sender: TObject);
begin
  // I[g[h
  Reload;
end;

destructor TTopicBrowser.Destroy;
begin
  FReloadTimer.Free;
  CompleteEvent.Free;
  RangeNoList.Free;

  inherited;
end;

procedure TTopicBrowser.Reload;
begin
  if Assigned(Topic) then
  begin
    //Topic.CancelDownload;
    Topic.OnStateChange := Topic_ChangeDownloadState;
    Topic.OnReceived    := Topic_MessageReceived;
    Topic.OnComplete    := Topic_Complete;

    gTaskManager.Request(Topic, self, reqtDownload);
  end;
end;

procedure TTopicBrowser.OpenTopic(tp: TTopic);
begin
  CompleteEvent.ResetEvent;

  if Assigned(Topic) then
  begin
    CloseTopic;
  end;

  if Assigned(tp) then
  begin
    FViewLimit := gConfig.ViewLimit;

    Topic := tp;
    gSynchronizer.DoSynchronize(SyncSetTitle);

    Topic.DoNotify := true;
    Topic.OnStateChange := Topic_ChangeDownloadState;
    Topic.OnReceived    := Topic_MessageReceived;
    Topic.OnComplete    := Topic_Complete;
    Include(Topic.Status, tsOpened);
    Topic.Download;
  end else
  begin
    gSynchronizer.DoSynchronize(SyncInit);
    CompleteEvent.SetEvent;
  end;
end;

procedure TTopicBrowser.CloseTopic;
begin
  if Assigned(Topic) then
  begin
    //Topic.CancelDownload;
    SaveScrollPosition;
    if gConfig.DoLogSave or (tsInBox in Topic.Status) or (tsGotLog in Topic.Status) then
    begin
      // Oۑ[h
      Topic.SaveIdx;
      if not (tsGotLog in Topic.Status) then
      begin
        // [hؑփ{^ꂽƂ
        // O܂ۑĂȂ̂łŕۑ
        Topic.SaveDat;
      end;
    end else
    begin
      // OۑȂ[h    
      Topic.GotMessageCount := 0;
      Topic.NewMessageCount := -1;
      Topic.Datsize         := 0;
      Topic.LastModified    := '';
    end;
    Exclude(Topic.Status, tsOpened);
    Topic.FreeMessage;
    SetAutoReload(false);
  end;
  RangeNoList.Clear;
  StartNo := 0;

  Topic := nil;
  gSynchronizer.DoSynchronize(SyncInit);
end;

procedure TTopicBrowser.SyncInit;
begin
  Caption  := '';
  ImageIndex := -1;
end;

procedure TTopicBrowser.SyncSetTitle;
begin
  Caption  := Topic.Title;
end;

procedure TTopicBrowser.Topic_Complete(Sender: TObject);
begin
  ChangeImageIndex;
  if Topic.NewMessageCount > 0 then
  begin
    SetAutoReload(false);
  end;

  if Assigned(OnComplete) then
  begin
    OnComplete(self);
  end;
  CompleteEvent.SetEvent;
end;

// ԕω
procedure TTopicBrowser.ChangeImageIndex;
begin
  case Topic.State of
    dsNone:
    begin
      if FAutoReload then
      begin
        ImageIndex := 1;
      end else
      begin
        ImageIndex := -1;
      end;
    end;
    dsComplete:
    begin
      if Topic.NewMessageCount > 0 then
      begin
        ImageIndex := 0
      end else
      begin
        if FAutoReload then
        begin
          ImageIndex := 1
        end else
        begin
          ImageIndex := -1;
          if tsError in Topic.Status then
          begin
            ImageIndex := 2;
          end;
          if tsFreezed in Topic.Status then
          begin
            ImageIndex := 3;
          end;
        end;
      end;

      if FDownloading = true then
      begin
        FDownloading := false;

        // vOANeBułȂ
        // ^Cgo[_ł
        if (Topic.NewMessageCount > 0) and
           not Application.Active then
        begin
          FlashWindow(Application.Handle, True);
        end;
      end;
    end;
    dsStarting:
      ImageIndex := 4;
    {dsProcessing:
      ImageIndex := 5;}
  end;
  Application.ProcessMessages;
end;

procedure TTopicBrowser.Topic_ChangeDownloadState(Sender: TObject);
begin
  ChangeImageIndex;
  if Assigned(OnChangeDownloadState) then
  begin
    OnChangeDownloadState(self);
  end;
end;

// I[g[hݒ
procedure TTopicBrowser.SetAutoReload(const Value: Boolean);
begin
  FAutoReload := Value;
  FReloadTimer.Enabled := Value;
  Topic_ChangeDownloadState(Topic);
end;

function TTopicBrowser.ChangePopup(const statustext: string): boolean;
var
  msgText: string;

begin
  msgText := TPopupManager.GetPopupMessages(Topic, statustext);

  if msgText <> '' then
  begin
    ToolTip.SetHint(msgText);
    Result := True;
  end else
  begin
    ToolTip.UnVisible;
    Result := False;
  end;
end;

{ TPopupManager }

class function TPopupManager.GetMessages(Msgs: TList; const First, Last: Integer): string;
var
  i: Integer;
  firstPos, lastPos: Integer;
  msg: string;

begin
  // eg. >>0
  if (First < 1) and (Last < 0) then
  begin
    Result := '';
    Exit;
  end;

  firstPos := Max(First - 1, 0);

  lastPos := Min(Last, Msgs.Count);
  Dec(LastPos);

  // eg. >>100000 or >>10-9
  if (firstPos >= msgs.Count) or ((lastPos >= 0) and (firstPos > lastPos)) then
  begin
    Result := '';
    Exit;
  end;

  if lastPos > 0 then
  begin
    for i := firstPos to lastPos do
    begin
      if I > firstPos + 20 then
      begin
        msg := msg + '(ȗ܂)';
        break;
      end else
      begin
        msg := msg + TTopicMessage(Msgs[i]).ForPopup + #10#10
      end;
    end
  end else
  begin
    msg := TTopicMessage(Msgs[FirstPos]).ForPopup;
  end;

  Result := msg;
end;

class function  TPopupManager.ParseMailto(const URI : string): string;
begin
  Result := CopyAfter(URI, 8);
end;

// \n+(-\n+)? only
class procedure  TPopupManager.ParseJump(const URI : string; var First, Last: Integer);
var
  Text : string;
  p : Integer;

begin
  Text := CopyAfter(URI, 13);

  p := Pos('-', Text);
  if p <> 0 then
  begin
    First := StrToInt(Copy(Text, 0, p - 1));
    Last  := StrToInt(Copy(Text, p + 1, Length(Text)));
  end else
  begin
    First := StrToInt(Text);
    Last  := -1;
  end;
end;

class function TPopupManager.GetPopupMessages(Topic: TTopic; const URI: string): string;
var
  first, last: integer;
  parser: T2chURLParser;

begin
  parser := T2chURLParser.Create;
  if Copy(URI, 1, 7) = 'mailto:' then
  begin
    Result := ParseMailto(URI);
  end else if Copy(URI, 1, 12) = 'jump://goto/' then
  begin
    ParseJump(URI, first, last);
    Result := GetMessages(Topic.MessageList, first, last);
  end else
  begin
    parser.Parse(URI);
    if parser.Is2chUrl then
    begin
      
    end else
    begin
    end;
    Result := '';
  end;
  parser.Free;
end;

end.
